/*
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: mdregmgr
 * File: md_info.c
 *
 * Description: This file contains all functions related to the common creation
 *              of info for MD volumes.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <plugin.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <time.h>
#include <unistd.h>
#include "md.h"
#include "raid5_mgr.h"


#define NUM_INFO_ENTRIES 5

static int md_get_volume_info( md_volume_t            * volume,
           	               extended_info_array_t ** info_array ) {

	int rc = 0;
	int i;
	int found = 0;
	extended_info_array_t * info = NULL;

	info = EngFncs->engine_alloc(sizeof(extended_info_array_t) + sizeof(extended_info_t) * ((NUM_INFO_ENTRIES - 1) + volume->nr_disks));
	if (info != NULL) {
		info->count = NUM_INFO_ENTRIES + volume->nr_disks;

		SET_STRING(info->info[0].name, "name");
		SET_STRING(info->info[0].title, "Name");
		SET_STRING(info->info[0].desc, "MD volume name");
		info->info[0].type = EVMS_Type_String;
		info->info[0].unit = EVMS_Unit_None;
		info->info[0].format = EVMS_Format_Normal;
		SET_STRING(info->info[0].value.s, volume->name);
		info->info[0].collection_type = EVMS_Collection_None;
		info->info[0].collection.list = NULL;
		info->info[0].group.group_number = 0;
		info->info[0].group.group_level = 0;
		info->info[0].group.group_name = NULL;
		info->info[0].flags = 0;
		
		SET_STRING(info->info[1].name, "state");
		SET_STRING(info->info[1].title, "State");
		SET_STRING(info->info[1].desc, "State of the MD region");
		info->info[1].type = EVMS_Type_String;
		info->info[1].unit = EVMS_Unit_None;
		info->info[1].format = EVMS_Format_Normal;
		message_buffer[0] = '\0';
		if (volume->flags & MD_DISCOVERED) {
			strcat(message_buffer, "Discovered");
		}
		if (volume->flags & MD_DEGRADED) {
			if (message_buffer[0] != '\0') {
				strcat(message_buffer, ", ");
			}
			strcat(message_buffer, "Degraded");
		}
		if (volume->flags & MD_CORRUPT) {
			if (message_buffer[0] != '\0') {
				strcat(message_buffer, ", ");
			}
			strcat(message_buffer, "Corrupt");
		}
		if (volume->flags & MD_DIRTY) {
			if (message_buffer[0] != '\0') {
				strcat(message_buffer, ", ");
			}
			strcat(message_buffer, "Dirty");
		}
		if (volume->flags & MD_ACTIVE) {
			if (message_buffer[0] != '\0') {
				strcat(message_buffer, ", ");
			}
			strcat(message_buffer, "Active");
		}
		SET_STRING(info->info[1].value.s, message_buffer);
		info->info[1].collection_type = EVMS_Collection_None;
		info->info[1].collection.list = NULL;
		info->info[1].group.group_number = 0;
		info->info[1].group.group_level = 0;
		info->info[1].group.group_name = NULL;
		info->info[1].flags = 0;

		SET_STRING(info->info[2].name, "personality");
		SET_STRING(info->info[2].title, "Personality");
		SET_STRING(info->info[2].desc, "MD personality");
		info->info[2].type = EVMS_Type_String;
		info->info[2].unit = EVMS_Unit_None;
		info->info[2].format = EVMS_Format_Normal;
		switch (volume->personality) {
		case MD_RESERVED:
			SET_STRING(info->info[2].value.s, "Reserved");
			break;
		case LINEAR:
			SET_STRING(info->info[2].value.s, "Linear");
			break;
		case RAID0:
			SET_STRING(info->info[2].value.s, "RAID0");
			break;
		case RAID1:
			SET_STRING(info->info[2].value.s, "RAID1");
			break;
		case RAID5:
			SET_STRING(info->info[2].value.s, "RAID5");
			break;
		case TRANSLUCENT:
			SET_STRING(info->info[2].value.s, "Translucent");
			break;
		case HSM:
			SET_STRING(info->info[2].value.s, "HSM");
			break;
		case MULTIPATH:
			SET_STRING(info->info[2].value.s, "Multipath");
			break;
		}
		info->info[2].collection_type = EVMS_Collection_None;
		info->info[2].collection.list = NULL;
		info->info[2].group.group_number = 0;
		info->info[2].group.group_level = 0;
		info->info[2].group.group_name = NULL;
		info->info[2].flags = 0;

		SET_STRING(info->info[3].name, "superblock");
		SET_STRING(info->info[3].title, "Working SuperBlock");
		SET_STRING(info->info[3].desc, "Copy of SuperBlock that is most up to date");
		info->info[3].type = EVMS_Type_String;
		info->info[3].unit = EVMS_Unit_None;
		info->info[3].format = EVMS_Format_Hex;
		info->info[3].value.s = NULL;
		info->info[3].collection_type = EVMS_Collection_None;
		info->info[3].collection.list = NULL;
		info->info[3].group.group_number = 0;
		info->info[3].group.group_level = 0;
		info->info[3].group.group_name = NULL;
		info->info[3].flags = EVMS_EINFO_FLAGS_MORE_INFO_AVAILABLE;

		SET_STRING(info->info[4].name, "nr_disks");
		SET_STRING(info->info[4].title, "Number of disks");
		SET_STRING(info->info[4].desc, "Number of disks that comprise this volume");
		info->info[4].type = EVMS_Type_Unsigned_Int32;
		info->info[4].unit = EVMS_Unit_None;
		info->info[4].format = EVMS_Format_Normal;
		info->info[4].value.ui32 = volume->nr_disks;
		info->info[4].collection_type = EVMS_Collection_None;
		info->info[4].collection.list = NULL;
		info->info[4].group.group_number = 0;
		info->info[4].group.group_level = 0;
		info->info[4].group.group_name = NULL;
		info->info[4].flags = 0;

		found = 0;
		for (i = 0; (i < MAX_MD_DEVICES) && (found < volume->nr_disks); i++ ) {
			if (volume->child_object[i]) {
				sprintf(message_buffer,"child_object%d", i);
				SET_STRING(info->info[NUM_INFO_ENTRIES + found].name, message_buffer);
				sprintf(message_buffer,"Disk %d", i);
				SET_STRING(info->info[NUM_INFO_ENTRIES + found].title, message_buffer);
				SET_STRING(info->info[NUM_INFO_ENTRIES + found].desc, "Disk that comprises this volume");
				info->info[NUM_INFO_ENTRIES + found].type = EVMS_Type_String;
				info->info[NUM_INFO_ENTRIES + found].unit = EVMS_Unit_None;
				info->info[NUM_INFO_ENTRIES + found].format = EVMS_Format_Normal;
				SET_STRING(info->info[NUM_INFO_ENTRIES + found].value.s, volume->child_object[i]->name);
				info->info[NUM_INFO_ENTRIES + found].collection_type = EVMS_Collection_None;
				info->info[NUM_INFO_ENTRIES + found].collection.list = NULL;
				info->info[NUM_INFO_ENTRIES + found].group.group_number = 0;
				info->info[NUM_INFO_ENTRIES + found].group.group_level = 0;
				info->info[NUM_INFO_ENTRIES + found].group.group_name = NULL;
				info->info[NUM_INFO_ENTRIES + found].flags = EVMS_EINFO_FLAGS_MORE_INFO_AVAILABLE;
				found++;
			}
		}

		*info_array = info;

	} else {
		LOG_CRITICAL("Error getting memory for an extended_info_array./n");
		rc = ENOMEM;
	}

	RETURN(rc);
}


#define NUM_DISK_ENTRIES 5

static int md_get_child_disk_info(md_volume_t            * volume,
		                  int                      child_index,
		                  extended_info_array_t ** info_array) {

	int rc = 0;
	extended_info_array_t * info = NULL;
	mdp_disk_t * disk = &(volume->super_block->disks[child_index]);

	info = EngFncs->engine_alloc(sizeof(extended_info_array_t) + sizeof(extended_info_t) * (NUM_DISK_ENTRIES - 1));
	if (info != NULL) {
		info->count = NUM_DISK_ENTRIES;

		sprintf(message_buffer, "number%d", child_index);
		SET_STRING(info->info[0].name, message_buffer);
		SET_STRING(info->info[0].title, "Number");
		SET_STRING(info->info[0].desc, "Disk number in the array");
		info->info[0].type = EVMS_Type_Unsigned_Int32;
		info->info[0].unit = EVMS_Unit_None;
		info->info[0].format = EVMS_Format_Normal;
		info->info[0].value.ui32 = disk->number;
		info->info[0].collection_type = EVMS_Collection_None;
		info->info[0].collection.list = NULL;
		info->info[0].group.group_number = 0;
		info->info[0].group.group_level = 0;
		info->info[0].group.group_name = NULL;
		info->info[0].flags = EVMS_EINFO_FLAGS_MORE_INFO_AVAILABLE;

		SET_STRING(info->info[1].name, "major");
		SET_STRING(info->info[1].title, "Major number");
		SET_STRING(info->info[1].desc, "Device major number");
		info->info[1].type = EVMS_Type_Unsigned_Int32;
		info->info[1].unit = EVMS_Unit_None;
		info->info[1].format = EVMS_Format_Normal;
		info->info[1].value.ui32 = disk->major;
		info->info[1].collection_type = EVMS_Collection_None;
		info->info[1].collection.list = NULL;
		info->info[1].group.group_number = 0;
		info->info[1].group.group_level = 0;
		info->info[1].group.group_name = NULL;
		info->info[1].flags = 0;

		SET_STRING(info->info[2].name, "minor");
		SET_STRING(info->info[2].title, "Minor number");
		SET_STRING(info->info[2].desc, "Device minor number");
		info->info[2].type = EVMS_Type_Unsigned_Int32;
		info->info[2].unit = EVMS_Unit_None;
		info->info[2].format = EVMS_Format_Normal;
		info->info[2].value.ui32 = disk->minor;
		info->info[2].collection_type = EVMS_Collection_None;
		info->info[2].collection.list = NULL;
		info->info[2].group.group_number = 0;
		info->info[2].group.group_level = 0;
		info->info[2].group.group_name = NULL;
		info->info[2].flags = 0;

		SET_STRING(info->info[3].name, "raid_disk");
		SET_STRING(info->info[3].title, "RAID disk");
		SET_STRING(info->info[3].desc, "The role of the device in the raid set");
		info->info[3].type = EVMS_Type_Unsigned_Int32;
		info->info[3].unit = EVMS_Unit_None;
		info->info[3].format = EVMS_Format_Normal;
		info->info[3].value.ui32 = disk->raid_disk;
		info->info[3].collection_type = EVMS_Collection_None;
		info->info[3].collection.list = NULL;
		info->info[3].group.group_number = 0;
		info->info[3].group.group_level = 0;
		info->info[3].group.group_name = NULL;
		info->info[3].flags = 0;

		SET_STRING(info->info[4].name, "state");
		SET_STRING(info->info[4].title, "State");
		SET_STRING(info->info[4].desc, "State flags");
		info->info[4].type = EVMS_Type_String;
		info->info[4].unit = EVMS_Unit_None;
		info->info[4].format = EVMS_Format_Normal;
		message_buffer[0] = '\0';
		if (disk->state & (1 << MD_DISK_FAULTY)) {
			strcat(message_buffer, "Faulty");
		}
		if (disk->state & (1 << MD_DISK_ACTIVE)) {
			if (message_buffer[0] != '\0') {
				strcat(message_buffer, ", ");
			}
			strcat(message_buffer, "Active");
		}
		if (disk->state & (1 << MD_DISK_SYNC)) {
			if (message_buffer[0] != '\0') {
				strcat(message_buffer, ", ");
			}
			strcat(message_buffer, "Sync");
		}
		if (disk->state & (1 << MD_DISK_REMOVED)) {
			if (message_buffer[0] != '\0') {
				strcat(message_buffer, ", ");
			}
			strcat(message_buffer, "Removed");
		}
		if (disk->state & (1 << MD_DISK_NEW)) {
			if (message_buffer[0] != '\0') {
				strcat(message_buffer, ", ");
			}
			strcat(message_buffer, "New");
		}
		if (disk->state & (1 << MD_DISK_PENDING_ACTIVE)) {
			if (message_buffer[0] != '\0') {
				strcat(message_buffer, ", ");
			}
			strcat(message_buffer, "Pending");
		}
		// If none of the flags are set, it must be a spare disk.
		if (message_buffer[0] == '\0') {
			strcpy(message_buffer, "Spare");
		}
		SET_STRING(info->info[4].value.s, message_buffer);
		info->info[4].collection_type = EVMS_Collection_None;
		info->info[4].collection.list = NULL;
		info->info[4].group.group_number = 0;
		info->info[4].group.group_level = 0;
		info->info[4].group.group_name = NULL;
		info->info[4].flags = 0;

		*info_array = info;
	} else {
		LOG_CRITICAL("Error getting memory for an extended_info_array./n");
		rc = ENOMEM;
	}

	RETURN(rc);
}


#define NUM_SB_ENTRIES 24

static int md_get_child_superblock_info( md_volume_t            * volume,
				         int                      child_index,
				         extended_info_array_t ** info_array ) {

	int rc = 0;
	int i, j, nr_disks, found;
	time_t time;
	extended_info_array_t * info = NULL;
	mdp_super_t * super;

	if (child_index == -1) {
		super = volume->super_block;
	} else
	{
		super = volume->super_array[child_index];
	}
	
	nr_disks = min(super->nr_disks, MAX_MD_DEVICES);

	info = EngFncs->engine_alloc(sizeof(extended_info_array_t) + sizeof(extended_info_t) * ((NUM_SB_ENTRIES - 1) + nr_disks*5));
	if (info != NULL) {
		info->count = NUM_SB_ENTRIES + nr_disks * 5;

		i = 0;
		SET_STRING(info->info[i].name, "md_magic");
		SET_STRING(info->info[i].title, "MD magic number");
		SET_STRING(info->info[i].desc, "MD identifier for the volume");
		info->info[i].type = EVMS_Type_Unsigned_Int32;
		info->info[i].unit = EVMS_Unit_None;
		info->info[i].format = EVMS_Format_Hex;
		info->info[i].value.ui32 = super->md_magic;
		info->info[i].collection_type = EVMS_Collection_None;
		info->info[i].collection.list = NULL;
		info->info[i].group.group_number = 0;
		info->info[i].group.group_level = 0;
		info->info[i].group.group_name = NULL;
		info->info[i].flags = 0;
		i++;

		SET_STRING(info->info[i].name, "version");
		SET_STRING(info->info[i].title, "Version");
		SET_STRING(info->info[i].desc, "Version of MD that wrote this superblock");
		info->info[i].type = EVMS_Type_String;
		info->info[i].unit = EVMS_Unit_None;
		info->info[i].format = EVMS_Format_Normal;
		sprintf(message_buffer, "%d.%d.%d", super->major_version,
			super->minor_version,
			super->patch_version);
		SET_STRING(info->info[i].value.s, message_buffer);
		info->info[i].collection_type = EVMS_Collection_None;
		info->info[i].collection.list = NULL;
		info->info[i].group.group_number = 0;
		info->info[i].group.group_level = 0;
		info->info[i].group.group_name = NULL;
		info->info[i].flags = 0;
		i++;

		SET_STRING(info->info[i].name, "set_uuid0");
		SET_STRING(info->info[i].title, "UUID0");
		SET_STRING(info->info[i].desc, "Lowest 32 bits of the UUID");
		info->info[i].type = EVMS_Type_Unsigned_Int32;
		info->info[i].unit = EVMS_Unit_None;
		info->info[i].format = EVMS_Format_Hex;
		info->info[i].value.ui32 = super->set_uuid0;
		info->info[i].collection_type = EVMS_Collection_None;
		info->info[i].collection.list = NULL;
		info->info[i].group.group_number = 0;
		info->info[i].group.group_level = 0;
		info->info[i].group.group_name = NULL;
		info->info[i].flags = 0;
		i++;

		SET_STRING(info->info[i].name, "ctime");
		SET_STRING(info->info[i].title, "Creation time");
		SET_STRING(info->info[i].desc, "The time the volume was created");
		info->info[i].type = EVMS_Type_String;
		info->info[i].unit = EVMS_Unit_None;
		info->info[i].format = EVMS_Format_Normal;
		time = (time_t) super->ctime;
		strcpy(message_buffer, asctime(localtime(&time)));
		// Strip off any trailing newline.
		if (message_buffer[strlen(message_buffer) - 1] == '\n') {
			message_buffer[strlen(message_buffer) - 1] = '\0';
		}
		SET_STRING(info->info[i].value.s, message_buffer);
		info->info[i].collection_type = EVMS_Collection_None;
		info->info[i].collection.list = NULL;
		info->info[i].group.group_number = 0;
		info->info[i].group.group_level = 0;
		info->info[i].group.group_name = NULL;
		info->info[i].flags = 0;
		i++;

		SET_STRING(info->info[i].name, "level");
		SET_STRING(info->info[i].title, "RAID level");
		SET_STRING(info->info[i].desc, "RAID level");
		info->info[i].type = EVMS_Type_String;
		info->info[i].unit = EVMS_Unit_None;
		info->info[i].format = EVMS_Format_Normal;
		if (super->level >= 0) {
			char raid_name[8];

			sprintf(raid_name, "RAID%d", super->level);
			SET_STRING(info->info[i].value.s, raid_name);

		} else {
			switch (level_to_pers(super->level)) {
			case MD_RESERVED:
				SET_STRING(info->info[i].value.s, "Reserved");
				break;
			case LINEAR:
				SET_STRING(info->info[i].value.s, "Linear");
				break;
			case TRANSLUCENT:
				SET_STRING(info->info[i].value.s, "Translucent");
				break;
			case HSM:
				SET_STRING(info->info[i].value.s, "HSM");
				break;
			case MULTIPATH:
				SET_STRING(info->info[i].value.s, "Multipath");
				break;
			default:
				{
					char message[32];

					sprintf(message, "Not valid (%d)", super->level);
					SET_STRING(info->info[i].value.s, message);
				}
			}
		}
		info->info[i].collection_type = EVMS_Collection_None;
		info->info[i].collection.list = NULL;
		info->info[i].group.group_number = 0;
		info->info[i].group.group_level = 0;
		info->info[i].group.group_name = NULL;
		info->info[i].flags = 0;
		i++;

		SET_STRING(info->info[i].name, "size");
		SET_STRING(info->info[i].title, "Size");
		SET_STRING(info->info[i].desc, "Apparent size of each individual disk");
		info->info[i].type = EVMS_Type_Unsigned_Int32;
		info->info[i].unit = EVMS_Unit_Kilobytes;
		info->info[i].format = EVMS_Format_Normal;
		info->info[i].value.ui32 = super->size;
		info->info[i].collection_type = EVMS_Collection_None;
		info->info[i].collection.list = NULL;
		info->info[i].group.group_number = 0;
		info->info[i].group.group_level = 0;
		info->info[i].group.group_name = NULL;
		info->info[i].flags = 0;
		i++;

		SET_STRING(info->info[i].name, "nr_disks");
		SET_STRING(info->info[i].title, "Number of disks");
		SET_STRING(info->info[i].desc, "Total disks in the RAID set");
		info->info[i].type = EVMS_Type_Unsigned_Int32;
		info->info[i].unit = EVMS_Unit_None;
		info->info[i].format = EVMS_Format_Normal;
		info->info[i].value.ui32 = super->nr_disks;
		info->info[i].collection_type = EVMS_Collection_None;
		info->info[i].collection.list = NULL;
		info->info[i].group.group_number = 0;
		info->info[i].group.group_level = 0;
		info->info[i].group.group_name = NULL;
		info->info[i].flags = 0;
		i++;

		SET_STRING(info->info[i].name, "raid_disks");
		SET_STRING(info->info[i].title, "RAID disks");
		SET_STRING(info->info[i].desc, "Number of disks in a fully functional RAID set");
		info->info[i].type = EVMS_Type_Unsigned_Int32;
		info->info[i].unit = EVMS_Unit_None;
		info->info[i].format = EVMS_Format_Normal;
		info->info[i].value.ui32 = super->raid_disks;
		info->info[i].collection_type = EVMS_Collection_None;
		info->info[i].collection.list = NULL;
		info->info[i].group.group_number = 0;
		info->info[i].group.group_level = 0;
		info->info[i].group.group_name = NULL;
		info->info[i].flags = 0;
		i++;

		SET_STRING(info->info[i].name, "md_minor");
		SET_STRING(info->info[i].title, "Minor number");
		SET_STRING(info->info[i].desc, "Preferred MD minor device number");
		info->info[i].type = EVMS_Type_Unsigned_Int32;
		info->info[i].unit = EVMS_Unit_None;
		info->info[i].format = EVMS_Format_Normal;
		info->info[i].value.ui32 = super->md_minor;
		info->info[i].collection_type = EVMS_Collection_None;
		info->info[i].collection.list = NULL;
		info->info[i].group.group_number = 0;
		info->info[i].group.group_level = 0;
		info->info[i].group.group_name = NULL;
		info->info[i].flags = 0;
		i++;

		SET_STRING(info->info[i].name, "not_persistent");
		SET_STRING(info->info[i].title, "Persistent superblock");
		SET_STRING(info->info[i].desc, "Does it have a persistent superblock?");
		info->info[i].type = EVMS_Type_String;
		info->info[i].unit = EVMS_Unit_None;
		info->info[i].format = EVMS_Format_Normal;
		if (super->not_persistent) {
			SET_STRING(info->info[i].value.s, "No");
		} else {
			SET_STRING(info->info[i].value.s, "Yes");
		}
		info->info[i].collection_type = EVMS_Collection_None;
		info->info[i].collection.list = NULL;
		info->info[i].group.group_number = 0;
		info->info[i].group.group_level = 0;
		info->info[i].group.group_name = NULL;
		info->info[i].flags = 0;
		i++;

		SET_STRING(info->info[i].name, "set_uuid1");
		SET_STRING(info->info[i].title, "UUID1");
		SET_STRING(info->info[i].desc, "Second to lowest 32 bits of the UUID");
		info->info[i].type = EVMS_Type_Unsigned_Int32;
		info->info[i].unit = EVMS_Unit_None;
		info->info[i].format = EVMS_Format_Hex;
		info->info[i].value.ui32 = super->set_uuid1;
		info->info[i].collection_type = EVMS_Collection_None;
		info->info[i].collection.list = NULL;
		info->info[i].group.group_number = 0;
		info->info[i].group.group_level = 0;
		info->info[i].group.group_name = NULL;
		info->info[i].flags = 0;
		i++;

		SET_STRING(info->info[i].name, "set_uuid2");
		SET_STRING(info->info[i].title, "UUID2");
		SET_STRING(info->info[i].desc, "Second to highest 32 bits of the UUID");
		info->info[i].type = EVMS_Type_Unsigned_Int32;
		info->info[i].unit = EVMS_Unit_None;
		info->info[i].format = EVMS_Format_Hex;
		info->info[i].value.ui32 = super->set_uuid2;
		info->info[i].collection_type = EVMS_Collection_None;
		info->info[i].collection.list = NULL;
		info->info[i].group.group_number = 0;
		info->info[i].group.group_level = 0;
		info->info[i].group.group_name = NULL;
		info->info[i].flags = 0;
		i++;

		SET_STRING(info->info[i].name, "set_uuid3");
		SET_STRING(info->info[i].title, "UUID3");
		SET_STRING(info->info[i].desc, "Highest 32 bits of the UUID");
		info->info[i].type = EVMS_Type_Unsigned_Int32;
		info->info[i].unit = EVMS_Unit_None;
		info->info[i].format = EVMS_Format_Hex;
		info->info[i].value.ui32 = super->set_uuid3;
		info->info[i].collection_type = EVMS_Collection_None;
		info->info[i].collection.list = NULL;
		info->info[i].group.group_number = 0;
		info->info[i].group.group_level = 0;
		info->info[i].group.group_name = NULL;
		info->info[i].flags = 0;
		i++;

		SET_STRING(info->info[i].name, "utime");
		SET_STRING(info->info[i].title, "Superblock update time");
		SET_STRING(info->info[i].desc, "Superblock update time");
		info->info[i].type = EVMS_Type_String;
		info->info[i].unit = EVMS_Unit_None;
		info->info[i].format = EVMS_Format_Normal;
		time = (time_t) super->utime;
		strcpy(message_buffer, asctime(localtime(&time)));
		// Strip off any trailing newline.
		if (message_buffer[strlen(message_buffer) - 1] == '\n') {
			message_buffer[strlen(message_buffer) - 1] = '\0';
		}
		SET_STRING(info->info[i].value.s, message_buffer);
		info->info[i].collection_type = EVMS_Collection_None;
		info->info[i].collection.list = NULL;
		info->info[i].group.group_number = 0;
		info->info[i].group.group_level = 0;
		info->info[i].group.group_name = NULL;
		info->info[i].flags = 0;
		i++;

		SET_STRING(info->info[i].name, "state");
		SET_STRING(info->info[i].title, "State");
		SET_STRING(info->info[i].desc, "Superblock state flags");
		info->info[i].type = EVMS_Type_String;
		info->info[i].unit = EVMS_Unit_None;
		info->info[i].format = EVMS_Format_Normal;
		message_buffer[0] = '\0';
		if (super->state & (1 << MD_SB_CLEAN)) {
			strcat(message_buffer, "Clean");
		}
		if (super->state & (1 << MD_SB_ERRORS)) {
			if (message_buffer[0] != '\0') {
				strcat(message_buffer, ", ");
			}
			strcat(message_buffer, "Errors");
		}
		SET_STRING(info->info[i].value.s, message_buffer);
		info->info[i].collection_type = EVMS_Collection_None;
		info->info[i].collection.list = NULL;
		info->info[i].group.group_number = 0;
		info->info[i].group.group_level = 0;
		info->info[i].group.group_name = NULL;
		info->info[i].flags = 0;
		i++;

		SET_STRING(info->info[i].name, "active_disks");
		SET_STRING(info->info[i].title, "Active disks");
		SET_STRING(info->info[i].desc, "Number of currently active disks");
		info->info[i].type = EVMS_Type_Unsigned_Int32;
		info->info[i].unit = EVMS_Unit_None;
		info->info[i].format = EVMS_Format_Normal;
		info->info[i].value.ui32 = super->active_disks;
		info->info[i].collection_type = EVMS_Collection_None;
		info->info[i].collection.list = NULL;
		info->info[i].group.group_number = 0;
		info->info[i].group.group_level = 0;
		info->info[i].group.group_name = NULL;
		info->info[i].flags = 0;
		i++;

		SET_STRING(info->info[i].name, "working_disks");
		SET_STRING(info->info[i].title, "Working disks");
		SET_STRING(info->info[i].desc, "Number of working disks");
		info->info[i].type = EVMS_Type_Unsigned_Int32;
		info->info[i].unit = EVMS_Unit_None;
		info->info[i].format = EVMS_Format_Normal;
		info->info[i].value.ui32 = super->working_disks;
		info->info[i].collection_type = EVMS_Collection_None;
		info->info[i].collection.list = NULL;
		info->info[i].group.group_number = 0;
		info->info[i].group.group_level = 0;
		info->info[i].group.group_name = NULL;
		info->info[i].flags = 0;
		i++;

		SET_STRING(info->info[i].name, "failed_disks");
		SET_STRING(info->info[i].title, "Failed disks");
		SET_STRING(info->info[i].desc, "Number of failed disks");
		info->info[i].type = EVMS_Type_Unsigned_Int32;
		info->info[i].unit = EVMS_Unit_None;
		info->info[i].format = EVMS_Format_Normal;
		info->info[i].value.ui32 = super->failed_disks;
		info->info[i].collection_type = EVMS_Collection_None;
		info->info[i].collection.list = NULL;
		info->info[i].group.group_number = 0;
		info->info[i].group.group_level = 0;
		info->info[i].group.group_name = NULL;
		info->info[i].flags = 0;
		i++;

		SET_STRING(info->info[i].name, "spare_disks");
		SET_STRING(info->info[i].title, "Spare disks");
		SET_STRING(info->info[i].desc, "Number of spare disks");
		info->info[i].type = EVMS_Type_Unsigned_Int32;
		info->info[i].unit = EVMS_Unit_None;
		info->info[i].format = EVMS_Format_Normal;
		info->info[i].value.ui32 = super->spare_disks;
		info->info[i].collection_type = EVMS_Collection_None;
		info->info[i].collection.list = NULL;
		info->info[i].group.group_number = 0;
		info->info[i].group.group_level = 0;
		info->info[i].group.group_name = NULL;
		info->info[i].flags = 0;
		i++;

		SET_STRING(info->info[i].name, "sb_csum");
		SET_STRING(info->info[i].title, "Check sum");
		SET_STRING(info->info[i].desc, "Superblock check sum");
		info->info[i].type = EVMS_Type_Unsigned_Int32;
		info->info[i].unit = EVMS_Unit_None;
		info->info[i].format = EVMS_Format_Normal;
		info->info[i].value.ui32 = super->sb_csum;
		info->info[i].collection_type = EVMS_Collection_None;
		info->info[i].collection.list = NULL;
		info->info[i].group.group_number = 0;
		info->info[i].group.group_level = 0;
		info->info[i].group.group_name = NULL;
		info->info[i].flags = 0;
		i++;

		SET_STRING(info->info[i].name, "events");
		SET_STRING(info->info[i].title, "Update count");
		SET_STRING(info->info[i].desc, "Superblock update count");
		info->info[i].type = EVMS_Type_Unsigned_Int64;
		info->info[i].unit = EVMS_Unit_None;
		info->info[i].format = EVMS_Format_Normal;
		info->info[i].value.ui64 = (u_int64_t) (super->events_hi);
		info->info[i].value.ui64 <<= 32;
                info->info[i].value.ui64 += (u_int64_t) (super->events_lo);
		info->info[i].collection_type = EVMS_Collection_None;
		info->info[i].collection.list = NULL;
		info->info[i].group.group_number = 0;
		info->info[i].group.group_level = 0;
		info->info[i].group.group_name = NULL;
		info->info[i].flags = 0;
		i++;

		SET_STRING(info->info[i].name, "layout");
		SET_STRING(info->info[i].title, "Layout");
		SET_STRING(info->info[i].desc, "The physical layout for a RAID5 array");
		info->info[i].type = EVMS_Type_String;
		info->info[i].unit = EVMS_Unit_None;
		info->info[i].format = EVMS_Format_Normal;
		if (super->level == 5) {
			switch (super->layout) {
			case ALGORITHM_LEFT_ASYMMETRIC:
				SET_STRING(info->info[i].value.s, ALGORITHM_LEFT_ASYMMETRIC_NAME);
				break;
			case ALGORITHM_RIGHT_ASYMMETRIC:
				SET_STRING(info->info[i].value.s, ALGORITHM_RIGHT_ASYMMETRIC_NAME);
				break;
			case ALGORITHM_LEFT_SYMMETRIC:
				SET_STRING(info->info[i].value.s, ALGORITHM_LEFT_SYMMETRIC_NAME);
				break;
			case ALGORITHM_RIGHT_SYMMETRIC:
				SET_STRING(info->info[i].value.s, ALGORITHM_RIGHT_SYMMETRIC_NAME);
				break;
			default:
				SET_STRING(info->info[i].value.s, "Unknown");
				break;
                        }
		} else {
			info->info[i].value.s = NULL;
		}
		info->info[i].collection_type = EVMS_Collection_None;
		info->info[i].collection.list = NULL;
		info->info[i].group.group_number = 0;
		info->info[i].group.group_level = 0;
		info->info[i].group.group_name = NULL;
		info->info[i].flags = 0;
		i++;

		SET_STRING(info->info[i].name, "chunk_size");
		SET_STRING(info->info[i].title, "Chunk size");
		SET_STRING(info->info[i].desc, "Chunk size in bytes");
		info->info[i].type = EVMS_Type_Unsigned_Int32;
		info->info[i].unit = EVMS_Unit_Bytes;
		info->info[i].format = EVMS_Format_Normal;
		info->info[i].value.ui32 = super->chunk_size;
		info->info[i].collection_type = EVMS_Collection_None;
		info->info[i].collection.list = NULL;
		info->info[i].group.group_number = 0;
		info->info[i].group.group_level = 0;
		info->info[i].group.group_name = NULL;
		info->info[i].flags = 0;
		i++;

		found = 0;
		for (j = 0; (j < MAX_MD_DEVICES) && (found < nr_disks); j++ ) {
			if (super->disks[j].major != 0) {
				mdp_disk_t * disk = &super->disks[j];

				sprintf(message_buffer, "disk%2dnumber", j);
				SET_STRING(info->info[i].name, message_buffer);
				sprintf(message_buffer, "Disk %d  Number", j);
				SET_STRING(info->info[i].title, message_buffer);
				SET_STRING(info->info[i].desc, "Disk index in the array");
				info->info[i].type = EVMS_Type_Unsigned_Int32;
				info->info[i].unit = EVMS_Unit_None;
				info->info[i].format = EVMS_Format_Normal;
				info->info[i].value.ui32 = super->disks[j].number;
				info->info[i].collection_type = EVMS_Collection_None;
				info->info[i].collection.list = NULL;
				info->info[i].group.group_number = 0;
				info->info[i].group.group_level = 0;
				info->info[i].group.group_name = NULL;
				info->info[i].flags = 0;
				i++;

				sprintf(message_buffer, "disk%2dmajor", j);
				SET_STRING(info->info[i].name, message_buffer);
				sprintf(message_buffer, "Disk %d  Major Number ", j);
				SET_STRING(info->info[i].title, message_buffer);
				SET_STRING(info->info[i].desc, "Device major number");
				info->info[i].type = EVMS_Type_Unsigned_Int32;
				info->info[i].unit = EVMS_Unit_None;
				info->info[i].format = EVMS_Format_Normal;
				info->info[i].value.ui32 = disk->major;
				info->info[i].collection_type = EVMS_Collection_None;
				info->info[i].collection.list = NULL;
				info->info[i].group.group_number = 0;
				info->info[i].group.group_level = 0;
				info->info[i].group.group_name = NULL;
				info->info[i].flags = 0;
				i++;

				sprintf(message_buffer, "disk%2dminor", j);
				SET_STRING(info->info[i].name, message_buffer);
				sprintf(message_buffer, "Disk %d  Minor Number ", j);
				SET_STRING(info->info[i].title, message_buffer);
				SET_STRING(info->info[i].desc, "Device minor number");
				info->info[i].type = EVMS_Type_Unsigned_Int32;
				info->info[i].unit = EVMS_Unit_None;
				info->info[i].format = EVMS_Format_Normal;
				info->info[i].value.ui32 = disk->minor;
				info->info[i].collection_type = EVMS_Collection_None;
				info->info[i].collection.list = NULL;
				info->info[i].group.group_number = 0;
				info->info[i].group.group_level = 0;
				info->info[i].group.group_name = NULL;
				info->info[i].flags = 0;
				i++;

				sprintf(message_buffer, "disk%2draiddisk", j);
				SET_STRING(info->info[i].name, message_buffer);
				sprintf(message_buffer, "Disk %d  RAID Disk", j);
				SET_STRING(info->info[i].title, message_buffer);
				SET_STRING(info->info[i].desc, "The role of the device in the raid set");
				info->info[i].type = EVMS_Type_Unsigned_Int32;
				info->info[i].unit = EVMS_Unit_None;
				info->info[i].format = EVMS_Format_Normal;
				info->info[i].value.ui32 = disk->raid_disk;
				info->info[i].collection_type = EVMS_Collection_None;
				info->info[i].collection.list = NULL;
				info->info[i].group.group_number = 0;
				info->info[i].group.group_level = 0;
				info->info[i].group.group_name = NULL;
				info->info[i].flags = 0;
				i++;

				sprintf(message_buffer, "disk%2dstate", j);
				SET_STRING(info->info[i].name, message_buffer);
				sprintf(message_buffer, "Disk %d  State", j);
				SET_STRING(info->info[i].title, message_buffer);
				SET_STRING(info->info[i].desc, "State flags");
				info->info[i].type = EVMS_Type_String;
				info->info[i].unit = EVMS_Unit_None;
				info->info[i].format = EVMS_Format_Normal;
				message_buffer[0] = '\0';
				if (disk->state & (1 << MD_DISK_FAULTY)) {
					strcat(message_buffer, "Faulty");
				}
				if (disk->state & (1 << MD_DISK_ACTIVE)) {
					if (message_buffer[0] != '\0') {
						strcat(message_buffer, ", ");
					}
					strcat(message_buffer, "Active");
				}
				if (disk->state & (1 << MD_DISK_SYNC)) {
					if (message_buffer[0] != '\0') {
						strcat(message_buffer, ", ");
					}
					strcat(message_buffer, "Sync");
				}
				if (disk->state & (1 << MD_DISK_REMOVED)) {
					if (message_buffer[0] != '\0') {
						strcat(message_buffer, ", ");
					}
					strcat(message_buffer, "Removed");
				}
				if (disk->state & (1 << MD_DISK_NEW)) {
					if (message_buffer[0] != '\0') {
						strcat(message_buffer, ", ");
					}
					strcat(message_buffer, "New");
				}
				if (disk->state & (1 << MD_DISK_PENDING_ACTIVE)) {
					if (message_buffer[0] != '\0') {
						strcat(message_buffer, ", ");
					}
					strcat(message_buffer, "Pending");
				}
				// If none of the flags are set, it must be a spare disk.
				if (message_buffer[0] == '\0') {
					strcpy(message_buffer, "Spare");
				}
				SET_STRING(info->info[i].value.s, message_buffer);
				info->info[i].collection_type = EVMS_Collection_None;
				info->info[i].collection.list = NULL;
				info->info[i].group.group_number = 0;
				info->info[i].group.group_level = 0;
				info->info[i].group.group_name = NULL;
				info->info[i].flags = 0;
				i++;

				found ++;
			}
		}

		SET_STRING(info->info[i].name, "this_disk");
		SET_STRING(info->info[i].title, "This disk");
		SET_STRING(info->info[i].desc, "This disk's index in the array");
		info->info[i].type = EVMS_Type_Unsigned_Int32;
		info->info[i].unit = EVMS_Unit_None;
		info->info[i].format = EVMS_Format_Normal;
		info->info[i].value.ui32 = super->this_disk.number;
		info->info[i].collection_type = EVMS_Collection_None;
		info->info[i].collection.list = NULL;
		info->info[i].group.group_number = 0;
		info->info[i].group.group_level = 0;
		info->info[i].group.group_name = NULL;
		info->info[i].flags = 0;

		*info_array = info;
	} else {
		LOG_CRITICAL("Error getting memory for an extended_info_array./n");
		rc = ENOMEM;
	}

	RETURN(rc);
}


int md_get_info(md_volume_t            * volume,
		char                   * name,
		extended_info_array_t ** info_array) {

	int rc = 0;
	int child_index = -1;

	if (! name) {
		// Default case. Return all basic info about the region.
		rc = md_get_volume_info(volume, info_array);
	}
	else if ( ! strncmp(name, "child_object", 12) ) {
                // "Extra" information about the child's superblock
		child_index = atoi(name + 12);
		if (child_index >= 0) {
			rc = md_get_child_disk_info(volume, child_index, info_array);
		}
		else {
			LOG_ERROR("No support for extra region information about \"%s\"\n", name);
			RETURN(EINVAL);
		}
	}
	else if ( ! strncmp(name, "number", 6) ) {
                // "Extra" information about a disk in the array
		child_index = atoi(name + 6);
		if (child_index >= 0) {
			rc = md_get_child_superblock_info(volume, child_index, info_array);
		}
		else {
			LOG_ERROR("No support for extra region information about \"%s\"\n", name);
			RETURN(EINVAL);
		}
	}
	else if ( ! strncmp(name, "superblock", 10) ) {
                // "Extra" information about the master superblock
			rc = md_get_child_superblock_info(volume, -1, info_array);
	}
	else {
		LOG_ERROR("No support for extra region information about \"%s\"\n", name);
		RETURN(EINVAL);
	}

	RETURN(rc);
}

