/*
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: lvmregmgr
 * File: lvm_uuid.c
 *
 * Description: This file contains all functions related to creation and
 *              checking of UUIDs in the LVM region manager.
 */


#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>
#include <fcntl.h>
#include <plugin.h>
#include "lvmregmgr.h"

// String containing allowable characters for a UUID
static unsigned char c[] = "0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ";


/* Function: lvm_check_for_uuid
 *
 *	Check all existing volume groups and physical volumes to see if the
 *	specified UUID is already in use.
 */
int lvm_check_for_uuid( char * uuid )
{
	lvm_volume_group_t	* group;
	int			i, rc;

	LOG_ENTRY;

	for ( rc = GoToStartOfList(lvm_group_list);
	      !rc && (group = lvm_get_list_item(lvm_group_list));
	      rc = NextItem(lvm_group_list) ) {

		// Check each group's UUID
		if ( ! memcmp(uuid, group->vg->vg_uuid, UUID_LEN) ) {
			LOG_ERROR("UUID %s already in use by VG %s\n", uuid, group->container->name);
			RETURN(EINVAL);
		}

		// Check each group's PVs' UUIDs
		for ( i = 1; i <= MAX_PV; i++ ) {
			if ( group->pv_list[i] ) {
				if ( ! memcmp(uuid, group->pv_list[i]->pv->pv_uuid, UUID_LEN) ) {
					LOG_ERROR("UUID %s already in use by PV %s\n", uuid, group->pv_list[i]->segment->name);
					RETURN(EINVAL);
				}
			}
		}
	}

	RETURN(0);
}


/* Function: lvm_create_uuid
 *
 *	Create a new UUID string. The uuid parameter must point to a char array
 *	at least 128 bytes in length. Make sure the generated UUID does not
 *	conflict with any UUIDs in the system. This function is based on the
 *	"lvm_create_uuid" function from the LVM tools library.
 */
int lvm_create_uuid( char * uuid )
{
	int random, i;

	LOG_ENTRY;

	memset( uuid, 0, UUID_LEN );
	if ( (random = open("/dev/urandom", O_RDONLY)) < 0 ) {
		LOG_ERROR("Error opening /dev/urandom\n");
		RETURN(EIO);
	}

	while ( 1 ) {
		i = read(random, uuid, UUID_LEN);
		if ( i < 0 ) {
			LOG_ERROR("Read error from /dev/urandom\n");
			close(random);
			RETURN(EIO);
		}
		for ( i = 0; i < UUID_LEN; i++ ) {
			uuid[i] = c[uuid[i] % (sizeof(c)-1)];
		}
		if ( ! lvm_check_for_uuid(uuid) ) {
			break;
		}
	}

	close(random);
	RETURN(0);
}


/* Function: lvm_print_uuid
 *
 *	Return a printable string of the specified UUID. This function is
 *	based on the "lvm_show_uuid" function from the LVM tools library.
 */
char * lvm_print_uuid ( char * uuid )
{
	int i, j;
	static char ret[NAME_LEN] = { 0, };

	LOG_ENTRY;

	memset(ret, 0, NAME_LEN);

	i = 6;
	memcpy(ret, uuid, i);
	uuid += i;

	for ( j = 0; j < 6; j++ ) {
		ret[i++] = '-';
		memcpy(&ret[i], uuid, 4);
		uuid += 4;
		i += 4;
	}

	memcpy(&ret[i], uuid, 2);

	RETURN(ret);
}


/* Function: lvm_clear_uuid_list_entry
 *
 *	Erase the specified UUID entry in the specified group.
 */
int lvm_clear_uuid_list_entry( lvm_volume_group_t * group, u_int32_t number )
{
	LOG_ENTRY;

	if ( number < 1 || number > MAX_PV ) {
		RETURN(EINVAL);
	}

	if ( group->uuid_list[number] ) {
		lvm_engine->engine_free(group->uuid_list[number]);
		group->uuid_list[number] = NULL;
	}

	RETURN(0);
}


/* Function: lvm_set_uuid_list_entry
 *
 *	Set the specified UUID entry in the specified group to the
 *	specified string.
 */
int lvm_set_uuid_list_entry( lvm_volume_group_t * group,
				u_int32_t	number,
				unsigned char	* uuid )
{
	LOG_ENTRY;

	if ( number < 1 || number > MAX_PV ) {
		RETURN(EINVAL);
	}

	if ( ! group->uuid_list[number] ) {
		group->uuid_list[number] = lvm_engine->engine_alloc(UUID_LEN);
		if ( ! group->uuid_list[number] ) {
			LOG_CRITICAL("Memory error creating string for UUID entry %d in container %s\n",
				number, group->container->name);
			RETURN(ENOMEM);
		}
	}
	memcpy(group->uuid_list[number], uuid, UUID_LEN);

	RETURN(0);
}


/* Function: lvm_verify_pv_uuid
 *
 *	pv_entry : The PV whose UUID we are verifying
 *	group : The group whose UUID list we are searching
 *
 *	Verify that the specified PV belongs to the specified group by
 *	searching for the PV's UUID in the group's list of PV UUIDs. If it is
 *	found, but in the wrong location, update the PV info to reflect the
 *	new location.
 */
int lvm_verify_pv_uuid(	lvm_physical_volume_t	* pv_entry,
			lvm_volume_group_t	* group )
{
	int i;

	LOG_ENTRY;

	// Obviously the UUID list must be present in order to search.
	if ( ! (group->flags & LVM_VG_FLAG_UUID_LIST_PRESENT) ) {
		LOG_ERROR("UUID list is missing from container %s\n", group->container->name);
		LOG_ERROR("Cannot verify UUID for PV %s\n", pv_entry->segment->name);
		RETURN(0);
	}

	// Start with the UUID entry for this PV's number
	if ( group->uuid_list[pv_entry->number] &&
	     ! memcmp(pv_entry->pv->pv_uuid, group->uuid_list[pv_entry->number], UUID_LEN) ) {
		RETURN(0);
	}

	// If it wasn't found there, then search the entire group's list.
	for ( i = 1; i <= MAX_PV; i++ ) {
		if ( group->uuid_list[i] &&
		     ! memcmp(pv_entry->pv->pv_uuid, group->uuid_list[i], UUID_LEN) ) {
			// Found the UUID. Update the PV to reflect the
			// correct PV number.
			LOG_ERROR("Detected UUID mismatch for PV %s\n", pv_entry->segment->name);
			LOG_ERROR("Moving PV %s from number %d to %d\n", pv_entry->segment->name, pv_entry->number, i);
			LOG_ERROR("If you have any snapshot regions in group %s\n", group->container->name);
			LOG_ERROR(" it is recommended that you delete them immediately!\n");
			pv_entry->number = i;
			pv_entry->pv->pv_number = i;
			group->flags |= SCFLAG_DIRTY;
			lvm_engine->changes_pending();
			RETURN(0);
		}
	}

	LOG_SERIOUS("Could not find UUID for PV %s in container %s\n", pv_entry->segment->name, group->container->name);
	RETURN(EINVAL);
}


