/*
 *
 *   Copyright (c) International Business Machines  Corp., 2000
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: bbroptions.c
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

#include <plugin.h>
#include <linux/evms/evms_user.h>
#include <linux/evms/evms_bbr.h>
#include <linux/genhd.h>

#include "bbr.h"
#include "bbroptions.h"




/*
 *  Returns the current object in the specified DLIST.  This call should
 *  be preceeded by the get_first_object_in_list() call or the caller
 *  should position the DLIST current pointer themselves.  Note, this call
 *  will not modify anything in the DLIST, including the CURRENT_ITEM ptr
 *  and so it will not advance in the DLIST.
 */
static  storage_object_t *  get_object_from_list( dlist_t list )
{
    int rc;
    storage_object_t *obj;
    storage_object_t *object_from_list = NULL;
    uint size;
    TAG  tag;

    LOGENTRY();

    rc = BlindGetObject( list, &size, &tag, NULL, FALSE, (void **)&obj );
    if (rc == DLIST_SUCCESS) {
        object_from_list = obj;
    }

    LOGEXIT();
    return  object_from_list;
}



/*
 *  Returns count of options for specified task
 */
int BBR_GetOptionCount(task_context_t * task)
{
    int count;

    LOGENTRY();

    switch (task->action) {

    case EVMS_Task_Create:
        count = BBR_CREATE_OPTION_COUNT;
        break;

    case EVMS_Task_Expand:
        count = BBR_EXPAND_OPTION_COUNT;
        break;

    case EVMS_Task_Shrink:
        count = BBR_SHRINK_OPTION_COUNT;
        break;

    default:
        count = 0;
        break;
    }

    LOGEXIT();
    return count;
}




static int allocate_create_option_descriptors( task_context_t * context )
{
    int rc = EINVAL;


    LOGENTRY();

    //
    //  If things look Ok it means we have a default size for the create operation and now
    //  simply need to initialize options.
    //
    if ( context ) {

        context->option_descriptors->count = BBR_CREATE_OPTION_COUNT ;      // must set count


        // Object Name
        context->option_descriptors->option[BBR_CREATE_NAME_INDEX].flags = EVMS_OPTION_FLAGS_INACTIVE ;
        context->option_descriptors->option[BBR_CREATE_NAME_INDEX].constraint.list = NULL;
        context->option_descriptors->option[BBR_CREATE_NAME_INDEX].constraint_type = EVMS_Collection_None;
        context->option_descriptors->option[BBR_CREATE_NAME_INDEX].help = NULL;
        SET_STRING_FIELD( context->option_descriptors->option[BBR_CREATE_NAME_INDEX].name, BBR_CREATE_NAME_NAME );
        context->option_descriptors->option[BBR_CREATE_NAME_INDEX].size = EVMS_VOLUME_NAME_SIZE;
        SET_STRING_FIELD( context->option_descriptors->option[BBR_CREATE_NAME_INDEX].tip, "The name to be given to the drivelink object.");
        SET_STRING_FIELD( context->option_descriptors->option[BBR_CREATE_NAME_INDEX].title, "Name" );
        context->option_descriptors->option[BBR_CREATE_NAME_INDEX].type = EVMS_Type_String;
        context->option_descriptors->option[BBR_CREATE_NAME_INDEX].unit = EVMS_Unit_None;
        context->option_descriptors->option[BBR_CREATE_NAME_INDEX].value.s = BBREngFncs->engine_alloc(EVMS_VOLUME_NAME_SIZE+1);
        if (context->option_descriptors->option[BBR_CREATE_NAME_INDEX].value.s) {
            memset(context->option_descriptors->option[BBR_CREATE_NAME_INDEX].value.s, 0, EVMS_VOLUME_NAME_SIZE);
        }
        else {
            rc = ENOMEM;
            LOGEXITRC();
            return rc;
        }



        rc = 0;
    }


    LOGEXITRC();
    return rc;
}

static int initialize_create_option_descriptors( task_context_t * context )
{
    int rc = EINVAL;



    LOGENTRY();

    // there should be exactly BBR_CREATE_OPTION_COUNT options
    if ( context->option_descriptors->count != BBR_CREATE_OPTION_COUNT) {
        LOG_ERROR("error, wrong number of option descriptors ... count= %d\n", context->option_descriptors->count);
        LOGEXITRC();
        return rc;
    }

    if (context) {

        // there should be exactly BBR_CREATE_OPTION_COUNT options
        if ( context->option_descriptors->count == BBR_CREATE_OPTION_COUNT ) {

            // Name
              context->option_descriptors->option[BBR_CREATE_NAME_INDEX].flags = 0;

            rc = 0;

        }
        else {

            LOG_ERROR("error, wrong number of option descriptors ... count= %d\n", context->option_descriptors->count);

        }

    }



    LOGEXITRC();
    return rc;
}





static int allocate_shrink_option_descriptors( task_context_t * context )
{
    int rc = EINVAL;


    LOGENTRY();

    //
    //  If things look Ok it means we have a default size for the create operation and now
    //  simply need to initialize options.
    //
    if ( context ) {

        context->option_descriptors->count = BBR_SHRINK_OPTION_COUNT ;      // must set count

        // Block replacement count
        context->option_descriptors->option[BBR_SHRINK_BLOCKCOUNT_INDEX].constraint.list = NULL;
        context->option_descriptors->option[BBR_SHRINK_BLOCKCOUNT_INDEX].constraint.range = BBREngFncs->engine_alloc( sizeof(value_range_t) );
        if (context->option_descriptors->option[BBR_SHRINK_BLOCKCOUNT_INDEX].constraint.range==NULL) {
            LOGEXIT();
            return ENOMEM;
        }
        context->option_descriptors->option[BBR_SHRINK_BLOCKCOUNT_INDEX].constraint_type                  = EVMS_Collection_Range;
        context->option_descriptors->option[BBR_SHRINK_BLOCKCOUNT_INDEX].constraint.range->min.ui64       = BBR_MIN_REPLACEMENT_BLOCKS;
        context->option_descriptors->option[BBR_SHRINK_BLOCKCOUNT_INDEX].constraint.range->max.ui64       = BBR_MAX_REPLACEMENT_BLOCKS;
        context->option_descriptors->option[BBR_SHRINK_BLOCKCOUNT_INDEX].constraint.range->increment.ui64 = 1;


        context->option_descriptors->option[BBR_SHRINK_BLOCKCOUNT_INDEX].flags = 0;
        context->option_descriptors->option[BBR_SHRINK_BLOCKCOUNT_INDEX].help = NULL;
        SET_STRING_FIELD( context->option_descriptors->option[BBR_SHRINK_BLOCKCOUNT_INDEX].name, BBR_SHRINK_BLOCKCOUNT_NAME );
        context->option_descriptors->option[BBR_SHRINK_BLOCKCOUNT_INDEX].size = sizeof(u_int64_t);
        SET_STRING_FIELD( context->option_descriptors->option[BBR_SHRINK_BLOCKCOUNT_INDEX].tip,"The number of replacement blocks you wish to use for the expanded object." );
        SET_STRING_FIELD( context->option_descriptors->option[BBR_SHRINK_BLOCKCOUNT_INDEX].title, "Count" );
        context->option_descriptors->option[BBR_SHRINK_BLOCKCOUNT_INDEX].type = EVMS_Type_Unsigned_Int64;
        context->option_descriptors->option[BBR_SHRINK_BLOCKCOUNT_INDEX].unit = EVMS_Unit_None;
        context->option_descriptors->option[BBR_SHRINK_BLOCKCOUNT_INDEX].value.ui64 = 0;

    }


    LOGEXITRC();
    return rc;
}

static int initialize_shrink_option_descriptors( storage_object_t *obj, task_context_t * context )
{
    int rc = EINVAL;
    u_int64_t  blocks=0;



    LOGENTRY();

    // there should be exactly BBR_SHRINK_OPTION_COUNT options
    if ( context->option_descriptors->count != BBR_SHRINK_OPTION_COUNT) {
        LOG_ERROR("error, wrong number of option descriptors ... count= %d\n", context->option_descriptors->count);
        LOGEXITRC();
        return rc;
    }


    // check that we claimed this storage object
    if (obj->plugin == BBR_PluginRecord_Ptr ) {

        if ( obj->private_data ) {

            // test for BBR signature
            if ( ((BBR_Private_Data *)obj->private_data)->signature == EVMS_BBR_SIGNATURE ) {

                blocks = ((BBR_Private_Data *)obj->private_data)->replacement_blocks_needed;

                rc = 0;
            }
        }

    }


    //
    //  initialize options.
    //
    if ( rc == 0 ) {
        // Block replacement count
        context->option_descriptors->option[BBR_SHRINK_BLOCKCOUNT_INDEX].value.ui64 = blocks;

    }


    LOGEXITRC();
    return rc;
}




/*
 *  Get the user selected object from the create context and init the option descriptors
 *  as appropriate for the object.
 *
 *  Note ... if there is more than 1 object in the source object list we will simply
 *           take the first good object and decline the remainder.  That is the reason for
 *           the BOOLEAN flag found_good_object.
 */
static int set_create_object( task_context_t * context,
                              dlist_t          declined_objects,
                              task_effect_t  * effect )
{
    int                 rc;
    uint                count;


    LOGENTRY();


    rc = GetListSize(context->selected_objects, &count );
    if (rc == DLIST_SUCCESS) {

        if ( count == 1 ) {

            rc = initialize_create_option_descriptors( context );

            *effect   |=  EVMS_Effect_Reload_Options;

        }

    }
    else {
        rc = EINVAL;
    }


    LOGEXITRC();
    return rc;
}



/*
 * Validate the objects in the source and target dlists in the context
 * record.  Remove from the selected objects lists any objects which are no
 * longer acceptable.  Return all acceptable objects in the parameter
 * dlists.  Also, for any object which is removed from the selected dlists,
 * or is otherwise not acceptable, create a declined_handle_t struct with
 * reason why not acceptable and add to the declined_objects dlist.
 */
static int set_shrink_object( task_context_t * context,
                              dlist_t          declined_objects,
                              task_effect_t  * effect )
{
    int  rc;
    storage_object_t  *obj;
    void              *handle;
    BOOLEAN            found_good_object = FALSE;
    declined_object_t *declined_object=NULL;


    LOGENTRY();

    *effect = 0;

    rc = GoToStartOfList( context->selected_objects );

    while (rc == DLIST_SUCCESS) {

        obj = get_object_from_list( context->selected_objects );
        if (obj) {

            if ( found_good_object == FALSE ) {

                found_good_object = TRUE;

                rc = initialize_shrink_option_descriptors(obj, context);

                if (rc == DLIST_SUCCESS) {
                    *effect |=  EVMS_Effect_Reload_Options;
                    rc = NextItem(context->selected_objects);
                }

            }
            else {

                LOG_ERROR("declining an object found in context->selected_objects, object name= %s\n", obj->name);

                declined_object = BBREngFncs->engine_alloc( sizeof(declined_object_t));

                if (declined_object) {

                    declined_object->object = obj;
                    declined_object->reason = rc; // REMINDER ... need to pickup reason codes the UI understands

                    rc = InsertObject ( declined_objects,
                                        sizeof(declined_object_t),
                                        declined_object,
                                        DECLINED_OBJECT_TAG,
                                        NULL,
                                        AppendToList,
                                        TRUE,
                                        &handle );

                    if (rc == DLIST_SUCCESS) {
                        *effect |=  EVMS_Effect_Reload_Objects;
                        rc = NextItem(context->selected_objects);
                    }
                    else {
                        BBREngFncs->engine_free(declined_object);
                    }

                }
                else {
                    LOG_ERROR("unable to malloc a declined object struct\n");
                    rc = ENOMEM;
                }

            }


        }
        else {
            rc = DLIST_END_OF_LIST;
        }

    };


    // throw away errors like ... DLIST_END_OF_LIST ... which are not real errors
    if ( ( rc == DLIST_EMPTY )||( rc == DLIST_END_OF_LIST )) {
        rc = DLIST_SUCCESS;
    }


    LOGEXITRC();
    return rc;
}




/*
 * Validate the objects in the selected_objects dlist in the task context.
 * Remove from the selected objects lists any objects which are not
 * acceptable.
 *
 * For unacceptable objects, create a declined_handle_t structure with the
 * reason why it is not acceptable, and add it to the declined_objects dlist.
 * Modify the accepatble_objects dlist in the task context as necessary
 * based on the selected objects and the current settings of the options.
 *
 * Modify any option settings as necessary based on the selected objects.
 * Return the appropriate task_effect_t settings if the object list(s),
 * minimum or maximum objects selected, or option settings have changed.
 */
int BBR_SetObjects( task_context_t * context,
                    dlist_t          declined_objects,
                    task_effect_t  * effect )
{

    int rc = EINVAL;

    LOGENTRY();

    if (context) {

        switch (context->action) {

        case EVMS_Task_Create:

            rc = set_create_object( context, declined_objects, effect );
            break;

        case EVMS_Task_Expand:

            // BBR is never an expansion point. It passes expands down to
            // child objects.
            rc = EINVAL;
            break;

        case EVMS_Task_Shrink:

            rc = set_shrink_object( context, declined_objects, effect );
            break;

        default:

            LOG_ERROR("context->action is unknown or unsupported\n");
            break;
        }
    }

    LOGEXITRC();
    return rc;
}




static int SetCreateOption( task_context_t * context,
                            u_int32_t        index,
                            value_t        * value,
                            u_int32_t      * info )
{
    int rc = EINVAL;

    LOGENTRY();

    if (index == BBR_CREATE_NAME_INDEX ) {


        if (value) {

            if (value->s) {

                if ( (strlen(value->s) > 0) &&
                     (strlen(value->s) <= EVMS_VOLUME_NAME_SIZE) ){

                    rc = BBREngFncs->validate_name( value->s );

                    if (rc == 0) {
                        strcpy( context->option_descriptors->option[index].value.s, value->s);
                    }

                }

            }

        }

    }


    LOGEXITRC();
    return rc;
}




static int SetShrinkOption( task_context_t * context,
                            u_int32_t        index,
                            value_t        * value,
                            u_int32_t      * info )
{
    int rc = EINVAL;

    LOGENTRY();

    if (index == BBR_SHRINK_BLOCKCOUNT_INDEX ) {

        context->option_descriptors->option[index].value.ui64 = value->ui64;
        rc = 0;

    }


    LOGEXITRC();
    return rc;
}


/*
 *
 */
int BBR_SetOption( task_context_t * context,
                   u_int32_t        index,
                   value_t        * value,
                   u_int32_t      * info )
{
    int rc=EINVAL;

    LOGENTRY();

    // a measure of protection ...
    if (context == NULL) {
        LOGEXITRC();
        return rc;
    }

    switch (context->action) {

    case EVMS_Task_Create:
        rc = SetCreateOption( context, index, value, info );
        break;

    case EVMS_Task_Expand:
        // BBR is never an expansion point ... so no options to set.
        rc = EINVAL;
        break;

    case EVMS_Task_Shrink:
        rc = SetShrinkOption( context, index, value, info );
        break;

    default:
        rc = EINVAL;
        break;
    }


    LOGEXITRC();

    return rc;
}




/*
 *  Called from get_acceptable_expand_objects to prune objects we
 *  cant layer BBR on top of.
 */
static BOOLEAN prune_noncreatable_objects( ADDRESS   Object,
                                           TAG       ObjectTag,
                                           uint      ObjectSize,
                                           ADDRESS   ObjectHandle,
                                           ADDRESS   Parameters,
                                           BOOLEAN  *FreeMemory,
                                           uint     *Error )
{
    storage_object_t * obj = (storage_object_t *) Object;

    *FreeMemory = FALSE;           // tells dlist not to free any memory
    *Error      = DLIST_SUCCESS;   // tells dlist we were successful

    if (obj) {

        if ( obj->plugin  != BBR_PluginRecord_Ptr ) {

                return FALSE;

        }

    }

    return TRUE;           // tells dlist to prune the segment from the list
}


/*
 * Validate the objects in the source and target dlists in the context
 * record.  Remove from the selected objects lists any objects which are no
 * longer acceptable.  Return all acceptable objects in the parameter
 * dlists.  Also, for any object which is removed from the selected dlists,
 * or is otherwise not acceptable, create a declined_handle_t struct with
 * reason why not acceptable and add to the declined_objects dlist.
 */
static int get_acceptable_create_objects( task_context_t * context )
{
    int   rc = EINVAL;
    uint  count;

    LOGENTRY();

    rc = GetListSize(context->acceptable_objects, &count);
    if (rc) {
        count = 0;
    }

    if (count == 0) {


        rc = BBREngFncs->get_object_list( EVMS_OBJECT | SEGMENT | REGION | DISK,
                                          DATA_TYPE,
                                          NULL,
                                          VALID_INPUT_OBJECT,
                                          &context->acceptable_objects );

        if (rc == DLIST_SUCCESS) {

            PruneList(context->acceptable_objects, prune_noncreatable_objects, NULL);

        }

    }
    else {
        LOG_ERROR("context already has acceptable objects\n");
    }

    LOGEXITRC();
    return rc;
}




/*
 *  Called from get_acceptable_shrink_objects
 */
static BOOLEAN prune_unshrinkable_objects( ADDRESS   Object,
                                           TAG       ObjectTag,
                                           uint      ObjectSize,
                                           ADDRESS   ObjectHandle,
                                           ADDRESS   Parameters,
                                           BOOLEAN  *FreeMemory,
                                           uint     *Error )
{
    int                        rc;
    storage_object_t          *child=NULL;
    dlist_t                    shrink_points=CreateList();
    struct plugin_functions_s *fncs;


    *FreeMemory = FALSE;           // tells dlist not to free any memory
    *Error      = DLIST_SUCCESS;   // tells dlist we were successful

    if ( (Object)&&
         (shrink_points)) {

        child = GET_BBR_CHILD( ((storage_object_t *)Object) );

        if ( child ) {

            fncs = (struct plugin_functions_s *)child->plugin->functions.plugin;

            rc = fncs->can_shrink(child, NULL, shrink_points);
            if (rc == 0) {
                DestroyList(&shrink_points,FALSE);
                return FALSE;
            }
        }

        DestroyList(&shrink_points,FALSE);

    }

    return TRUE;           // tells dlist to prune the segment from the list
}


/*
 * Validate the objects in the source and target dlists in the context
 * record.  Remove from the selected objects lists any objects which are no
 * longer acceptable.  Return all acceptable objects in the parameter
 * dlists.  Also, for any object which is removed from the selected dlists,
 * or is otherwise not acceptable, create a declined_handle_t struct with
 * reason why not acceptable and add to the declined_objects dlist.
 */
static int get_acceptable_shrink_objects( task_context_t * context  )
{
    int   rc = EINVAL;
    uint  count;


    LOGENTRY();

    rc = GetListSize(context->acceptable_objects, &count);
    if (rc) {
        count = 0;
    }

    if (count == 0) {

        rc = BBREngFncs->get_object_list( EVMS_OBJECT,
                                          DATA_TYPE,
                                          BBR_PluginRecord_Ptr,
                                          VALID_INPUT_OBJECT,
                                          &context->acceptable_objects );

        if (rc == DLIST_SUCCESS) {

            PruneList(context->acceptable_objects, prune_unshrinkable_objects, NULL);

        }

    }
    else {
        LOG_ERROR("context already has acceptable objects\n");
    }


    LOGEXITRC();
    return rc;
}



/*
 * Validate the objects in the source and target dlists in the context
 * record.  Remove from the selected objects lists any objects which are no
 * longer acceptable.  Return all acceptable objects in the parameter
 * dlists.  Also, for any object which is removed from the selected dlists,
 * or is otherwise not acceptable, create a declined_handle_t struct with
 * reason why not acceptable and add to the declined_objects dlist.
 */
int BBR_InitTask( task_context_t * context )
{
    int rc = EINVAL;

    LOGENTRY();


    if (context) {

        switch (context->action) {

        case EVMS_Task_Create:

            context->min_selected_objects = 1;
            context->max_selected_objects = 1;

            rc = allocate_create_option_descriptors( context );
            if (rc == 0) {
                rc = get_acceptable_create_objects( context );
            }
            break;

        case EVMS_Task_Expand:

            // BBR is never an expansion point. It always passes expand
            // requests down to child objects who become expansion points.
            rc = EINVAL;
            break;

        case EVMS_Task_Shrink:

            context->min_selected_objects = 1;
            context->max_selected_objects = 1;

            rc = allocate_shrink_option_descriptors( context );
            if (rc == 0) {
                rc = get_acceptable_shrink_objects( context );
            }
            break;

        default:
               LOG_ERROR("context->action is unknown or unsupported\n");
               break;
        }
    }


    LOGEXITRC();
    return rc;
}


/*
 * Returns information about a BBR object
 */
int BBR_GetInfo( storage_object_t  * object, char *name,  extended_info_array_t * * info)
{
    int rc = EINVAL;
    extended_info_array_t   *Info;
    BBR_Private_Data        *pdata = (BBR_Private_Data *) object->private_data;


    LOGENTRY();

    // a measure of protection ...
    if ((info == NULL)||(pdata->signature != EVMS_BBR_SIGNATURE)) {
        LOGEXITRC();
        return rc;
    }

    rc    = ENOMEM;  // init to failed calloc
    *info = NULL;     // init to no info returned

    if ( object->object_type == EVMS_OBJECT ) {

        Info = BBREngFncs->engine_alloc( sizeof(extended_info_array_t) + (BBR_INFO_COUNT * sizeof(extended_info_t) ) );
        if (Info) {

            Info->count = BBR_INFO_COUNT;

            SET_STRING_FIELD( Info->info[BBR_INFO_NAME_INDEX].name, "Name" );
            SET_STRING_FIELD( Info->info[BBR_INFO_NAME_INDEX].title, "Name" );
            SET_STRING_FIELD( Info->info[BBR_INFO_NAME_INDEX].desc, "This is the name given to the storage object. It must be unique on the system.");
            Info->info[BBR_INFO_NAME_INDEX].type               = EVMS_Type_String;
            Info->info[BBR_INFO_NAME_INDEX].unit               = EVMS_Unit_None;
            SET_STRING_FIELD( Info->info[BBR_INFO_NAME_INDEX].value.s, object->name );
            Info->info[BBR_INFO_NAME_INDEX].collection_type    = EVMS_Collection_None;
            memset( &Info->info[BBR_INFO_NAME_INDEX].group, 0, sizeof(group_info_t));

            SET_STRING_FIELD( Info->info[BBR_INFO_SIZE_INDEX].name, "Size" );
            SET_STRING_FIELD( Info->info[BBR_INFO_SIZE_INDEX].title, "Size" );
            SET_STRING_FIELD( Info->info[BBR_INFO_SIZE_INDEX].desc, "This is the size of the storage object after reserving space for metadata.");
            Info->info[BBR_INFO_SIZE_INDEX].type               = EVMS_Type_Unsigned_Int64;
            Info->info[BBR_INFO_SIZE_INDEX].unit               = EVMS_Unit_None;
            Info->info[BBR_INFO_SIZE_INDEX].value.ui64         = object->size;
            Info->info[BBR_INFO_SIZE_INDEX].collection_type    = EVMS_Collection_None;
            memset( &Info->info[BBR_INFO_SIZE_INDEX].group, 0, sizeof(group_info_t));

            SET_STRING_FIELD( Info->info[BBR_INFO_BLKS_INDEX].name, "Reserve Blocks" );
            SET_STRING_FIELD( Info->info[BBR_INFO_BLKS_INDEX].title, "Blocks" );
            SET_STRING_FIELD( Info->info[BBR_INFO_BLKS_INDEX].desc, "This is the number of replacement blocks BBR is reserving for this storage object.");
            Info->info[BBR_INFO_BLKS_INDEX].type               = EVMS_Type_Unsigned_Int64;
            Info->info[BBR_INFO_BLKS_INDEX].unit               = EVMS_Unit_None;
            Info->info[BBR_INFO_BLKS_INDEX].value.ui64         = pdata->replacement_blocks_needed;
            Info->info[BBR_INFO_BLKS_INDEX].collection_type    = EVMS_Collection_None;
            memset( &Info->info[BBR_INFO_BLKS_INDEX].group, 0, sizeof(group_info_t));

            SET_STRING_FIELD( Info->info[BBR_INFO_BLKSZ_INDEX].name, "Block Size" );
            SET_STRING_FIELD( Info->info[BBR_INFO_BLKSZ_INDEX].title, "Block Size" );
            SET_STRING_FIELD( Info->info[BBR_INFO_BLKSZ_INDEX].desc, "This value tells you the size of a replacement block.");
            Info->info[BBR_INFO_BLKSZ_INDEX].type               = EVMS_Type_Unsigned_Int64;
            Info->info[BBR_INFO_BLKSZ_INDEX].unit               = EVMS_Unit_None;
            Info->info[BBR_INFO_BLKSZ_INDEX].value.ui64         = pdata->block_size;
            Info->info[BBR_INFO_BLKSZ_INDEX].collection_type    = EVMS_Collection_None;
            memset( &Info->info[BBR_INFO_BLKSZ_INDEX].group, 0, sizeof(group_info_t));

            SET_STRING_FIELD( Info->info[BBR_INFO_TBLSZ_INDEX].name, "Table Size" );
            SET_STRING_FIELD( Info->info[BBR_INFO_TBLSZ_INDEX].title, "Size of BBR table" );
            SET_STRING_FIELD( Info->info[BBR_INFO_TBLSZ_INDEX].desc, "This is the number of sectors being used by the BBR remapping table.");
            Info->info[BBR_INFO_TBLSZ_INDEX].type               = EVMS_Type_Unsigned_Int64;
            Info->info[BBR_INFO_TBLSZ_INDEX].unit               = EVMS_Unit_None;
            Info->info[BBR_INFO_TBLSZ_INDEX].value.ui64         = pdata->bbr_table_size_in_sectors;
            Info->info[BBR_INFO_TBLSZ_INDEX].collection_type    = EVMS_Collection_None;
            memset( &Info->info[BBR_INFO_TBLSZ_INDEX].group, 0, sizeof(group_info_t));

            *info = Info;

            rc = 0;
        }
        else {
            LOG_ERROR("unable to malloc memory for extended info array\n");
        }
    }

    LOGEXITRC();
    return rc;
}


/*
 *
 */
int BBR_GetPluginInfo( char * descriptor_name, extended_info_array_t * * info )
{
    int rc = EINVAL;
    extended_info_array_t   *Info;
    char                     version_string[64];
    char                     required_version_string[64];


    LOGENTRY();

    // a measure of protection ...
    if (info == NULL) {
        LOGEXITRC();
        return rc;
    }

    rc    = ENOMEM;  // init to failed calloc
    *info = NULL;     // init to no info returned

    Info = BBREngFncs->engine_alloc( sizeof(extended_info_array_t) + (BBR_PLUGIN_INFO_COUNT*sizeof(extended_info_t))  );
    if (Info) {

        Info->count = BBR_PLUGIN_INFO_COUNT;

        sprintf(version_string, "%d.%d.%d",
                MAJOR_VERSION,
                MINOR_VERSION,
                PATCH_LEVEL );

        sprintf(required_version_string, "%d.%d.%d",
                BBR_PluginRecord_Ptr->required_api_version.major,
                BBR_PluginRecord_Ptr->required_api_version.minor,
                BBR_PluginRecord_Ptr->required_api_version.patchlevel );

        SET_STRING_FIELD( Info->info[BBR_PLUGIN_INFO_SNAME_INDEX].name, "Short Name" );
        SET_STRING_FIELD( Info->info[BBR_PLUGIN_INFO_SNAME_INDEX].title, "Short Name" );
        SET_STRING_FIELD( Info->info[BBR_PLUGIN_INFO_SNAME_INDEX].desc, "A short name given to this plugin.");
        Info->info[BBR_PLUGIN_INFO_SNAME_INDEX].type               = EVMS_Type_String;
        Info->info[BBR_PLUGIN_INFO_SNAME_INDEX].unit               = EVMS_Unit_None;
        SET_STRING_FIELD( Info->info[BBR_PLUGIN_INFO_SNAME_INDEX].value.s, BBR_PluginRecord_Ptr->short_name );
        Info->info[BBR_PLUGIN_INFO_SNAME_INDEX].collection_type    = EVMS_Collection_None;
        memset( &Info->info[BBR_PLUGIN_INFO_SNAME_INDEX].group, 0, sizeof(group_info_t));

        SET_STRING_FIELD( Info->info[BBR_PLUGIN_INFO_LNAME_INDEX].name, "Long Name" );
        SET_STRING_FIELD( Info->info[BBR_PLUGIN_INFO_LNAME_INDEX].title, "Long Name" );
        SET_STRING_FIELD( Info->info[BBR_PLUGIN_INFO_LNAME_INDEX].desc, "A long name given to this plugin.");
        Info->info[BBR_PLUGIN_INFO_LNAME_INDEX].type               = EVMS_Type_String;
        Info->info[BBR_PLUGIN_INFO_LNAME_INDEX].unit               = EVMS_Unit_None;
        SET_STRING_FIELD( Info->info[BBR_PLUGIN_INFO_LNAME_INDEX].value.s, BBR_PluginRecord_Ptr->long_name );
        Info->info[BBR_PLUGIN_INFO_LNAME_INDEX].collection_type    = EVMS_Collection_None;
        memset( &Info->info[BBR_PLUGIN_INFO_LNAME_INDEX].group, 0, sizeof(group_info_t));

        SET_STRING_FIELD( Info->info[BBR_PLUGIN_INFO_TYPE_INDEX].name, "Type" );
        SET_STRING_FIELD( Info->info[BBR_PLUGIN_INFO_TYPE_INDEX].title, "Plugin Type" );
        SET_STRING_FIELD( Info->info[BBR_PLUGIN_INFO_TYPE_INDEX].desc, "There are various types of plugins; each responsible for some kind of storage object.");
        Info->info[BBR_PLUGIN_INFO_TYPE_INDEX].type               = EVMS_Type_String;
        Info->info[BBR_PLUGIN_INFO_TYPE_INDEX].unit               = EVMS_Unit_None;
        SET_STRING_FIELD( Info->info[BBR_PLUGIN_INFO_TYPE_INDEX].value.s, "EVMS Feature" );
        Info->info[BBR_PLUGIN_INFO_TYPE_INDEX].collection_type    = EVMS_Collection_None;
        memset( &Info->info[BBR_PLUGIN_INFO_TYPE_INDEX].group, 0, sizeof(group_info_t));

        SET_STRING_FIELD( Info->info[BBR_PLUGIN_INFO_VERSION_INDEX].name, "Version" );
        SET_STRING_FIELD( Info->info[BBR_PLUGIN_INFO_VERSION_INDEX].title, "Plugin Version" );
        SET_STRING_FIELD( Info->info[BBR_PLUGIN_INFO_VERSION_INDEX].desc, "This is the version number of the plugin.");
        Info->info[BBR_PLUGIN_INFO_VERSION_INDEX].type               = EVMS_Type_String;
        Info->info[BBR_PLUGIN_INFO_VERSION_INDEX].unit               = EVMS_Unit_None;
        SET_STRING_FIELD( Info->info[BBR_PLUGIN_INFO_VERSION_INDEX].value.s, version_string );
        Info->info[BBR_PLUGIN_INFO_VERSION_INDEX].collection_type    = EVMS_Collection_None;
        memset( &Info->info[BBR_PLUGIN_INFO_VERSION_INDEX].group, 0, sizeof(group_info_t));

        SET_STRING_FIELD( Info->info[BBR_PLUGIN_INFO_REQVERSION_INDEX].name, "Required Version" );
        SET_STRING_FIELD( Info->info[BBR_PLUGIN_INFO_REQVERSION_INDEX].title, "Required Version" );
        SET_STRING_FIELD( Info->info[BBR_PLUGIN_INFO_REQVERSION_INDEX].desc, "This is the version of the engine that the plugin requires. It will not run on older versions of the Engine.");
        Info->info[BBR_PLUGIN_INFO_REQVERSION_INDEX].type               = EVMS_Type_String;
        Info->info[BBR_PLUGIN_INFO_REQVERSION_INDEX].unit               = EVMS_Unit_None;
        SET_STRING_FIELD( Info->info[BBR_PLUGIN_INFO_REQVERSION_INDEX].value.s, required_version_string );
        Info->info[BBR_PLUGIN_INFO_REQVERSION_INDEX].collection_type    = EVMS_Collection_None;
        memset( &Info->info[BBR_PLUGIN_INFO_REQVERSION_INDEX].group, 0, sizeof(group_info_t));

        *info = Info;

        rc = 0;
    }


    LOGEXITRC();
    return rc;
}
