/*
 *
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: appAPI.h
 */

#ifndef EVMS_APP_API_H_INCLUDED
#define EVMS_APP_API_H_INCLUDED 1

#include <dlist.h>
#include <common.h>
#include <options.h>
#include <appstructs.h>

#define ENGINE_API_MAJOR_VERSION    4
#define ENGINE_API_MINOR_VERSION    0
#define ENGINE_API_PATCH_LEVEL      1

/*
 * Global notes:
 *
 * All the evms_can_?????() functions return 0 for "yes", otherwise they
 * return an error code that indicates why the operation is not allowed.
 *
 * All handle_array_t structures that are returned by APIs are allocated by
 * the Engine but it is up to the application to free them when they are no
 * longer needed.
 */

/******************************
* EVMS Engine Application API *
******************************/

#define DEFAULT_LOG_FILE        "/var/log/evmsEngine.log"

/*
 * Get the version of the API that this Engine supports.
 */
void evms_get_api_version(evms_version_t * version);

/*
 * Update the /dev/evms tree so that its nodes reflect the volumes that
 * are exported by the EVMS kernel.  This command will remove nodes and
 * directories that are no longer exported by the EVMS kernel.  It will
 * create/update the nodes in the tree so that there is a node with the
 * correct minor number for each volume exported by the EVMS kernel.
 * This API can be run without calling evms_open_engine() first.
 * The verbose_level says how much information to display.
 */

typedef enum {
    NO_MESSAGES = 0,    /* no output */
    ERRORS_ONLY,        /* errors only to stderr */
    ERRORS_AND_INFO     /* errors to stderr, information to stdout */
} evms_verbose_level_t;

int evms_update_evms_dev_tree(evms_verbose_level_t verbose_level);

/*
 * The EVMS Engine must be opened before it can do any work.
 * message_callbacks is used to register callback routines for messages that the
 * Engine or plug-ins may want to display.  If message_callbacks is NULL, no
 * callbacks will be registered.
 * If log_name is NULL, the Engine will use its default log file.
 */
int evms_open_engine(engine_mode_t                mode,
                     engine_message_functions_t * message_callbacks,
                     debug_level_t                level,
                     char                       * log_name);

/*
 * Get a list of plug-in handles.  If a type code is specified, the Engine will
 * return handles only for plug-ins of that type.  If the type code is 0, the
 * Engine will return handles for all the plug-ins that are installed.  See the
 * plugin_search_flags_t definitions in common.h
 */
int evms_get_plugin_list(evms_plugin_code_t    type,
                         plugin_search_flags_t flags,
                         handle_array_t    * * plugin_list);

/*
 * Get the handle for a plug-in with a given ID.
 */
int evms_get_plugin_by_ID(plugin_id_t       plugin_ID,
                          object_handle_t * plugin_handle);

/*
 * Get the handle for a plug-in with a given short name.
 */
int evms_get_plugin_by_name(char            * plugin_name,
                            object_handle_t * plugin_handle);

/*
 * Free a structure that was allocated and returned by the Engine,
 * such as a handle_array_t, a handle_object_info_t,
 * an extended_info_array_t, a declined_handle_array_t,
 * or an option_descriptor_t.
 */
void evms_free(void * buffer);

/*
 * Are there any changes pending in the Engine, i.e., does anything need to be
 * committed to disk?
 */
BOOLEAN evms_changes_pending();

/*
 * Rescan the system to determine the logical disk, disk segments, storage
 * containers, storage regions, storage objects, and volumes that are currently
 * active in the system.
 */
int evms_rediscover(void);

/*
 * Write an entry to the Engine's log file.
 */
int evms_write_log_entry(debug_level_t level,
                         char        * module_name,
                         char        * fmt,
                         ...);

/*
 * Get the Engine's current setting of the debug level.
 */
int evms_get_debug_level(debug_level_t * level);

/*
 * Set the Engine's debug level.
 */
int evms_set_debug_level(debug_level_t level);

/*
 * Get the EVMS kernel's current setting of its info level.
 */
int evms_get_kernel_info_level(debug_level_t * level);

/*
 * Set the EVMS kernel's info level.
 */
int evms_set_kernel_info_level(debug_level_t level);


/*
 * Commit any changes to disk.  Until this API is called, all changes are kept
 * in memory only.  If you close the Engine without calling
 * evms_commit_changes(), all changes will be lost and will not be applied to
 * the system.
 * The status_callback is an optional pointer to a function that the Engine will
 * call during various stages of the commit process with a status message.  You
 * can set status_callback to NULL if you do not want any status callbacks.
 */
int evms_commit_changes(commit_status_func_t status_callback);

/*
 * Close the EVMS Engine
 */
int evms_close_engine(void);

/********************************************
* Functions common to several object types, *
* e.g., volumes and storage objects         *
********************************************/

/*
 * evms_can_remove will determine the type of "thing" (volume, object, region,
 * container, segment, or logical disk).  If the thing is a volume, the Engine
 * will ask the feature that manages the top level object of the volume if it
 * can handle having the volume removed.  If the thing is a storage object, it
 * must be a top level storage object, i.e., it does not have a parent object.
 * If the thing is a top level storage object, the Engine will call the
 * can_delete() function of the feature that manages the storage object to ask
 * it if it can be removed.
 */
int evms_can_remove(object_handle_t thing);

/*
 * evms_can_destroy will determine the type of "thing" (volume, object, region,
 * container, segment, or logical disk) and will call the can_delete() function
 * of every object that comprises the thing.
 */
int evms_can_destroy(object_handle_t thing);

/*
 * evms_can_expand will determine if the "thing" can be expanded.  "thing"
 * must be a handle to a volume or to a top level object.
 */
int evms_can_expand(object_handle_t thing);

/*
 * evms_can_shrink will determine if the "thing" can be shrunk.  "thing"
 * must be a handle to a volume or to a top level object.
 */
int evms_can_shrink(object_handle_t thing);

/*
 * evms_can_move will determine the types of the source and target objects and
 * will find out if the source object can be moved to the target object.
 */
int evms_can_move(object_handle_t source,
                  object_handle_t target);

/*
 * evms_can_set_info will determine if a thing's information can be set.
 */
int evms_can_set_info(object_handle_t thing);


/*
 * evms_remove will remove the "thing".  If the "thing" is a volume, the volume
 * will be deleted leaving the underlying object by itself.  If the "thing" is
 * an object, the object will be deleted, leaving any child objects that the
 * thing had as free objects.
 */
int evms_remove(object_handle_t thing);

/* evms_destroy will determine the type of "thing" to be destroyed (volume,
 * object, region, container, segment, or logical disk) and will call the
 * corresponding delete() function for the thing and the delete() function on
 * all objects that comprise the thing.  All objects that comprise the thing
 * will be destroyed as well.  For example, an evms_destroy on a volume will
 * destroy all the objects that comprise the volume, and free the regions that
 * comprise the objects.  If the regions are backed directly by segments (i.e.
 * the regions are not created by a volume group manager) the segments will be
 * destroyed as well.
 */
int evms_destroy(object_handle_t thing);

/*
 * evms_get_expand_points is targeted at a "thing" which must be either a
 * volume or a top level object.  It is used to find which objects that comprise
 * "thing" can be expanded.  It returns an array of expand_handle_t entries,
 * each of which specifies the handle of an object that can be expanded and
 * the maximum size by which it can be expanded.  The UI can then select the
 * handle of the object to be expanded and use it to create an EVMS_Task_Expand
 * task, or to call evms_expand() directly if the UI knows what options to
 * supply to exms_expand().
 */
int evms_get_expand_points(object_handle_t           thing,
                           expand_handle_array_t * * expand_points);

/*
 * evms_expand will determine the type of "thing" to be expanded (volume, evms
 * object, region, or segment) and will call the corresponding expand function
 * for the thing.  The "objects" parameter is used by aggregating features that
 * expand by adding new objects to the aggregate.  "objects" should be set to
 * NULL for a "thing" that is not the product of an aggregating feature.
 */
int evms_expand(object_handle_t  thing,
                handle_array_t * objects,
                option_array_t * options);

/*
 * evms_get_shrink_points is targeted at a "thing" which must be either a
 * volume or a top level object.  It is used to find which objects that comprise
 * "thing" can be shrunk.  It returns an array of shrink_handle_t entries,
 * each of which specifies the handle of an object that can be shrunk and
 * the maximum size by which it can be shrunk.  The UI can then select the
 * handle of the object to be shrunk and use it to create an EVMS_Task_Shrink
 * task, or to call evms_shrink() directly if the UI knows what options to
 * supply to exms_shrink().
 */
int evms_get_shrink_points(object_handle_t           thing,
                           shrink_handle_array_t * * shrink_points);

/*
 * evms_shrink will determine the type of "thing" to be shrunk (volume, object,
 * region, or segment) and will call the corresponding shrink function for the
 * thing.  The "objects" parameter is used by aggregating features that shrink
 * by removing objects from the aggregate.  "objects" should be set to NULL for
 * a "thing" that is not the product of an aggregating feature.
 */
int evms_shrink(object_handle_t  thing,
                handle_array_t * objects,
                option_array_t * options);

/*
 * evms_move will determine the types of the source and target objects and will
 * move the contents of the source to the target.
 */
int evms_move(object_handle_t  source,
              object_handle_t  target);

/*
 * evms_get_handle_object_type returns the object type of the object with a
 * given handle.
 */

int evms_get_handle_object_type(object_handle_t handle,
                                object_type_t * type);

/*
 * evms_get_info will determine the type of "thing" (volume, object, region,
 * container, segment, or logical disk) and will return corresponding
 * information for the thing.
 */
int evms_get_info(object_handle_t          thing,
                  handle_object_info_t * * info);

/*
 * evms_get_extended_info will determine the type of "thing" (volume, object,
 * region, container, segment, logical disk or plug-in) and will return
 * corresponding plugin-specific information for the thing.  The information
 * returned is essentially a flat structure of basic data types.  Some of the
 * fields returned may have more information about that field.  To get the
 * additional information, call evms_get_extended_info() on the same object and
 * specify the name of the descriptor.  When the name parameter is NULL the
 * plug-in returns the basic level of extended information.
 */
int evms_get_extended_info(object_handle_t           thing,
                           char                    * descriptor_name,
                           extended_info_array_t * * info);

/*
 * evms_set_info will set information, specified in options, for a thing.
 */
int evms_set_info(object_handle_t  object,
                  option_array_t * options);

/*
 * Find the object in the given list that has the specified name and return a
 * handle to the object.
 */
int evms_get_object_handle_for_name(object_type_t     list,
                                    char            * name,
                                    object_handle_t * object_handle);

/*
 * Get a list of the handles of the feature plug-ins that are installed on this
 * volume or storage object.  The "thing" parameter can only be a handle for a
 * volume or for a storage object.  The order of the handles in the list does
 * not imply the order of the features used to build the volume or object.
 * Since the volume/object can actually have a tree of child objects, the scalar
 * list returned by this API has no way of indicating which features are
 * installed on which nodes of the object tree.
 */
int evms_get_feature_list(object_handle_t    thing,
                          handle_array_t * * plugin_list);

/*
 * TBD
 */
int evms_direct_plugin_communication(plugin_handle_t plugin,
                                     object_handle_t thing,
                                     BOOLEAN         target_kernel_plugin,
                                     void          * arg);


/***************************
* Storage object functions *
***************************/

/*
 * evms_create will create an object from the given list objects, using the
 * specified options.  A handle array to the new object(s) is returned in
 * output_objects.
 */
int evms_create(plugin_handle_t    plugin,
                handle_array_t   * input_objects,
                option_array_t   * options,
                handle_array_t * * output_objects);

/*
 * evms_transfer will transfer an object to a particular plug-in and/or
 * container.  If container is 0, the plug-in will assign the object to its
 * default or unassigned container.
 */
int evms_transfer(object_handle_t  object,
                  plugin_handle_t  plugin,
                  object_handle_t  container,
                  option_array_t * options);


/*
 * Get a list of handles for objects of the specified type.  See the
 * object_type_t, data_type_t, and object_search_flags_t enums in common.h.
 * If object_type is 0, objects of any type will be returned.  If data_type is
 * 0, objects of any data type will be returned.  If plugin is 0, objects
 * managed by any plug-in will be returned.
 */
int evms_get_object_list(object_type_t         object_type,
                         data_type_t           data_type,
                         plugin_handle_t       plugin,
                         object_search_flags_t flags,
                         handle_array_t    * * objects);

/*******************
* Volume functions *
*******************/

/*
 * Can this object be made into an EVMS volume?
 */
int evms_can_create_volume(object_handle_t object);

/*
 * Can this object be made into a compatibility volume?  The object must have no
 * features installed on it in order to be made into a compatibility volume.
 */
int evms_can_create_compatibility_volume(object_handle_t object);

/*
 * Can this volume have its name changed?
 */
int evms_can_set_volume_name(object_handle_t volume);

/*
 * Create an EVMS volume from this object.
 */
int evms_create_volume(object_handle_t object,
                       char          * name);

/*
 * Set the name of an EVMS volume.  (Compatibility volumes cannot have their
 * names changed.)
 */
int evms_set_volume_name(object_handle_t volume,
                         char          * name);

/*
 * Create a compatibility volume from this object.
 */
int evms_create_compatibility_volume(object_handle_t object);

/*
 * Can this file system be installed on this volume?
 */
int evms_can_mkfs(object_handle_t volume,
                  plugin_handle_t fsim);

/*
 * Can the file system be removed from the volume.
 */
int evms_can_unmkfs(object_handle_t volume);

/*
 * Can fsck be run on the volume?
 */
int evms_can_fsck(object_handle_t volume);

/*
 * Can the volume be defragmented?
 */
int evms_can_defrag(object_handle_t volume);

/*
 * Install a file system on a volume.
 */
int evms_mkfs(object_handle_t  volume,
              plugin_handle_t  fsim,
              option_array_t * options);

/*
 * Remove a file system from a volume.
 */
int evms_unmkfs(object_handle_t volume);

/*
 * Run fsck on a volume.
 */
int evms_fsck(object_handle_t  volume,
              option_array_t * options);

/*
 * Defragment a volume.
 */
int evms_defrag(object_handle_t  volume,
                option_array_t * options);

/*
 * Get a list of handles for the volumes that are in the Engine's current
 * working set.  If the object handle of an FSIM is specified, the Engine will
 * only return handles for volumes that are managed by that FSIM.  If the FSIM
 * parameter is 0, the Engine will return the handles for all of the volumes.
 */
int evms_get_volume_list(object_handle_t    fsim,
                         handle_array_t * * volume_list);


/******************************
* Storage Container functions *
******************************/

/*
 * Can this object be added to the specified container?
 */
int evms_can_add_to_container(object_handle_t object,
                              object_handle_t container);

/*
 * Can this object be removed from the container that currently consumes it?
 */
int evms_can_remove_from_container(object_handle_t object);

/*
 * Create a container for the given list objects, using the specified options.
 * A handle the new container is returned in output_container.
 */
int evms_create_container(plugin_handle_t   plugin,
                          handle_array_t  * input_objects,
                          option_array_t  * options,
                          object_handle_t * output_container);

/*
 * Get a list of handles for the containers that are in the Engine's current
 * working set.  If the object handle of a region manager is specified, the
 * Engine will only return handles for containers that are managed by that
 * region manager.  If the region_manager parameter is 0, the Engine will return
 * the handles for all of the containers.
 */
int evms_get_container_list(object_handle_t     plugin,
                            handle_array_t * * container_list);


/*************************
* Disk Segment functions *
*************************/


int evms_can_slide(object_handle_t segment,
                   int64_t         offset);

int evms_slide(object_handle_t  segment,
               option_array_t * options);


/*****************
* Task functions *
*****************/

int evms_create_task(engine_handle_t thing,
                     task_action_t   action,
                     task_handle_t * new_task_context);

int evms_invoke_task(task_handle_t      task,
                     handle_array_t * * resulting_objects);

int evms_destroy_task(task_handle_t task);

int evms_get_task_action (task_handle_t handle, task_action_t *action);

/************************
* Task object functions *
************************/

int evms_get_acceptable_objects(task_handle_t      task,
                                handle_array_t * * acceptable_object_list);

int evms_get_selected_objects(task_handle_t      task,
                              handle_array_t * * selected_object_list);

int evms_get_selected_object_limits(task_handle_t task,
                                    u_int32_t   * miminum,
                                    u_int32_t   * maximum);

int evms_set_selected_objects(task_handle_t               task,
                              handle_array_t            * selected_object_list,
                              declined_handle_array_t * * declined_list,
                              task_effect_t             * effect);


/************************
* Task option functions *
************************/

int evms_get_option_count(task_handle_t task,
                          int         * count);

int evms_get_option_descriptor(task_handle_t           task,
                               u_int32_t               option,
                               option_descriptor_t * * descriptor);

int evms_set_option_value(task_handle_t   task,
                          u_int32_t       option,
                          value_t       * value,
                          task_effect_t * effect);

int evms_get_option_descriptor_by_name(task_handle_t           task,
                                       const char            * option,
                                       option_descriptor_t * * descriptor);

int evms_set_option_value_by_name(task_handle_t   task,
                                  const char    * option_name,
                                  value_t       * value,
                                  task_effect_t * effect);

#endif
