/*
 *
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: views.c
 */ 

#include <frontend.h>
#include <gtk/gtk.h>

#include "support.h"
#include "readable.h"
#include "pixmap.h"
#include "volume_cb.h"
#include "destroy.h"
#include "revert.h"
#include "resize.h"
#include "setinfo.h"
#include "transfer.h"
#include "thing.h"
#include "fsutils.h"
#include "logging.h"
#include "views.h"

/*
 * The GtkCTree widget pointers for the various "views".
 */
 
static GtkCTree *view_ctree[MAX_VIEWS];

/*
 * Array of hash tables for each of the different views.
 *
 * Each hash table contains the GtkCTreeNode pointers
 * for the current unique and topmost nodes in one of the
 * GtkCTree widgets corresponding to a view. We use this
 * hash table in order to determine the expand/collapse
 * state of a node before destroying a view and rebuilding
 * it. That way, the rebuild has some knowledge of whether
 * the node was expanded or collapsed earlier. This hash
 * table also helps in quickly finding a parent node for
 * a child to be placed under.
 */

static GHashTable *ctree_hash_table[MAX_VIEWS];

/*
 *
 *   void set_browser_notebook_tab_pixmap (GtkWidget *, gchar *, object_type_t)
 *   
 *   Description:
 *      This routine does a lookup for the GtkHBox widget
 *      in the notebook tab contains the tab pixmaps. It
 *      sets the pixmap according to the object type.
 * 
 *   Entry:
 *      widget  - address of the GtkNotebook widget
 *      key     - the string used to retrieve the hbox widget
 *      type    - the object type
 *      topmost - TRUE if getting the topmost objects pixmap
 *
 *   Exit:
 *      The corresponding notebook tab icon is set.
 *
 */
void set_browser_notebook_tab_pixmap (GtkWidget *widget, gchar *key, object_type_t type, gboolean topmost)
{
    GdkBitmap *mask;
    GtkWidget *pixmap;
    GtkWidget *tab_hbox;
    GdkPixmap *gdk_pixmap;

    if ((tab_hbox = lookup_widget (widget, key)))
    {
        if (topmost)
            get_topmost_pixmap (&gdk_pixmap, &mask);
        else
            get_pixmap_for_object_type (type, &gdk_pixmap, &mask);
        
        pixmap = gtk_pixmap_new (gdk_pixmap, mask);
        gtk_widget_show (pixmap);
        gtk_box_pack_start (GTK_BOX (tab_hbox), pixmap, TRUE, TRUE, 0);
    }
}

/*
 *
 *   inline void set_ctree_column_checkbox (GtkCTree *, GtkCTreeNode *, 
 *                                          gint, gboolean, gboolean)
 *
 *   Description:
 *      This routine simply sees if the checked boolean
 *      is non-zero. If it is, it sets the pixmap
 *      for the given GtkCTreeNode to the checked pixmap.
 *      Otherwise, it sets it to the unchecked pixmap.
 * 
 *   Entry:
 *      ctree    - address of GtkCTree widget column and node are in
 *      node     - node corresponding to row to place pixmap in
 *      column   - GtkCTree column to place pixmap in
 *      critical - if TRUE and checked is TRUE, use the red checkmark
 *      checked  - if FALSE, use the unchecked pixmap otherwise 
 *                 use the checkmark pixmap
 *
 *   Exit:
 *      Returns nothing.
 *
 */
inline void set_ctree_column_checkbox (GtkCTree *ctree, GtkCTreeNode *node, 
                                       gint column, gboolean critical, 
                                       gboolean checked)
{
    GdkBitmap *mask;
    GdkPixmap *pixmap;
   
    get_checkmark_box_pixmap (critical, checked, &pixmap, &mask);
    gtk_ctree_node_set_pixmap (ctree, node, column, pixmap, mask);
}

/*
 *
 *   inline void change_ctree_properties (GtkCTree *)
 *
 *   Description:
 *      This routine simply updates properties of a
 *      a GtkCTree to set things that can't be set
 *      through Glade.
 * 
 *   Entry:
 *      ctree - address of GtkCTree widget to tweak
 *
 *   Exit:
 *      Returns nothing.
 *
 */
inline void change_ctree_properties (GtkCTree *ctree)
{
    gtk_ctree_set_line_style (ctree, GTK_CTREE_LINES_DOTTED);
    gtk_ctree_set_expander_style (ctree, GTK_CTREE_EXPANDER_CIRCULAR);

    /*
     * First column of tree has names so specify auto resize
     * to allow viewing the longest name to avoid having to
     * manually resize the column.
     *
     * Also, second column has size. Set the text justification
     * for the size column to right so it looks better (IMHO).
     * Auto resize this column as well.
     */

    gtk_clist_set_column_auto_resize (GTK_CLIST (ctree), 0, TRUE);
    gtk_clist_set_column_auto_resize (GTK_CLIST (ctree), 1, TRUE);
    gtk_clist_set_column_justification (GTK_CLIST (ctree), 1, GTK_JUSTIFY_RIGHT);
}

/*
 *
 *   inline void paint_pending_ctree_updates (GtkCTree *)
 *
 *   Description:
 *      This routine is usually called at certain
 *      points by the populate view routines to
 *      allow any updates they may have to occur to 
 *      avoid the blank/grey screens when processing
 *      hundreds of list/tree items.
 * 
 *   Entry:
 *      ctree - address of GtkCTree widget to repaint
 *
 *   Exit:
 *      Returns nothing.
 *
 */
inline void paint_pending_ctree_updates (GtkCTree *ctree)
{    
    gboolean clist_frozen = GTK_CLIST (ctree)->freeze_count == 0;
    
    if (clist_frozen) 
        gtk_clist_thaw (GTK_CLIST (ctree));
    
    while (gtk_events_pending ())
        gtk_main_iteration_do (FALSE);

    if (clist_frozen)
        gtk_clist_freeze (GTK_CLIST (ctree));
}

/*
 *
 *   inline void initialize_view (GtkCTree *, view_type_t)
 *
 *   Description:
 *      This routine handles some common initialization steps
 *      required by all the tree type views.
 * 
 *   Entry:
 *      ctree - address of GtkCTree widget for view
 *      view  - the id of the view
 *
 *   Exit:
 *      Returns nothing.
 *
 */
inline void initialize_view (GtkCTree *ctree, view_type_t view)
{
    change_ctree_properties (ctree);
    view_ctree[view] = ctree;    
    ctree_hash_table[view] = g_hash_table_new (g_direct_hash, g_direct_equal);        
}

/*
 *
 *   gchar * get_object_plugin_name (storage_object_info_t *)
 *
 *   Description:
 *      This routine retrieves the shortname of the
 *      plugin associated with a storage object.
 * 
 *   Entry:
 *      object - address of the storage_object_info_t 
 *
 *   Exit:
 *      Returns a dynamically allocated string
 *      containing the shortname of the plugin
 *      or NULL if unable to get the name.
 *
 */
gchar * get_object_plugin_name (storage_object_info_t *object)
{
    gint                  rc;
    gchar                *plugin_name=NULL;
    handle_object_info_t *plugin;

    rc = evms_get_info (object->plugin, &plugin);

    if (rc != SUCCESS)
    {
        log_error ("%s: evms_get_info() returned error %d.\n", __FUNCTION__, rc);        
    }
    else
    {
        plugin_name = g_strdup (plugin->info.plugin.short_name);
        evms_free (plugin);
    }

    return plugin_name;
}

/*
 *
 *   gchar **init_column_text_array (view_type_t)
 *
 *   Description:
 *      This routine initialized an array of pointers
 *      to point to empty strings according to the
 *      number needed for a particular view. 
 * 
 *   Entry:
 *      view - the view we need the string array for
 *
 *   Exit:
 *      Returns address of text array with pointers initialized to
 *      point to an empty string constant.
 *
 */
gchar **init_column_text_array (view_type_t view)
{
    static gchar *text[MAX_COLUMNS];

    switch (view)
    {
        default:
        case VOLUME_VIEW:
            text[8] = "";
        
        case SEGMENT_VIEW:        
            text[7] = "";

        case DISK_VIEW:
        case OBJECT_VIEW:
        case REGION_VIEW:                        
            text[6] = "";
        
        case PLUGIN_VIEW:
            text[5] = "";
        
        case CONTAINER_VIEW:
            text[4] = "";
            text[3] = "";
            text[2] = "";
            text[1] = "";
            text[0] = "";
            break;        
    }

    return text;
}

/*
 *
 *   GtkCTreeNode* insert_volume_into_ctree (GtkCTree *, GtkCTreeNode *, 
 *                                           logical_volume_info_t *,
 *                                           GdkPixmap *, GdkBitmap *, 
 *                                           gchar **, view_type_t)
 *
 *   Description:
 *      This routine inserts volume information into a GtkCTree
 *      and conforms the information displayed to a certain view.
 * 
 *   Entry:
 *      ctree  - the GtkCTree to insert this node in
 *      parent - the immediate parent GtkCTreeNode
 *      volume - volume information
 *      pixmap - pixmap corresponding to object type
 *      mask   - pixmap color mask
 *      text   - array of pointers for column text strings sized to current view
 *      view   - the view corresponding to the GtkCTree
 *
 *   Exit:
 *      Returns the GtkCTreeNode * for the volume inserted
 *
 */
GtkCTreeNode* insert_volume_into_ctree (GtkCTree *ctree, GtkCTreeNode *parent, 
                                        logical_volume_info_t *volume, 
                                        GdkPixmap *pixmap, GdkBitmap *mask, 
                                        gchar **text, view_type_t view)
{
    GtkCTreeNode *node=NULL;
    GdkPixmap    *open_volume_pixmap;
    GdkBitmap    *open_volume_pixmap_mask;

    switch (view)
    {
        case VOLUME_VIEW:
            text[VV_NAME_COLUMN]   = volume->name;
            text[VV_SIZE_COLUMN]   = make_sectors_readable_string (volume->vol_size);
            text[VV_PLUGIN_COLUMN] = get_volume_fsim_name (volume);
            text[VV_MINOR_COLUMN]  = g_strdup_printf ("%u", volume->minor_number);
        
            if (volume->mount_point)
                text[VV_MOUNTPOINT_COLUMN] = volume->mount_point;

            get_open_volume_pixmap (&open_volume_pixmap, &open_volume_pixmap_mask);

            node = gtk_ctree_insert_node (ctree, parent, NO_SIBLING, text, 
                                          COLUMN_SPACING, pixmap, mask,
                                          open_volume_pixmap, open_volume_pixmap_mask,
                                          FALSE, FALSE);

            set_ctree_column_checkbox (ctree, node, VV_DIRTY_COLUMN, 
                                       FALSE, (volume->flags & VOLFLAG_DIRTY));
            set_ctree_column_checkbox (ctree, node, VV_NEW_COLUMN, 
                                       FALSE, (volume->flags & VOLFLAG_NEW));
            set_ctree_column_checkbox (ctree, node, VV_NEEDS_DEVNODE_COLUMN, 
                                       FALSE, (volume->flags & VOLFLAG_NEEDS_DEV_NODE));
            set_ctree_column_checkbox (ctree, node, VV_READ_ONLY_COLUMN, 
                                       FALSE, (volume->flags & VOLFLAG_READ_ONLY));

            g_free (text[VV_SIZE_COLUMN]);
            g_free (text[VV_PLUGIN_COLUMN]);
            g_free (text[VV_MINOR_COLUMN]);

            break;
            
        default:
            log_warning ("%s: Unexpected view supplied: %d", __FUNCTION__, view);
            break;
    }

    return node;
}

/*
 *
 *   GtkCTreeNode* insert_storage_object_into_ctree (GtkCTree *, GtkCTreeNode *, 
 *                                                   storage_object_info_t *,
 *                                                   GdkPixmap *, GdkBitmap *, 
 *                                                   gchar **, view_type_t, 
 *                                                   object_type_t)
 *
 *   Description:
 *      This routine inserts a storage object, which is either a
 *      disk, segment, region, or feature object, into a GtkCTree
 *      and conforms the information displayed to a certain view.
 * 
 *   Entry:
 *      ctree  - the GtkCTree to insert this node in
 *      parent - the immediate parent GtkCTreeNode
 *      object - storage object information
 *      pixmap - pixmap corresponding to object type
 *      mask   - pixmap color mask
 *      text   - array of pointers for column text strings sized to current view
 *      view   - the view corresponding to the GtkCTree
 *      type   - the object type (DISK, SEGMENT, REGION, or EVMS_OBJECT)
 *
 *   Exit:
 *      Returns the GtkCTreeNode * for the storage object inserted
 *
 */
GtkCTreeNode* insert_storage_object_into_ctree (GtkCTree *ctree, GtkCTreeNode *parent, 
                                                storage_object_info_t *object, 
                                                GdkPixmap *pixmap, GdkBitmap *mask, 
                                                gchar **text, view_type_t view,
                                                object_type_t type)
{
    GtkCTreeNode *node;
        
    switch (view)
    {
        case VOLUME_VIEW:
            text[VV_NAME_COLUMN]   = object->name;
            text[VV_SIZE_COLUMN]   = make_sectors_readable_string (object->size);
            text[VV_PLUGIN_COLUMN] = get_object_plugin_name (object);
            break;

        case OBJECT_VIEW:
            text[OV_NAME_COLUMN]   = object->name;
            text[OV_SIZE_COLUMN]   = make_sectors_readable_string (object->size);
            text[OV_PLUGIN_COLUMN] = get_object_plugin_name (object);
            break;

        case REGION_VIEW:
            text[RV_NAME_COLUMN] = object->name;
            text[RV_SIZE_COLUMN] = make_sectors_readable_string (object->size);
            break;

        case CONTAINER_VIEW:
            text[CV_NAME_COLUMN] = object->name;
            text[CV_SIZE_COLUMN] = make_sectors_readable_string (object->size);
            break;

        case SEGMENT_VIEW:
            text[SV_NAME_COLUMN] = object->name;
            text[SV_SIZE_COLUMN] = make_sectors_readable_string (object->size);
        
            /*
             * Segments get extra information displayed in their view
             */
        
            if (type == SEGMENT)
            {
                text[SV_LBA_COLUMN]  = g_strdup_printf ("%llu", object->start);
                text[SV_TYPE_COLUMN] = make_data_type_readable_string (object->data_type);            
            }

            break;

        case DISK_VIEW:
            text[DV_NAME_COLUMN] = object->name;
            text[DV_SIZE_COLUMN] = make_sectors_readable_string (object->size);
        
            /*
             * Disks get extra information displayed in their view
             */
        
            if (type == DISK)
                text[DV_GEOMETRY_COLUMN] = g_strdup_printf ("(%llu,%u,%u)", object->geometry.cylinders,
                                                            object->geometry.heads,
                                                            object->geometry.sectors_per_track);
            
            break;
            
         default:
            log_warning ("%s: Unexpected view supplied: %d", __FUNCTION__, view);
            break;
    }

    node = gtk_ctree_insert_node (ctree, parent, NO_SIBLING, text, COLUMN_SPACING, 
                                  pixmap, mask, pixmap, mask, FALSE, FALSE);

    switch (view)
    {
        case VOLUME_VIEW:
            set_ctree_column_checkbox (ctree, node, VV_DIRTY_COLUMN, 
                                       FALSE, (object->flags & SOFLAG_DIRTY));
            set_ctree_column_checkbox (ctree, node, VV_READ_ONLY_COLUMN, 
                                       FALSE, (object->flags & SOFLAG_READ_ONLY));
        
            g_free (text[VV_PLUGIN_COLUMN]);
            g_free (text[VV_SIZE_COLUMN]);
            break;

        case OBJECT_VIEW:
            set_ctree_column_checkbox (ctree, node, OV_DIRTY_COLUMN, 
                                       FALSE, (object->flags & SOFLAG_DIRTY));
            set_ctree_column_checkbox (ctree, node, OV_CORRUPT_COLUMN,
                                       TRUE, (object->flags & SOFLAG_CORRUPT));
            set_ctree_column_checkbox (ctree, node, OV_READ_ONLY_COLUMN,
                                       FALSE, (object->flags & SOFLAG_READ_ONLY));        
            set_ctree_column_checkbox (ctree, node, OV_BIOS_READABLE_COLUMN, 
                                       FALSE, (object->flags & SOFLAG_BIOS_READABLE));

            g_free (text[OV_PLUGIN_COLUMN]);
            g_free (text[OV_SIZE_COLUMN]);
            break;

        case REGION_VIEW:
            set_ctree_column_checkbox (ctree, node, RV_ALLOCATED_COLUMN, 
                                       FALSE, (object->data_type == DATA_TYPE));
            set_ctree_column_checkbox (ctree, node, RV_DIRTY_COLUMN, 
                                       FALSE, (object->flags & SOFLAG_DIRTY));
            set_ctree_column_checkbox (ctree, node, RV_CORRUPT_COLUMN,
                                       TRUE, (object->flags & SOFLAG_CORRUPT));
            set_ctree_column_checkbox (ctree, node, RV_READ_ONLY_COLUMN,
                                       FALSE, (object->flags & SOFLAG_READ_ONLY));
            set_ctree_column_checkbox (ctree, node, RV_BIOS_READABLE_COLUMN, 
                                       FALSE, (object->flags & SOFLAG_BIOS_READABLE));

            g_free (text[RV_SIZE_COLUMN]);
            break;

        case CONTAINER_VIEW:
            set_ctree_column_checkbox (ctree, node, CV_DIRTY_COLUMN, 
                                       FALSE, (object->flags & SOFLAG_DIRTY));
            set_ctree_column_checkbox (ctree, node, CV_CORRUPT_COLUMN, 
                                       TRUE, (object->flags & SOFLAG_CORRUPT));
            set_ctree_column_checkbox (ctree, node, CV_READ_ONLY_COLUMN, 
                                       FALSE, (object->flags & SOFLAG_READ_ONLY));

            g_free (text[CV_SIZE_COLUMN]);
            break;

        case SEGMENT_VIEW:
            set_ctree_column_checkbox (ctree, node, SV_DIRTY_COLUMN, 
                                       FALSE, (object->flags & SOFLAG_DIRTY));
            set_ctree_column_checkbox (ctree, node, SV_CORRUPT_COLUMN, 
                                       TRUE, (object->flags & SOFLAG_CORRUPT));
            set_ctree_column_checkbox (ctree, node, SV_READ_ONLY_COLUMN, 
                                       FALSE, (object->flags & SOFLAG_READ_ONLY));
            set_ctree_column_checkbox (ctree, node, SV_BIOS_READABLE_COLUMN, 
                                       FALSE, (object->flags & SOFLAG_BIOS_READABLE));

            g_free (text[SV_SIZE_COLUMN]);
        
            if (type == SEGMENT)
                g_free (text[SV_LBA_COLUMN]);
            
            break;

        case DISK_VIEW:
            set_ctree_column_checkbox (ctree, node, DV_DIRTY_COLUMN, 
                                       FALSE, (object->flags & SOFLAG_DIRTY));
            set_ctree_column_checkbox (ctree, node, DV_CORRUPT_COLUMN,
                                       TRUE, (object->flags & SOFLAG_CORRUPT));        
            set_ctree_column_checkbox (ctree, node, DV_ERRORS_COLUMN,
                                       TRUE, (object->flags & SOFLAG_IO_ERROR));
            set_ctree_column_checkbox (ctree, node, DV_READ_ONLY_COLUMN,
                                       FALSE, (object->flags & SOFLAG_READ_ONLY));
        
            g_free (text[DV_SIZE_COLUMN]);
        
            if (type == DISK)
                g_free (text[DV_GEOMETRY_COLUMN]);
            
            break;
            
         default:
            break;
    }

    return node;
}

/*
 *
 *   GtkCTreeNode* insert_container_into_ctree (GtkCTree *, GtkCTreeNode *, 
 *                                              storage_container_info_t *,
 *                                              GdkPixmap *, GdkBitmap *, 
 *                                              gchar **, view_type_t)
 *
 *   Description:
 *      This routine inserts a storage container into a GtkCTree
 *      and conforms the information displayed to a certain view.
 * 
 *   Entry:
 *      ctree     - the GtkCTree to insert this node in
 *      parent    - the immediate parent GtkCTreeNode
 *      container - storage container information
 *      pixmap    - pixmap corresponding to object type
 *      mask      - pixmap color mask
 *      text      - array of pointers for column text strings sized to current view
 *      view      - the view corresponding to the GtkCTree
 *
 *   Exit:
 *      Returns the GtkCTreeNode * for the storage container inserted
 *
 */
GtkCTreeNode* insert_container_into_ctree (GtkCTree *ctree, GtkCTreeNode *parent, 
                                           storage_container_info_t *container, 
                                           GdkPixmap *pixmap, GdkBitmap *mask, 
                                           gchar **text, view_type_t view)
{
    GtkCTreeNode *node;

    /*
     * Yes, it does look rather stupid to have a switch statement with almost
     * identical code but this is here for future changes so there. It also
     * separates the views in case the columns get rearranged or we want 
     * more or less info for a new or existing view.
     */
    
    switch (view)
    {
        case REGION_VIEW:
            text[RV_NAME_COLUMN] = container->name;
            text[RV_SIZE_COLUMN] = make_sectors_readable_string (container->size);
            break;
        
        case CONTAINER_VIEW:
            text[CV_NAME_COLUMN] = container->name;
            text[CV_SIZE_COLUMN] = make_sectors_readable_string (container->size);
            break;
        
        default:
            log_warning ("%s: Unexpected view supplied: %d", __FUNCTION__, view);
            break;
    }
    
    node = gtk_ctree_insert_node (ctree, parent, NO_SIBLING, text, COLUMN_SPACING, 
                                  pixmap, mask, pixmap, mask, FALSE, FALSE);

    switch (view)
    {
        case REGION_VIEW:
            set_ctree_column_checkbox (ctree, node, RV_DIRTY_COLUMN, 
                                       FALSE, (container->flags & SCFLAG_DIRTY));
        
            g_free (text[RV_SIZE_COLUMN]);
            break;
        
        case CONTAINER_VIEW:
            set_ctree_column_checkbox (ctree, node, CV_DIRTY_COLUMN, 
                                       FALSE, (container->flags & SCFLAG_DIRTY));
        
            g_free (text[CV_SIZE_COLUMN]);
            break;
        
        default:
            break;
    }
    
    return node;
}

/*
 *
 *   GtkCTreeNode* insert_plugin_into_ctree (GtkCTree *, GtkCTreeNode *,
 *                                           plugin_info_t *, 
 *                                           GdkPixmap *, GdkBitmap *,
 *                                           gchar **, view_type_t)
 *
 *   Description:
 *      This routine inserts plugin information into a GtkCTree. At this
 *      time, it always gets placed in the first column on the root of
 *      the GtkCTree and it is expanded to show its contents.
 * 
 *   Entry:
 *      ctree  - the GtkCTree to insert this node into
 *      parent - address of parent GtkCTreeNode (ignored)
 *      plugin - plugin information
 *      pixmap - pixmap corresponding to object type
 *      mask   - pixmap color mask
 *      text   - array of pointers for column text strings sized to current view
 *      view   - the view corresponding to the GtkCTree
 *
 *   Exit:
 *      Returns the GtkCTreeNode * for the plugin inserted
 *
 */
GtkCTreeNode* insert_plugin_into_ctree (GtkCTree *ctree, GtkCTreeNode *parent,
                                        plugin_info_t *plugin, 
                                        GdkPixmap *pixmap, GdkBitmap *mask, 
                                        gchar **text, view_type_t view)
{
    GtkCTreeNode *node;
    
    text[NAME_COLUMN] = plugin->short_name;

    node = gtk_ctree_insert_node (ctree, NO_PARENT, NO_SIBLING, text, COLUMN_SPACING,
                                  pixmap, mask, pixmap, mask, FALSE, TRUE);
    
    return node;
}

/*
 *
 *   GtkCTreeNode* insert_object_into_ctree (GtkCTree *, GtkCTreeNode *, 
 *                                           object_handle_t, view_type_t)
 *
 *   Description:
 *      This routine inserts an object into a GtkCTree corresponding
 *      and conforming to a certain view. This is a high level routine
 *      that takes care of some common work and calls a subordinate
 *      routine to actually format the information for a particular
 *      object type within a particular view.
 * 
 *   Entry:
 *      ctree  - the GtkCTree to insert this node in
 *      parent - the immediate parent GtkCTreeNode
 *      handle - handle of object we want to insert
 *      view   - the view corresponding to the GtkCTree
 *
 *   Exit:
 *      Returns the GtkCTreeNode * for the object inserted
 *
 */
GtkCTreeNode* insert_object_into_ctree (GtkCTree *ctree, GtkCTreeNode *parent, 
                                        object_handle_t handle, view_type_t view)
{
    gint                  rc;
    GtkCTreeNode         *node = NULL;
    handle_object_info_t *object;

    rc = evms_get_info (handle, &object);

    if (rc != SUCCESS)
    {
        log_error ("%s: evms_get_info() returned error %d.\n", __FUNCTION__, rc);
    }
    else
    {  
        gchar    **text;
        GdkBitmap *mask;
        GdkPixmap *pixmap;

        text = init_column_text_array (view);

        get_pixmap_for_object_type (object->type, &pixmap, &mask);

        switch (object->type)
        {
            case DISK:            
            case REGION:
            case SEGMENT:
            case EVMS_OBJECT:                
                node = insert_storage_object_into_ctree (ctree, parent, &(object->info.object),
                                                         pixmap, mask, text, view, object->type);
                break;

            case VOLUME:
                node = insert_volume_into_ctree (ctree, parent, &(object->info.volume), 
                                                 pixmap, mask, text, view);
                break;

            case CONTAINER:
                node = insert_container_into_ctree (ctree, parent, &(object->info.container), 
                                                    pixmap, mask, text, view);
                break;

            case PLUGIN:
                node = insert_plugin_into_ctree (ctree, parent, &(object->info.plugin), 
                                                 pixmap, mask, text, view);
                break;

            default:
                log_error ("%s: Attempt to insert unknown type %d in the GtkCTree.\n", __FUNCTION__, object->type);
                break;
        }

        /*
         * Associate object handle with row data so that we can support a 
         * context popup menu of actions that can be done.
         */

        if (node)
            gtk_ctree_node_set_row_data (ctree, node, GUINT_TO_POINTER (handle));

        evms_free (object);
    }

    return node;
}

/*
 *
 *   void update_volume_in_ctree (GtkCTree *, GtkCTreeNode *, 
 *                                logical_volume_info_t *,
 *                                GdkPixmap *, GdkBitmap *, 
 *                                view_type_t)
 *
 *   Description:
 *      This routine updates volume information in a GtkCTree
 *      and conforms the information displayed to a certain view.
 * 
 *   Entry:
 *      ctree  - the GtkCTree the node is in
 *      node   - the volume's GtkCTreeNode
 *      volume - volume information
 *      pixmap - pixmap corresponding to object type
 *      mask   - pixmap color mask
 *      view   - the view corresponding to the GtkCTree
 *
 *   Exit:
 *      Updates the volume information
 *
 */
void update_volume_in_ctree (GtkCTree *ctree, GtkCTreeNode *node, 
                             logical_volume_info_t *volume, 
                             GdkPixmap *pixmap, GdkBitmap *mask, 
                             view_type_t view)
{
    gchar *text;
    
    switch (view)
    {
        case VOLUME_VIEW: {
                guint8     spacing;
                gboolean   is_leaf;
                gboolean   expanded;
                GdkPixmap *open_book_pixmap;
                GdkBitmap *open_book_mask;
                
                gtk_ctree_get_node_info (ctree, node, &text, &spacing, 
                                         &pixmap, &mask, 
                                         &open_book_pixmap, &open_book_mask,
                                         &is_leaf, &expanded);
                
                if (expanded)            
                {
                    /*
                     * The volume is expanded so it is using the open book
                     * pixmap, set pixmap and bitmap to these for our
                     * set_pixtext call below.
                     */
                    
                    mask = open_book_mask;                    
                    pixmap = open_book_pixmap;
                }
                
                gtk_ctree_node_set_pixtext (ctree, node, VV_NAME_COLUMN, volume->name,
                                            COLUMN_SPACING, pixmap, mask);
            
                text = make_sectors_readable_string (volume->vol_size);
                gtk_ctree_node_set_text (ctree, node, VV_SIZE_COLUMN, text);
                g_free (text);
                
                text = g_strdup_printf ("%u", volume->minor_number);
                gtk_ctree_node_set_text (ctree, node, VV_MINOR_COLUMN, text);
                g_free (text);        
    
                if (volume->mount_point)
                    text = volume->mount_point;
                else
                    text = "";
                    
                gtk_ctree_node_set_text (ctree, node, VV_MOUNTPOINT_COLUMN, text);
                
                set_ctree_column_checkbox (ctree, node, VV_DIRTY_COLUMN, 
                                           FALSE, (volume->flags & VOLFLAG_DIRTY));
                set_ctree_column_checkbox (ctree, node, VV_NEW_COLUMN, 
                                           FALSE, (volume->flags & VOLFLAG_NEW));
                set_ctree_column_checkbox (ctree, node, VV_NEEDS_DEVNODE_COLUMN, 
                                           FALSE, (volume->flags & VOLFLAG_NEEDS_DEV_NODE));
                set_ctree_column_checkbox (ctree, node, VV_READ_ONLY_COLUMN, 
                                           FALSE, (volume->flags & VOLFLAG_READ_ONLY));
            }

            break;
            
        default:
            log_warning ("%s: Unexpected view supplied: %d", __FUNCTION__, view);
            break;
    }
}

/*
 *
 *   void update_storage_object_in_ctree (GtkCTree *, GtkCTreeNode *, 
 *                                        storage_object_info_t *,
 *                                        GdkPixmap *, GdkBitmap *, 
 *                                        view_type_t, object_type_t)
 *
 *   Description:
 *      This routine updates a storage object, which is either a
 *      disk, segment, region, or feature object, in a GtkCTree
 *      and conforms the information displayed to a certain view.
 * 
 *   Entry:
 *      ctree  - the GtkCTree the node is in
 *      node   - the object's GtkCTreeNode
 *      object - storage object information
 *      pixmap - pixmap corresponding to object type
 *      mask   - pixmap color mask
 *      view   - the view corresponding to the GtkCTree
 *      type   - the object type (DISK, SEGMENT, REGION, or EVMS_OBJECT)
 *
 *   Exit:
 *      Updates the information for a storage object
 *
 */
void update_storage_object_in_ctree (GtkCTree *ctree, GtkCTreeNode *node, 
                                     storage_object_info_t *object, 
                                     GdkPixmap *pixmap, GdkBitmap *mask, 
                                     view_type_t view, object_type_t type)
{
    gchar *text;
        
    switch (view)
    {
        case VOLUME_VIEW:
            gtk_ctree_node_set_pixtext (ctree, node, VV_NAME_COLUMN, object->name,
                                        COLUMN_SPACING, pixmap, mask);
        
            text = make_sectors_readable_string (object->size);
            gtk_ctree_node_set_text (ctree, node, VV_SIZE_COLUMN, text);
            g_free (text);

            text = get_object_plugin_name (object);
            gtk_ctree_node_set_text (ctree, node, VV_PLUGIN_COLUMN, text);
            g_free (text);
        
            set_ctree_column_checkbox (ctree, node, VV_DIRTY_COLUMN, 
                                       FALSE, (object->flags & SOFLAG_DIRTY));
            set_ctree_column_checkbox (ctree, node, VV_READ_ONLY_COLUMN, 
                                       FALSE, (object->flags & SOFLAG_READ_ONLY));        
            break;

        case OBJECT_VIEW:
            gtk_ctree_node_set_pixtext (ctree, node, OV_NAME_COLUMN, object->name,
                                        COLUMN_SPACING, pixmap, mask);
        
            text = make_sectors_readable_string (object->size);
            gtk_ctree_node_set_text (ctree, node, OV_SIZE_COLUMN, text);
            g_free (text);

            text = get_object_plugin_name (object);
            gtk_ctree_node_set_text (ctree, node, OV_PLUGIN_COLUMN, text);
            g_free (text);
        
            set_ctree_column_checkbox (ctree, node, OV_DIRTY_COLUMN, 
                                       FALSE, (object->flags & SOFLAG_DIRTY));
            set_ctree_column_checkbox (ctree, node, OV_CORRUPT_COLUMN, 
                                       TRUE, (object->flags & SOFLAG_CORRUPT));
            set_ctree_column_checkbox (ctree, node, OV_READ_ONLY_COLUMN, 
                                       FALSE, (object->flags & SOFLAG_READ_ONLY));        
            set_ctree_column_checkbox (ctree, node, OV_BIOS_READABLE_COLUMN, 
                                       FALSE, (object->flags & SOFLAG_BIOS_READABLE));        
            break;

        case REGION_VIEW:
            gtk_ctree_node_set_pixtext (ctree, node, RV_NAME_COLUMN, object->name,
                                        COLUMN_SPACING, pixmap, mask);
            
            text = make_sectors_readable_string (object->size);
            gtk_ctree_node_set_text (ctree, node, RV_SIZE_COLUMN, text);
            g_free (text);
        
            set_ctree_column_checkbox (ctree, node, RV_ALLOCATED_COLUMN, 
                                       FALSE, (object->data_type == DATA_TYPE));
            set_ctree_column_checkbox (ctree, node, RV_DIRTY_COLUMN, 
                                       FALSE, (object->flags & SOFLAG_DIRTY));
            set_ctree_column_checkbox (ctree, node, RV_CORRUPT_COLUMN, 
                                       TRUE, (object->flags & SOFLAG_CORRUPT));
            set_ctree_column_checkbox (ctree, node, RV_READ_ONLY_COLUMN, 
                                       FALSE, (object->flags & SOFLAG_READ_ONLY));        
            set_ctree_column_checkbox (ctree, node, RV_BIOS_READABLE_COLUMN, 
                                       FALSE, (object->flags & SOFLAG_BIOS_READABLE));        
            break;

        case CONTAINER_VIEW:
            gtk_ctree_node_set_pixtext (ctree, node, CV_NAME_COLUMN, object->name,
                                        COLUMN_SPACING, pixmap, mask);
            
            text = make_sectors_readable_string (object->size);
            gtk_ctree_node_set_text (ctree, node, CV_SIZE_COLUMN, text);
            g_free (text);
        
            set_ctree_column_checkbox (ctree, node, CV_DIRTY_COLUMN, 
                                       FALSE, (object->flags & SOFLAG_DIRTY));
            set_ctree_column_checkbox (ctree, node, CV_CORRUPT_COLUMN, 
                                       TRUE, (object->flags & SOFLAG_CORRUPT));
            set_ctree_column_checkbox (ctree, node, CV_READ_ONLY_COLUMN, 
                                       FALSE, (object->flags & SOFLAG_READ_ONLY));
            break;

        case SEGMENT_VIEW:
            gtk_ctree_node_set_pixtext (ctree, node, SV_NAME_COLUMN, object->name,
                                        COLUMN_SPACING, pixmap, mask);
        
            text = make_sectors_readable_string (object->size);
            gtk_ctree_node_set_text (ctree, node, SV_SIZE_COLUMN, text);
            g_free (text);

            /*
             * Segments get extra information displayed in their view
             */
        
            if (type == SEGMENT)
            {
                text = g_strdup_printf ("%llu", object->start);
                gtk_ctree_node_set_text (ctree, node, SV_LBA_COLUMN, text);
                g_free (text);
                
                text = make_data_type_readable_string (object->data_type);
                gtk_ctree_node_set_text (ctree, node, SV_TYPE_COLUMN, text);            
            }
            
            set_ctree_column_checkbox (ctree, node, SV_DIRTY_COLUMN, 
                                       FALSE, (object->flags & SOFLAG_DIRTY));
            set_ctree_column_checkbox (ctree, node, SV_CORRUPT_COLUMN, 
                                       TRUE, (object->flags & SOFLAG_CORRUPT));
            set_ctree_column_checkbox (ctree, node, SV_READ_ONLY_COLUMN, 
                                       FALSE, (object->flags & SOFLAG_READ_ONLY));
            set_ctree_column_checkbox (ctree, node, SV_BIOS_READABLE_COLUMN, 
                                       FALSE, (object->flags & SOFLAG_BIOS_READABLE));            
            break;

        case DISK_VIEW:
            gtk_ctree_node_set_pixtext (ctree, node, DV_NAME_COLUMN, object->name,
                                        COLUMN_SPACING, pixmap, mask);
        
            text = make_sectors_readable_string (object->size);
            gtk_ctree_node_set_text (ctree, node, DV_SIZE_COLUMN, text);
            g_free (text);
        
            /*
             * Disks get extra information displayed in their view
             */
        
            if (type == DISK)
            {
                text = g_strdup_printf ("(%llu,%u,%u)", object->geometry.cylinders,
                                                        object->geometry.heads,
                                                        object->geometry.sectors_per_track);
                gtk_ctree_node_set_text (ctree, node, DV_GEOMETRY_COLUMN, text);
                g_free (text);        
            }
            
            set_ctree_column_checkbox (ctree, node, DV_DIRTY_COLUMN, 
                                       FALSE, (object->flags & SOFLAG_DIRTY));
            set_ctree_column_checkbox (ctree, node, DV_CORRUPT_COLUMN,
                                       TRUE, (object->flags & SOFLAG_CORRUPT));            
            set_ctree_column_checkbox (ctree, node, DV_ERRORS_COLUMN,
                                       TRUE, (object->flags & SOFLAG_IO_ERROR));
            set_ctree_column_checkbox (ctree, node, DV_READ_ONLY_COLUMN,
                                       FALSE, (object->flags & SOFLAG_READ_ONLY));            
            break;
            
         default:
            log_warning ("%s: Unexpected view supplied: %d", __FUNCTION__, view);
            break;
    }
}

/*
 *
 *   void update_container_in_ctree (GtkCTree *, GtkCTreeNode *, 
 *                                   storage_container_info_t *,
 *                                   GdkPixmap *, GdkBitmap *, 
 *                                   view_type_t)
 *
 *   Description:
 *      This routine updates a storage container in a GtkCTree
 *      and conforms the information displayed to a certain view.
 * 
 *   Entry:
 *      ctree     - the GtkCTree where the container is located
 *      node      - the container's GtkCTreeNode
 *      container - storage container information
 *      pixmap    - pixmap corresponding to object type
 *      mask      - pixmap color mask
 *      view      - the view corresponding to the GtkCTree
 *
 *   Exit:
 *      Updates the column text with the information from the container
 *
 */
void update_container_in_ctree (GtkCTree *ctree, GtkCTreeNode *node, 
                                storage_container_info_t *container, 
                                GdkPixmap *pixmap, GdkBitmap *mask, 
                                view_type_t view)
{
    gchar *text;
    
    switch (view)
    {
        case REGION_VIEW:        
            gtk_ctree_node_set_pixtext (ctree, node, RV_NAME_COLUMN, container->name,
                                        COLUMN_SPACING, pixmap, mask);
            
            text = make_sectors_readable_string (container->size);
            gtk_ctree_node_set_text (ctree, node, RV_SIZE_COLUMN, text);
            g_free (text);
        
            set_ctree_column_checkbox (ctree, node, RV_DIRTY_COLUMN, 
                                       FALSE, (container->flags & SCFLAG_DIRTY));
        
            break;
        
        case CONTAINER_VIEW:
            gtk_ctree_node_set_pixtext (ctree, node, CV_NAME_COLUMN, container->name,
                                        COLUMN_SPACING, pixmap, mask);
            
            text = make_sectors_readable_string (container->size);
            gtk_ctree_node_set_text (ctree, node, CV_SIZE_COLUMN, text);
            g_free (text);
        
            set_ctree_column_checkbox (ctree, node, CV_DIRTY_COLUMN, 
                                       FALSE, (container->flags & SCFLAG_DIRTY));
        
            break;
        
        default:
            log_warning ("%s: Unexpected view supplied: %d", __FUNCTION__, view);
            break;
    }    
}

/*
 *
 *   void update_plugin_in_ctree (GtkCTree *, GtkCTreeNode *, plugin_info_t *, view_type_t)
 *
 *   Description:
 *      This routine updates plugin information in a GtkCTree.
 * 
 *   Entry:
 *      ctree  - the GtkCTree the node is in
 *      node   - address of plugin's GtkCTreeNode
 *      plugin - plugin information
 *      pixmap - pixmap corresponding to object type
 *      mask   - pixmap color mask
 *      view   - the view corresponding to the GtkCTree
 *
 *   Exit:
 *      Updates the column text with the information from the plugin
 *
 */
void update_plugin_in_ctree (GtkCTree *ctree, GtkCTreeNode *node,
                             plugin_info_t *plugin,
                             GdkPixmap *pixmap, GdkBitmap *mask,                              
                             view_type_t view)
{    
    gtk_ctree_node_set_pixtext (ctree, node, NAME_COLUMN, plugin->short_name, 
                                COLUMN_SPACING, pixmap, mask);    
}

/*
 *
 *   void update_object_in_ctree (GtkCTree *, GtkCTreeNode *, object_handle_t, view_type_t)
 *
 *   Description:
 *      This routine updates an object in a GtkCTree corresponding
 *      and conforming to a certain view. This is a high level routine
 *      that takes care of some common work and calls a subordinate
 *      routine to actually format the information for a particular
 *      object type within a particular view.
 * 
 *   Entry:
 *      ctree  - the GtkCTree this node is in
 *      node   - GtkCTreeNode corresponding to the object to update
 *      handle - handle of object we want to update
 *      view   - the view corresponding to the GtkCTree
 *
 *   Exit:
 *      Update the text information in the view for the object
 *
 */
void update_object_in_ctree (GtkCTree *ctree, GtkCTreeNode *node, 
                             object_handle_t handle, view_type_t view)
{
    gint                  rc;
    handle_object_info_t *object;

    rc = evms_get_info (handle, &object);

    if (rc != SUCCESS)
    {
        log_error ("%s: evms_get_info() returned error %d.\n", __FUNCTION__, rc);
    }
    else
    {  
        GdkBitmap *mask;
        GdkPixmap *pixmap;
                
        get_pixmap_for_object_type (object->type, &pixmap, &mask);

        switch (object->type)
        {
            case DISK:            
            case REGION:
            case SEGMENT:
            case EVMS_OBJECT:                
                update_storage_object_in_ctree (ctree, node, &(object->info.object),
                                                pixmap, mask, view, object->type);
                break;

            case VOLUME:
                update_volume_in_ctree (ctree, node, &(object->info.volume), 
                                        pixmap, mask, view);
                break;

            case CONTAINER:
                update_container_in_ctree (ctree, node, &(object->info.container), 
                                           pixmap, mask, view);
                break;

            case PLUGIN:
                update_plugin_in_ctree (ctree, node, &(object->info.plugin), 
                                        pixmap, mask, view);
                break;

            default:
                log_error ("%s: Attempt to update unknown type %d in the GtkCTree.\n", 
                           __FUNCTION__, object->type);
                break;
        }

        evms_free (object);
    }
}

/*
 *
 *   void update_object_ctree_node (GtkCTree *, GtkCTreeNode *, gpointer)
 *
 *   Description:
 *      This routine is invoked by gtk_ctree_pre_recursive() when we
 *      have initiated a refresh of screen information of an object
 *      tree.
 * 
 *   Entry:
 *      ctree     - address of the GtkCTree widget for the current view
 *      node      - address of the GtkCTreeNode for the object to update
 *      user_data - identifies the view to know what text to update in which column
 *
 *   Exit:
 *      Updates the text for the object corresponding to the tree node 
 *      and object handle
 *
 */
void update_object_ctree_node (GtkCTree *ctree, GtkCTreeNode *node, gpointer user_data)
{
    view_type_t     view;
    object_handle_t handle;

    view = GPOINTER_TO_UINT (user_data);
    
    handle = GPOINTER_TO_UINT (gtk_ctree_node_get_row_data (ctree, node));
    
    if (handle)
        update_object_in_ctree (ctree, node, handle, view);
}

/*
 *
 *   GtkCTreeNode* populate_object_tree (GtkCTree *, GtkCTreeNode *, 
 *                                       object_handle_t, view_type_t)
 *
 *   Description:
 *      This routine calls evms_get_info() on the given handle.
 *      It inserts a new node in the ctree underneath its parent 
 *      node. If the handle info is of a storage object which
 *      indicates that there are additional object children, this
 *      function then calls itself to recursively populate this 
 *      subtree. If the handle is for an object that has no children
 *      then it simply displays it for the given view and returns.
 * 
 *   Entry:
 *      ctree   - address of the GtkCTree widget for the current view
 *      parent  - address of the GtkCTreeNode which is our parent node
 *      handle  - handle of the storage object in the tree
 *      view    - identifies the view to know what text to insert in which column
 *
 *   Exit:
 *      Inserts storage objects in their proper places in the GtkCTree
 *      provided and builds a node subtree as necessary. Returns the
 *      GtkCTreeNode * of the topmost object inserted (the first one).
 *
 */
GtkCTreeNode* populate_object_tree (GtkCTree *ctree, GtkCTreeNode *parent, 
                                    object_handle_t handle, view_type_t view)
{
    gint                  rc;
    GtkCTreeNode         *node=NULL;
    handle_object_info_t *object;

    rc = evms_get_info (handle, &object);

    if (rc != SUCCESS) 
    {
        log_error ("%s: evms_get_info() returned error %d.\n", __FUNCTION__, rc);
    }
    else
    {
        guint i;

        /*
         * First, insert the given node then loop through
         * each element in the array of child objects that
         * may comprise this object. For each handle call
         * populate_object_tree() recursively to insert
         * additional nodes as required.
         */

        node = insert_object_into_ctree (ctree, parent, handle, view);

        if (object->info.object.child_objects != NULL)
        {
            for (i=0; i < object->info.object.child_objects->count; i++)
            {
                populate_object_tree (ctree, node, object->info.object.child_objects->handle[i], view);
            }
        }
        
        evms_free (object);
    }
    
    return node;
}

/*
 *
 *   inline void set_node_expand_state (GtkCTree *, GtkCTreeNode *, object_handle_t, GHashTable *)
 *
 *   Description:
 *      This routine ensures that a given node is either expanded or collapsed 
 *      according to the state of its predecessor view.
 * 
 *   Entry:
 *      ctree  - the address of the GtkCTree widget
 *      node   - the address of the GtkCTreeNode for the object
 *      handle - the object's handle/key to hash table entry
 *      table  - address of refresh hash table
 *
 *   Exit:
 *      Returns nothing.
 *
 */
inline void set_node_expand_state (GtkCTree *ctree, GtkCTreeNode *node, object_handle_t handle, GHashTable *table)
{
    if (table)
    {
        guint state;
        
        state = GPOINTER_TO_UINT (g_hash_table_lookup (table, GUINT_TO_POINTER (handle)));
        
        if (state)
        { 
            if (state == NODE_IS_EXPANDED)
                gtk_ctree_expand (ctree, node);
            else if (state == NODE_IS_COLLAPSED)
                gtk_ctree_collapse (ctree, node);
        }
    }
}

/*
 *
 *   void remove_empty_plugin_hiearchy (GtkCTree *, evms_plugin_code_t, 
 *                                      plugin_search_flags_t, view_type_t)
 *
 *   Description:
 *      This routine removes a plugin at the top of a view hiearchy
 *      that had no nodes placed under it during the building of the
 *      view.
 * 
 *   Entry:
 *      ctree - the address of the GtkCTree widget
 *      type  - the type of plugins we want
 *      flags - additional search flags on the plugin type
 *      view  - the id of the view
 *
 *   Exit:
 *      Returns nothing.
 *
 */
void remove_empty_plugin_hiearchy (GtkCTree *ctree, evms_plugin_code_t type, 
                                   plugin_search_flags_t flags, view_type_t view)
{
    gint            rc;
    GHashTable     *table;
    handle_array_t *plugins; 
    
    table = ctree_hash_table[view];

    if (table)
    {    
        rc = evms_get_plugin_list (type, flags, &plugins);
        
        if (rc != SUCCESS) 
        {
            log_error ("%s: evms_get_plugin_list() returned error %d.\n", __FUNCTION__, rc);
        }
        else
        {   
            guint         i;
            GtkCTreeNode *node;
            GtkCTreeRow  *row;
            
            for (i=0; i < plugins->count; i++) 
            {            
                node = g_hash_table_lookup (table, GUINT_TO_POINTER (plugins->handle[i]));
             
                if (node)
                {
                    row = GTK_CTREE_ROW (node);

                    /*
                     * If we have no children then remove the ctree
                     * node from the view and remove the hash table
                     * entry as well.
                     */
                    
                    if (!(row->children))
                    {
                        gtk_ctree_remove_node (ctree, node);
                        g_hash_table_remove (table, GUINT_TO_POINTER (plugins->handle[i]));
                    }
                }
            }
                    
            evms_free (plugins);
        }
    }
}

/*
 *
 *   void populate_ctree_with_plugins (GtkCTree *, evms_plugin_code_t,
 *                                     plugin_search_flags_t, GHashTable *,
 *                                     view_type_t)
 *
 *   Description:
 *      This routine populates the root of the view with the names of given plugins.
 * 
 *   Entry:
 *      ctree         - the address of the GtkCTree widget
 *      type          - the type of plugins we want
 *      flags         - additional search flags on the plugin type
 *      refresh_table - address of refresh hash table containing expand/collapse
 *                      state for nodes that still exist
 *      view          - the id of the view
 *
 *   Exit:
 *      Returns nothing.
 *
 */
void populate_ctree_with_plugins (GtkCTree *ctree, evms_plugin_code_t type,
                                  plugin_search_flags_t flags, GHashTable *refresh_table,
                                  view_type_t view)
{
    gint            rc;
    handle_array_t *plugins; 

    rc = evms_get_plugin_list (type, flags, &plugins);
    
    if (rc != SUCCESS) 
    {
        log_error ("%s: evms_get_plugin_list() returned error %d.\n", __FUNCTION__, rc);
    }
    else
    {   
        guint         i;
        GtkCTreeNode *node;
        
        for (i=0; i < plugins->count; i++) 
        {            
            node = insert_object_into_ctree (ctree, NO_PARENT, plugins->handle[i], view);
                        
            if (node)
            {
                set_node_expand_state (ctree, node, plugins->handle[i], refresh_table);
                
                g_hash_table_insert (ctree_hash_table[view], GUINT_TO_POINTER (plugins->handle[i]), node);
            }
        }
                
        evms_free (plugins);
    }            
}

/*
 *
 *   void populate_plugin_view_with_plugins (GtkCList *clist)
 *
 *   Description:
 *      This routine populates the plugin view with all plugins
 *      that are currently loaded.
 * 
 *   Entry:
 *      clist - the address of the plugin view GtkCList widget
 *
 *   Exit:
 *      Returns nothing.
 *
 */
void populate_plugin_view_with_plugins (GtkCList *clist)
{
    gint            rc;
    handle_array_t *plugins; 
    
    rc = evms_get_plugin_list (EVMS_NO_PLUGIN, 0, &plugins);
    
    if (rc != SUCCESS) 
    {
        log_error ("%s: evms_get_plugin_list() returned error %d.\n", __FUNCTION__, rc);
    }
    else
    {
        guint                 i;
        guint                 row;
        gchar               **text;          
        handle_object_info_t *object;
    
        text = init_column_text_array (PLUGIN_VIEW);
        
        log_debug ("%s: evms_get_plugin_list() returned %d entries.\n", __FUNCTION__, plugins->count);
        
        for (i=0; i < plugins->count; i++)
        {
            rc = evms_get_info (plugins->handle[i], &object);
            
            if (rc != SUCCESS)
            {
                log_error ("%s: evms_get_info() returned error %d.\n", __FUNCTION__, rc);
            }
            else
            {                               
                text[PV_ID_COLUMN]          = g_strdup_printf ("%08X", object->info.plugin.id);
                text[PV_OEM_COLUMN]         = object->info.plugin.oem_name;
                text[PV_NAME_COLUMN]        = object->info.plugin.short_name;                
                text[PV_DESCRIPTION_COLUMN] = object->info.plugin.long_name;
                text[PV_VERSION_COLUMN]     = g_strdup_printf ("%u.%u.%u", 
                                                           object->info.plugin.version.major,
                                                           object->info.plugin.version.minor,
                                                           object->info.plugin.version.patchlevel);
                
                row = gtk_clist_append (clist, text);

                /*
                 * Associate plugin handle with row data so that we can
                 * support a popup context menu of actions that can be
                 * done.
                 */

                if (row != -1) 
                {
                    set_clist_row_pixmap (clist, row, object->type);
                    gtk_clist_set_row_data (clist, row, GUINT_TO_POINTER (plugins->handle[i]));
                }

                g_free (text[PV_ID_COLUMN]);
                g_free (text[PV_VERSION_COLUMN]);

                evms_free (object);
            }
        }

        evms_free (plugins);
    }    
}

/*
 *
 *   void populate_volume_view_with_volumes (GtkCTree *, GHashTable *)
 *
 *   Description:
 *      This routine populates the volume view with volumes and the
 *      objects that comprise it. 
 *
 *      Below each volume will be the tree of objects that comprise
 *      it.
 * 
 *   Entry:
 *      ctree         - the address of the volume view GtkCTree widget
 *      refresh_table - address of refresh hash table containing expand/collapse
 *                      state for nodes that still exist
 *
 *   Exit:
 *      Returns nothing.
 *
 */
void populate_volume_view_with_volumes (GtkCTree *ctree, GHashTable *refresh_table)
{
    gint            rc;
    handle_array_t *volumes; 
    
    rc = evms_get_volume_list (0, &volumes);
    
    if (rc != SUCCESS) 
    {
        log_error ("%s: evms_get_volume_list() returned error %d.\n", __FUNCTION__, rc);
    }
    else
    {
        guint                 i;
        GHashTable           *table=ctree_hash_table[VOLUME_VIEW];
        GtkCTreeNode         *node;
        handle_object_info_t *object;        
        
        log_debug ("%s: evms_get_volume_list() returned %d entries.\n", __FUNCTION__, volumes->count);
        
        for (i=0; i < volumes->count; i++)
        {
            rc = evms_get_info (volumes->handle[i], &object);
            
            if (rc != SUCCESS)
            {
                log_error ("%s: evms_get_info() returned error code %d.\n", __FUNCTION__, rc);
            }
            else
            {                                
                node = insert_object_into_ctree (ctree, NO_PARENT, volumes->handle[i], 
                                                 VOLUME_VIEW);
                if (node)
                {    
                    set_node_expand_state (ctree, node, volumes->handle[i], refresh_table);
                    
                    g_hash_table_insert (table, GUINT_TO_POINTER (volumes->handle[i]), node);                
   
                    populate_object_tree (ctree, node, object->info.volume.object, 
                                          VOLUME_VIEW);
                }
                
                evms_free (object);
            }           
        }

        evms_free (volumes);
    }    
}

/*
 *
 *   void populate_object_view_with_objects (GtkCTree *, GSList **)
 *
 *   Description:
 *      This routine populates the object view with topmost objects 
 *      and with the objects that comprise it. 
 * 
 *   Entry:
 *      ctree         - the address of the available object view GtkCTree widget
 *      refresh_table - address of refresh hash table containing expand/collapse
 *                      state for nodes that still exist
 *
 *   Exit:
 *      Returns nothing.
 *
 */
void populate_object_view_with_objects (GtkCTree *ctree, GHashTable *refresh_table)
{
    gint            rc;
    handle_array_t *objects; 
    
    rc = evms_get_object_list (0, DATA_TYPE, 0, TOPMOST, &objects);
    
    if (rc != SUCCESS) 
    {
        log_error ("%s: evms_get_object_list() returned error %d.\n", __FUNCTION__, rc);
    }
    else
    {
        guint         i;
        GHashTable   *table=ctree_hash_table[OBJECT_VIEW];
        GtkCTreeNode *node;
            
        log_debug ("%s: evms_get_object_list() returned %d entries.\n", __FUNCTION__, objects->count);
        
        for (i=0; i < objects->count; i++)
        {
            node = populate_object_tree (ctree, NO_PARENT, objects->handle[i], OBJECT_VIEW);
            
            if (node)
            {
                set_node_expand_state (ctree, node, objects->handle[i], refresh_table);
                
                g_hash_table_insert (table, GUINT_TO_POINTER (objects->handle[i]), node);
            }
        }

        evms_free (objects);
    }    
}

/*
 *
 *   void populate_feature_object_view_with_objects (GtkCTree *, GSList **)
 *
 *   Description:
 *      This routine populates the feature object view with feature objects 
 *      and with the objects that comprise it. 
 * 
 *   Entry:
 *      ctree         - the address of the feature object view GtkCTree widget
 *      refresh_table - address of refresh hash table containing expand/collapse
 *                      state for nodes that still exist
 *
 *   Exit:
 *      Returns nothing.
 *
 */
void populate_feature_object_view_with_objects (GtkCTree *ctree, GHashTable *refresh_table)
{
    gint            rc;
    handle_array_t *objects; 
    
    rc = evms_get_object_list (EVMS_OBJECT, DATA_TYPE, 0, 0, &objects);
    
    if (rc != SUCCESS) 
    {
        log_error ("%s: evms_get_object_list() returned error %d.\n", __FUNCTION__, rc);
    }
    else
    {
        guint         i;
        GHashTable   *table=ctree_hash_table[FEATURE_VIEW];
        GtkCTreeNode *node;
            
        log_debug ("%s: evms_get_object_list() returned %d entries.\n", __FUNCTION__, objects->count);
        
        for (i=0; i < objects->count; i++)
        {
            /*
             * Since the feature object view was cloned from the topmost object view
             * with just the "BIOS Readable" column hidden, populate the ctree using
             * the same code path that would be used for the available objects view.
             */
            node = populate_object_tree (ctree, NO_PARENT, objects->handle[i], OBJECT_VIEW);
            
            if (node)
            {
                set_node_expand_state (ctree, node, objects->handle[i], refresh_table);
                
                g_hash_table_insert (table, GUINT_TO_POINTER (objects->handle[i]), node);
            }
        }

        evms_free (objects);
    }    
}

/*
 *
 *   void populate_region_view_with_regions (GtkCTree *, GHashTable *)
 *
 *   Description:
 *      This routine populates the region view with regions. The regions
 *      are placed under their managing region manager plugin. The tree
 *      of child objects are also placed underneath the region.
 * 
 *   Entry:
 *      ctree         - address of the GtkCTree widget for the region view
 *      refresh_table - address of refresh hash table containing expand/collapse
 *                      state for nodes that still exist
 *
 *   Exit:
 *      Returns nothing and inserts regions into their proper places in
 *      the GtkCTree provided.
 *
 */
void populate_region_view_with_regions (GtkCTree *ctree, GHashTable *refresh_table)
{
    gint            rc;
    handle_array_t *regions; 
    
    rc = evms_get_object_list (REGION, 0, 0, 0, &regions);
    
    if (rc != SUCCESS) 
    {
        log_error ("%s: evms_get_object_list() returned error %d.\n", __FUNCTION__, rc);
    }
    else
    {
        guint                 i;
        GHashTable           *table=ctree_hash_table[REGION_VIEW];
        GtkCTreeNode         *node;
        GtkCTreeNode         *parent;
        handle_object_info_t *object;        
                    
        log_debug ("%s: evms_get_object_list() returned %d entries.\n", __FUNCTION__, regions->count);
        
        for (i=0; i < regions->count; i++)
        {
            rc = evms_get_info (regions->handle[i], &object);
            
            if (rc != SUCCESS)
            {
                log_error ("%s: evms_get_info() returned error %d.\n", __FUNCTION__, rc);
            }
            else
            {                
                parent = g_hash_table_lookup (table, GUINT_TO_POINTER (object->info.region.plugin));
                        
                node = populate_object_tree (ctree, parent, regions->handle[i], REGION_VIEW);
             
                if (node)
                {
                    set_node_expand_state (ctree, node, regions->handle[i], refresh_table);
                    
                    g_hash_table_insert (table, GUINT_TO_POINTER (regions->handle[i]), node);
                }
                
                evms_free (object);
            }            
        }

        evms_free (regions);
    }    
}

/*
 *
 *   void populate_container_with_objects_produced (GtkCTree *, GtkCTreeNode *, 
 *                                                  handle_array_t *, GHashTable *)
 *
 *   Description:
 *      This routine places objects produced by a container under
 *      the node for the parent container.
 * 
 *   Entry:
 *      ctree         - address of the GtkCTree widget where the parent
 *                      container is located
 *      parent        - address of the GtkCTreeNode of the parent tree node
 *      objects       - address of the handle array containing the object
 *                      handles of objects produced by the container
 *      table         - address of hash table to track node expand state
 *
 *   Exit:
 *      Returns nothing and inserts objects produced from a container into
 *      their proper places in the GtkCTree provided.
 *
 */
void populate_container_with_objects_produced (GtkCTree *ctree, GtkCTreeNode *parent, 
                                               handle_array_t *objects, GHashTable *table)
{            
    gint          i;
    GtkCTreeNode *node;
            
    for (i=0; i < objects->count; i++) 
    {
        node = populate_object_tree (ctree, parent, objects->handle[i], 
                                     CONTAINER_VIEW);                    
        if (node)
        {
            set_node_expand_state (ctree, node, objects->handle[i], table);
            
            g_hash_table_insert (ctree_hash_table[CONTAINER_VIEW], 
                                 GUINT_TO_POINTER (objects->handle[i]), node);
        }
    }
}

/*
 *
 *   void populate_container_with_objects_consumed (GtkCTree *, GtkCTreeNode *, 
 *                                                  handle_array_t *, GHashTable *)
 *
 *   Description:
 *      This routine places objects consumed by a container under
 *      the node for the parent container.
 * 
 *   Entry:
 *      ctree        - address of the GtkCTree widget where the parent
 *                     container is located
 *      parent       - address of the GtkCTreeNode of the parent tree node
 *      objects      - address of the handle array containing the object
 *                     handles of objects consumed by the container
 *      table        - address of hash table to track node expand state
 *
 *   Exit:
 *      Returns nothing and inserts objects that comprise a container into
 *      their proper places in the GtkCTree provided.
 *
 */
void populate_container_with_objects_consumed (GtkCTree *ctree, GtkCTreeNode *parent, 
                                               handle_array_t *objects, GHashTable *table)
{            
    if (objects->count > 0)
    {    
        gint          i;
        gchar       **text;        
        GtkCTreeNode *node;
        GtkCTreeNode *label_node;
        
        text = init_column_text_array (CONTAINER_VIEW);
        
        text[CV_NAME_COLUMN] = _("Objects Consumed...");
        
        label_node = gtk_ctree_insert_node (ctree, parent, NO_SIBLING, text, 
                                            COLUMN_SPACING, NULL, NULL, 
                                            NULL, NULL, FALSE, FALSE);
                    
        for (i=0; i < objects->count; i++) 
        {            
            node = populate_object_tree (ctree, label_node, objects->handle[i], 
                                         CONTAINER_VIEW);
                
            if (node)
            {
                set_node_expand_state (ctree, node, objects->handle[i], table);
                
                g_hash_table_insert (ctree_hash_table[CONTAINER_VIEW], 
                                     GUINT_TO_POINTER (objects->handle[i]), node);
            }
        }
    }
}

/*
 *
 *   void populate_ctree_with_containers (GtkCTree *, GHashTable *, view_type_t view)
 *
 *   Description:
 *      This routine populates the region or container view with containers.
 *      If this is the container view then the we also populate the container
 *      with the objects it produces and consumes.
 * 
 *   Entry:
 *      ctree         - the address of the view GtkCTree widget
 *      refresh_table - address of refresh hash table containing expand/collapse
 *                      state for nodes that still exist
 *      view          - whether this is the REGION_VIEW or CONTAINER_VIEW
 *
 *   Exit:
 *      Returns nothing.
 *
 */
void populate_ctree_with_containers (GtkCTree *ctree, GHashTable *refresh_table, 
                                     view_type_t view)
{
    gint            rc;
    handle_array_t *containers;
    
    rc = evms_get_container_list (0, &containers);
    
    if (rc != SUCCESS) 
    {
        log_error ("%s: evms_get_container_list() returned error %d.\n", __FUNCTION__, rc);
    }
    else
    {
        guint                 i;
        GHashTable           *table=ctree_hash_table[view];
        GtkCTreeNode         *node;
        GtkCTreeNode         *parent;
        handle_object_info_t *object;
        
        log_debug ("%s: evms_get_container_list() returned %d entries.\n", __FUNCTION__, containers->count);
        
        for (i=0; i < containers->count; i++)
        {            
            rc = evms_get_info (containers->handle[i], &object);
            
            if (rc != SUCCESS)
            {
                log_error ("%s: evms_get_info() returned error %d.\n", __FUNCTION__, rc);
            }
            else
            {   
                log_debug ("%s: Container handle is %#x\n", __FUNCTION__, containers->handle[i]);
                
                /* 
                 * First check that the plugin for this container is in the view. If
                 * not then don't bother showing it.
                 */
                
                parent = g_hash_table_lookup (table, GUINT_TO_POINTER (object->info.container.plugin));
                
                if (parent)
                {                    
                    node = insert_object_into_ctree (ctree, parent, containers->handle[i], view);
                    
                    if (node)
                    {    
                        /*
                         * Display the storage objects the container consumes and produces if
                         * this is the container view.
                         */
                         
                        if (view == CONTAINER_VIEW)
                        {                                                                 
                            populate_container_with_objects_consumed (ctree, node, 
                                               object->info.container.objects_consumed, 
                                               refresh_table); 
                            
                            populate_container_with_objects_produced (ctree, node,
                                               object->info.container.objects_produced, 
                                               refresh_table);                        
                        }
                        
                        set_node_expand_state (ctree, node, containers->handle[i], refresh_table);
        
                        g_hash_table_insert (table, GUINT_TO_POINTER (containers->handle[i]), node);
                    }
                }
                
                evms_free (object);
            }
        }

        evms_free (containers);
    }
}

/*
 *
 *   void populate_segment_view_with_segments (GtkCTree *, GHashTable *)
 *
 *   Description:
 *      This routine populates the view of segments and groups them by 
 *      segment manager. 
 * 
 *   Entry:
 *      ctree         - the address of the segment view GtkCTree widget
 *      refresh_table - address of refresh hash table containing expand/collapse
 *                      state for nodes that still exist
 *
 *   Exit:
 *      Returns nothing.
 *
 */
void populate_segment_view_with_segments (GtkCTree *ctree, GHashTable *refresh_table)
{
    gint            rc;
    handle_array_t *segments;
    
    rc = evms_get_object_list (SEGMENT, 0, 0, 0, &segments);        
                 
    if (rc != SUCCESS) 
    {
        log_error ("%s: evms_get_object_list() returned error %d.\n", __FUNCTION__, rc);
    }
    else
    {
        guint                 i;
        GHashTable           *table=ctree_hash_table[SEGMENT_VIEW];
        GtkCTreeNode         *node;
        GtkCTreeNode         *parent;        
        handle_object_info_t *object;        

        log_debug ("%s: evms_get_object_list() returned %d entries.\n", __FUNCTION__, segments->count);
        
        for (i=0; i < segments->count; i++) 
        {
            rc = evms_get_info (segments->handle[i], &object);
            
            if (rc != SUCCESS)
            {
                log_error ("%s: evms_get_info() returned error %d.\n", __FUNCTION__, rc);
            }
            else
            {
                parent = g_hash_table_lookup (table, GUINT_TO_POINTER (object->info.segment.plugin));

                node = populate_object_tree (ctree, parent, segments->handle[i], SEGMENT_VIEW);
             
                if (node)
                {
                    set_node_expand_state (ctree, node, segments->handle[i], refresh_table);
                    
                    g_hash_table_insert (table, GUINT_TO_POINTER (segments->handle[i]), node);
                }
                
                evms_free (object);
            }                        
        }

        evms_free (segments);
    }
}

/*
 *
 *   void populate_disk_view_with_disks (GtkCTree *, GHashTable *)
 *
 *   Description:
 *      This routine populates the view of logical disks and groups them
 *      by device manager.
 * 
 *   Entry:
 *      ctree         - the address of the disk view GtkCTree widget
 *      refresh_table - address of refresh hash table containing expand/collapse
 *                      state for nodes that still exist
 *
 *   Exit:
 *      Returns nothing.
 *
 */
void populate_disk_view_with_disks (GtkCTree *ctree, GHashTable *refresh_table)
{
    gint            rc;
    handle_array_t *disks;
    
    rc = evms_get_object_list (DISK, 0, 0, 0, &disks);        
                 
    if (rc != SUCCESS) 
    {
        log_error ("%s: evms_get_object_list() returned error %d.\n", __FUNCTION__, rc);
    }
    else
    {
        guint                 i;
        guint                 j;
        GHashTable           *table=ctree_hash_table[DISK_VIEW]; 
        GtkCTreeNode         *node;
        GtkCTreeNode         *parent;        
        handle_array_t       *users;
        handle_object_info_t *object;        

        log_debug ("%s: evms_get_object_list() returned %d entries.\n", __FUNCTION__, disks->count);
        
        for (i=0; i < disks->count; i++) 
        {            
            rc = evms_get_info (disks->handle[i], &object);
            
            if (rc != SUCCESS)
            {
                log_error ("%s: evms_get_info() returned error %d.\n", __FUNCTION__, rc);
            }
            else
            {
                parent = g_hash_table_lookup (table, GUINT_TO_POINTER (object->info.disk.plugin));
                                        
                node = insert_object_into_ctree (ctree, parent, disks->handle[i], DISK_VIEW);
                
                if (node)
                {    
                    /*
                     * Underneath the logical disk, place the parent_objects which
                     * are the immediate users of the disk. Typically, we'll see
                     * these objects to be segments but it is possible they are
                     * feature objects or regions.
                     */
                    
                    users = object->info.disk.parent_objects;
                    
                    for (j=0; j < users->count; j++)
                    {
                        insert_object_into_ctree (ctree, node, users->handle[j], DISK_VIEW);
                    }
                    
                    set_node_expand_state (ctree, node, disks->handle[i], refresh_table);
                        
                    g_hash_table_insert (table, GUINT_TO_POINTER (disks->handle[i]), node); 
                }
                
                evms_free (object);
            }
        }

        evms_free (disks);
    }
}

void yet_to_be_implemented_menu_item (GtkMenuItem *menuitem, gpointer user_data)
{
    display_not_implemented_yet_popup ();
}

/*
 *
 *   void on_jump_to_object_menu_item_activate (GtkMenuItem *, gpointer)
 *
 *   Description:
 *      This routine jumps to the object in the view corresponding to the
 *      object's type.
 * 
 *   Entry:
 *      menuitem  - the menu item that was activated
 *      user_data - contains the handle of the object we want to go to
 *
 *   Exit:
 *      Returns nothing.
 *
 */
void on_jump_to_object_menu_item_activate (GtkMenuItem *menuitem, gpointer user_data)
{
    object_type_t   type;    
    object_handle_t handle = GPOINTER_TO_UINT (user_data);

    if (evms_get_handle_object_type (handle, &type) == SUCCESS)
    {
        view_type_t   view;
        GtkCTree     *ctree;        
        GtkCTreeNode *node;
                
        view = set_current_view_by_object_type (type);
                
        ctree = view_ctree[view];

        if (ctree)
        {    
            node = g_hash_table_lookup (ctree_hash_table[view], GUINT_TO_POINTER (handle));
            
            if (node)
            {
                GtkCTreeRow  *ctree_row = GTK_CTREE_ROW (node);
                GtkCTreeNode *parent;
                        
                /*
                 * Expand the ancestors of the node, move it into view and
                 * select it.
                 */
                
                while (ctree_row->parent)
                {
                    parent = ctree_row->parent;
                    ctree_row = GTK_CTREE_ROW (parent);
                    
                    if (!ctree_row->expanded)
                        gtk_ctree_expand (ctree, parent);
                }    
                
                gtk_ctree_node_moveto (ctree, node, 0, 0.0, 0.0);
                gtk_ctree_select (ctree, node);
            }
        }
    }
}

/*
 *
 *   inline void get_object_handles (object_handle_t, object_handle_t *, 
 *                                   object_handle_t *, object_handle_t *)
 *
 *   Description:
 *      This routine returns the handles for an object's parent, producing
 *      and consuming containers. If the object is topmost (no parent) and
 *      has a volume then the volume handle is returned as the parent handle.
 * 
 *   Entry:
 *      handle              - handle to the object
 *      parent              - address where to store the parent's object handle
 *      producing_container - address where to store the producing container handle
 *      consuming_container - address where to store the consuming container handle
 *
 *   Exit:
 *      Returns the object's parent and container handles.
 *
 */
inline void get_object_handles (object_handle_t handle, object_handle_t *parent,
                                object_handle_t *producing_container,
                                object_handle_t *consuming_container)
{
    gint                  rc;
    handle_object_info_t *object;

    *parent = 0;
    *producing_container = 0;
    *consuming_container = 0;
    
    rc = evms_get_info (handle, &object);

    if (rc == SUCCESS)
    {
        switch (object->type)
        {
            case DISK:
            case SEGMENT:
            case REGION:
            case EVMS_OBJECT:
                *producing_container = object->info.object.producing_container;
                *consuming_container = object->info.object.consuming_container;
            
                if (object->info.object.parent_objects)
                {
                    if (object->info.object.parent_objects->count == 1)
                    {    
                        *parent = object->info.object.parent_objects->handle[0];
                    }
                    else if (object->info.object.parent_objects->count == 0 &&
                             object->info.object.volume != 0)
                    {
                        *parent = object->info.object.volume;
                    }
                }
                else
                {
                    if (object->info.object.volume != 0)
                        *parent = object->info.object.volume;
                }
                
                break;

            default:
                break;
        }

        evms_free (object);
    }
}

/*
 *
 *   void display_action_popup_menu_for_handle (object_handle_t, GdkEventButton *)
 *
 *   Description:
 *      This routine gets invoked when there is a "button-press-event" in any
 *      of the information viewer GtkCTree or GtkCList views. We display a 
 *      context popup of actions that can be done with the row data.
 * 
 *   Entry:
 *      handle - object handle associated with row popup is for
 *      event  - everything we want to know about the event
 *
 *   Exit:
 *      Creates a context sensitive popup menu populated with
 *      actions suitable for the row data the user clicked on.
 *
 */
void display_action_popup_menu_for_handle (object_handle_t handle, GdkEventButton *event)
{
    object_type_t type;

    if (evms_get_handle_object_type (handle, &type) == SUCCESS)
    {
        GtkMenu   *menu;
        GtkWidget *item;

        menu = GTK_MENU (gtk_menu_new ());

        /*
         * Display "navigation" menu items first
         */
        
        if (type != PLUGIN && type != VOLUME && type != CONTAINER)
        {
            object_handle_t parent;
            object_handle_t consuming_container;
            object_handle_t producing_container;
            
            get_object_handles (handle, &parent, &producing_container, &consuming_container);
            
            if (parent)
            {
                item = gtk_menu_item_new_with_label (_("Go to parent object"));
                gtk_menu_append (menu, item);
                gtk_signal_connect (GTK_OBJECT (item), "activate",
                                    GTK_SIGNAL_FUNC (on_jump_to_object_menu_item_activate),
                                    GUINT_TO_POINTER (parent));                
            }
            
            if (producing_container)
            {
                item = gtk_menu_item_new_with_label (_("Go to producing container"));
                gtk_menu_append (menu, item);
                gtk_signal_connect (GTK_OBJECT (item), "activate",
                                    GTK_SIGNAL_FUNC (on_jump_to_object_menu_item_activate),
                                    GUINT_TO_POINTER (producing_container));                
            }

            if (consuming_container)
            {
                item = gtk_menu_item_new_with_label (_("Go to consuming container"));
                gtk_menu_append (menu, item);
                gtk_signal_connect (GTK_OBJECT (item), "activate",
                                    GTK_SIGNAL_FUNC (on_jump_to_object_menu_item_activate),
                                    GUINT_TO_POINTER (consuming_container));                
            }            
        }
        
        /*
         * For a volume we always display generic info
         * along with any extended info so we have 
         * something to show even if there is no FSIM.
         */

        if (type == VOLUME)
        {    
            item = gtk_menu_item_new_with_label (_("Display details..."));
            gtk_menu_append (menu, item);
            gtk_signal_connect (GTK_OBJECT (item), "activate",
                                GTK_SIGNAL_FUNC (on_display_volume_details_menu_item_activate),
                                GUINT_TO_POINTER (handle));
        }
        else
        {    
            item = gtk_menu_item_new_with_label (_("Display details..."));
            gtk_menu_append (menu, item);
            gtk_signal_connect (GTK_OBJECT (item), "activate",
                                GTK_SIGNAL_FUNC (on_display_thing_details_menu_item_activate),
                                GUINT_TO_POINTER (handle));
        }        

        if (type != PLUGIN)
        {
            if (type == VOLUME)
            {
                if (evms_can_set_volume_name (handle) == 0)
                {
                    item = gtk_menu_item_new_with_label (_("Modify Properties..."));
                    gtk_menu_append (menu, item);
                    gtk_signal_connect (GTK_OBJECT (item), "activate",
                                        GTK_SIGNAL_FUNC (on_modify_volume_popup_menu_item_activate),
                                        GUINT_TO_POINTER (handle));                    
                }

                if (volume_has_fsim (handle))
                {    
                    if (evms_can_fsck (handle) == 0)
                    {
                        item = gtk_menu_item_new_with_label (_("Check/Repair File System..."));
                        gtk_menu_append (menu, item);
                        gtk_signal_connect (GTK_OBJECT (item), "activate",
                                            GTK_SIGNAL_FUNC (on_fsck_popup_menu_item_activate),
                                            GUINT_TO_POINTER (handle));                    
                    }
    
                    if (evms_can_defrag (handle) == 0)
                    {
                        item = gtk_menu_item_new_with_label (_("Defragment File System..."));
                        gtk_menu_append (menu, item);
                        gtk_signal_connect (GTK_OBJECT (item), "activate",
                                            GTK_SIGNAL_FUNC (on_defrag_popup_menu_item_activate),
                                            GUINT_TO_POINTER (handle));                    
                    }
                    
                    if (evms_can_unmkfs (handle) == 0)
                    {
                        item = gtk_menu_item_new_with_label (_("Remove File System..."));
                        gtk_menu_append (menu, item);
                        gtk_signal_connect (GTK_OBJECT (item), "activate",
                                            GTK_SIGNAL_FUNC (on_unmkfs_popup_menu_item_activate),
                                            GUINT_TO_POINTER (handle));                    
                    }                
                }
                
                if (can_volume_be_formatted (handle))
                {
                    item = gtk_menu_item_new_with_label (_("Make File System..."));
                    gtk_menu_append (menu, item);
                    gtk_signal_connect (GTK_OBJECT (item), "activate",
                                        GTK_SIGNAL_FUNC (on_mkfs_popup_menu_item_activate),
                                        GUINT_TO_POINTER (handle));                    
                }                    
            }
            else
            {
                if (evms_can_set_info (handle) == 0)
                {
                    item = gtk_menu_item_new_with_label (_("Modify Properties..."));
                    gtk_menu_append (menu, item);
                    gtk_signal_connect (GTK_OBJECT (item), "activate",
                                        GTK_SIGNAL_FUNC (on_modify_thing_properties_menu_item_activate),
                                        GUINT_TO_POINTER (handle));
                }
            }
            
            if (evms_can_create_volume (handle) == 0)
            {
                item = gtk_menu_item_new_with_label (_("Create EVMS Volume..."));
                gtk_menu_append (menu, item);
                gtk_signal_connect (GTK_OBJECT (item), "activate",
                                    GTK_SIGNAL_FUNC (on_create_volume_popup_menu_item_activate),
                                    GUINT_TO_POINTER (handle));
            }

            if (evms_can_create_compatibility_volume (handle) == 0)
            {
                item = gtk_menu_item_new_with_label (_("Create Compatibility Volume..."));
                gtk_menu_append (menu, item);
                gtk_signal_connect (GTK_OBJECT (item), "activate",
                                    GTK_SIGNAL_FUNC (on_create_compatibility_volume_popup_menu_item_activate),
                                    GUINT_TO_POINTER (handle));
            }

            if (evms_can_destroy (handle) == 0)
            {
                item = gtk_menu_item_new_with_label (_("Destroy..."));
                gtk_menu_append (menu, item);
                gtk_signal_connect (GTK_OBJECT (item), "activate",
                                    GTK_SIGNAL_FUNC (on_destroy_thing_menu_item_activate),
                                    GUINT_TO_POINTER (handle));
            }

            if (evms_can_remove (handle) == 0)
            {
                item = gtk_menu_item_new_with_label (_("Remove..."));
                gtk_menu_append (menu, item);
                gtk_signal_connect (GTK_OBJECT (item), "activate",
                                    GTK_SIGNAL_FUNC (on_revert_thing_menu_item_activate),
                                    GUINT_TO_POINTER (handle));
            }
            
            if (evms_can_expand (handle) == 0)
            {                    
                item = gtk_menu_item_new_with_label (_("Expand..."));
                gtk_menu_append (menu, item);
                gtk_signal_connect (GTK_OBJECT (item), "activate",
                                    GTK_SIGNAL_FUNC (on_expand_thing_menu_item_activate),
                                    GUINT_TO_POINTER (handle));
            }
            
            if (evms_can_shrink (handle) == 0)
            {
                item = gtk_menu_item_new_with_label (_("Shrink..."));
                gtk_menu_append (menu, item);
                gtk_signal_connect (GTK_OBJECT (item), "activate",
                                    GTK_SIGNAL_FUNC (on_shrink_thing_menu_item_activate),
                                    GUINT_TO_POINTER (handle));
            }
            
            if (evms_can_remove_from_container (handle) == 0)
            {
                item = gtk_menu_item_new_with_label (_("Remove from Container..."));
                gtk_menu_append (menu, item);
                gtk_signal_connect (GTK_OBJECT (item), "activate",
                                    GTK_SIGNAL_FUNC (on_remove_thing_from_container_menu_item_activate),
                                    GUINT_TO_POINTER (handle));
            }            
        }

        if (GTK_MENU_SHELL (menu)->children == NULL)
        {
                item = gtk_menu_item_new_with_label (_("No actions currently available"));
                gtk_menu_append (menu, item);            
        }
        
        gtk_menu_popup (menu, NULL, NULL, NULL, NULL, event->button, event->time);

        gtk_widget_show_all (GTK_WIDGET (menu));
    }
}

/*
 *
 *   void populate_volume_view_ctree (GtkCTree *, GHashTable *)
 *
 *   Description:
 *      This routine makes the appropriate calls to insert
 *      new nodes in the volume view and preserve still
 *      existing nodes' expand/collapse state.
 * 
 *   Entry:
 *      ctree         - the GtkCTree for the volume view
 *      refresh_table - address of hash table if refresh request
 *
 *   Exit:
 *      Returns nothing.
 *
 */
void populate_volume_view_ctree (GtkCTree *ctree, GHashTable *refresh_table)
{
    populate_volume_view_with_volumes (ctree, refresh_table);
}

/*
 *
 *   void populate_object_view_ctree (GtkCTree *, GHashTable *)
 *
 *   Description:
 *      This routine makes the appropriate calls to insert
 *      new nodes in the topmost object view and preserve still
 *      existing nodes' expand/collapse state.
 * 
 *   Entry:
 *      ctree         - the GtkCTree for the object view
 *      refresh_table - address of hash table if refresh request
 *
 *   Exit:
 *      Returns nothing.
 *
 */
void populate_object_view_ctree (GtkCTree *ctree, GHashTable *refresh_table)
{
    populate_object_view_with_objects (ctree, refresh_table);
}

/*
 *
 *   void populate_feature_object_view_ctree (GtkCTree *, GHashTable *)
 *
 *   Description:
 *      This routine makes the appropriate calls to insert
 *      new nodes in the feature object view and preserve still
 *      existing nodes' expand/collapse state.
 * 
 *   Entry:
 *      ctree         - the GtkCTree for the object view
 *      refresh_table - address of hash table if refresh request
 *
 *   Exit:
 *      Returns nothing.
 *
 */
void populate_feature_object_view_ctree (GtkCTree *ctree, GHashTable *refresh_table)
{
    populate_feature_object_view_with_objects (ctree, refresh_table);
}

/*
 *
 *   void populate_region_view_ctree (GtkCTree *, GHashTable *)
 *
 *   Description:
 *      This routine makes the appropriate calls to insert
 *      new nodes in the region view and preserve still
 *      existing nodes' expand/collapse state.
 * 
 *   Entry:
 *      ctree         - the GtkCTree for the region view
 *      refresh_table - address of hash table if refresh request
 *
 *   Exit:
 *      Returns nothing.
 *
 */
void populate_region_view_ctree (GtkCTree *ctree, GHashTable *refresh_table)
{
    populate_ctree_with_plugins (ctree, EVMS_REGION_MANAGER, 0, 
                                 refresh_table, REGION_VIEW);
    populate_region_view_with_regions (ctree, refresh_table);
    remove_empty_plugin_hiearchy (ctree, EVMS_REGION_MANAGER, 0, REGION_VIEW);
}

/*
 *
 *   void populate_container_view_ctree (GtkCTree *, GSList **)
 *
 *   Description:
 *      This routine makes the appropriate calls to insert
 *      new nodes in the container view and preserve still
 *      existing nodes' expand/collapse state. 
 * 
 *   Entry:
 *      ctree         - the GtkCTree for the container view
 *      refresh_table - address of hash table for refresh request
 *
 *   Exit:
 *      Returns nothing.
 *
 */
void populate_container_view_ctree (GtkCTree *ctree, GHashTable *refresh_table)
{
    populate_ctree_with_plugins (ctree, 0, SUPPORTS_CONTAINERS, 
                                 refresh_table, CONTAINER_VIEW);
    populate_ctree_with_containers (ctree, refresh_table, CONTAINER_VIEW);
    remove_empty_plugin_hiearchy (ctree, 0, SUPPORTS_CONTAINERS, CONTAINER_VIEW);
}

/*
 *
 *   void populate_segment_view_ctree (GtkCTree *, GHashTable *)
 *
 *   Description:
 *      This routine makes the appropriate calls to insert
 *      new nodes in the segment view and preserve still
 *      existing nodes' expand/collapse state.
 * 
 *   Entry:
 *      ctree         - the GtkCTree for the segment view
 *      refresh_table - address of hash table for refresh request
 *
 *   Exit:
 *      Returns nothing.
 *
 */
void populate_segment_view_ctree (GtkCTree *ctree, GHashTable *refresh_table)
{
    populate_ctree_with_plugins (ctree, EVMS_SEGMENT_MANAGER, 0, 
                                 refresh_table, SEGMENT_VIEW);
    populate_segment_view_with_segments (ctree, refresh_table);
    remove_empty_plugin_hiearchy (ctree, EVMS_SEGMENT_MANAGER, 0, SEGMENT_VIEW);
}

/*
 *
 *   void populate_disk_view_ctree (GtkCTree *,  GHashTable *)
 *
 *   Description:
 *      This routine makes the appropriate calls to insert
 *      new nodes in the disk view and preserve still
 *      existing nodes' expand/collapse state.
 * 
 *   Entry:
 *      ctree         - the GtkCTree for the disk view
 *      refresh_table - address of hash table for refresh request
 *
 *   Exit:
 *      Returns nothing.
 *
 */
void populate_disk_view_ctree (GtkCTree *ctree, GHashTable *refresh_table)
{
    populate_ctree_with_plugins (ctree, EVMS_DEVICE_MANAGER, 0, 
                                 refresh_table, DISK_VIEW);
    populate_disk_view_with_disks (ctree, refresh_table);
    remove_empty_plugin_hiearchy (ctree, EVMS_DEVICE_MANAGER, 0, DISK_VIEW);
}

/*
 *
 *   void process_ctree_node_state (gpointer, gpointer, GHashTable *)
 *
 *   Description:
 *      This routine gets creates entries in the refresh hash table
 *      to correlate the expand/collapse state of a node to its
 *      object handle for processing of a tree rebuild.
 * 
 *   Entry:
 *      key           - is the object handle for the node in the
 *                      master ctree hash table
 *      node          - the GtkCTreeNode for the object handle
 *      refresh_table - address of hash table to insert node expand/collapse
 *                      state.
 *
 *   Exit:
 *      Returns nothing.
 *
 */
void process_ctree_node_state (gpointer key, GtkCTreeNode *node, GHashTable *refresh_table)
{
    guint        state;
    GtkCTreeRow *ctree_row = GTK_CTREE_ROW (node);
    
    state = ctree_row->expanded ? NODE_IS_EXPANDED : NODE_IS_COLLAPSED;
    
    g_hash_table_insert (refresh_table, key, GUINT_TO_POINTER (state));
}

/*
 *
 *   void refresh_view (view_type_t)
 *
 *   Description:
 *      This routine rebuilds a view from scratch while
 *      attempting to maintain some of the expanded/collapsed
 *      nodes from the previous view.
 *
 *      It gathers information from the ctree_hash_table for the
 *      view and places the expand/collapse state information
 *      in a refresh hash table that is also accessed through
 *      object handle.
 * 
 *   Entry:
 *      view - the id of the view to attempt to refresh
 *
 *   Exit:
 *      Returns nothing.
 *
 */
void refresh_view (view_type_t view)
{
    GtkCTree *ctree = NULL;
    
    if (view < MAX_VIEWS)
        ctree = view_ctree[view];
    
    if (ctree != NULL)
    {
        GHashTable *refresh_table;

        /*
         * Allocate the hash table to hold the expand/collapse
         * state for each node in the ctree_hash_table[view].
         * We track the expand/collapse for so nodes that
         * will still exist after we rebuild of the view still
         * remain expanded/collapsed as before the rebuild.
         */
        
        refresh_table = g_hash_table_new (g_direct_hash, g_direct_equal);
        
        g_hash_table_foreach (ctree_hash_table[view], (GHFunc) process_ctree_node_state, 
                              refresh_table);
                
        gtk_clist_freeze (GTK_CLIST (ctree));                
        gtk_clist_clear (GTK_CLIST (ctree));
        
        g_hash_table_destroy (ctree_hash_table[view]);
        ctree_hash_table[view] = g_hash_table_new (g_direct_hash, g_direct_equal);
        
        switch (view)
        {
            case VOLUME_VIEW:
                populate_volume_view_ctree (ctree, refresh_table);
                break;
            
            case OBJECT_VIEW:
                populate_object_view_ctree (ctree, refresh_table);
                break;

            case FEATURE_VIEW:
                populate_feature_object_view_ctree (ctree, refresh_table);
                break;
            
            case REGION_VIEW:
                populate_region_view_ctree (ctree, refresh_table);
                break;
            
            case CONTAINER_VIEW:
                populate_container_view_ctree (ctree, refresh_table);
                break;
            
            case SEGMENT_VIEW:
                populate_segment_view_ctree (ctree, refresh_table);
                break;
            
            case DISK_VIEW:
                populate_disk_view_ctree (ctree, refresh_table);
                break;
            
            default:
                log_warning ("%s: Unexpected view supplied: %d", __FUNCTION__, view);
                break;
        }
                        
        gtk_clist_thaw (GTK_CLIST (ctree));
               
        g_hash_table_destroy (refresh_table);
    }
}

/*
 *
 *   void refresh_main_window_views (void)
 *
 *   Description:
 *      This routine initiates a refresh of all 
 *      realized views on the main window.
 * 
 *   Entry:
 *      Nothing
 *
 *   Exit:
 *      Returns nothing.
 *
 */
void refresh_main_window_views (void)
{
    refresh_view (VOLUME_VIEW);
    refresh_view (OBJECT_VIEW);
    refresh_view (FEATURE_VIEW);
    refresh_view (REGION_VIEW);
    refresh_view (CONTAINER_VIEW);
    refresh_view (SEGMENT_VIEW);
    refresh_view (DISK_VIEW);
}

/*
 *
 *   void set_current_view_by_object_type (object_type_t)
 *
 *   Description:
 *      This routine changes the current view to the page corresponding
 *      to the view for the object type supplied.
 * 
 *   Entry:
 *      type - a valid type from the object_type_t enumeration
 *
 *   Exit:
 *      Set the current notebook page to the view corresponding
 *      to view for the object type supplied. Returns the page
 *      number.
 *
 */
gint set_current_view_by_object_type (object_type_t type)
{
    gint         page;
    GtkWidget   *child=NULL;
    GtkNotebook *notebook;

    notebook = GTK_NOTEBOOK (lookup_widget (get_main_window_id (), "browser_notebook"));

    switch (type)
    {
        case VOLUME:
            child = lookup_widget (GTK_WIDGET (notebook), "view_volumes_tree_frame");
            page  = gtk_notebook_page_num (notebook, child);
            break;

        case EVMS_OBJECT:
            child = lookup_widget (GTK_WIDGET (notebook), "view_feature_objects_frame");
            page  = gtk_notebook_page_num (notebook, child);
            break;

        case REGION:
            child = lookup_widget (GTK_WIDGET (notebook), "view_regions_tree_frame");
            page  = gtk_notebook_page_num (notebook, child);
            break;

        case CONTAINER:
            child = lookup_widget (GTK_WIDGET (notebook), "view_containers_tree_frame");
            page  = gtk_notebook_page_num (notebook, child);
            break;

        case SEGMENT:
            child = lookup_widget (GTK_WIDGET (notebook), "view_segments_tree_frame");
            page  = gtk_notebook_page_num (notebook, child);
            break;

        case DISK:
            child = lookup_widget (GTK_WIDGET (notebook), "view_disks_tree_frame");
            page  = gtk_notebook_page_num (notebook, child);
            break;

        case PLUGIN:
            child = lookup_widget (GTK_WIDGET (notebook), "view_plugins_list_frame");
            page  = gtk_notebook_page_num (notebook, child);
            break;

        default:
            page = gtk_notebook_get_current_page (notebook);
            log_warning ("%s: Unknown object type %d supplied.\n", __FUNCTION__, type);
            break;
    }

    gtk_notebook_set_page (notebook, page);
    
    return page;
}

/*
 *
 *   void refresh_current_view (void)
 *
 *   Description:
 *      This routine determines the current page and calls
 *      refresh_view () to refresh its contents.
 * 
 *   Entry:
 *      Nothing
 *
 *   Exit:
 *      Refreshes the contents of the current view.
 *
 */
void refresh_current_view (void)
{
    gint         page;
    GtkNotebook *notebook;

    notebook = GTK_NOTEBOOK (lookup_widget (get_main_window_id (), "browser_notebook"));
    page = gtk_notebook_get_current_page (notebook);

    if (page < PLUGIN_VIEW)
        refresh_view (page);
}

/*
 *
 *   void set_refresh_views_pixmap (GtkWidget *)
 *
 *   Description:
 *      This routine sets the pixmap in the eventbox that
 *      handles refreshing either the current view or all of
 *      the realized views.
 * 
 *   Entry:
 *      window - the id of the main window
 *
 *   Exit:
 *      Finds the refresh_pixmap_eventbox and adds the 
 *      refresh views pixmap to the eventbox.
 *
 */
void set_refresh_views_pixmap (GtkWidget *window)
{
    GdkBitmap *mask;
    GtkWidget *pixmap;
    GtkWidget *eventbox;
    GdkPixmap *gdk_pixmap;

    if ((eventbox = lookup_widget (window, "refresh_pixmap_eventbox")))
    {
        get_refresh_pixmap (&gdk_pixmap, &mask);
        pixmap = gtk_pixmap_new (gdk_pixmap, mask);
        gtk_widget_show (pixmap);
        gtk_container_add (GTK_CONTAINER (eventbox), pixmap);
    }    
}

/*
 *
 *   void on_refresh_view_menu_item_activate (GtkMenuItem *, gpointer)
 *
 *   Description:
 *      This routine is invoked by a choice in the refresh views
 *      context menu popup to refresh either the current view or
 *      all of the realized main window views.
 * 
 *   Entry:
 *      menuitem  - the menuitem that was activated
 *      user_data - contains the id of what view(s) to refresh
 *
 *   Exit:
 *      Refreshes either the current view or all of the main window views.
 *
 */
void on_refresh_view_menu_item_activate (GtkMenuItem *menuitem, gpointer user_data)
{
    guint view_to_refresh = GPOINTER_TO_UINT (user_data);
    
    if (view_to_refresh == ALL_VIEWS)
        refresh_main_window_views ();
    else
        refresh_current_view ();
}

/*
 *
 *   void display_refresh_view_popup_menu (GdkEventButton *)
 *
 *   Description:
 *      This routine displays a popup menu with menu items choices
 *      as to which main window views should get refreshed.
 * 
 *   Entry:
 *      event - everything we want to know about the event
 *
 *   Exit:
 *      Creates a context sensitive popup menu populated with
 *      menu items to refresh either all the views or just the
 *      current view.
 *
 */
void display_refresh_view_popup_menu (GdkEventButton *event)
{
    GtkMenu   *menu;
    GtkWidget *item;

    menu = GTK_MENU (gtk_menu_new ());

    item = gtk_menu_item_new_with_label (_("Refresh all views"));
    gtk_menu_append (menu, item);
    gtk_signal_connect (GTK_OBJECT (item), "activate",
                        GTK_SIGNAL_FUNC (on_refresh_view_menu_item_activate),
                        GUINT_TO_POINTER (ALL_VIEWS));

    item = gtk_menu_item_new_with_label (_("Refresh current view"));
    gtk_menu_append (menu, item);
    gtk_signal_connect (GTK_OBJECT (item), "activate",
                        GTK_SIGNAL_FUNC (on_refresh_view_menu_item_activate),
                        GUINT_TO_POINTER (CURRENT_VIEW));
    
    gtk_menu_popup (menu, NULL, NULL, NULL, NULL, event->button, event->time);

    gtk_widget_show_all (GTK_WIDGET (menu));
}
