/*
 *
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: value.c
 */ 
 
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <frontend.h>
#include <gtk/gtk.h>

#include "value.h"
#include "readable.h"
#include "logging.h"
#include "support.h"

#define MAX_UNITS                5
#define KB_TO_SECTORS_MULTIPLIER ((u_int64_t)2)
#define MB_TO_SECTORS_MULTIPLIER ((u_int64_t)(KB_TO_SECTORS_MULTIPLIER * (u_int64_t)1024))
#define GB_TO_SECTORS_MULTIPLIER ((u_int64_t)(MB_TO_SECTORS_MULTIPLIER * (u_int64_t)1024))
#define TB_TO_SECTORS_MULTIPLIER ((u_int64_t)(GB_TO_SECTORS_MULTIPLIER * (u_int64_t)1024))
#define PB_TO_SECTORS_MULTIPLIER ((u_int64_t)(TB_TO_SECTORS_MULTIPLIER * (u_int64_t)1024))

static const gdouble conversion_factor [MAX_UNITS] = { KB_TO_SECTORS_MULTIPLIER,
                                                       MB_TO_SECTORS_MULTIPLIER,
                                                       GB_TO_SECTORS_MULTIPLIER,
                                                       TB_TO_SECTORS_MULTIPLIER,
                                                       PB_TO_SECTORS_MULTIPLIER };

static const value_unit_t units [MAX_UNITS] = { EVMS_Unit_Kilobytes,
                                                EVMS_Unit_Megabytes,
                                                EVMS_Unit_Gigabytes,
                                                EVMS_Unit_Terabytes,
                                                EVMS_Unit_Petabytes };

/*
 *
 *   gboolean value_less_than_integer (value_t, value_type_t, gint)
 *
 *   Description:
 *      This routine takes a value_t of type value_type_t and
 *      and integer value and determines if the value is less
 *      than the supplied integer.
 *
 *   Entry:
 *      value  - the union of the basic data types
 *      type   - the id of the data type
 *      number - the integer value to check against
 *
 *   Exit:
 *      Returns TRUE if the value is less than integer supplied
 *      otherwise returns FALSE if it isn't (duh).
 *
 */
gboolean value_less_than_integer (value_t value, value_type_t type, gint number)
{
    gboolean result = FALSE;

    switch (type)
    {
        case EVMS_Type_String:
            /*
             * BUGBUG: Convert string to int or int to string to compare?
             */
            break;

        case EVMS_Type_Char:
            result = value.c < number;
            break;

        case EVMS_Type_Unsigned_Char:
            result = value.uc < number;
            break;

        case EVMS_Type_Real32:
            result = value.r32 < number;
            break;

        case EVMS_Type_Real64:
            result = value.r64 < number;
            break;
            
        case EVMS_Type_Int8:
            result = value.i8 < number;
            break;

        case EVMS_Type_Unsigned_Int8:
            result = value.ui8 < number;
            break;

        case EVMS_Type_Int16:
            result = value.i16 < number;
            break;

        case EVMS_Type_Unsigned_Int16:
            result = value.ui16 < number;
            break;

        case EVMS_Type_Int:
        case EVMS_Type_Int32:
            result = value.i32 < number;
            break;

        case EVMS_Type_Unsigned_Int:
        case EVMS_Type_Unsigned_Int32:
            result = value.ui32 < number;
            break;

        case EVMS_Type_Int64:
            result = value.i64 < number;
            break;

        case EVMS_Type_Unsigned_Int64:
            result = value.ui64 < number;
            break;

        default:
            log_warning ("%s: Value type of %d was not handled.\n", __FUNCTION__, type);
    }

    return result;
}

/*
 *
 *   gchar* convert_value_to_string (value_t, value_type_t, value_format_t)
 *
 *   Description:
 *      This routine takes a value_t of type value_type_t and
 *      produces a string that represents the value. The string
 *      should be freed with g_free() when no longer needed.
 *
 *   Entry:
 *      value  - the union of the basic data types
 *      type   - the id of the data type
 *      format - the suggested format (currently only hex supported)
 *
 *   Exit:
 *      A string is produced that represents the value or NULL
 *      if an error occurred.
 *
 */
gchar* convert_value_to_string (value_t value, value_type_t type, value_format_t format)
{
    gchar *string = NULL;

    switch (type)
    {
        case EVMS_Type_Boolean:
            string = value.bool ? g_strdup (_("Yes")) : g_strdup (_("No"));
            break;
        
        case EVMS_Type_String:
            string = g_strdup (value.s);
            break;

        case EVMS_Type_Char:
        case EVMS_Type_Unsigned_Char:
            if (format == EVMS_Format_Hex)
                string = g_strdup_printf ("%#hhx", value.uc);
            else
                string = g_strdup_printf ("%c", value.uc);
            break;

        case EVMS_Type_Real32:
            if (format == EVMS_Format_Hex)
                string = g_strdup_printf ("%.2a", value.r32);
            else
                string = g_strdup_printf ("%.2f", value.r32);
            break;

        case EVMS_Type_Real64:
            if (format == EVMS_Format_Hex)
                string = g_strdup_printf ("%.2a", value.r64);
            else
                string = g_strdup_printf ("%.2f", value.r64);
            break;

        case EVMS_Type_Int8:
            if (format == EVMS_Format_Hex)
                string = g_strdup_printf ("%#hhx", value.i8);
            else
                string = g_strdup_printf ("%hhd", value.i8);
            break;

        case EVMS_Type_Unsigned_Int8:
            if (format == EVMS_Format_Hex)
                string = g_strdup_printf ("%#hhx", value.ui8);
            else
                string = g_strdup_printf ("%hhu", value.ui8);
            break;

        case EVMS_Type_Int16:
            if (format == EVMS_Format_Hex)
                string = g_strdup_printf ("%#hx", value.i16);
            else
                string = g_strdup_printf ("%hd", value.i16);
            break;

        case EVMS_Type_Unsigned_Int16:
            if (format == EVMS_Format_Hex)
                string = g_strdup_printf ("%#hx", value.ui16);
            else
                string = g_strdup_printf ("%hu", value.ui16);
            break;

        case EVMS_Type_Int:
        case EVMS_Type_Int32:
            if (format == EVMS_Format_Hex)
                string = g_strdup_printf ("%#x", value.i32);
            else
                string = g_strdup_printf ("%d", value.i32);
            break;

        case EVMS_Type_Unsigned_Int:
        case EVMS_Type_Unsigned_Int32:
            if (format == EVMS_Format_Hex)
                string = g_strdup_printf ("%#x", value.ui32);
            else
                string = g_strdup_printf ("%u", value.ui32);
            break;

        case EVMS_Type_Int64:
            if (format == EVMS_Format_Hex)
                string = g_strdup_printf ("%#llx", value.i64);
            else
                string = g_strdup_printf ("%lld", value.i64);
            break;

        case EVMS_Type_Unsigned_Int64:
            if (format == EVMS_Format_Hex)
                string = g_strdup_printf ("%#llx", value.ui64);
            else
                string = g_strdup_printf ("%llu", value.ui64);
            break;

        default:
            log_warning ("%s: Value type of %d was not handled.\n", __FUNCTION__, type);
    }

    return string;
}

/*
 *
 *   void convert_string_to_value (gchar *, value_type_t *, value_t *)
 *
 *   Description:
 *      This routine takes a string and a value_type_t identifier
 *      and produces the corresponding basic data type.
 *
 *   Entry:
 *      string - the value in text form
 *      type   - the id of the data type
 *      maxlen - maximum length of buffer if a string value
 *      value  - the address of where to store the value
 *
 *   Exit:
 *      A value_t is filled in with the proper value from the string
 *      supplied.
 *
 */
void convert_string_to_value (gchar *string, value_type_t type, gint maxlen, value_t *value)
{
    switch (type)
    {
        case EVMS_Type_Boolean:
            if (g_strcasecmp (_("True"), string) == 0)
                value->bool = TRUE;
            else
                value->bool = FALSE;
            break;
            
        case EVMS_Type_String:
            value->s = g_malloc (maxlen + 1);
            strcpy (value->s, string);
            break;

        case EVMS_Type_Char:
        case EVMS_Type_Unsigned_Char:
            value->uc = *string;
            break;

        case EVMS_Type_Real32:
        case EVMS_Type_Real64:
            value->r64 = atof (string);
            break;
            
        case EVMS_Type_Int8:
        case EVMS_Type_Unsigned_Int8:
            value->ui8 = (u_int8_t) atoi (string);
            break;

        case EVMS_Type_Int16:
        case EVMS_Type_Unsigned_Int16:
            value->i16 = atoi (string);
            break;

        case EVMS_Type_Int:
        case EVMS_Type_Int32:
        case EVMS_Type_Unsigned_Int:
        case EVMS_Type_Unsigned_Int32:
            value->i32 = atol (string);
            break;

        case EVMS_Type_Int64:
        case EVMS_Type_Unsigned_Int64:
            value->i64 = atoll (string);
            break;

        default:
            log_warning ("%s: Value type of %d was not handled.\n", __FUNCTION__, type);
    }
}

/*
 *
 *   value_list_t* allocate_value_list (guint, value_type_t, guint)
 *
 *   Description:
 *      This routine creates a value list. If the list is
 *      a list of string types then it also allocates 
 *      string buffers of the given size + 1.
 *
 *   Entry:
 *      count - number of entries in the list
 *      type  - the id of the data type
 *      size  - max string size
 *
 *   Exit:
 *      The value list is created and any string buffers as well.
 *
 */
value_list_t* allocate_value_list (guint count, value_type_t type, guint size)
{
    value_list_t *vlist;

    vlist = g_malloc0 (sizeof (value_list_t) + ((count-1) * sizeof (value_t)) );

    if (type == EVMS_Type_String)
    {
        guint i;

        size++;

        for (i=0; i < count; i++)
        {
            vlist->value[i].s    = g_malloc (size);
            *(vlist->value[i].s) = '\0';
        }
    }

    return vlist;
}

/*
 *
 *   void duplicate_value (value_t, value_type_t, gboolean, guint, gint, value_t *)
 *
 *   Description:
 *      This routine takes a value and copies the contents
 *      to another value_t. If the value type indicates it
 *      is a string then the string is allocated of the max
 *      length and the string is duplicated. If the value
 *      type describes a value list then the list is duplicated.
 *
 *   Entry:
 *      source_value - the union of the basic data types
 *      type         - the id of the data type
 *      is_list      - if the value is actually a value_list_t *
 *      list_count   - if a list, the actual allocated item count,
 *                     not the used entries count
 *      maxlen       - the maximum length of the string buffer
 *      target_value - address of value_t to store duplicated contents
 *
 *   Exit:
 *      The source value is copied to the target and strings 
 *      and lists are duplicated as necessary.
 *
 */
void duplicate_value (value_t source_value, value_type_t type, gboolean is_list, 
                      guint list_count, gint maxlen, value_t *target_value)
{
    /*
     * Add one to maxlen as it is used for strings so always
     * ensure one more byte for the terminator byte.
     */

    maxlen += 1;

    if (is_list)
    {
        if (type == EVMS_Type_String)
        {
            guint i;

            target_value->list = allocate_value_list (list_count, type, maxlen);

            for (i=0; i < source_value.list->count; i++)
            {
                strcpy (target_value->list->value[i].s, source_value.list->value[i].s);
            }
        }
        else
        {
            target_value->list = g_memdup (source_value.list, 
                                           (sizeof (value_list_t) + (list_count * (sizeof (value_t) - 1))));
        }
    }
    else if (type == EVMS_Type_String)
    { 
        target_value->s = g_malloc (maxlen);
        strcpy (target_value->s, source_value.s);
    }
    else
    {
        memcpy (target_value, &source_value, sizeof (value_t));
    }
}

/*
 *
 *   void clear_value (value_t *, value_type_t, gboolean, guint)
 *
 *   Description:
 *      This routine takes the address of a value and clears
 *      the value contents. If the value is a list of values
 *      then the list is freed. If the value contains a string
 *      then the string is freed.
 *
 *   Entry:
 *      value      - the address of the union of basic data types
 *      type       - the id of the data type
 *      is_list    - if the value is actually a value_list_t *
 *      list_count - if this is a list, this is the allocated item count
 *
 *   Exit:
 *      The value is cleared. Any lists or strings are freed
 *      prior to setting the value contents to zeroes.
 *
 */
void clear_value (value_t *value, value_type_t type, gboolean is_list, guint list_count)
{
    if (is_list && value->list != NULL)
    {
        if (type == EVMS_Type_String)
        {
            guint i;

            for (i=0; i < list_count; i++)
                g_free (value->list->value[i].s);
        }

        g_free (value->list);
    }
    else if (type == EVMS_Type_String)
        g_free (value->s);

    memset (value, 0, sizeof (value_t));
}

/*
 *
 *   gfloat convert_value_to_float (value_t, value_type_t)
 *
 *   Description:
 *      This routine takes a value (hopefully numeric) and converts
 *      it to a float.
 *
 *   Entry:
 *      value - the union of the basic data types
 *      type  - the id of the data type
 *
 *   Exit:
 *      Returns the numeric value as a float.
 *
 */
gfloat convert_value_to_float (value_t value, value_type_t type)
{
    gfloat float_value = 0.0;

    switch (type)
    {
        case EVMS_Type_Real32:
            float_value = value.r32;
            break;

        case EVMS_Type_Real64:
            float_value = value.r64;
            break;

        case EVMS_Type_Char:
            float_value = value.c * 1.0;
            break;

        case EVMS_Type_Unsigned_Char:
            float_value = value.uc * 1.0;
            break;
            
        case EVMS_Type_Int8:
            float_value = value.i8 * 1.0;
            break;

        case EVMS_Type_Unsigned_Int8:
            float_value = value.ui8 * 1.0;
            break;

        case EVMS_Type_Int16:
            float_value = value.i16 * 1.0;
            break;

        case EVMS_Type_Unsigned_Int16:
            float_value = value.ui16 * 1.0;
            break;

        case EVMS_Type_Int:
        case EVMS_Type_Int32:
            float_value = value.i32 * 1.0;
            break;

        case EVMS_Type_Unsigned_Int:
        case EVMS_Type_Unsigned_Int32:
            float_value = value.ui32 * 1.0;
            break;

        case EVMS_Type_Int64:
            float_value = value.i64 * 1.0;
            break;

        case EVMS_Type_Unsigned_Int64:
            float_value = value.ui64 * 1.0;
            break;

        default:
            log_warning ("%s: Value type of %d was not handled.\n", __FUNCTION__, type);
    }

    return float_value;
}

/*
 *
 *   void convert_float_to_value (gfloat, value_type_t, value_t *)
 *
 *   Description:
 *      This routine takes a float and converts it to a value_t.
 *
 *   Entry:
 *      float_value - the float to convert to a value_t type
 *      type        - the id of the data type
 *      value       - addresss to store converted value type
 *
 *   Exit:
 *      Converts the float to a value_t containing value_type_t
 *
 */
void convert_float_to_value (gfloat float_value, value_type_t type, value_t *value)
{
    switch (type)
    {
        case EVMS_Type_Real32:
            value->r32 = float_value;
            break;

        case EVMS_Type_Real64:
            value->r64 = float_value;
            break;

        case EVMS_Type_Char:
        case EVMS_Type_Unsigned_Char:
        case EVMS_Type_Int8:
        case EVMS_Type_Unsigned_Int8:        
            value->ui8 = float_value;
            break;

        case EVMS_Type_Int16:
        case EVMS_Type_Unsigned_Int16:
            value->ui16 = float_value;
            break;

        case EVMS_Type_Int:
        case EVMS_Type_Int32:
        case EVMS_Type_Unsigned_Int:
        case EVMS_Type_Unsigned_Int32:
            value->ui32 = float_value;
            break;

        case EVMS_Type_Int64:
        case EVMS_Type_Unsigned_Int64:
            value->ui64 = float_value;
            break;

        default:
            log_warning ("%s: Value type of %d was not handled.\n", __FUNCTION__, type);
    }
}

/*
 *
 *   gdouble convert_value_to_double (value_t, value_type_t)
 *
 *   Description:
 *      This routine takes a value (hopefully numeric) and converts
 *      it to a double.
 *
 *   Entry:
 *      value - the union of the basic data types
 *      type  - the id of the data type
 *
 *   Exit:
 *      Returns the numeric value as a double.
 *
 */
gdouble convert_value_to_double (value_t value, value_type_t type)
{
    gdouble double_value = 0.0;

    switch (type)
    {
        case EVMS_Type_Real32:
            double_value = value.r32;
            break;

        case EVMS_Type_Real64:
            double_value = value.r64;
            break;

        case EVMS_Type_Char:
            double_value = value.c * 1.0;
            break;

        case EVMS_Type_Unsigned_Char:
            double_value = value.uc * 1.0;
            break;
            
        case EVMS_Type_Int8:
            double_value = value.i8 * 1.0;
            break;

        case EVMS_Type_Unsigned_Int8:
            double_value = value.ui8 * 1.0;
            break;

        case EVMS_Type_Int16:
            double_value = value.i16 * 1.0;
            break;

        case EVMS_Type_Unsigned_Int16:
            double_value = value.ui16 * 1.0;
            break;

        case EVMS_Type_Int:
        case EVMS_Type_Int32:
            double_value = value.i32 * 1.0;
            break;

        case EVMS_Type_Unsigned_Int:
        case EVMS_Type_Unsigned_Int32:
            double_value = value.ui32 * 1.0;
            break;

        case EVMS_Type_Int64:
            double_value = value.i64 * 1.0;
            break;

        case EVMS_Type_Unsigned_Int64:
            double_value = value.ui64 * 1.0;
            break;

        default:
            log_warning ("%s: Value type of %d was not handled.\n", __FUNCTION__, type);
    }

    return double_value;
}

/*
 *
 *   void convert_double_to_value (gdouble, value_type_t, value_t *)
 *
 *   Description:
 *      This routine takes a double and converts it to a value_t type.
 *
 *   Entry:
 *      double_value - the double to convert to a value_t
 *      type         - the id of the data type
 *      value        - addresss to store converted value type
 *
 *   Exit:
 *      Converts the double to a value_t containing value_type_t
 *
 */
void convert_double_to_value (gdouble double_value, value_type_t type, value_t *value)
{
    switch (type)
    {
        case EVMS_Type_Real32:
            value->r32 = double_value;
            break;

        case EVMS_Type_Real64:
            value->r64 = double_value;
            break;

        case EVMS_Type_Char:
        case EVMS_Type_Unsigned_Char:
        case EVMS_Type_Int8:
        case EVMS_Type_Unsigned_Int8:        
            value->ui8 = double_value;
            break;

        case EVMS_Type_Int16:
        case EVMS_Type_Unsigned_Int16:
            value->ui16 = double_value;
            break;

        case EVMS_Type_Int:
        case EVMS_Type_Int32:
        case EVMS_Type_Unsigned_Int:
        case EVMS_Type_Unsigned_Int32:
            value->ui32 = double_value;
            break;

        case EVMS_Type_Int64:
        case EVMS_Type_Unsigned_Int64:
            value->ui64 = double_value;
            break;

        default:
            log_warning ("%s: Value type of %d was not handled.\n", __FUNCTION__, type);
    }
}

/*
 *
 *   u_int64_t convert_value_to_ui64 (value_t, value_type_t)
 *
 *   Description:
 *      This routine takes a value (hopefully numeric) and converts
 *      it to a 64-bit unsigned value.
 *
 *   Entry:
 *      value - the union of the basic data types
 *      type  - the id of the data type
 *
 *   Exit:
 *      Returns the numeric value as an unsigned 64-bit.
 *
 */
u_int64_t convert_value_to_ui64 (value_t value, value_type_t type)
{
    u_int64_t ui64_value = 0;

    switch (type)
    {
        case EVMS_Type_Real32:
            ui64_value = value.r32;
            break;

        case EVMS_Type_Real64:
            ui64_value = value.r64;
            break;

        case EVMS_Type_Char:
            ui64_value = value.c;
            break;

        case EVMS_Type_Unsigned_Char:
            ui64_value = value.uc;
            break;

        case EVMS_Type_Int8:
            ui64_value = value.i8;
            break;

        case EVMS_Type_Unsigned_Int8:
            ui64_value = value.ui8;
            break;

        case EVMS_Type_Int16:
            ui64_value = value.i16;
            break;

        case EVMS_Type_Unsigned_Int16:
            ui64_value = value.ui16;
            break;

        case EVMS_Type_Int:
        case EVMS_Type_Int32:
            ui64_value = value.i32;
            break;

        case EVMS_Type_Unsigned_Int:
        case EVMS_Type_Unsigned_Int32:
            ui64_value = value.ui32;
            break;

        case EVMS_Type_Int64:
            ui64_value = value.i64;
            break;

        case EVMS_Type_Unsigned_Int64:
            ui64_value = value.ui64;
            break;

        default:      
            log_warning ("%s: Value type of %d was not handled.\n", __FUNCTION__, type);
    }

    return ui64_value;
}

/*
 *
 *   gchar* convert_sector_value_to_string (value_t, value_type_t, value_unit_t *)
 *
 *   Description:
 *      This routine takes a value (hopefully numeric) which is a value
 *      expressed in sectors and converts it to a string expressed in
 *      bytes, kilobytes, megabytes, gigabytes, or terabytes.
 *
 *   Entry:
 *      value    - the union of the basic data types
 *      type     - the id of the data type
 *      new_unit - address to write the unit type converted to
 *
 *   Exit:
 *      Returns the sector size converted to bytes, KB, MB, GB or TB along with
 *      the new unit value.
 *
 */
gchar* convert_sector_value_to_string (value_t value, value_type_t type, value_unit_t *new_unit)
{      
    u_int64_t size_in_sectors;

    size_in_sectors = convert_value_to_ui64 (value, type);
    return make_sectors_readable_size_string (size_in_sectors, new_unit);
}

/*
 *
 *   value_unit_t get_unit_from_mnemonic (gchar *)
 *
 *   Description:
 *      This routine takes a string which is hopefully a word
 *      or mnemonic such as "bytes", "KB", "MB", "GB", or "TB"
 *      and if recognized returns the unit type this corresponds
 *      to.
 *
 *   Entry:
 *      mnemonic - the string containing a mnemonic or word representing
 *                 a unit measurement
 *
 *   Exit:
 *      Returns the value_unit_t type corresponding to the unit of 
 *      measurement the string represents.
 *
 */
value_unit_t get_unit_from_mnemonic (gchar *mnemonic)
{
    value_unit_t unit;

    unit = EVMS_Unit_None;
    g_strstrip (mnemonic);

    if (g_strcasecmp (mnemonic, "bytes") == 0)
        unit = EVMS_Unit_Bytes;
    else if (g_strcasecmp (mnemonic, "KB") == 0)
        unit = EVMS_Unit_Kilobytes;
    else if (g_strcasecmp (mnemonic, "MB") == 0)
        unit = EVMS_Unit_Megabytes;
    else if (g_strcasecmp (mnemonic, "GB") == 0)
        unit = EVMS_Unit_Gigabytes;
    else if (g_strcasecmp (mnemonic, "TB") == 0)
        unit = EVMS_Unit_Terabytes;
    else if (g_strcasecmp (mnemonic, "PB") == 0)
        unit = EVMS_Unit_Petabytes;
    else if (g_strcasecmp (mnemonic, "sectors") == 0)
        unit = EVMS_Unit_Sectors;
    else
        log_warning ("%s: mnemonic or word not recognized: %s.\n", __FUNCTION__, mnemonic);

    return unit;
}

/*
 *
 *   gfloat convert_sector_value_to_float_with_unit (value_t, value_type_t, value_unit_t)
 *
 *   Description:
 *      This routine takes a value expressed in sector units and
 *      converts it to a real number that expresses the value in
 *      the conversion unit, i.e. KB, GB, TB, etc.
 *
 *   Entry:
 *      value           - the value expressed in sectors
 *      type            - the value type, e.g. EVMS_Type_Int
 *      conversion_unit - the unit of measure to convert to, e.g. KB, TB, etc.
 *
 *   Exit:
 *      Returns the value expressed in the conversion unit as a real number
 *
 */
gfloat convert_sector_value_to_float_with_unit (value_t value, value_type_t type, value_unit_t conversion_unit)
{
    gfloat new_size;

    new_size = convert_value_to_float (value, type);

    switch (conversion_unit)
    {
        case EVMS_Unit_Kilobytes:
            new_size /= KB_TO_SECTORS_MULTIPLIER;
            break;

        case EVMS_Unit_Megabytes:
            new_size /= MB_TO_SECTORS_MULTIPLIER;
            break;

        case EVMS_Unit_Gigabytes:
            new_size /= GB_TO_SECTORS_MULTIPLIER;
            break;

        case EVMS_Unit_Terabytes:
            new_size /= TB_TO_SECTORS_MULTIPLIER;
            break;

        case EVMS_Unit_Petabytes:
            new_size /= PB_TO_SECTORS_MULTIPLIER;
            break;

        default:
            log_debug ("%s: We are not going to convert this unit of measurement: %d\n", 
                       __FUNCTION__, conversion_unit);
            break;
    }

    return new_size;
}

/*
 *
 *   value_unit_t select_readable_unit_for_sector_value (value_t, value_type_t)
 *
 *   Description:
 *      This routine takes a value in sector units and determines the
 *      highest unit of measure (KB,MB,GB) that we can use to express
 *      the value in a more meaningful way.
 *
 *   Entry:
 *      value - the input size value
 *      type  - id of the size type, i.e. EVMS_Type_Real32 or EVMS_Type_Unsigned_Int64
 *
 *   Exit:
 *      Returns the highest unit (KB,MB,GB,etc.) that we can use to express the
 *      number given.
 *
 */
value_unit_t select_readable_unit_for_sector_value (value_t value, value_type_t type)
{
    gint          i;
    u_int64_t     value_as_ui64;
    u_int64_t     divisor;
    gdouble       result;
    value_unit_t  unit = EVMS_Unit_Sectors;

    value_as_ui64 = convert_value_to_ui64 (value, type);

    for (i=0; i < MAX_UNITS; i++)
    {
        divisor = conversion_factor[i];
        result  = value_as_ui64 / divisor;

        if (result > 0.0)
            unit = units[i];      
        else
            break;
    }

    return unit;
}

/*
 *
 *   void convert_size_value_to_sector_value (value_t, value_type_t, value_unit_t, 
 *                                            value_t *, value_type_t)
 *
 *   Description:
 *      This routine takes a value representing a size in some user understandable
 *      unit of measurement such as KB, TB or GB and typically either as a real
 *      or integer number and it gets converted to a value expressed in terms of
 *      EVMS_Unit_Sectors unit of measurement.
 *
 *   Entry:
 *      size_value - the input size value
 *      size_type  - id of the size type, i.e. EVMS_Type_Real32 or EVMS_Type_Unsigned_Int64
 *      size_unit  - the size value unit (hopefully bytes, KB, MB, GB, TB, or PB)
 *      sec_value  - address to write the size_value as a value in sector units
 *      sec_type   - id of the sector value type, i.e. EVMS_Type_Int64 or EVMS_Type_Unsigned_Int
 *
 *   Exit:
 *      Returns the value convert to sector units if possible.
 *
 */
void convert_size_value_to_sector_value (value_t size_value, value_type_t size_type, 
                                         value_unit_t size_unit, 
                                         value_t *sec_value, value_type_t sec_type)
{
    gdouble size_as_double;
    gdouble size_in_sectors;

    /*
     * For the sake of simplicity. Do calculations with reals then
     * convert to a integer value if necessary.
     */

    size_as_double = convert_value_to_double (size_value, size_type);

    switch (size_unit)
    {
        case EVMS_Unit_Bytes:
            size_in_sectors = size_as_double / EVMS_VSECTOR_SIZE;
            break;

        case EVMS_Unit_Kilobytes:
            size_in_sectors = size_as_double * KB_TO_SECTORS_MULTIPLIER;
            break;

        case EVMS_Unit_Megabytes:
            size_in_sectors = size_as_double * MB_TO_SECTORS_MULTIPLIER;
            break;

        case EVMS_Unit_Gigabytes:
            size_in_sectors = size_as_double * GB_TO_SECTORS_MULTIPLIER;
            break;

        case EVMS_Unit_Terabytes:
            size_in_sectors = size_as_double * TB_TO_SECTORS_MULTIPLIER;
            break;

        case EVMS_Unit_Petabytes:
            size_in_sectors = size_as_double * PB_TO_SECTORS_MULTIPLIER;
            break;

        default:
            log_debug ("%s: We are not going to convert this unit of measurement: %d\n", 
                       __FUNCTION__, size_unit);

        case EVMS_Unit_Sectors:
            size_in_sectors = size_as_double;
            break;

    }
    
    convert_double_to_value (size_in_sectors, sec_type, sec_value);
}

/*
 *
 *   void convert_float_to_sector_value (gfloat, value_type_t, value_unit_t, value_t *)
 *
 *   Description:
 *      This routine takes a float that expresses a size unit
 *      (KB,MB,TB,etc.) and converts it to a value_t of value_type_t
 *      expressed in a EVMS_Unit_Sectors unit of measurement.
 *
 *   Entry:
 *      size  - the sector unit size as a real number
 *      type  - the type of the sector value, e.g. EVMS_Type_Int64
 *      unit  - the unit of the size, e.g. KB, TB, etc.
 *      value - address to write sector value to 
 *
 *   Exit:
 *      Returns the size real number converted to a sector unit value
 *      of the requested type.
 *
 */
void convert_float_to_sector_value (gfloat size, value_type_t type, value_unit_t unit, value_t *value)
{
    gfloat size_in_sectors;

    switch (unit)
    {
        case EVMS_Unit_Bytes:
            size_in_sectors = size / EVMS_VSECTOR_SIZE;
            break;

        case EVMS_Unit_Kilobytes:
            size_in_sectors = size * KB_TO_SECTORS_MULTIPLIER;
            break;

        case EVMS_Unit_Megabytes:
            size_in_sectors = size * MB_TO_SECTORS_MULTIPLIER;
            break;

        case EVMS_Unit_Gigabytes:
            size_in_sectors = size * GB_TO_SECTORS_MULTIPLIER;
            break;

        case EVMS_Unit_Terabytes:
            size_in_sectors = size * TB_TO_SECTORS_MULTIPLIER;
            break;

        case EVMS_Unit_Petabytes:
            size_in_sectors = size * PB_TO_SECTORS_MULTIPLIER;
            break;

        default:
            log_debug ("%s: We are not going to convert this unit of measurement: %d\n", 
                       __FUNCTION__, unit);

        case EVMS_Unit_Sectors:
            size_in_sectors = size;
            break;

    }
    
    convert_float_to_value (size_in_sectors, type, value);
}

/*
 *
 *   void convert_size_string_to_sector_value (gchar *, value_type_t, value_t *)
 *
 *   Description:
 *      This routine takes a string which is hopefully a numeric
 *      followed by the word "bytes", "KB", "MB", "GB", or "TB"
 *      and strips the word off to convert the string to a numeric.
 *      The numeric is then converted to a sector multiple based
 *      on the trailing mnemonic unit value.
 *
 *   Entry:
 *      string    - the string containing a size, i.e. "10 MB" or "1.4GB"
 *      type_out  - value_type_t to convert the value to on return
 *      value_out - address to write sector value to 
 *
 *   Exit:
 *      Returns the size converted to a sector unit value.
 *
 */
void convert_size_string_to_sector_value (gchar *string, value_type_t type_out, value_t *value_out)
{
    value_out->i64 = 0;

    if (string != NULL)
    {
        guint        length=0;
        gchar       *number;
        value_t      value_in;
        value_type_t type_in;
        value_unit_t unit;

        /*
         * Copy digits, blank space and '.' to temp string until we hit an
         * alpha character.
         */

        number       = g_malloc (strlen (string) + 1);
        type_in      = EVMS_Type_Unsigned_Int64;
        unit         = EVMS_Unit_None;
        value_in.i64 = 0;

        while (*string && (isdigit (*string) || *string == '.' || isspace (*string)))
        {
            if (*string == '.')
                type_in = EVMS_Type_Real64;

            number[length] = *string;
            length++;
            string++;
        }

        number[length] = '\0';

        convert_string_to_value (number, type_in, 256, &value_in);

        /*
         * If there is something remaining in the string check to see
         * if it is a recognizable size measurement and if so, set the
         * the unit to this so we can convert it to a sector unit value
         * of the expected output value type.
         */

        if (*string != '\0')
            unit = get_unit_from_mnemonic (string);

        convert_size_value_to_sector_value (value_in, type_in, unit, value_out, type_out);

        g_free (number);
    }
}

/*
 *
 *   gchar *convert_value_to_string_with_unit (value_t, value_type_t, value_unit_t, value_format_t)
 *
 *   Description:
 *      This routine takes a value and converts it to a string with
 *      the short form of the unit value concatenated to the string.
 *
 *   Entry:
 *      value  - the union of the basic data types
 *      type   - the id of the data type
 *      unit   - the value unit measurement
 *      format - the format for the the display value
 *
 *   Exit:
 *      Returns the value converted to a string followed by the
 *      unit value (if it is something other than EVMS_Unit_None).
 *
 */
gchar *convert_value_to_string_with_unit (value_t value, value_type_t type, value_unit_t unit, value_format_t format)
{
    gchar *string;
    gchar *unit_string;

    unit_string  = make_unit_readable_string (unit, TRUE);

    if (unit_string != NULL)
    {
        gchar *value_string;

        value_string = convert_value_to_string (value, type, format);

        string = g_strjoin (" ", value_string, unit_string, NULL);

        g_free (unit_string);
        g_free (value_string);
    }
    else
    {
        string = convert_value_to_string (value, type, format);
    }

    return string;
}

/*
 *
 *   gchar *make_value_readable_string (value_t, value_type_t, value_unit_t, value_format_t, gboolean)
 *
 *   Description:
 *      This routine takes a value and converts it to a string with
 *      the short form of the unit value concatenated to the string.
 *      If ok_to_convert is TRUE, it converts it first to a more
 *      readable form, e.g. sectors or bytes to MB or GB.
 *
 *   Entry:
 *      value         - the union of the basic data types
 *      type          - the id of the data type
 *      unit          - the value unit measurement
 *      format        - the format for the the display value
 *      ok_to_convert - TRUE if we are allowed to convert 
 *                      the value to something more readable
 *
 *   Exit:
 *      Returns the value converted to a string followed by the
 *      unit value (if it is something other than EVMS_Unit_None).
 *
 */
gchar *make_value_readable_string (value_t value, value_type_t type, value_unit_t unit, 
                                   value_format_t format, gboolean ok_to_convert)
{
    gchar *string;

    switch (unit)
    {
        case EVMS_Unit_Sectors:
            if (ok_to_convert)
                string = make_sectors_readable_string (convert_value_to_ui64 (value, type));
            else
                string = convert_value_to_string_with_unit (value, type, unit, format);
            break;

        case EVMS_Unit_Kilobytes:
            if (ok_to_convert)
                string = make_kbytes_readable_string (convert_value_to_ui64 (value, type));
            else
                string = convert_value_to_string_with_unit (value, type, unit, format);
            break;

        case EVMS_Unit_Bytes: {
            u_int64_t bytes = convert_value_to_ui64 (value, type);

            /*
             * While admittedly not very accurate we will convert
             * the bytes to a sector unit multiple to display in
             * KB, MB, etc. if we are allowed to convert and the
             * number of bytes is greater than 64K.
             */

            if (ok_to_convert && bytes > 65535)
                string = make_sectors_readable_string (bytes/EVMS_VSECTOR_SIZE);
            else
                string = convert_value_to_string_with_unit (value, type, unit, format);
        }
            break;

        default:
            string = convert_value_to_string_with_unit (value, type, unit, format);
            break;
    }

    return string;
}

/*
 *
 *   gchar* validate_string_as_numeric_value (gchar *, value_type_t)
 *
 *   Description:
 *      This routine takes a string and a value_type_t identifier
 *      and validates that the string as numeric is within bounds
 *      for the basic numeric type it represents.
 *
 *   Entry:
 *      string - the value in text form
 *      type   - the id of the data type
 *
 *   Exit:
 *      Returns NULL if the string represents a valid numeric type
 *      or a message indicating the acceptable range otherwise.
 *      Any string returned should be freed with g_free() when no
 *      longer needed.
 *
 */
gchar* validate_string_as_numeric_value (gchar *string, value_type_t type)
{
    gchar *acceptable_values_msg = NULL;

    return acceptable_values_msg;
}

/*
 *
 *   gboolean values_are_equal (value_t, value_t, value_type_t)
 *
 *   Description:
 *      This routine compares to values of the same type for equality.
 *
 *   Entry:
 *      value1 - the first value
 *      value2 - the second value
 *      type   - the id of the data type of the values
 *
 *   Exit:
 *      Returns TRUE if the values are equal, FALSE otherwise.
 *
 */
gboolean values_are_equal (value_t value1, value_t value2, value_type_t type)
{
    gboolean result = FALSE;

    switch (type)
    {
        case EVMS_Type_String:
            if (value1.s == value2.s)
                result = TRUE;
            else if (value1.s && value2.s)
                result = g_strcasecmp (value1.s, value2.s) == 0;
            break;

        case EVMS_Type_Char:
        case EVMS_Type_Unsigned_Char:
            result = value1.uc == value2.uc;
            break;

        case EVMS_Type_Real32:
            result = value1.r32 == value2.r32;
            break;

        case EVMS_Type_Real64:
            result = value1.r64 == value2.r64;
            break;

        case EVMS_Type_Int8:
        case EVMS_Type_Unsigned_Int8:
            result = value1.ui8 == value2.ui8;
            break;

        case EVMS_Type_Int16:
        case EVMS_Type_Unsigned_Int16:
            result = value1.ui16 == value2.ui16;
            break;

        case EVMS_Type_Int:
        case EVMS_Type_Int32:
        case EVMS_Type_Unsigned_Int:
        case EVMS_Type_Unsigned_Int32:
            result = value1.ui32 == value2.ui32;
            break;

        case EVMS_Type_Int64:
        case EVMS_Type_Unsigned_Int64:
            result = value1.ui64 == value2.ui64;
            break;

        default:
            log_warning ("%s: Value type of %d was not handled.\n", __FUNCTION__, type);
    }

    return result;
}

/*
 *
 *   gboolean value_in_value_list (value_t, value_list_t *, value_type_t)
 *
 *   Description:
 *      This routine determines if a value like the one given exists
 *      in the given value list.
 *
 *   Entry:
 *      value - the value
 *      vlist - the value list to search through
 *      type  - the id of the data type of the values
 *
 *   Exit:
 *      Returns TRUE if the value is in the list, FALSE otherwise.
 *
 */
gboolean value_in_value_list (value_t value, value_list_t *vlist, value_type_t type)
{
    guint    i;
    gboolean is_value_in_list=FALSE;

    for (i=0; i < vlist->count; i++)
    {
        if (values_are_equal (value, vlist->value[i], type))
        {
            is_value_in_list = TRUE;
            break;
        }
    }

    return is_value_in_list;
}

