/*
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: LvmUtils
 * File: evms_lvdisplay.c
 *
 *	Emulates LVM's 'lvdisplay' utility using the EVMS Engine. All options
 *	and several status messages are based on the original lvscan command
 *	from Heinz Mauelshagen and Sistina Software (www.sistina.com).
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <getopt.h>
#include <libgen.h>
#include <frontend.h>

typedef struct cmd_options_s {
	int debug;
	int help;
	int verbose;
	int version;
} cmd_options_t;

static char * cmd = NULL;
static cmd_options_t opts;

#include "helpers/get_region_handle.c"
#include "helpers/is_lvm_region.c"
#include "helpers/open_engine.c"
#include "helpers/print_ext_info_item.c"
#include "helpers/remove_duplicates.c"


static int showheader( void )
{
	// VERSION and DATE are defined in the top-level make.rules
	printf("Enterprise Volume Management System\n");
	printf("International Business Machines  %s\n", DATE);
	printf("LVM Emulation Utilities %s\n\n", VERSION);
	return 0;
}


static int showhelp( void )
{
	showheader();
	printf("\n");
	printf("%s - Logical Volume Display\n\n", cmd);
	printf("Synopsis:\n");
	printf("---------\n\n");
	printf("%s\n", cmd);
	printf("\t[-c/--colon]\n");
	printf("\t[-d/--debug]\n");
	printf("\t[-D/--disk]\n");
	printf("\t[-h/-?/--help]\n");
	printf("\t[-v[v]/--verbose [--verbose]]\n");
	printf("\t[-V/--version]\n");
	printf("\tLogicalVolume[Path] [LogicalVolume[Path]...]\n\n");
	return 0;
}


static int parse_options( int		argc,
			char		** argv )
{
	int		c;
	char		* short_opts = "cdDh?vV";
	struct option	long_opts[] = {
				{ "colon",	no_argument, NULL, 'c'},
				{ "debug",	no_argument, NULL, 'd'},
				{ "disk",	no_argument, NULL, 'D'},
				{ "help",	no_argument, NULL, 'h'},
				{ "verbose",	no_argument, NULL, 'v'},
				{ "version",	no_argument, NULL, 'V'},
				{ NULL, 0, NULL, 0} };

	while ( (c = getopt_long(argc, argv, short_opts,
				long_opts, NULL)) != EOF ) {
		switch (c) {
		case 'c':
			// -c is ignored by EVMS
			break;
		case 'd':
			opts.debug++;
			opts.verbose++;
			break;
		case 'D':
			// -D is ignored by EVMS
			break;
		case 'h':
		case '?':
			opts.help++;
			break;
		case 'v':
			opts.verbose++;
			break;
		case 'V':
			opts.version++;
			break;
		default:
			printf("%s -- unrecognized option \"%c\"\n\n", cmd, c);
			return EINVAL;
		}
	}

	return 0;
}


int main( int argc, char * argv[] )
{
	object_handle_t			region_handle;
	extended_info_array_t		* region_ext_info = NULL;
	extended_info_array_t		* extent_info = NULL;
	char				* lv_names[256] = {NULL};
	int				number_of_lvs;
	int				log_level = DEFAULT;
	int				i, j, rc;

	memset(&opts, 0, sizeof(cmd_options_t));
	cmd = basename(argv[0]);

	// Get the command line options.
	rc = parse_options(argc, argv);
	if (rc) {
		showhelp();
		return rc;
	}
	if ( opts.help ) {
		showhelp();
		return 0;
	}
	if ( opts.version ) {
		showheader();
		return 0;
	}
	if ( opts.verbose ) {
		log_level = DEBUG;
	}
	if ( opts.debug ) {
		log_level = ENTRY_EXIT;
	}

	// Check for LV names
	if ( optind == argc ) {
		printf("%s -- please enter a logical volume path\n\n", cmd);
		return EINVAL;
	}
	number_of_lvs = argc - optind;

	// Copy the LV names from the command line, detecting any duplicates.
	remove_duplicates( &argv[optind], lv_names, number_of_lvs);

	// Open the engine.
	rc = open_engine(ENGINE_READWRITE, log_level);
	if (rc) {
		return rc;
	}

	// Examine each logical volume
	for ( i = 0; i < number_of_lvs; i++ ) {

		// Get the handle for this LV.
		rc = get_region_handle(lv_names[i], &region_handle);
		if (rc) {
			continue;
		}

		// Does this region belong to LVM?
		if ( ! is_lvm_region(region_handle) ) {
			printf("%s -- Region %s does not belong to LVM.\n", cmd, lv_names[i]);
			continue;
		}

		// Get the extended info for the LV.
		rc = evms_get_extended_info(region_handle, NULL, &region_ext_info);
		if (rc) {
			printf("%s -- Error getting extended info for LV %s (%d)\n", cmd, lv_names[i], rc);
			continue;
		}

		// Display all available information
		printf("--- Logical Volume ---\n");
		for ( j = 0; j < region_ext_info->count; j++ ) {
			print_ext_info_item(&(region_ext_info->info[j]));
		}
		printf("\n");
		evms_free(region_ext_info);

		if ( opts.verbose ) {
			// Get the info for the LE maps.
			rc = evms_get_extended_info(region_handle, "Extents", &extent_info);
			if (rc) {
				printf("%s -- Error getting LE mapping information for LV %s (%d)\n", cmd, lv_names[i], rc);
				continue;
			}

			for ( j = 0; j < extent_info->count; j++ ) {
				print_ext_info_item(&(extent_info->info[j]));
			}
			printf("\n");
			evms_free(extent_info);
		}
	}

	evms_close_engine();
	return 0;
}

