/*
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: lvmregmgr
 * File: lvm_info.c
 *
 * Description: 
 */


#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <plugin.h>
#include "lvmregmgr.h"


/* Function: lvm_get_volume_info
 *
 *	Fill in all the basic information about this volume.
 */
int lvm_get_volume_info(lvm_logical_volume_t	* volume,
			extended_info_array_t	** info_array )
{
	extended_info_array_t	* info = NULL;
	lvm_logical_volume_t	* snap_volume = NULL;
	int			j,i = 0;

	LOG_ENTRY;

	if ( volume->lv->lv_access & LV_SNAPSHOT ) {
		lvm_update_snapshot_stats(volume);
	}

	// Get memory for the info array
	if ( ! (info = lvm_engine->engine_alloc(sizeof(extended_info_array_t) + sizeof(extended_info_t)*14)) ) {
		LOG_CRITICAL("Memory error creating info array\n");
		RETURN(ENOMEM);
	}

	// Region Name
	SET_STRING(info->info[i].name, "LV_Name");
	SET_STRING(info->info[i].title, "Region Name");
	SET_STRING(info->info[i].desc, "Name of LVM Region (LV)");
	info->info[i].type = EVMS_Type_String;
	SET_STRING(info->info[i].value.s, volume->region->name);
	i++;

	// Container Name
	SET_STRING(info->info[i].name, "VG_Name");
	SET_STRING(info->info[i].title, "Container Name");
	SET_STRING(info->info[i].desc, "Name of LVM Container (VG)");
	info->info[i].type = EVMS_Type_String;
	SET_STRING(info->info[i].value.s, volume->group->container->name);
	i++;

	// LV Number
	SET_STRING(info->info[i].name, "LV_Number");
	SET_STRING(info->info[i].title, "Region Number");
	SET_STRING(info->info[i].desc, "ID number for this region in this container");
	info->info[i].type = EVMS_Type_Int;
	info->info[i].value.i = volume->number;
	i++;

	// LV Size
	SET_STRING(info->info[i].name, "LV_Size");
	SET_STRING(info->info[i].title, "Region Size");
	SET_STRING(info->info[i].desc, "Usable space for this region");
	info->info[i].type = EVMS_Type_Unsigned_Int32;
	info->info[i].unit = EVMS_Unit_Sectors;
	info->info[i].value.ui32 = volume->region->size;
	i++;

	// Allocated LEs
	SET_STRING(info->info[i].name, "Extents");
	SET_STRING(info->info[i].title, "Logical Extents");
	SET_STRING(info->info[i].desc, "Number of logical extents used by this region");
	info->info[i].type = EVMS_Type_Unsigned_Int32;
	info->info[i].value.ui32 = volume->lv->lv_allocated_le;
	info->info[i].flags = EVMS_EINFO_FLAGS_MORE_INFO_AVAILABLE;
	i++;

	// LV Status
	SET_STRING(info->info[i].name, "LV_Status");
	SET_STRING(info->info[i].title, "Region Status");
	info->info[i].type = EVMS_Type_String;
	if ( volume->lv->lv_status & LV_ACTIVE ) {
		SET_STRING(info->info[i].value.s, "Available");
	}
	else {
		SET_STRING(info->info[i].value.s, "Unavailable");
	}
	i++;

	// Allocation Scheme
	SET_STRING(info->info[i].name, "Allocation");
	SET_STRING(info->info[i].title, "Allocation Policy");
	SET_STRING(info->info[i].desc, "How extents are allocated to this region");
	info->info[i].type = EVMS_Type_String;
	if ( volume->lv->lv_allocation & LV_CONTIGUOUS ) {
		SET_STRING(info->info[i].value.s, "Contiguous");
	}
	else {
		SET_STRING(info->info[i].value.s, "Next-free");
	}
	i++;

	// Write Permissions
	SET_STRING(info->info[i].name, "Permissions");
	SET_STRING(info->info[i].title, "Access Permissions");
	info->info[i].type = EVMS_Type_String;
	if ( volume->lv->lv_access & LV_WRITE && volume->lv->lv_access & LV_READ ) {
		SET_STRING(info->info[i].value.s, "Read-Write");
	}
	else if ( volume->lv->lv_access & LV_WRITE ) {
		SET_STRING(info->info[i].value.s, "Write-Only");
	}
	else if ( volume->lv->lv_access & LV_READ ) {
		SET_STRING(info->info[i].value.s, "Read-Only");
	}
	else {
		SET_STRING(info->info[i].value.s, "No-Access");
	}
	i++;

	// Striping Info
	if ( volume->lv->lv_stripes > 1 ) {
		// Number of Stripes
		SET_STRING(info->info[i].name, "Stripes");
		SET_STRING(info->info[i].title, "Stripes");
		SET_STRING(info->info[i].desc, "Number of objects this region is striped across");
		info->info[i].type = EVMS_Type_Unsigned_Int32;
		info->info[i].value.ui32 = volume->lv->lv_stripes;
		i++;

		// Stripe Size
		SET_STRING(info->info[i].name, "Stripe_Size");
		SET_STRING(info->info[i].title, "Stripe Size");
		info->info[i].type = EVMS_Type_Unsigned_Int32;
		info->info[i].unit = EVMS_Unit_Sectors;
		info->info[i].value.ui32 = volume->lv->lv_stripesize;
		i++;
	}
	
	// Snapshot Info
	if ( volume->lv->lv_access & LV_SNAPSHOT_ORG ) {
		// Count number of snapshots
		for ( j = 0, snap_volume = volume->snapshot_next; snap_volume; j++, snap_volume = snap_volume->snapshot_next ) ;

		// Snapshot names
		SET_STRING(info->info[i].name, "Snapshot_Org");
		SET_STRING(info->info[i].title, "Snapshotted By");
		SET_STRING(info->info[i].desc, "Regions that are snapshotting this region");
		info->info[i].type = EVMS_Type_String;
		info->info[i].collection_type = EVMS_Collection_List;
		info->info[i].collection.list = lvm_engine->engine_alloc(sizeof(value_list_t) + sizeof(value_t)*(j));
		info->info[i].collection.list->count = j;
		for ( j = 0, snap_volume = volume->snapshot_next; snap_volume; j++, snap_volume = snap_volume->snapshot_next ) {
			SET_STRING(info->info[i].collection.list->value[j].s, snap_volume->region->name);
		}
		i++;
	}
	else if ( volume->lv->lv_access & LV_SNAPSHOT ) {
		// Name of Original
		SET_STRING(info->info[i].name, "Snapshot");
		SET_STRING(info->info[i].title, "Snapshot Of");
		SET_STRING(info->info[i].desc, "Region that this region is snapshotting");
		info->info[i].type = EVMS_Type_String;
		SET_STRING(info->info[i].value.s, volume->snapshot_org->region->name);
		i++;

		// Chunk size
		SET_STRING(info->info[i].name, "Chunk_Size");
		SET_STRING(info->info[i].title, "Chunk Size");
		SET_STRING(info->info[i].desc, "Size of each snapshot-remapped piece of data");
		info->info[i].type = EVMS_Type_Unsigned_Int32;
		info->info[i].unit = EVMS_Unit_Sectors;
		info->info[i].value.ui32 = volume->lv->lv_chunk_size;
		i++;

		// Percent Full
		SET_STRING(info->info[i].name, "Percent_Full");
		SET_STRING(info->info[i].title, "Percent Full");
		SET_STRING(info->info[i].desc, "Percentage of space that has been consumed on the snapshot region");
		info->info[i].type = EVMS_Type_Real32;
		info->info[i].unit = EVMS_Unit_Percent;
		info->info[i].value.r32 = ((float)(volume->next_free_chunk) / (float)(volume->lv->lv_size)) * 100.0;
		i++;

		if ( volume->next_free_chunk >= volume->lv->lv_size &&
		     ! (volume->lv->lv_status & LV_ACTIVE) ) {
			// Snapshot is full
			SET_STRING(info->info[i].name, "Snapshot_Full");
			SET_STRING(info->info[i].title, "Snapshot region is full and should be deleted.");
			info->info[i].type = EVMS_Type_String;
			SET_STRING(info->info[i].value.s, "");
			i++;
		}
	}

	// Minor Number
	if ( volume->region->volume ) {
		SET_STRING(info->info[i].name, "Minor_Number");
		SET_STRING(info->info[i].title, "Device Minor Number");
		info->info[i].type = EVMS_Type_Unsigned_Int32;
		info->info[i].value.ui32 = volume->region->volume->minor_number;
		i++;
	}

	if ( volume->flags & LVM_LV_FLAG_INCOMPLETE ) {
		SET_STRING(info->info[i].name, "Incomplete_LV");
		SET_STRING(info->info[i].title, "INCOMPLETE REGION!!!");
		SET_STRING(info->info[i].desc, "This region has extents which are not mapped! One or more objects are probably missing from this container!");
		info->info[i].type = EVMS_Type_String;
		SET_STRING(info->info[i].value.s, "");
		i++;
	}

	info->count = i;
	*info_array = info;

	RETURN(0);
}


static inline int is_next_le_consecutive(lvm_logical_volume_t	* volume,
					int			this_le )
{
	if ( volume->le_map[this_le].owning_pv == volume->le_map[this_le+1].owning_pv &&
	     volume->le_map[this_le].pe_number + 1 == volume->le_map[this_le+1].pe_number ) {
		return TRUE;
	}
	else {
		return FALSE;
	}
}


/* Function: lvm_get_volume_extent_info
 *
 *	Fill in the "extra" extended info about the LE-to-PE mapping for
 *	this volume.
 */
int lvm_get_volume_extent_info(	lvm_logical_volume_t	* volume,
				extended_info_array_t	** info_array )
{
	extended_info_array_t	* info = NULL;
	char			buffer[150] = {0};
	int			printed_dots = FALSE;
	int			consecutive_run = FALSE;
	int			info_line = 0;
	int			j, i = 0;

	LOG_ENTRY;

	// Get memory for the info array
	if ( ! (info = lvm_engine->engine_alloc(sizeof(extended_info_array_t))) ) {
		LOG_CRITICAL("Memory error creating info array\n");
		RETURN(ENOMEM);
	}

	// Extent mappings
	SET_STRING(info->info[i].name, "Extent_Map");
	SET_STRING(info->info[i].title, "Logical Extents");
	SET_STRING(info->info[i].desc, "LE Number : PV Name : PE Number");
	info->info[i].type = EVMS_Type_String;
	info->info[i].collection_type = EVMS_Collection_List;
	info->info[i].collection.list = lvm_engine->engine_alloc(sizeof(value_list_t) + sizeof(value_t)*(volume->lv->lv_allocated_le));
	snprintf(buffer, 150, "%-5s : %-15s : %-5s", "LE #", "PV Name", "PE #");
	SET_STRING(info->info[i].collection.list->value[info_line].s, buffer);
	info_line++;

	// Print info for each LE. Condense runs of consecutive LEs into
	// the first and last, with a ... in between.
	for ( j = 0; j < volume->lv->lv_allocated_le; j++ ) {
		if ( ! volume->le_map[j].owning_pv ) {
			snprintf(buffer, 150, "%-5d : %-15s : %-5s", j, "Missing PV", "");
			consecutive_run = FALSE;
		}
		// Are we at the end of the LV or at the end of a run
		// of consecutive LEs?
		else if ( j == volume->lv->lv_allocated_le - 1 ||
		          ! is_next_le_consecutive(volume, j) ) {
			snprintf(buffer, 150, "%-5d : %-15s : %-5d", j, volume->le_map[j].owning_pv->segment->name, volume->le_map[j].pe_number);
			consecutive_run = FALSE;
		}
		// Are we at the start of the LV or at the start of a
		// run of consecutive LEs?
		else if ( ! consecutive_run ) {
			snprintf(buffer, 150, "%-5d : %-15s : %-5d", j, volume->le_map[j].owning_pv->segment->name, volume->le_map[j].pe_number);
			consecutive_run = TRUE;
			printed_dots = FALSE;
		}
		// Have we printed the ... yet?
		else if ( ! printed_dots ) {
			snprintf(buffer, 150, ".....");
			printed_dots = TRUE;
		}
		else {
			continue;
		}
		SET_STRING(info->info[i].collection.list->value[info_line].s, buffer);
		info_line++;
	}
	info->info[i].collection.list->count = info_line;
	i++;

	info->count = i;
	*info_array = info;

	RETURN(0);
}


/* Function: lvm_get_group_info
 */
int lvm_get_group_info( lvm_volume_group_t	* group,
			extended_info_array_t	** info_array )
{
	extended_info_array_t	* info = NULL;
	char			* buf = NULL;
	int			i = 0;

	LOG_ENTRY;

	// Get memory for the info array
	if ( ! (info = lvm_engine->engine_alloc(sizeof(extended_info_array_t) + sizeof(extended_info_t)*12)) ) {
		LOG_CRITICAL("Memory error creating info array\n");
		RETURN(ENOMEM);
	}

	// Container Name
	SET_STRING(info->info[i].name, "VG_Name");
	SET_STRING(info->info[i].title, "Container Name");
	SET_STRING(info->info[i].desc, "Name of LVM Container (VG)");
	info->info[i].type = EVMS_Type_String;
	SET_STRING(info->info[i].value.s, group->container->name);
	i++;

	// VG Number
	SET_STRING(info->info[i].name, "VG_Number");
	SET_STRING(info->info[i].title, "Container Number");
	SET_STRING(info->info[i].desc, "ID number for this container in the LVM plugin");
	info->info[i].type = EVMS_Type_Int;
	info->info[i].value.i = group->vg->vg_number;
	i++;

	// VG Size
	SET_STRING(info->info[i].name, "VG_Size");
	SET_STRING(info->info[i].title, "Container Size");
	SET_STRING(info->info[i].desc, "Total accumulated space in this container");
	info->info[i].type = EVMS_Type_Unsigned_Int32;
	info->info[i].unit = EVMS_Unit_Sectors;
	info->info[i].value.ui32 = group->container->size;
	i++;

	// Available VG Size
	SET_STRING(info->info[i].name, "VG_Free_Size");
	SET_STRING(info->info[i].title, "Available Size");
	SET_STRING(info->info[i].desc, "Amount of space currently available for allocating to regions");
	info->info[i].type = EVMS_Type_Unsigned_Int32;
	info->info[i].unit = EVMS_Unit_Sectors;
	info->info[i].value.ui32 = group->freespace->lv->lv_size;
	i++;

	// Percent Allocated
	SET_STRING(info->info[i].name, "VG_Percent_Allocated");
	SET_STRING(info->info[i].title, "Percent Allocated");
	SET_STRING(info->info[i].desc, "Percentage of space currently allocated to regions");
	info->info[i].type = EVMS_Type_Real32;
	info->info[i].unit = EVMS_Unit_Percent;
	info->info[i].value.r32 = ((float)(group->vg->pe_allocated) / (float)(group->vg->pe_total)) * 100.0;
	i++;

	// PE Size
	SET_STRING(info->info[i].name, "PE_Size");
	SET_STRING(info->info[i].title, "Extent Size");
	SET_STRING(info->info[i].desc, "Size of each extent available for allocating to regions");
	info->info[i].type = EVMS_Type_Unsigned_Int32;
	info->info[i].unit = EVMS_Unit_Sectors;
	info->info[i].value.ui32 = group->vg->pe_size;
	i++;

	// Total PEs
	SET_STRING(info->info[i].name, "Total_PEs");
	SET_STRING(info->info[i].title, "Total PEs");
	SET_STRING(info->info[i].desc, "Total number of extents in the container");
	info->info[i].type = EVMS_Type_Unsigned_Int32;
	info->info[i].value.ui32 = group->vg->pe_total;
	i++;

	// Available PEs
	SET_STRING(info->info[i].name, "Available_PEs");
	SET_STRING(info->info[i].title, "Available PEs");
	SET_STRING(info->info[i].desc, "Number of extents available for allocating to regions");
	info->info[i].type = EVMS_Type_Unsigned_Int32;
	info->info[i].value.ui32 = group->freespace->lv->lv_allocated_le;
	i++;

	// Number of PVs
	SET_STRING(info->info[i].name, "Current_PVs");
	SET_STRING(info->info[i].title, "Number of Objects (PVs)");
	SET_STRING(info->info[i].desc, "Number of objects in this container");
	info->info[i].type = EVMS_Type_Unsigned_Int32;
	info->info[i].value.ui32 = group->pv_count;
	info->info[i].flags = EVMS_EINFO_FLAGS_MORE_INFO_AVAILABLE;
	i++;

	// Number of LVs
	SET_STRING(info->info[i].name, "Current_LVs");
	SET_STRING(info->info[i].title, "Number of Regions (LVs)");
	SET_STRING(info->info[i].desc, "Number of regions in this container");
	info->info[i].type = EVMS_Type_Unsigned_Int32;
	info->info[i].value.ui32 = group->volume_count;
	info->info[i].flags = EVMS_EINFO_FLAGS_MORE_INFO_AVAILABLE;
	i++;

	// Max LV Size
	SET_STRING(info->info[i].name, "Max_LV_Size");
	SET_STRING(info->info[i].title, "Maximum Region Size");
	SET_STRING(info->info[i].desc, "Maximum possible size of any region in this container for the given PE size");
	info->info[i].type = EVMS_Type_Unsigned_Int32;
	info->info[i].unit = EVMS_Unit_Sectors;
	info->info[i].value.ui32 = group->vg->pe_size * 64 * 1024;
	i++;

	// VG UUID
	SET_STRING(info->info[i].name, "VG_UUID");
	SET_STRING(info->info[i].title, "Container UUID");
	info->info[i].type = EVMS_Type_String;
	buf = lvm_print_uuid(group->vg->vg_uuid);
	SET_STRING(info->info[i].value.s, buf);
	i++;

	if ( group->pv_count < group->vg->pv_cur ) {
		SET_STRING(info->info[i].name, "Incomplete_VG");
		SET_STRING(info->info[i].title, "INCOMPLETE CONTAINER!!!");
		SET_STRING(info->info[i].desc, "This container seems to be missing one or more objects!");
		info->info[i].type = EVMS_Type_String;
		SET_STRING(info->info[i].value.s, "");
		i++;
	}

	info->count = i;
	*info_array = info;

	RETURN(0);
}


/* Function: lvm_get_group_pv_list_info
 */
int lvm_get_group_pv_list_info( lvm_volume_group_t	* group,
				extended_info_array_t	** info_array )
{
	extended_info_array_t	* info = NULL;
	char			buf[50] = {0};
	int			j, i = 0;

	LOG_ENTRY;

	// Get memory for the info array
	if ( ! (info = lvm_engine->engine_alloc(sizeof(extended_info_array_t) + sizeof(extended_info_t)*group->pv_count-1)) ) {
		LOG_CRITICAL("Memory error creating info array\n");
		RETURN(ENOMEM);
	}

	for ( j = 1; j <= MAX_PV; j++ ) {
		if ( group->pv_list[j] ) {
			sprintf(buf, "PV%d", j);
			SET_STRING(info->info[i].name, buf);
			sprintf(buf, "Object (PV) %d", j);
			SET_STRING(info->info[i].title, buf);
			info->info[i].type = EVMS_Type_String;
			SET_STRING(info->info[i].value.s, group->pv_list[j]->segment->name);
			info->info[i].flags = EVMS_EINFO_FLAGS_MORE_INFO_AVAILABLE;
			i++;
		}
	}

	info->count = i;
	*info_array = info;

	RETURN(0);
}


/* Function: lvm_get_group_lv_list_info
 */
int lvm_get_group_lv_list_info( lvm_volume_group_t	* group,
				extended_info_array_t	** info_array )
{
	extended_info_array_t	* info = NULL;
	char			buf[50] = {0};
	int			j, i = 0;

	LOG_ENTRY;

	// Get memory for the info array
	if ( ! (info = lvm_engine->engine_alloc(sizeof(extended_info_array_t) + sizeof(extended_info_t)*group->volume_count-1)) ) {
		LOG_CRITICAL("Memory error creating info array\n");
		RETURN(ENOMEM);
	}

	for ( j = 1; j <= MAX_LV; j++ ) {
		if ( group->volume_list[j] ) {
			sprintf(buf, "LV%d", j);
			SET_STRING(info->info[i].name, buf);
			sprintf(buf, "Region (LV) %d", j);
			SET_STRING(info->info[i].title, buf);
			info->info[i].type = EVMS_Type_String;
			SET_STRING(info->info[i].value.s, group->volume_list[j]->region->name);
			i++;
		}
	}

	info->count = i;
	*info_array = info;

	RETURN(0);
}


/* Function: lvm_get_pv_info
 */
int lvm_get_pv_info(	lvm_physical_volume_t	* pv_entry,
			extended_info_array_t	** info_array )
{
	extended_info_array_t	* info = NULL;
	char			* buf1 = NULL;
	char			buf2[50] = {0};
	int			i = 0;

	LOG_ENTRY;

	// Get memory for the info array
	if ( ! (info = lvm_engine->engine_alloc(sizeof(extended_info_array_t) + sizeof(extended_info_t)*9)) ) {
		LOG_CRITICAL("Memory error creating info array\n");
		RETURN(ENOMEM);
	}

	// PV Name
	SET_STRING(info->info[i].name, "PV_Name");
	SET_STRING(info->info[i].title, "Object Name");
	SET_STRING(info->info[i].desc, "Name of LVM Object (PV)");
	info->info[i].type = EVMS_Type_String;
	SET_STRING(info->info[i].value.s, pv_entry->segment->name);
	i++;

	// VG Name
	SET_STRING(info->info[i].name, "VG_Name");
	SET_STRING(info->info[i].title, "Container Name");
	SET_STRING(info->info[i].desc, "Name of LVM Container (VG)");
	info->info[i].type = EVMS_Type_String;
	SET_STRING(info->info[i].value.s, pv_entry->group->container->name);
	i++;

	// PV Number
	SET_STRING(info->info[i].name, "PV_Number");
	SET_STRING(info->info[i].title, "Object Number");
	SET_STRING(info->info[i].desc, "ID number for this object (PV) in this container");
	info->info[i].type = EVMS_Type_Int;
	info->info[i].value.i = pv_entry->number;
	i++;

	// PV Size
	SET_STRING(info->info[i].name, "PV_Size");
	SET_STRING(info->info[i].title, "Object Size");
	info->info[i].type = EVMS_Type_Unsigned_Int32;
	info->info[i].unit = EVMS_Unit_Sectors;
	info->info[i].value.ui32 = pv_entry->segment->size;
	i++;

	// Usable Size
	SET_STRING(info->info[i].name, "Usable_PV_Size");
	SET_STRING(info->info[i].title, "Usable Space");
	SET_STRING(info->info[i].desc, "PE Size * Number of PEs. Some object space is used for metadata, some is unusable due to PE size");
	info->info[i].type = EVMS_Type_Unsigned_Int32;
	info->info[i].unit = EVMS_Unit_Sectors;
	info->info[i].value.ui32 = pv_entry->pv->pe_size * pv_entry->pv->pe_total;
	i++;

	// Number of Regions
	SET_STRING(info->info[i].name, "Current_Regions");
	SET_STRING(info->info[i].title, "Current Regions");
	SET_STRING(info->info[i].desc, "Number of regions currently using space on this object");
	info->info[i].type = EVMS_Type_Unsigned_Int32;
	info->info[i].value.ui32 = pv_entry->pv->lv_cur;
	i++;

	// PE Size
	SET_STRING(info->info[i].name, "PE_Size");
	SET_STRING(info->info[i].title, "Extent Size");
	SET_STRING(info->info[i].desc, "Size of each extent available for allocating to regions");
	info->info[i].type = EVMS_Type_Unsigned_Int32;
	info->info[i].unit = EVMS_Unit_Sectors;
	info->info[i].value.ui32 = pv_entry->pv->pe_size;
	i++;

	// Total PEs
	sprintf(buf2, "PEMapPV%ld", pv_entry->number);
	SET_STRING(info->info[i].name, buf2);
	SET_STRING(info->info[i].title, "Total PEs");
	SET_STRING(info->info[i].desc, "Total number of extents in this object");
	info->info[i].type = EVMS_Type_Unsigned_Int32;
	info->info[i].value.ui32 = pv_entry->pv->pe_total;
	info->info[i].flags = EVMS_EINFO_FLAGS_MORE_INFO_AVAILABLE;
	i++;

	// Available PEs
	SET_STRING(info->info[i].name, "Available_PEs");
	SET_STRING(info->info[i].title, "Available PEs");
	SET_STRING(info->info[i].desc, "Number of extents available for allocating to regions");
	info->info[i].type = EVMS_Type_Unsigned_Int32;
	info->info[i].value.ui32 = pv_entry->pv->pe_total - pv_entry->pv->pe_allocated;
	i++;

	// PV UUID
	SET_STRING(info->info[i].name, "PV_UUID");
	SET_STRING(info->info[i].title, "Object (PV) UUID");
	info->info[i].type = EVMS_Type_String;
	buf1 = lvm_print_uuid(pv_entry->pv->pv_uuid);
	SET_STRING(info->info[i].value.s, buf1);
	i++;

	info->count = i;
	*info_array = info;
	RETURN(0);
}


static inline int is_next_pe_consecutive(lvm_physical_volume_t	* pv_entry,
					int			this_pe )
{
	if ( pv_entry->pe_map[this_pe].lv_num == pv_entry->pe_map[this_pe+1].lv_num ) {
		if ( pv_entry->pe_map[this_pe].lv_num ) {
			if ( pv_entry->pe_map[this_pe].le_num + 1 == pv_entry->pe_map[this_pe+1].le_num ) {
				return TRUE;
			}
			else {
				return FALSE;
			}
		}
		else {
			return TRUE;
		}
	}
	else {
		return FALSE;
	}
}

/* Function: lvm_get_pv_extent_info
 */
int lvm_get_pv_extent_info(	lvm_physical_volume_t	* pv_entry,
				extended_info_array_t	** info_array )
{
	extended_info_array_t	* info = NULL;
	char			buffer[150] = {0};
	int			consecutive_run = FALSE;
	int			printed_dots = FALSE;
	int			info_line = 1;
	int			j,i = 0;

	LOG_ENTRY;

	// Get memory for the info array
	if ( ! (info = lvm_engine->engine_alloc(sizeof(extended_info_array_t))) ) {
		LOG_CRITICAL("Memory error creating info array\n");
		RETURN(ENOMEM);
	}

	// Extent mappings
	SET_STRING(info->info[i].name, "Extent_Map");
	SET_STRING(info->info[i].title, "Physical Extents");
	SET_STRING(info->info[i].desc, "PE Number : LV Name : LE Number: Sector");
	info->info[i].type = EVMS_Type_String;
	info->info[i].collection_type = EVMS_Collection_List;
	info->info[i].collection.list = lvm_engine->engine_alloc(sizeof(value_list_t) + sizeof(value_t)*(pv_entry->pv->pe_total));
	snprintf(buffer, 150, "%-5s : %-20s : %-5s : %-5s", "PE #", "LV Name", "LE #", "Sector");
	SET_STRING(info->info[i].collection.list->value[0].s, buffer);

	// Print info for each PE. Condense runs of consecutive PEs into
	// the first and last, with a ... in between.
	for ( j = 0; j < pv_entry->pv->pe_total; j++ ) {
		if ( pv_entry->group->volume_list[pv_entry->pe_map[j].lv_num] ) {
			// PE is mapped to a volume.
			if ( j == pv_entry->pv->pe_total ||
			     ! is_next_pe_consecutive(pv_entry, j) ) {
				snprintf(buffer, 150, "%-5d : %-20s : %-5d : %-5d",
					j, pv_entry->group->volume_list[pv_entry->pe_map[j].lv_num]->region->name,
					pv_entry->pe_map[j].le_num, pv_entry->pv->pe_start+pv_entry->pv->pe_size*j);
				consecutive_run = FALSE;
			}
			else if ( ! consecutive_run ) {
				snprintf(buffer, 150, "%-5d : %-20s : %-5d : %-5d",
					j, pv_entry->group->volume_list[pv_entry->pe_map[j].lv_num]->region->name,
					pv_entry->pe_map[j].le_num, pv_entry->pv->pe_start+pv_entry->pv->pe_size*j);
				consecutive_run = TRUE;
				printed_dots = FALSE;
			}
			else if ( ! printed_dots ) {
				snprintf(buffer, 150, ".....");
				printed_dots = TRUE;
			}
			else {
				continue;
			}
		}
		else {
			// PE is unallocated.
			if ( j == pv_entry->pv->pe_total - 1 ||
			     ! is_next_pe_consecutive(pv_entry, j) ) {
				snprintf(buffer, 150, "%-5d : %-20s : %-5s : %-5d",
					j, "none", "n/a", pv_entry->pv->pe_start+pv_entry->pv->pe_size*j);
				consecutive_run = FALSE;
			}
			else if ( ! consecutive_run ) {
				snprintf(buffer, 150, "%-5d : %-20s : %-5s : %-5d",
					j, "none", "n/a", pv_entry->pv->pe_start+pv_entry->pv->pe_size*j);
				consecutive_run = TRUE;
				printed_dots = FALSE;
			}
			else if ( ! printed_dots ) {
				snprintf(buffer, 150, ".....");
				printed_dots = TRUE;
			}
			else {
				continue;
			}
		}
		SET_STRING(info->info[i].collection.list->value[info_line].s, buffer);
		info_line++;
	}
	info->info[i].collection.list->count = info_line;
	i++;

	info->count = i;
	*info_array = info;
	RETURN(0);
}


