/*
 * etPan! -- a mail user agent
 *
 * Copyright (C) 2001, 2002 - DINH Viet Hoa
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the libEtPan! project nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * $Id: etpan-console-app.c,v 1.15 2005/02/01 02:31:58 hoa Exp $
 */

#include "etpan-console-app.h"
#include "etpan-app-subapp.h"
#include "etpan-subapp.h"
#include "etpan-errors.h"
#include <stdlib.h>
#include <string.h>
#include <ncurses.h>
#include "etpan-app.h"
#include <unistd.h>
#include "etpan-help-viewer.h"

static void set_fd(struct etpan_subapp * app, fd_set * fds, int * maxfd);
static void handle_fd(struct etpan_subapp * app, fd_set * fds);
static void handle_key(struct etpan_subapp * app, int key);
static void handle_resize(struct etpan_subapp * app);
static int display_init(struct etpan_subapp * app);
static void display(struct etpan_subapp * app, WINDOW * w);
static void set_color(struct etpan_subapp * app);
static int init(struct etpan_subapp * subapp);
static void done(struct etpan_subapp * subapp);
static void leave(struct etpan_subapp * app, struct etpan_subapp * new_app);


static struct etpan_subapp_driver etpan_console_app_driver = {
  .name = "console",
  .always_handle_key = 1,
  .always_on_top = 1,
  .get_idle_delay = NULL,
  .idle = NULL,
  .set_fd = set_fd,
  .handle_fd = handle_fd,
  .handle_key = handle_key,
  .handle_resize = handle_resize,
  .display = display,
  .set_color = set_color,
  .init = init,
  .done = done,
  .enter = NULL,
  .leave = leave,
  .display_init = display_init,
  .display_done = NULL,
};

#define MAX_MESSAGES 2000

struct app_state {
  struct etpan_subapp * last_subapp;
  int enabled;
  
  /* store messages */
  carray * message_list;
  unsigned int first;
  unsigned int count;
  pthread_mutex_t lock;
  
  /* cursor */
  unsigned int index;
  
  /* display */
  int message_attr;
  
  /* update fd */
  int update_fd[2];
  int flood;
};

static void set_fd(struct etpan_subapp * app, fd_set * fds, int * maxfd)
{
  struct app_state * state;
  
  state = app->data;
  
  FD_SET(state->update_fd[0], fds);
  if (state->update_fd[0] > * maxfd)
    * maxfd = state->update_fd[0];
}

static void handle_fd(struct etpan_subapp * app, fd_set * fds)
{
  struct app_state * state;
  int ack;
  
  state = app->data;
  
  if (FD_ISSET(state->update_fd[0], fds)) {
    state->flood = 0;
    read(state->update_fd[0], &ack, sizeof(ack));
  }
}

static void set_index(struct etpan_subapp * app);

static int show_help(struct etpan_subapp * app);

static void handle_key(struct etpan_subapp * app, int key)
{
  struct app_state * state;
  struct etpan_subapp * parent;
  int has_console;

  state = app->data;
  
  if (app->app->current_subapp == app) {
    switch (key) {
    case KEY_CTRL('L'):
      if (state->last_subapp != NULL) {
        etpan_subapp_set_title(app, NULL);
        etpan_app_switch_subapp(state->last_subapp, 0);
        etpan_subapp_handle_resize(app);
      }
      break;

#if 0
    case 'a':
      {
        static int count = 0;
        char toto[256];
        
        snprintf(toto, sizeof(toto), "test %i", count);
        count ++;
        etpan_console_log(app, toto);
        break;
      }
#endif
      
    case 'k':
    case KEY_UP:
      if (state->index > 0)
        state->index --;
      break;
      
    case 'j':
    case KEY_DOWN:
      if (state->index + app->display_height < state->count)
        state->index ++;
      break;
      
    case KEY_PPAGE:
      if (state->index > (unsigned int) (app->display_height - 1))
        state->index -= app->display_height - 1;
      else
        state->index = 0;
      break;
      
    case KEY_NPAGE:
      if (state->index + app->display_height
          + app->display_height - 1 < state->count)
        state->index += app->display_height - 1;
      else if (state->index + app->display_height > state->count)
        state->index = 0;
      else
        state->index = state->count - app->display_height;
      break;

    case KEY_HOME:
      state->index = 0;
      break;

    case KEY_END:
      state->index = state->count - app->display_height;
      break;
      
    case KEY_F(1):
    case '?':
      show_help(app);
      break;
      
    case 'd':
      mailstream_debug = !mailstream_debug;
      break;
      
    default:
      ETPAN_APP_DEBUG((app->app, "%o", key));
      break;
    }
  }
  else {
    switch (key) {
    case KEY_CTRL('L'):
      etpan_subapp_set_title(app, "etPan! - console");
      parent = app->app->current_subapp;
      has_console = 0;
      while (parent != NULL) {
        if (parent == app)
          has_console = 1;
        parent = parent->parent;
      }
      if (!has_console)
        state->last_subapp = app->app->current_subapp;
      etpan_app_switch_subapp(app, 0);
      
      etpan_subapp_handle_resize(app);
      break;
      
    default:
      state->enabled = 0;
      set_index(app);
      break;
    }
  }
}

#define CONSOLE_HEIGHT 1

static void set_index(struct etpan_subapp * app)
{
  struct app_state * state;
  
  state = app->data;
  if (app->app->current_subapp == app) {
    if (state->count > (unsigned int) app->display_height)
      state->index = state->count - app->display_height;
    else
      state->index = 0;
  }
  else {
    if (state->enabled) {
      if (state->count > (unsigned int) app->display_height)
        state->index = state->count - app->display_height;
      else
        state->index = 0;
    }
    else {
      if (state->count > (unsigned int) app->display_height - 1)
        state->index = state->count - app->display_height + 1;
      else
        state->index = 0;
    }
  }
}

static void handle_resize(struct etpan_subapp * app)
{
  struct app_state * state;
  
  state = app->data;
  
  if (app->app->current_subapp == app) {
    app->left = 0;
    app->top = 0;
    app->width = app->app->width;
    app->height = app->app->height;
  }
  else {
    app->left = 0;
    app->top = app->app->height - CONSOLE_HEIGHT;
    app->width = app->app->width;
    app->height = CONSOLE_HEIGHT;
  }
  etpan_subapp_set_relative_coord(app);
  etpan_subapp_update_display_size(app);
  set_index(app);
}

static int display_init(struct etpan_subapp * app)
{
  app->show_cursor = 1;
  etpan_subapp_handle_resize(app);
  
  return NO_ERROR;
}

static int display_full(struct etpan_subapp * app, WINDOW * w);

static void display(struct etpan_subapp * app, WINDOW * w)
{
  display_full(app, w);
}

static void set_color(struct etpan_subapp * app)
{
  struct app_state * state;
  
  state = app->data;
  
  etpan_app_set_color(app->app, "message", &state->message_attr, A_NORMAL);
}

static void leave(struct etpan_subapp * app, struct etpan_subapp * new_app)
{
  etpan_subapp_handle_resize(app);
}

static int init(struct etpan_subapp * subapp)
{
  struct app_state * state;
  int r;
  unsigned int i;
  
  state = malloc(sizeof(* state));
  if (state == NULL)
    goto err;
  
  subapp->data = state;
  
  state->last_subapp = NULL;
  state->enabled = 0;
  
  /* store messages */
  state->message_list = carray_new(MAX_MESSAGES);
  if (state->message_list == NULL)
    goto free;
  
  carray_set_size(state->message_list, MAX_MESSAGES);
  for(i = 0 ; i < carray_count(state->message_list) ; i ++)
    carray_set(state->message_list, i, NULL);

  state->first = 0;
  state->count = 0;
  
  /* cursor */
  state->index = 0;
  
  /* display */
  state->message_attr = A_NORMAL;
  
  r = pthread_mutex_init(&state->lock, NULL);
  if (r < 0)
    goto free_array;
  
  r = pipe(state->update_fd);
  if (r < 0)
    goto destroy_lock;
  
  state->flood = 0;
  
  return NO_ERROR;
  
 destroy_lock:
  pthread_mutex_destroy(&state->lock);
 free_array:
  carray_free(state->message_list);
 free:
  free(state);
  subapp->data = NULL;
 err:
  return ERROR_MEMORY;
}

static void done(struct etpan_subapp * subapp)
{
  struct app_state * state;
  unsigned int i;
  
  state = subapp->data;
  
  close(state->update_fd[1]);
  close(state->update_fd[0]);
  pthread_mutex_destroy(&state->lock);

  for(i = 0 ; i < carray_count(state->message_list) ; i ++) {
    char * str;
    
    str = carray_get(state->message_list, i);
    if (str != NULL)
      free(str);
  }
  carray_free(state->message_list);
  free(state);
}


static int display_full(struct etpan_subapp * app, WINDOW * w)
{
  struct app_state * state;
  int y;
  unsigned int index;
  unsigned int count;
  char * output;
  char * fill;
  
  output = app->app->output;
  fill = app->app->fill;

  state = app->data;
  
  y = 0;
  
  index = (state->first + state->index) %
    carray_count(state->message_list);
  count = state->count - state->index;

  pthread_mutex_lock(&state->lock);
  wattron(w, state->message_attr);
  while (y < app->display_height) {
    if (count == 0) {
      mvwprintw(w, y, 0, "%s", fill);
    }
    else {
      char * str;
      
      str = carray_get(state->message_list, index);
      snprintf(app->app->output, app->display_width + 1, "%s%s", str, fill);
      mvwprintw(w, y, 0, "%s", app->app->output);
      app->cursor_x = strlen(str);
      app->cursor_y = y;
    }
    
    if (count > 0) {
      if (index >= carray_count(state->message_list) - 1)
        index = 0;
      else
        index ++;
      count --;
    }
    y ++;
  }
  wattroff(w, state->message_attr);
  pthread_mutex_unlock(&state->lock);
  
  return NO_ERROR;
}

struct etpan_subapp * etpan_console_app_new(struct etpan_app * app)
{
  return etpan_subapp_new(app, &etpan_console_app_driver);
}

int etpan_console_log(struct etpan_subapp * console, char * message)
{
  struct app_state * state;
  char * dup_msg;
  int ack;
  
  state = console->data;
  
  dup_msg = strdup(message);
  if (dup_msg == NULL)
    return ERROR_MEMORY;
  
  pthread_mutex_lock(&state->lock);
  if (state->count == carray_count(state->message_list)) {
    char * old_str;
    
    old_str = carray_get(state->message_list, state->first);
    free(old_str);
    carray_set(state->message_list, state->first, dup_msg);
    state->first ++;
    state->first %= carray_count(state->message_list);
  }
  else {
    carray_set(state->message_list,
        (state->first + state->count) % carray_count(state->message_list),
        dup_msg);
    state->count ++;
  }
  pthread_mutex_unlock(&state->lock);
  
  state->enabled = 1;
  set_index(console);
  
  /* notify to update of display */
  
  if (!state->flood) {
    state->flood = 1;
    ack = 0;
    write(state->update_fd[1], &ack, sizeof(ack));
  }
  
  return NO_ERROR;
}

#define HELP_TEXT \
"\
Help for console log viewer\n\
---------------------------\n\
\n\
Console log viewer will show you all the messages displayed in\n\
under the status bar.\n\
\n\
- Ctrl-L           : close log viewer\n\
- arrow keys       : scroll\n\
- d                : dump all your trafic to libetpan-stream-debug.log\n\
                     WARNING! this will also dump your passwords\n\
\n\
- ?                : help\n\
\n\
(? or q to exit help)\n\
"

static int show_help(struct etpan_subapp * app)
{
  return etpan_show_help(app, HELP_TEXT, sizeof(HELP_TEXT) - 1);
}
