#include "etpan-folder-discover.h"

#include <libetpan/libetpan.h>
#include <stdlib.h>
#include <string.h>
#include <dirent.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <fcntl.h>
#include <libgen.h>

#include "etpan-imap.h"
#include "etpan-config.h"
#include "etpan-errors.h"
#include "etpan-tools.h"
#include "etpan-cfg-storage-types.h"
#include "etpan-cfg-vfolder.h"
#include "etpan-folder-params.h"

/*
  auto mbox
  auto mh
  auto imap
  auto maildir
*/


enum {
  NO_STATE,
  STATE_ADDED,
  STATE_DELETED,
};

struct path_info {
  char * mb;
  char * vpath;
  int state;
};

struct path_list {
  carray * mb_list;
};

/*
  STORAGE_TYPE_IMAP = 0,
  STORAGE_TYPE_POP3,
  STORAGE_TYPE_NNTP,
  STORAGE_TYPE_MH,
  STORAGE_TYPE_MBOX,
  STORAGE_TYPE_MAILDIR,
  STORAGE_TYPE_DB,
*/


static inline void lock(struct etpan_discovery_info * info)
{
  pthread_mutex_lock(&info->lock);
}

static inline void unlock(struct etpan_discovery_info * info)
{
  pthread_mutex_unlock(&info->lock);
}


static struct path_info * path_info_new(char * mb, char * vpath)
{
  struct path_info * info;
  
  info = malloc(sizeof(* info));
  if (info == NULL)
    return NULL;
  
  info->mb = mb;
  info->vpath = vpath;
  info->state = NO_STATE;
  
  return info;
}

static int add_to_array(carray * list, char * mb, char * vpath)
{
  struct path_info * info;
  int r;
  
  info = path_info_new(mb, vpath);
  if (info == NULL)
    return -1;
  
  r = carray_add(list, info, NULL);
  if (r < 0)
    return -1;
  
  return 0;
}

static void free_array(carray * list)
{
  unsigned int i;
  
  for(i = 0 ; i < carray_count(list) ; i ++) {
    struct path_info * info;
    
    info = carray_get(list, i);
    free(info->vpath);
    free(info->mb);
    free(info);
  }
  carray_free(list);
}

static int compare_path(const void * elt1, const void * elt2)
{
  struct path_info * const * ppath1;
  struct path_info * const * ppath2;
  struct path_info * path1;
  struct path_info * path2;
  
  ppath1 = elt1;
  ppath2 = elt2;
  path1 = * ppath1;
  path2 = * ppath2;
  
  return strcmp(path1->mb, path2->mb);
}

/* mbox */

static int mbox_add_to_list(carray * list, char * base_vpath, char * pathname)
{
  DIR * dir;
  struct dirent * ent;
  int r;
  
  dir = opendir(pathname);
  if (dir == NULL)
    return NO_ERROR;
  
  while ((ent = readdir(dir)) != NULL) {
    char * name;
    struct stat stat_buf;
    int file_type;
    char filename[PATH_MAX];
    char tmp_vpath[PATH_MAX];
    char * vpath;
    ssize_t size;
    char from_space[6];
    int fd;
    
    if (ent->d_name[0] == '.')
      continue;
    
    snprintf(filename, sizeof(filename), "%s/%s", pathname, ent->d_name);
    
    r = stat(filename, &stat_buf);
    if (r < 0)
      continue;
    
    file_type = stat_buf.st_mode & S_IFMT;
    if (file_type == S_IFDIR) {
      
      if (* base_vpath == '\0')
        snprintf(tmp_vpath, sizeof(tmp_vpath), "%s", ent->d_name);
      else
        snprintf(tmp_vpath, sizeof(tmp_vpath), "%s/%s", base_vpath, ent->d_name);
      
      r = mbox_add_to_list(list, tmp_vpath, filename);
      if (r != NO_ERROR)
        goto err;
    }
    
    if (file_type != S_IFREG)
      continue;
    
    if (stat_buf.st_size != 0) {
      fd = open(filename, O_RDONLY);
      if (fd < 0)
        continue;
      
      /* WARNING - system should be able to return 5 bytes when reading */
      size = read(fd, from_space, 5);
      if (size < 5)
        continue;
      
      from_space[5] = '\0';
      
      close(fd);
      
      if (strcmp(from_space, "From ") != 0)
        continue;
    }
    
    name = strdup(filename);
    if (name == NULL) {
      goto err;
    }
    
    if (* base_vpath == '\0')
      snprintf(tmp_vpath, sizeof(tmp_vpath), "%s", ent->d_name);
    else
      snprintf(tmp_vpath, sizeof(tmp_vpath), "%s/%s", base_vpath, ent->d_name);
    vpath = strdup(tmp_vpath);
    if (vpath == NULL) {
      free(name);
      goto err;
    }
    
    r = add_to_array(list, name, vpath);
    if (r < 0) {
      free(vpath);
      free(name);
      goto err;
    }
  }
  
  closedir(dir);
  
  return NO_ERROR;
  
 err:
  return ERROR_MEMORY;
}

static struct path_list * get_mbox_list(char * base_vpath, char * pathname)
{
  struct path_list * list;
  int r;
  
  list = malloc(sizeof(* list));
  if (list == NULL)
    goto err;
  
  list->mb_list = carray_new(8);
  if (list->mb_list == NULL)
    goto free;
  
  r = mbox_add_to_list(list->mb_list, base_vpath, pathname);
  if (r != NO_ERROR)
    goto free_list;
  
  qsort(carray_data(list->mb_list),
      carray_count(list->mb_list),
      sizeof(void *),
      compare_path);
  
  return list;
  
 free_list:
  free_array(list->mb_list);
 free:
  free(list);
 err:
  return NULL;
}


/* mh list */

static int mh_add_to_list(carray * list, char * base_vpath, char * pathname)
{
  DIR * dir;
  struct dirent * ent;
  int r;
  
  dir = opendir(pathname);
  if (dir == NULL)
    return NO_ERROR;
  
  while ((ent = readdir(dir)) != NULL) {
    char * name;
    struct stat stat_buf;
    int file_type;
    char filename[PATH_MAX];
    char tmp_vpath[PATH_MAX];
    char * vpath;
    
    if (ent->d_name[0] == '.')
      continue;
    
    snprintf(filename, sizeof(filename), "%s/%s", pathname, ent->d_name);
    
    r = stat(filename, &stat_buf);
    if (r < 0)
      continue;
    
    file_type = stat_buf.st_mode & S_IFMT;
    if (file_type == S_IFDIR) {
      if (* base_vpath == '\0')
        snprintf(tmp_vpath, sizeof(tmp_vpath), "%s", ent->d_name);
      else
        snprintf(tmp_vpath, sizeof(tmp_vpath), "%s/%s", base_vpath, ent->d_name);
      
      mh_add_to_list(list, tmp_vpath, filename);
      
      name = strdup(filename);
      if (name == NULL) {
        goto err;
      }
      
      vpath = strdup(tmp_vpath);
      if (vpath == NULL) {
        free(name);
        goto err;
      }
      
      r = add_to_array(list, name, vpath);
      if (r < 0) {
        free(vpath);
        free(name);
        goto err;
      }
    }
  }
  
  closedir(dir);
  
  return NO_ERROR;
  
 err:
  return ERROR_MEMORY;
}


static struct path_list * get_mh_list(char * base_vpath, char * pathname)
{
  struct path_list * list;
  int r;
  
  list = malloc(sizeof(* list));
  if (list == NULL)
    goto err;
  
  list->mb_list = carray_new(8);
  if (list->mb_list == NULL)
    goto free;
  
  r = mh_add_to_list(list->mb_list, base_vpath, pathname);
  if (r != NO_ERROR)
    goto free_list;
  
  qsort(carray_data(list->mb_list),
      carray_count(list->mb_list),
      sizeof(void *),
      compare_path);
  
  return list;
  
 free_list:
  free_array(list->mb_list);
 free:
  free(list);
 err:
  return NULL;
}

/* maildir list */

static int maildir_add_to_list(carray * list, char * base_vpath,
    char * pathname)
{
  DIR * dir;
  struct dirent * ent;
  int r;
  int file_type;
  struct stat stat_buf;
  char filename[PATH_MAX];
  
  snprintf(filename, sizeof(filename), "%s/cur", pathname);
  r = stat(filename, &stat_buf);
  if (r == 0) {
    r = stat(pathname, &stat_buf);
    if (r < 0)
      goto err;
    
    file_type = stat_buf.st_mode & S_IFMT;
    if (file_type == S_IFDIR) {
      char * name;
      char * vpath;
      char tmp_vpath[PATH_MAX];
      
      name = strdup(pathname);
      if (name == NULL) {
        goto err;
      }
      
      if (* base_vpath == '\0')
        snprintf(tmp_vpath, sizeof(tmp_vpath), "INBOX");
      else
        snprintf(tmp_vpath, sizeof(tmp_vpath), "%s/INBOX", base_vpath);
      vpath = strdup(tmp_vpath);
      if (vpath == NULL) {
        free(name);
        goto err;
      }
      
      r = add_to_array(list, name, vpath);
      if (r < 0) {
        free(vpath);
        free(name);
        goto err;
      }
    }
  }
  
  dir = opendir(pathname);
  if (dir == NULL)
    return NO_ERROR;
  
  while ((ent = readdir(dir)) != NULL) {
    if (ent->d_name[0] == '.') {
      if (ent->d_name[1] == '\0')
        continue;
      else if (ent->d_name[1] == '.') {
        if (ent->d_name[2] == '\0')
          continue;
      }
      else {
        snprintf(filename, sizeof(filename), "%s/%s/cur",
            pathname, ent->d_name);
        r = stat(filename, &stat_buf);
        if (r == 0) {
          snprintf(filename, sizeof(filename), "%s/%s",
              pathname, ent->d_name);
          
          r = stat(filename, &stat_buf);
          if (r < 0)
            continue;
          
          file_type = stat_buf.st_mode & S_IFMT;
          if (file_type == S_IFDIR) {
            char tmp_vpath[PATH_MAX];
            char * name;
            char * vpath;
            char tmp_dname[PATH_MAX];
            char * p;
            
            snprintf(tmp_dname, sizeof(tmp_dname), "%s", ent->d_name + 1);
            p = tmp_dname;
            while ((p = strchr(p, '.')) != NULL) {
              * p = '/';
              p ++;
            }
            
            if (* base_vpath == '\0')
              snprintf(tmp_vpath, sizeof(tmp_vpath), "%s", tmp_dname);
            else
              snprintf(tmp_vpath, sizeof(tmp_vpath), "%s/%s",
                  base_vpath, tmp_dname);
            
            name = strdup(filename);
            if (name == NULL) {
              goto err;
            }
            
            vpath = strdup(tmp_vpath);
            if (vpath == NULL) {
              free(name);
              goto err;
            }
            
            r = add_to_array(list, name, vpath);
            if (r < 0) {
              free(vpath);
              free(name);
              goto err;
            }
          }
        }
      }
    }
  }
  
  closedir(dir);
  
  return NO_ERROR;
  
 err:
  return ERROR_MEMORY;
}


static struct path_list * get_maildir_list(char * base_vpath, char * pathname)
{
  struct path_list * list;
  int r;
  
  list = malloc(sizeof(* list));
  if (list == NULL)
    goto err;
  
  list->mb_list = carray_new(8);
  if (list->mb_list == NULL)
    goto free;
  
  r = maildir_add_to_list(list->mb_list, base_vpath, pathname);
  if (r != NO_ERROR)
    goto free_list;
  
  qsort(carray_data(list->mb_list),
      carray_count(list->mb_list),
      sizeof(void *),
      compare_path);
  
  return list;
  
 free_list:
  free_array(list->mb_list);
 free:
  free(list);
 err:
  return NULL;
}


/* imap list */

static int imap_add_to_list(struct mailstorage * storage,
    carray * list, char * base_vpath,
    char * dir)
{
  carray * imap_mb_list;
  int r;
  unsigned int i;
  char pattern[PATH_MAX];
  char tmp_vpath[PATH_MAX];
  char * vpath;
  char * name;
  
  snprintf(pattern, sizeof(pattern), "*");
  
  if (dir != NULL) {
    if (* dir != '\0')
      snprintf(pattern, sizeof(pattern), "%s/*", dir);
  }
  
  r = etpan_imap_lsub_mailboxes(storage->sto_session,
      pattern, &imap_mb_list);
  if (r != NO_ERROR)
    return r;
  
  name = strdup("INBOX");
  if (name == NULL) {
    goto free_list;
  }
  
  if (* base_vpath == '\0')
    snprintf(tmp_vpath, sizeof(tmp_vpath), "INBOX");
  else
    snprintf(tmp_vpath, sizeof(tmp_vpath), "%s/INBOX", base_vpath);
  
  vpath = strdup(tmp_vpath);
  if (vpath == NULL) {
    free(name);
    goto free_list;
  }
  
  r = add_to_array(list, name, vpath);
  if (r < 0) {
    free(vpath);
    free(name);
    goto free_list;
  }
  
  for(i = 0 ; i < carray_count(imap_mb_list) ; i ++) {
    struct etpan_imap_mailbox_info * info;
    char mb[PATH_MAX];
    
    info = carray_get(imap_mb_list, i);
    
    if ((info->flags & ETPAN_IMAP_MB_NOSELECT) != 0)
      continue;
    
    snprintf(mb, sizeof(mb), "%s", info->mb);
    
    if (info->delimiter != '/') {
      char * p;
      
      p = mb;
      while ((p = strchr(p, '/')) != NULL) {
        * p = '-';
        p ++;
      }
      
      p = mb;
      while ((p = strchr(p, info->delimiter)) != NULL) {
        * p = '/';
        p ++;
      }
    }
    
    name = strdup(info->mb);
    if (name == NULL) {
      goto free_list;
    }
    
    if (* base_vpath == '\0')
      snprintf(tmp_vpath, sizeof(tmp_vpath), "%s", mb);
    else
      snprintf(tmp_vpath, sizeof(tmp_vpath), "%s/%s", base_vpath, mb);
    
    vpath = strdup(tmp_vpath);
    if (vpath == NULL) {
      free(name);
      goto free_list;
    }
    
    r = add_to_array(list, name, vpath);
    if (r < 0) {
      free(vpath);
      free(name);
      goto free_list;
    }
  }
  
  etpan_imap_list_mailboxes_free(imap_mb_list);
  
  return NO_ERROR;
  
 free_list:
  etpan_imap_list_mailboxes_free(imap_mb_list);
 err:
  return ERROR_MEMORY;
}

static struct path_list * get_imap_list(struct mailstorage * storage,
    char * base_vpath, char * pathname)
{
  struct path_list * list;
  int r;
  
  list = malloc(sizeof(* list));
  if (list == NULL)
    goto err;
  
  list->mb_list = carray_new(8);
  if (list->mb_list == NULL)
    goto free;
  
  r = imap_add_to_list(storage, list->mb_list, base_vpath, pathname);
  if (r != NO_ERROR)
    goto free_list;
  
  qsort(carray_data(list->mb_list),
      carray_count(list->mb_list),
      sizeof(void *),
      compare_path);
  
  return list;
  
 free_list:
  free_array(list->mb_list);
 free:
  free(list);
 err:
  return NULL;
}

static int path_list_equal(struct path_list * list1,
    struct path_list * list2)
{
  unsigned int i;
  unsigned int j;
  int is_equal;
  
  if (list1 == list2)
    return 1;
  
  if ((list1 == NULL) || (list2 == NULL))
    return 1;
  
  i = 0;
  j = 0;
  
  is_equal = 1;
  
  while (1) {
    struct path_info * path1;
    struct path_info * path2;
    int result;
    
    if (i < carray_count(list1->mb_list))
      path1 = carray_get(list1->mb_list, i);
    else
      path1 = NULL;
    
    if (j < carray_count(list2->mb_list))
      path2 = carray_get(list2->mb_list, j);
    else
      path2 = NULL;
    
    if ((path1 != NULL) && (path2 != NULL)) {
      result = compare_path(&path1, &path2);
    }
    else if (path1 != NULL)
      result = -1;
    else if (path2 != NULL)
      result = 1;
    else
      break;
      
    if (result == 0) {
      i ++;
      j ++;
    }
    else {
      is_equal = 0;
      
      if (result < 0) {
        i ++;
        path1->state = STATE_DELETED;
      }
      else {
        j ++;
        path2->state = STATE_ADDED;
      }
    }
  }
  
  return is_equal;
}

/*
  auto-discover : poll
  virtual-path, start-path =>  changed flag, list of folders
*/

struct etpan_discovery_info * etpan_discovery_info_new(int type,
    struct mailstorage * storage, char * start_path, char * vpath,
    struct etpan_account_info * account,
    char * sent_folder,
    char * draft_folder)
{
  struct etpan_discovery_info * info;
  
  info = malloc(sizeof(* info));
  if (info == NULL)
    goto err;
  
  info->type = type;
  info->storage = storage;
  info->start_path = start_path;
  info->changed = 0;
  info->path_list = NULL;
  info->updated_path_list = NULL;
  info->vpath = vpath;
  pthread_mutex_init(&info->lock, NULL);
  info->account = account;
  info->sent_folder = sent_folder;
  info->draft_folder = draft_folder;
  
  return info;
  
 err:
  return NULL;
}

void etpan_discovery_info_free(struct etpan_discovery_info * info)
{
  if (info->path_list != NULL) {
    free_array(info->path_list->mb_list);
    free(info->path_list);
  }
  if (info->updated_path_list != NULL) {
    free_array(info->updated_path_list->mb_list);
    free(info->updated_path_list);
  }
  pthread_mutex_destroy(&info->lock);
  free(info->vpath);
  free(info->start_path);
  free(info);
}

struct etpan_discovery_manager *
etpan_discovery_manager_new(struct etpan_app * app)
{
  struct etpan_discovery_manager * manager;
  
  manager = malloc(sizeof(* manager));
  if (manager == NULL)
    goto err;
  
  manager->app = app;
  manager->vpath_hash = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (manager->vpath_hash == NULL)
    goto free;
  manager->seq = 0;
  
  return manager;
  
 free:
  free(manager);
 err:
  return NULL;
}

void etpan_discovery_manager_free(struct etpan_discovery_manager * manager)
{
  chashiter * iter;
  
  for(iter = chash_begin(manager->vpath_hash) ; iter != NULL ;
      iter = chash_next(manager->vpath_hash, iter)) {
    struct etpan_discovery_info * info;
    chashdatum data;
    
    chash_value(iter, &data);
    info = data.data;
    etpan_discovery_info_free(info);
  }
  
  chash_free(manager->vpath_hash);
  free(manager);
}

int etpan_discovery_manager_add(struct etpan_discovery_manager * manager,
    struct etpan_discovery_info * info)
{
  chashdatum data;
  chashdatum key;
  int r;
  
  data.data = info;
  data.len = 0;
  key.data = info->vpath;
  key.len = strlen(info->vpath);
  r = chash_set(manager->vpath_hash, &key, &data, NULL);
  if (r < 0)
    goto err;
  
  return NO_ERROR;
  
 err:
  return ERROR_MEMORY;
}

int etpan_discovery_info_update(struct etpan_discovery_info * info)
{
  struct path_list * new_path_list;
  
  new_path_list = NULL;
  
  switch (info->type) {
  case STORAGE_TYPE_MBOX:
    new_path_list = get_mbox_list(info->vpath, info->start_path);
    break;
    
  case STORAGE_TYPE_MH:
    new_path_list = get_mh_list(info->vpath, info->start_path);
    break;
    
  case STORAGE_TYPE_MAILDIR:
    new_path_list = get_maildir_list(info->vpath, info->start_path);
    break;
    
  case STORAGE_TYPE_IMAP:
    new_path_list = get_imap_list(info->storage,
        info->vpath, info->start_path);
    break;
  }
  
  if (new_path_list == NULL)
    return NO_ERROR;
  
  if (info->path_list == NULL) {
    unsigned int i;
    
    for(i = 0 ; i < carray_count(new_path_list->mb_list) ; i ++) {
      struct path_info * pinfo;
      
      pinfo = carray_get(new_path_list->mb_list, i);
      pinfo->state = STATE_ADDED;
    }
  }
  else if (path_list_equal(info->path_list, new_path_list)) {
    free_array(new_path_list->mb_list);
    free(new_path_list);
    return NO_ERROR;
  }
  
  lock(info);
  info->changed = 1;
  
  if (info->updated_path_list != NULL) {
    free_array(info->updated_path_list->mb_list);
    free(info->updated_path_list);
  }
  info->updated_path_list = new_path_list;
  unlock(info);
  
  return NO_ERROR;
}

static int mbox_generate(struct etpan_discovery_manager * manager,
    struct etpan_discovery_info * discovery_info,
    struct path_info * info, struct mailfolder ** pfolder);

static int mh_generate(struct etpan_discovery_manager * manager,
    struct etpan_discovery_info * discovery_info,
    struct path_info * info, struct mailfolder ** pfolder);

static int maildir_generate(struct etpan_discovery_manager * manager,
    struct etpan_discovery_info * discovery_info,
    struct path_info * info, struct mailfolder ** pfolder);

static int imap_generate(struct etpan_discovery_manager * manager,
    struct etpan_discovery_info * discovery_info,
    struct path_info * info, struct mailfolder ** pfolder);

static int recursive_add_to_hash(chash * path_hash,
    struct mailfolder * folder)
{
  unsigned int i;
  char * path;
  chashdatum key;
  chashdatum value;
  int r;
  
  path = etpan_folder_get_virtual_path(folder);
  key.data = path;
  key.len = strlen(path);
  value.data = folder;
  value.len = 0;
  r = chash_set(path_hash, &key, &value, NULL);
  free(path);
  if (r < 0)
    return ERROR_MEMORY;
  
  for(i = 0 ; i < carray_count(folder->fld_children) ; i ++) {
    struct mailfolder * child;
    
    child = carray_get(folder->fld_children, i);
    recursive_add_to_hash(path_hash, child);
  }
  
  return NO_ERROR;
}

static struct mailfolder * vfolder_new_folder(chash * folder_hash,
    carray * path_tab,
    struct mailstorage * storage, char * virtualpath, char * location)
{
  char * p;
  char * name;
  int r;
  struct mailfolder * folder;
  chashdatum key;
  chashdatum value;
  
  /* get base name */
  p = virtualpath;
  while (1) {
    name = p;
    p = strchr(name, '/');
    if (p == NULL)
      break;
    p ++;
  }
  
  folder = mailfolder_new(storage, location, name);
  if (folder == NULL) {
    goto err;
  }
  
  key.data = virtualpath;
  key.len = strlen(virtualpath);
  value.data = folder;
  value.len = 0;
  
  r = chash_set(folder_hash, &key, &value, NULL);
  if (r < 0) {
    mailfolder_free(folder);
    goto err;
  }
  
  r = carray_add(path_tab, folder, NULL);
  if (r < 0) {
    key.data = virtualpath;
    key.len = strlen(virtualpath);
    chash_delete(folder_hash, &key, NULL);
    mailfolder_free(folder);
    goto err;
  }
  
  return folder;
  
 err:
  return NULL;
}

int etpan_discover_update_config(struct etpan_discovery_manager * manager,
    struct etpan_folder_params * folder_params,
    struct etpan_discovery_info * info)
{
  unsigned int i;
  int r;
  int res;
  chash * path_hash;
  chash * folder_to_path_hash;
  chash * discover_hash;
  carray * path_tab;
  struct mailfolder * root;
  struct mailfolder * sent_folder;
  struct mailfolder * draft_folder;
  
  lock(info);
  
  if (!info->changed) {
    res = NO_ERROR;
    goto unlock;
  }
  
  path_hash = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (path_hash == NULL) {
    res = ERROR_MEMORY;
    goto unlock;
  }
  
  folder_to_path_hash = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYALL);
  if (folder_to_path_hash == NULL) {
    res = ERROR_MEMORY;
    goto free_path_hash;
  }

  discover_hash = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (discover_hash == NULL) {
    res = ERROR_MEMORY;
    goto free_folder_to_path_hash;
  }
  
  path_tab = carray_new(16);
  if (path_tab == NULL) {
    res = ERROR_MEMORY;
    goto free_hash;
  }
  
  root = manager->app->config.vfolder_config->root;
  
  r = recursive_add_to_hash(path_hash, root);
  if (r != NO_ERROR) {
    res = r;
    goto free_array;
  }
  
  for(i = 0 ; i < carray_count(info->updated_path_list->mb_list) ; i ++) {
    struct path_info * pinfo;
    chashdatum key;
    chashdatum value;
    
    pinfo = carray_get(info->updated_path_list->mb_list, i);
    
    /* test if the folder already exists */
    key.data = pinfo->vpath;
    key.len = strlen(pinfo->vpath);
    r = chash_get(path_hash, &key, &value);
    if (r == 0)
      continue;
    
    if (pinfo->state == STATE_ADDED) {
      chashdatum key;
      chashdatum value;
      struct mailfolder * new_folder;
      
      switch (info->type) {
      case STORAGE_TYPE_MBOX:
        r = mbox_generate(manager, info, pinfo, &new_folder);
        if (r != NO_ERROR) {
          res = r;
          goto free_array;
        }
        break;
        
      case STORAGE_TYPE_MH:
        r = mh_generate(manager, info, pinfo, &new_folder);
        if (r != NO_ERROR) {
          res = r;
          goto free_array;
        }
        break;
        
      case STORAGE_TYPE_MAILDIR:
        r = maildir_generate(manager, info, pinfo, &new_folder);
        if (r != NO_ERROR) {
          res = r;
          goto free_array;
        }
        break;
        
      case STORAGE_TYPE_IMAP:
        r = imap_generate(manager, info, pinfo, &new_folder);
        if (r != NO_ERROR) {
          res = r;
          goto free_array;
        }
        break;
      }
      
      key.data = &new_folder;
      key.len = sizeof(new_folder);
      value.data = info;
      value.len = 0;
      r = chash_set(discover_hash, &key, &value, NULL);
      /* ignore errors */
      
      key.data = pinfo->vpath;
      key.len = strlen(pinfo->vpath);
      value.data = new_folder;
      value.len = 0;
      r = chash_set(path_hash, &key, &value, NULL);
      /* ignore errors */
      
      key.data = &new_folder;
      key.len = sizeof(new_folder);
      value.data = pinfo->vpath;
      value.len = strlen(pinfo->vpath);
      r = chash_set(folder_to_path_hash, &key, &value, NULL);
      /* ignore errors */
      
      carray_add(path_tab, new_folder, NULL);
      /* ignore errors */
      
      r = etpan_folder_params_add(folder_params, new_folder);
      /* ignore errors */
    }
  }
  
  /* link folders to their parent */
  
  for(i = 0 ; i < carray_count(path_tab) ; i ++) {
    struct mailfolder * folder;
    chashdatum key;
    chashdatum value;
    char * path;
    size_t len;
    struct mailfolder * child;
    char * end;
    
    folder = carray_get(path_tab, i);
    
    key.data = &folder;
    key.len = sizeof(folder);
    r = chash_get(folder_to_path_hash, &key, &value);
    if (r < 0)
      continue;
    
    path = malloc(value.len + 1);
    strncpy(path, value.data, value.len);
    path[value.len] = '\0';
    
    len = value.len;
    child = NULL;
    
    while (len > 0) {
      chashdatum key;
      chashdatum value;
      
      ETPAN_APP_DEBUG((manager->app, "adding %s", path));
      
      key.data = path;
      key.len = len;
      
      r = chash_get(path_hash, &key, &value);
      if (r == 0) {
        folder = value.data;
      }
      else {
        folder = vfolder_new_folder(path_hash, path_tab, NULL, path, NULL);
        if (folder == NULL) {
          res = ERROR_MEMORY;
          goto free_array;
        }
        
        r = etpan_folder_params_add(folder_params, folder);
        /* ignore errors */
      }
      
      if (child != NULL) {
        ETPAN_APP_DEBUG((manager->app, "adding %p", child));
        
        r = mailfolder_add_child(folder, child);
        if (r != MAIL_NO_ERROR) {
          res = r;
          goto free_array;
        }
        
#if 0
        etpan_folder_set_opened(folder_params, folder, 1);
#endif
      }
      
      if (folder->fld_parent != NULL)
        break;
      
      end = path + len - 1;
      while ((end != path) && (* end != '/')) {
	end --;
	len --;
      }
      len --;
      * end = 0;
      
      child = folder;
    }
    free(path);
  }
  
  /* add folders that have no parent to the root folder */
  
  for(i = 0 ; i < carray_count(path_tab) ; i ++) {
    struct mailfolder * folder;
    chashdatum key;
    chashdatum value;
    char * path;
    size_t len;
    struct mailfolder * child;
    char * end;
    
    folder = carray_get(path_tab, i);
    if (folder->fld_parent == NULL) {
      r = mailfolder_add_child(root, folder);
      if (r != MAIL_NO_ERROR) {
        res = r;
          goto free_array;
      }
    }
  }
  
  /* set discovery additionnal info */
  
  sent_folder = NULL;
  if (info->sent_folder != NULL)
    sent_folder =
      etpan_vfolder_get_folder(manager->app->config.vfolder_config,
          info->sent_folder);
  
  draft_folder = NULL;
  if (info->draft_folder != NULL)
    draft_folder =
      etpan_vfolder_get_folder(manager->app->config.vfolder_config,
          info->draft_folder);
  
  for(i = 0 ; i < carray_count(path_tab) ; i ++) {
    struct mailfolder * folder;
    chashdatum key;
    chashdatum value;
    char * path;
    size_t len;
    char * end;
    
    folder = carray_get(path_tab, i);
    
    key.data = &folder;
    key.len = sizeof(folder);
    
    r = chash_get(discover_hash, &key, &value);
    if (r < 0)
      continue;
    
    etpan_vfolder_set_account(manager->app->config.vfolder_config,
        folder, info->account);
    
    etpan_vfolder_set_sent_folder(manager->app->config.vfolder_config,
        folder, sent_folder);
    
    etpan_vfolder_set_draft_folder(manager->app->config.vfolder_config,
        folder, draft_folder);
    
    if (folder->fld_parent != NULL)
      etpan_folder_set_opened(folder_params, folder->fld_parent, 1);
  }
  
  carray_free(path_tab);
  chash_free(discover_hash);
  chash_free(folder_to_path_hash);
  chash_free(path_hash);
  
  if (info->path_list != NULL) {
    free_array(info->path_list->mb_list);
    free(info->path_list);
  }
  
  info->path_list = info->updated_path_list;
  info->updated_path_list = NULL;
  info->changed = 0;
  unlock(info);
  
  etpan_config_write(manager->app, &manager->app->config);
  
  return NO_ERROR;
  
 free_array:
  carray_free(path_tab);
 free_hash:
  chash_free(discover_hash);
 free_folder_to_path_hash:
  chash_free(folder_to_path_hash);
 free_path_hash:
  chash_free(path_hash);
 unlock:
  unlock(info);
 err:
  return res;
}

static char * quote_path(char * mb)
{
  MMAPString * gstr;
  char * str;

  gstr = mmap_string_new("");
  if (gstr == NULL)
    return NULL;
  
  while (* mb != 0) {
    char hex[3];

    if (((* mb >= 'a') && (* mb <= 'z')) ||
	((* mb >= 'A') && (* mb <= 'Z')) ||
	((* mb >= '0') && (* mb <= '9')))
      mmap_string_append_c(gstr, * mb);
    else {
      if (mmap_string_append_c(gstr, '%') == NULL)
	goto free;
      snprintf(hex, 3, "%02x", (unsigned char) (* mb));
      if (mmap_string_append(gstr, hex) == NULL)
	goto free;
    }
    mb ++;
  }

  str = strdup(gstr->str);
  if (str == NULL)
    goto free;

  mmap_string_free(gstr);
  
  return str;

 free:
  mmap_string_free(gstr);
  return NULL;
}

static int mbox_generate(struct etpan_discovery_manager * manager,
    struct etpan_discovery_info * discovery_info,
    struct path_info * info, struct mailfolder ** pfolder)
{
  struct mailstorage * storage;
  int r;
  char * id;
  int res;
  char cache_directory[PATH_MAX];
  char flags_directory[PATH_MAX];
  char basename_buf[PATH_MAX];
  char * name;
  struct mailfolder * new_folder;
  char * parent_name;
  struct mailfolder * folder;
  chashdatum key;
  chashdatum value;
  
  id = quote_path(info->mb);
  if (id == NULL) {
    res = ERROR_MEMORY;
    goto err;
  }
  
  storage = etpan_storage_get(manager->app->config.storage_config, id);
  if (storage == NULL) {
    storage = mailstorage_new(id);
    if (storage == NULL) {
      res = ERROR_MEMORY;
      goto free_id;
    }
  
    snprintf(cache_directory,
        PATH_MAX, "%s/%s/%s", etpan_get_home_dir(), ETPAN_CACHE_PATH, id);
    snprintf(flags_directory,
        PATH_MAX, "%s/%s/%s", etpan_get_home_dir(), ETPAN_FLAGS_PATH, id);
  
    r = mbox_mailstorage_init(storage, info->mb, 1,
        cache_directory, flags_directory);
    if (r != NO_ERROR) {
      res = ERROR_MEMORY;
      goto free_storage;
    }
  
    r = etpan_storage_config_add(manager->app->config.storage_config,
        storage, 1);
    if (r != NO_ERROR) {
      res = r;
      goto free_storage;
    }
  
    etpan_storage_config_set_generated(manager->app->config.storage_config,
        storage, 1);
  }
  
  snprintf(basename_buf, sizeof(basename_buf), "%s", info->vpath);
  name = basename(basename_buf);
  
  new_folder = mailfolder_new(storage, info->mb, name);
  if (new_folder == NULL) {
    res = ERROR_MEMORY;
    goto free_id;
  }
  
  r = etpan_cfg_vfolder_add_prop(manager->app->config.vfolder_config,
      new_folder);
  if (r != NO_ERROR) {
    res = r;
    goto free_new_folder;
  }
  
  free(id);
  
  * pfolder = new_folder;
  
  return NO_ERROR;
  
 free_new_folder:
  mailfolder_free(new_folder);
  goto free_id;
 free_storage:
  mailstorage_free(storage);
 free_id:
  free(id);
 err:
  return res;
}

static int mh_generate(struct etpan_discovery_manager * manager,
    struct etpan_discovery_info * discovery_info,
    struct path_info * info, struct mailfolder ** pfolder)
{
  struct mailstorage * storage;
  int r;
  char * id;
  int res;
  char cache_directory[PATH_MAX];
  char flags_directory[PATH_MAX];
  char basename_buf[PATH_MAX];
  char * name;
  char * parent_name;
  struct mailfolder * new_folder;
  struct mailfolder * folder;
  chashdatum key;
  chashdatum value;
  
  id = quote_path(info->mb);
  if (id == NULL) {
    res = ERROR_MEMORY;
    goto err;
  }
  
  storage = etpan_storage_get(manager->app->config.storage_config, id);
  if (storage == NULL) {
    storage = mailstorage_new(id);
    if (storage == NULL) {
      res = ERROR_MEMORY;
      goto free_id;
    }
  
    snprintf(cache_directory,
        PATH_MAX, "%s/%s/%s", etpan_get_home_dir(), ETPAN_CACHE_PATH, id);
    snprintf(flags_directory,
        PATH_MAX, "%s/%s/%s", etpan_get_home_dir(), ETPAN_FLAGS_PATH, id);
  
    r = mh_mailstorage_init(storage, info->mb, 1,
        cache_directory, flags_directory);
    if (r != NO_ERROR) {
      res = ERROR_MEMORY;
      goto free_storage;
    }
  
    r = etpan_storage_config_add(manager->app->config.storage_config,
        storage, 1);
    if (r != NO_ERROR) {
      res = r;
      goto free_storage;
    }
  
    etpan_storage_config_set_generated(manager->app->config.storage_config,
        storage, 1);
  }
  
  snprintf(basename_buf, sizeof(basename_buf), "%s", info->vpath);
  name = basename(basename_buf);
  
  new_folder = mailfolder_new(storage, info->mb, name);
  if (new_folder == NULL) {
    res = ERROR_MEMORY;
    goto free_id;
  }
  
  r = etpan_cfg_vfolder_add_prop(manager->app->config.vfolder_config,
      new_folder);
  if (r != NO_ERROR) {
    res = r;
    goto free_new_folder;
  }
  
  free(id);
  
  * pfolder = new_folder;
  
  return NO_ERROR;
  
 free_new_folder:
  mailfolder_free(new_folder);
  goto free_id;
 free_storage:
  mailstorage_free(storage);
 free_id:
  free(id);
 err:
  return res;
}

static int maildir_generate(struct etpan_discovery_manager * manager,
    struct etpan_discovery_info * discovery_info,
    struct path_info * info, struct mailfolder ** pfolder)
{
  struct mailstorage * storage;
  int r;
  char * id;
  int res;
  char cache_directory[PATH_MAX];
  char flags_directory[PATH_MAX];
  char basename_buf[PATH_MAX];
  char * name;
  char * parent_name;
  struct mailfolder * new_folder;
  struct mailfolder * folder;
  chashdatum key;
  chashdatum value;
  
  id = quote_path(info->mb);
  if (id == NULL) {
    res = ERROR_MEMORY;
    goto err;
  }
  
  storage = etpan_storage_get(manager->app->config.storage_config, id);
  if (storage == NULL) {
    storage = mailstorage_new(id);
    if (storage == NULL) {
      res = ERROR_MEMORY;
      goto free_id;
    }
  
    snprintf(cache_directory,
        PATH_MAX, "%s/%s/%s", etpan_get_home_dir(), ETPAN_CACHE_PATH, id);
    snprintf(flags_directory,
        PATH_MAX, "%s/%s/%s", etpan_get_home_dir(), ETPAN_FLAGS_PATH, id);
  
    r = maildir_mailstorage_init(storage, info->mb, 1,
        cache_directory, flags_directory);
    if (r != NO_ERROR) {
      res = ERROR_MEMORY;
      goto free_storage;
    }
  
    r = etpan_storage_config_add(manager->app->config.storage_config,
        storage, 1);
    if (r != NO_ERROR) {
      res = r;
      goto free_storage;
    }
  
    etpan_storage_config_set_generated(manager->app->config.storage_config,
        storage, 1);
  }
  
  snprintf(basename_buf, sizeof(basename_buf), "%s", info->vpath);
  name = basename(basename_buf);
  
  new_folder = mailfolder_new(storage, info->mb, name);
  if (new_folder == NULL) {
    res = ERROR_MEMORY;
    goto free_id;
  }
  
  r = etpan_cfg_vfolder_add_prop(manager->app->config.vfolder_config,
      new_folder);
  if (r != NO_ERROR) {
    res = r;
    goto free_new_folder;
  }
  
  free(id);
  
  * pfolder = new_folder;
  
  return NO_ERROR;
  
 free_new_folder:
  mailfolder_free(new_folder);
  goto free_id;
 free_storage:
  mailstorage_free(storage);
 free_id:
  free(id);
 err:
  return res;
}


static int imap_generate(struct etpan_discovery_manager * manager,
    struct etpan_discovery_info * discovery_info,
    struct path_info * info, struct mailfolder ** pfolder)
{
  struct mailstorage * storage;
  int r;
  int res;
  char basename_buf[PATH_MAX];
  char * name;
  char * parent_name;
  struct mailfolder * new_folder;
  struct mailfolder * folder;
  chashdatum key;
  chashdatum value;
  
  storage = discovery_info->storage;
  
  snprintf(basename_buf, sizeof(basename_buf), "%s", info->vpath);
  name = basename(basename_buf);
  
  new_folder = mailfolder_new(storage, info->mb, name);
  if (new_folder == NULL) {
    res = ERROR_MEMORY;
    goto err;
  }
  
  r = etpan_cfg_vfolder_add_prop(manager->app->config.vfolder_config,
      new_folder);
  if (r != NO_ERROR) {
    res = r;
    goto free_new_folder;
  }
  
  * pfolder = new_folder;
  
  return NO_ERROR;
  
 free_new_folder:
  mailfolder_free(new_folder);
 err:
  return res;
}
