/*
 * etPan! -- a mail user agent
 *
 * Copyright (C) 2001, 2002 - DINH Viet Hoa
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the libEtPan! project nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * $Id: etpan-nntp.c,v 1.9 2004/12/27 15:27:38 hoa Exp $
 */

#include "etpan-nntp.h"

#include <libetpan/libetpan.h>
#include <stdlib.h>
#include <string.h>
#include "etpan-errors.h"
#include "etpan-app.h"
#include <stdio.h>
#include "etpan-tools.h"

/* this is a NNTP specific operation */

#define GROUPSLIST "groupslist"

static struct etpan_nntp_group_info *
group_info_new(char * name, int count, char type)
{
  struct etpan_nntp_group_info * info;
  
  info = malloc(sizeof(* info));
  if (info == NULL)
    goto err;
  
  info->name = strdup(name);
  if (info->name == NULL)
    goto free;
  
  info->count = count;
  info->type = type;
  info->description = strdup("no description");
  if (info->description == NULL)
    goto free_name;

  return info;

 free_name:
  free(info->name);
 free:
  free(info);
 err:
  return NULL;
}

static void group_info_free(struct etpan_nntp_group_info * info)
{
  if (info->description != NULL)
    free(info->description);
  free(info->name);
  free(info);
}

static int store_in_hash(chash * group_hash, clist * info_list)
{
  clistiter * cur;
  int res;
  int r;
    
  for(cur = clist_begin(info_list) ; cur != NULL ; cur = clist_next(cur)) {
    chashdatum key;
    chashdatum value;
    struct newsnntp_group_info * nntp_info;
    struct etpan_nntp_group_info * info;
    int type;

    nntp_info = clist_content(cur);
      
    switch (nntp_info->grp_type) {
    case 'y':
      type = ETPAN_NNTP_POST_AUTHORIZED;
      break;
    case 'n':
      type = ETPAN_NNTP_POST_UNAUTHORIZED;
      break;
    case 'm':
      type = ETPAN_NNTP_POST_MODERATED;
      break;
    default:
      type = ETPAN_NNTP_POST_UNKNOWN;
      break;
    }
    info = group_info_new(nntp_info->grp_name, nntp_info->grp_count, type);
    if (info == NULL) {
      res = ERROR_MEMORY;
      goto err;
    }

    key.data = info->name;
    key.len = strlen(info->name);
    value.data = info;
    value.len = 0;
      
    r = chash_set(group_hash, &key, &value, NULL);
    if (r < 0) {
      group_info_free(info);
      res = ERROR_MEMORY;
      goto err;
    }
  }
  
  return NO_ERROR;
  
 err:
  return res;
}

static int read_string(FILE * f, char ** pstr)
{
  char * str;
  size_t len;
  size_t readbytes;
  
  readbytes = fread(&len, sizeof(len), 1, f);
  if (readbytes == 0)
    goto err;
  
  str = malloc(len + 1);
  if (str == NULL)
    goto err;
  
  readbytes = fread(str, 1, len, f);
  if (readbytes != len)
    goto err;
  
  str[len] = '\0';
  
  * pstr = str;
  
  return NO_ERROR;
  
 err:
  return ERROR_FILE;
}

static int get_cached_list(struct mailstorage * storage, char * filter,
    carray ** result)
{
  char cache_filename[PATH_MAX];
  FILE * f;
  int r;
  unsigned int i;
  carray * info_array;
  int res;
  size_t len;
  unsigned int info_count;
  
  if (filter == NULL) {
    snprintf(cache_filename, PATH_MAX, "%s/%s/%s/%s",
        etpan_get_home_dir(), ETPAN_CACHE_PATH, storage->sto_id, GROUPSLIST);
  }
  else {
    char * quoted_filter;
    
    quoted_filter = etpan_quote_mailbox(filter);
    snprintf(cache_filename, PATH_MAX, "%s/%s/%s/%s%s",
        etpan_get_home_dir(), ETPAN_CACHE_PATH,
        storage->sto_id, GROUPSLIST, quoted_filter);
    free(quoted_filter);
  }
  
  info_array = carray_new(1024);
  if (info_array == NULL) {
    res = ERROR_MEMORY;
    goto err;
  }
  
  f = fopen(cache_filename, "r");
  if (f == NULL) {
    res = ERROR_FILE;
    goto free_array;
  }
  
  len = fread(&info_count, sizeof(info_count), 1, f);
  if (len == 0) {
    res = ERROR_FILE;
    goto close;
  }

  for(i = 0 ; i < info_count ; i ++) {
    struct etpan_nntp_group_info * info;
    char * name;
    char * description;
    uint32_t count;
    int type;
    size_t len;
    
    r = read_string(f, &name);
    if (r != NO_ERROR) {
      res = r;
      goto close;
    }
    
    r = read_string(f, &description);
    if (r != NO_ERROR) {
      free(name);
      res = r;
      goto close;
    }
    
    len = fread(&count, sizeof(count), 1, f);
    if (len == 0) {
      free(description);
      free(name);
      res = ERROR_FILE;
      goto close;
    }
    
    len = fread(&type, sizeof(type), 1, f);
    if (len == 0) {
      free(description);
      free(name);
      res = ERROR_FILE;
      goto close;
    }
    
    info = group_info_new(name, count, type);
    if (info == NULL) {
      free(description);
      free(name);
      res = ERROR_MEMORY;
      goto close;
    }
    
    info->description = description;
    
    r = carray_add(info_array, info, NULL);
    if (r < 0) {
      group_info_free(info);
      res = ERROR_MEMORY;
      goto close;
    }
  }
  
  fclose(f);
  
  * result = info_array;
  
  return NO_ERROR;
  
 close:
  fclose(f);
 free_array:
  for(i = 0 ; i < carray_count(info_array) ; i ++) {
    struct etpan_nntp_group_info * info;
    
    info = carray_get(info_array, i);
    group_info_free(info);
  }
  carray_free(info_array);
 err:
  return res;
}

static int write_string(FILE * f, char * str, size_t len)
{
  size_t written;
  
  written = fwrite(&len, sizeof(len), 1, f);
  if (written == 0)
    goto err;
  written = fwrite(str, 1, len, f);
  if (written != len)
    goto err;
  
  return NO_ERROR;
  
 err:
  return ERROR_FILE;
}

static int store_cached_list(struct mailstorage * storage, char * filter,
    carray * info_array)
{
  char cache_filename[PATH_MAX];
  unsigned int i;
  unsigned int count;
  FILE * f;
  int r;
  size_t len;
  
  if (filter == NULL) {
    snprintf(cache_filename, PATH_MAX, "%s/%s/%s/%s",
        etpan_get_home_dir(), ETPAN_CACHE_PATH, storage->sto_id, GROUPSLIST);
  }
  else {
    char * quoted_filter;
    
    quoted_filter = etpan_quote_mailbox(filter);
    snprintf(cache_filename, PATH_MAX, "%s/%s/%s/%s%s",
        etpan_get_home_dir(), ETPAN_CACHE_PATH, storage->sto_id,
        GROUPSLIST, quoted_filter);
    free(quoted_filter);
  }
  
  f = fopen(cache_filename, "w");
  if (f == NULL)
    goto err;
  
  count = carray_count(info_array);
  len = fwrite(&count, sizeof(count), 1, f);
  if (len == 0)
    goto close;

  for(i = 0 ; i < carray_count(info_array) ; i ++) {
    struct etpan_nntp_group_info * info;
    
    info = carray_get(info_array, i);

    r = write_string(f, info->name, strlen(info->name));
    if (r != NO_ERROR)
      goto close;
    
    r = write_string(f, info->description, strlen(info->description));
    if (r != NO_ERROR)
      goto close;
    
    len = fwrite(&info->count, sizeof(info->count), 1, f);
    if (len == 0)
      goto close;

    len = fwrite(&info->type, sizeof(info->type), 1, f);
    if (len == 0)
      goto close;
  }

  fclose(f);
  
  return NO_ERROR;
  
 close:
  fclose(f);
  unlink(cache_filename);
 err:
  return ERROR_FILE;
}

static int get_nntp_group_info_list(struct mailstorage * storage,
    newsnntp * nntp,
    char * filter, carray ** result)
{
  int res;
  int r;
  clist * info_list;
  clist * description_list;
  chash * group_hash;
  carray * info_array;
  chashiter * iter_hash;
  
  r = get_cached_list(storage, filter, &info_array);
  if (r == NO_ERROR) {
    * result = info_array;
    return NO_ERROR;
  }
  
  group_hash = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (group_hash == NULL) {
    res = ERROR_MEMORY;
    goto err;
  }
  
  if (filter == NULL) {
    /* old NNTP server support */
    r = newsnntp_list(nntp, &info_list);
    if (r != NEWSNNTP_NO_ERROR)
      return ERROR_NEWSGROUPS_LIST;
    
    if (r == NEWSNNTP_NO_ERROR) {
      r = store_in_hash(group_hash, info_list);
      if (r != NO_ERROR) {
        res = r;
        goto free_hash;
      }
      
      newsnntp_list_free(info_list);
    }
  }
  else {
    /* get count and group type */
    r = newsnntp_list_active(nntp, filter, &info_list);
    if (r != NEWSNNTP_NO_ERROR) {
      res = ERROR_NEWSGROUPS_LIST;
      goto free_hash;
    }
    
    r = store_in_hash(group_hash, info_list);
    if (r != NO_ERROR) {
      res = r;
      goto free_hash;
    }
    
    newsnntp_list_active_free(info_list);
  
    /* get description */
    r = newsnntp_list_newsgroups(nntp, filter,
        &description_list);
    if (r == NEWSNNTP_NO_ERROR) {
      clistiter * cur;
    
      for(cur = clist_begin(description_list) ; cur != NULL ;
          cur = clist_next(cur)) {
        struct newsnntp_group_description * group_description;
        chashdatum key;
        chashdatum value;
      
        group_description = clist_content(cur);
      
        key.data = group_description->grp_name;
        key.len = strlen(group_description->grp_name);
      
        r = chash_get(group_hash, &key, &value);
        if (r == 0) {
          struct etpan_nntp_group_info * info;
          char * description;
        
          info = value.data;
        
          description = strdup(group_description->grp_description);
          if (description != NULL) {
            if (info->description != NULL)
              free(info->description);
            info->description = description;
          }
        }
      }
    
      newsnntp_list_newsgroups_free(description_list);
    }
  }
  
  info_array = carray_new(chash_count(group_hash));
  if (info_array == NULL) {
    res = ERROR_MEMORY;
    goto free_hash;
  }
  
  for(iter_hash = chash_begin(group_hash) ; iter_hash != NULL ;
      iter_hash = chash_next(group_hash, iter_hash)) {
    struct etpan_nntp_group_info * info;
    chashdatum value;

    chash_value(iter_hash, &value);
    info = value.data;
    r = carray_add(info_array, info, NULL);
    if (r < 0) {
      res = ERROR_MEMORY;
      goto free_array;
    }
  }
  
  chash_free(group_hash);
  
  store_cached_list(storage, filter, info_array);
  
  * result = info_array;
  
  return NO_ERROR;

 free_array:
  carray_free(info_array);
 free_hash:
  for(iter_hash = chash_begin(group_hash) ; iter_hash != NULL ;
      iter_hash = chash_next(group_hash, iter_hash)) {
    struct etpan_nntp_group_info * info;
    chashdatum value;
    
    chash_value(iter_hash, &value);
    info = value.data;
    group_info_free(info);
  }
  chash_clear(group_hash);
 err:
  return res;
}

static int store_in_tab(carray * info_array,
    carray * info_filter_array)
{
  unsigned int i;
  int r;
  int res;
  
  for(i = 0 ; i < carray_count(info_filter_array) ; i ++) {
    struct etpan_nntp_group_info * info;
      
    info = carray_get(info_filter_array, i);
    r = carray_add(info_array, info, NULL);
    if (r < 0) {
      unsigned int j;
      
      for(j = 0 ; j < carray_count(info_filter_array) ; j ++) {
        info = carray_get(info_filter_array, j);
        if (info != NULL)
          group_info_free(info);
      }
        
      res = ERROR_MEMORY;
      goto err;
    }
    carray_set(info_filter_array, i, NULL);
  }
  
  return NO_ERROR;
  
 err:
  return res;
}


static int cmp_name(const void * a, const void *b)
{
  struct etpan_nntp_group_info * info_a;
  struct etpan_nntp_group_info * info_b;
  
  info_a = * (struct etpan_nntp_group_info **) a;
  info_b = * (struct etpan_nntp_group_info **) b;
  
  return strcmp(info_a->name, info_b->name);
}

#define NEWSGROUP_NAME_MAX 512

int etpan_nntp_list_groups(struct mailstorage * storage,
    mailsession * session,
    char * filters_list, carray ** result)
{
  newsnntp * nntp;
  struct nntp_session_state_data * data;
  carray * info_array;
  int old_support_enabled;
  char filter[NEWSGROUP_NAME_MAX];
  unsigned int i;
  int res;
  int r;
  
  if (strcasecmp(session->sess_driver->sess_name, "nntp-cached") == 0) {
    struct nntp_cached_session_state_data * cached_data;
    
    cached_data = session->sess_data;
    session = cached_data->nntp_ancestor;
  }
  
  data = session->sess_data;
  nntp = data->nntp_session;
  
  old_support_enabled = 0;
  
  info_array = carray_new(1024);
  if (info_array == NULL) {
    res = ERROR_MEMORY;
    goto err;
  }
  
  /* iterate through given newsgroups filters */
  while (filters_list != NULL) {
    char * p;
    carray * info_filter_array;
    
    strncpy(filter, filters_list, sizeof(filter));
    filter[sizeof(filter) - 1] = '\0';
    p = strchr(filter, ',');
    if (p != NULL) {
      * p = '\0';
      filters_list += p - filter + 1;
    }
    else {
      filters_list = NULL;
    }
    
    r = get_nntp_group_info_list(storage, nntp, filter, &info_filter_array);
    if (r == ERROR_NEWSGROUPS_LIST) {
      old_support_enabled = 1;
      break;
    }
    
    r = store_in_tab(info_array, info_filter_array);
    if (r != NO_ERROR) {
      res = r;
      goto free_tab;
    }
    carray_free(info_filter_array);
  }
  
  if (old_support_enabled) {
    carray * info_filter_array;
    
    /* old NNTP server support */
    r = get_nntp_group_info_list(storage, nntp, NULL, &info_filter_array);
    if (r == ERROR_NEWSGROUPS_LIST) {
      res = r;
      goto free_tab;
    }
    
    r = store_in_tab(info_array, info_filter_array);
    if (r != NO_ERROR) {
      res = r;
      goto free_tab;
    }
    carray_free(info_filter_array);
  }
  
  qsort(carray_data(info_array), carray_count(info_array),
      sizeof(carray_data(info_array)[0]), cmp_name);
  
  * result = info_array;
  
  return NO_ERROR;
  
 free_tab:
  for(i = 0 ; i < carray_count(info_array) ; i ++) {
    struct etpan_nntp_group_info * info;
    
    info = carray_get(info_array, i);
    group_info_free(info);
  }
  carray_free(info_array);
 err:
  return res;
}


void etpan_nntp_list_groups_free(carray * info_array)
{
  unsigned int i;
  
  for(i = 0 ; i < carray_count(info_array) ; i ++) {
    struct etpan_nntp_group_info * info;
    
    info = carray_get(info_array, i);
    group_info_free(info);
  }
  carray_free(info_array);
}

int etpan_nntp_set_max_articles(mailsession * session, uint32_t max)
{
  return mailsession_parameters(session,
      NNTPDRIVER_SET_MAX_ARTICLES, &max);
}

