/*
 * etPan! -- a mail user agent
 *
 * Copyright (C) 2001, 2002 - DINH Viet Hoa
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the libEtPan! project nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * $Id: etpan-imap.c,v 1.10 2004/11/12 13:57:02 hoa Exp $
 */

#include "etpan-imap.h"

#include <libetpan/libetpan.h>
#include <stdlib.h>
#include <string.h>
#include "etpan-errors.h"

static struct etpan_imap_mailbox_info * mailbox_info_new(char * mb,
    int flags, char delimiter)
{
  struct etpan_imap_mailbox_info * mb_info;
  
  mb_info = malloc(sizeof(* mb_info));
  if (mb_info == NULL)
    goto err;
  
  mb_info->mb = strdup(mb);
  if (mb_info->mb == NULL)
    goto free;
  
  mb_info->flags = flags;
  mb_info->delimiter = delimiter;
  
  return mb_info;
  
 free:
  free(mb_info);
 err:
  return NULL;
}

static void mailbox_info_free(struct etpan_imap_mailbox_info * mb_info)
{
  free(mb_info->mb);
  free(mb_info);
}

static int imap_flags_to_flags(struct mailimap_mbx_list_flags * imap_flags)
{
  int flags;
  clistiter * cur;
  
  flags = 0;
  if (imap_flags->mbf_type == MAILIMAP_MBX_LIST_FLAGS_SFLAG) {
    switch (imap_flags->mbf_sflag) {
    case MAILIMAP_MBX_LIST_SFLAG_MARKED:
      flags |= ETPAN_IMAP_MB_MARKED;
      break;
    case MAILIMAP_MBX_LIST_SFLAG_NOSELECT:
      flags |= ETPAN_IMAP_MB_NOSELECT;
      break;
    case MAILIMAP_MBX_LIST_SFLAG_UNMARKED:
      flags |= ETPAN_IMAP_MB_UNMARKED;
      break;
    }
  }
  
  for(cur = clist_begin(imap_flags->mbf_oflags) ; cur != NULL ;
      cur = clist_next(cur)) {
    struct mailimap_mbx_list_oflag * oflag;
    
    oflag = clist_content(cur);
    
    switch (oflag->of_type) {
    case MAILIMAP_MBX_LIST_OFLAG_NOINFERIORS:
      flags |= ETPAN_IMAP_MB_NOINFERIORS;
      break;
    }
  }
  
  return flags;
}

int etpan_imap_lsub_mailboxes(mailsession * session,
    char * wildcard, carray ** result)
{
  clist * list;
  int r;
  clistiter * cur;
  struct imap_session_state_data * data;
  mailimap * imap;
  carray * info_array;
  unsigned int i;
  int res;
  
  if (strcasecmp(session->sess_driver->sess_name, "imap-cached") == 0) {
    struct imap_cached_session_state_data * cached_data;
    
    cached_data = session->sess_data;
    session = cached_data->imap_ancestor;
  }

  data = session->sess_data;
  imap = data->imap_session;
  
  r = mailimap_lsub(imap, "", wildcard, &list);
  if (r == MAILIMAP_ERROR_STREAM) {
    res = ERROR_STREAM;
    goto err;
  }
  else if (r != MAILIMAP_NO_ERROR) {
    res = ERROR_IMAP_MAILBOX_LIST;
    goto err;
  }

  info_array = carray_new(1024);
  if (info_array == NULL) {
    res = ERROR_MEMORY;
    goto free_list;
  }

  for(cur = clist_begin(list) ; cur != NULL ; cur = cur->next) {
    struct mailimap_mailbox_list * mb_list;
    struct etpan_imap_mailbox_info * mb_info;
    int flags;
    
    mb_list = cur->data;
    
    flags = 0;
    if (mb_list->mb_flag != NULL)
      flags = imap_flags_to_flags(mb_list->mb_flag);
    
    mb_info = mailbox_info_new(mb_list->mb_name, flags, mb_list->mb_delimiter);
    if (mb_info == NULL) {
      res = ERROR_MEMORY;
      goto free_array;
    }
    
    r = carray_add(info_array, mb_info, NULL);
    if (r < 0) {
      mailbox_info_free(mb_info);
      res = ERROR_MEMORY;
      goto free_array;
    }
  }
  
  mailimap_list_result_free(list);
  
  * result = info_array;
  
  return NO_ERROR;
  
 free_array:
  for(i = 0 ; i < carray_count(info_array) ; i ++) {
    struct etpan_imap_mailbox_info * mb_info;
    
    mb_info = carray_get(info_array, i);
    mailbox_info_free(mb_info);
  }
  carray_free(info_array);
 free_list:
  mailimap_list_result_free(list);
 err:
  return res;
}

int etpan_imap_list_mailboxes(mailsession * session,
    char * wildcard, carray ** result)
{
  clist * list;
  int r;
  clistiter * cur;
  struct imap_session_state_data * data;
  mailimap * imap;
  carray * info_array;
  unsigned int i;
  int res;
  
  if (strcasecmp(session->sess_driver->sess_name, "imap-cached") == 0) {
    struct imap_cached_session_state_data * cached_data;
    
    cached_data = session->sess_data;
    session = cached_data->imap_ancestor;
  }

  data = session->sess_data;
  imap = data->imap_session;
  
  r = mailimap_list(imap, "", wildcard, &list);
  if (r == MAILIMAP_ERROR_STREAM) {
    res = ERROR_STREAM;
    goto err;
  }
  else if (r != MAILIMAP_NO_ERROR) {
    res = ERROR_IMAP_MAILBOX_LIST;
    goto err;
  }

  info_array = carray_new(1024);
  if (info_array == NULL) {
    res = ERROR_MEMORY;
    goto free_list;
  }

  for(cur = clist_begin(list) ; cur != NULL ; cur = cur->next) {
    struct mailimap_mailbox_list * mb_list;
    struct etpan_imap_mailbox_info * mb_info;
    int flags;
    
    mb_list = cur->data;
    
    flags = 0;
    if (mb_list->mb_flag != NULL)
      flags = imap_flags_to_flags(mb_list->mb_flag);
    
    mb_info = mailbox_info_new(mb_list->mb_name, flags, mb_list->mb_delimiter);
    if (mb_info == NULL) {
      res = ERROR_MEMORY;
      goto free_array;
    }
    
    r = carray_add(info_array, mb_info, NULL);
    if (r < 0) {
      mailbox_info_free(mb_info);
      res = ERROR_MEMORY;
      goto free_array;
    }
  }
  
  mailimap_list_result_free(list);
  
  * result = info_array;
  
  return NO_ERROR;
  
 free_array:
  for(i = 0 ; i < carray_count(info_array) ; i ++) {
    struct etpan_imap_mailbox_info * mb_info;
    
    mb_info = carray_get(info_array, i);
    mailbox_info_free(mb_info);
  }
  carray_free(info_array);
 free_list:
  mailimap_list_result_free(list);
 err:
  return res;
}

void etpan_imap_list_mailboxes_free(carray * info_array)
{
  unsigned int i;
  
  for(i = 0 ; i < carray_count(info_array) ; i ++) {
    struct etpan_imap_mailbox_info * mb_info;
    
    mb_info = carray_get(info_array, i);
    mailbox_info_free(mb_info);
  }
  carray_free(info_array);
}

int etpan_imap_select(mailsession * session,
    char * mailbox)
{
  int r;
  struct imap_session_state_data * data;
  mailimap * imap;
  
  if (strcasecmp(session->sess_driver->sess_name, "imap-cached") == 0) {
    struct imap_cached_session_state_data * cached_data;
    
    cached_data = session->sess_data;
    session = cached_data->imap_ancestor;
  }

  data = session->sess_data;
  imap = data->imap_session;
  
  r = mailimap_select(imap, mailbox);
  if (r == MAILIMAP_ERROR_STREAM)
    return ERROR_STREAM;
  else if (r != MAILIMAP_NO_ERROR)
    return ERROR_IMAP_SELECT;
  
  r = mailimap_subscribe(imap, mailbox);
  /* ignore errors */
  
  return NO_ERROR;
}

int etpan_imap_create(mailsession * session,
    char * mailbox)
{
  int r;
  struct imap_session_state_data * data;
  mailimap * imap;
  
  if (strcasecmp(session->sess_driver->sess_name, "imap-cached") == 0) {
    struct imap_cached_session_state_data * cached_data;
    
    cached_data = session->sess_data;
    session = cached_data->imap_ancestor;
  }

  data = session->sess_data;
  imap = data->imap_session;
  
  r = mailimap_create(imap, mailbox);
  if (r == MAILIMAP_ERROR_STREAM)
    return ERROR_STREAM;
  else if (r != MAILIMAP_NO_ERROR)
    return ERROR_IMAP_CREATE;
  
  r = mailimap_subscribe(imap, mailbox);
  /* ignore errors */
  
  return NO_ERROR;
}
