/* ethos-js-plugin-loader.c
 *
 * Copyright (C) 2009 Christian Hergert <chris@dronelabs.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston MA 
 * 02110-1301 USA
 */

#include <gmodule.h>
#include <gjs/gjs.h>
#include <jsapi.h>

#include <ethos/ethos-error.h>
#include <ethos/ethos-plugin-loader.h>
#include <ethos/ethos-plugin-info-private.h>

#include "ethos-js-plugin-loader.h"

struct _EthosJSPluginLoaderPrivate
{
	GjsContext  *context;
	EthosPlugin *created_plugin;
};

static G_CONST_RETURN gchar*
get_name (EthosPluginLoader *plugin_loader)
{
	return "js";
}

static void
gc (EthosPluginLoader *plugin_loader)
{
}

static void
initialize (EthosPluginLoader *plugin_loader,
            EthosManager      *manager)
{
	EthosJSPluginLoaderPrivate *priv;

	g_return_if_fail (ETHOS_IS_JS_PLUGIN_LOADER (plugin_loader));

	priv = ETHOS_JS_PLUGIN_LOADER (plugin_loader)->priv;

	priv->context = gjs_context_new ();
}

static void
unload (EthosPluginLoader *plugin_loader)
{
	EthosJSPluginLoaderPrivate *priv;

	g_return_if_fail (ETHOS_IS_JS_PLUGIN_LOADER (plugin_loader));

	priv = ETHOS_JS_PLUGIN_LOADER (plugin_loader)->priv;

	g_object_unref (priv->context);
}

static GType
find_new_type (GType *before,
               guint  n_before,
               GType *after,
               guint  n_after)
{
	gboolean found;
	gint i, j;

	for (i = 0; i < n_after; i++) {
		found = FALSE;
		for (j = 0; j < n_before; j++) {
			if (before [j] == after [i]) {
				found = TRUE;
				break;
			}
		}
		if (!found)
			return after [i];
	}

	return G_TYPE_INVALID;
}

static EthosPlugin*
ethos_js_plugin_loader_load (EthosPluginLoader  *plugin_loader,
                             EthosPluginInfo    *plugin_info,
                             GError            **error)
{
	EthosJSPluginLoaderPrivate *priv;
	EthosPlugin                *plugin = NULL;
	const gchar                *filename,
	                           *module;
	gchar                      *dirname,
				   *basename,
				   *scriptname;
	gboolean                    success = FALSE;
	gint                        exit_status = 0;

	g_return_val_if_fail (ETHOS_IS_JS_PLUGIN_LOADER (plugin_loader), NULL);
	g_return_val_if_fail (ETHOS_IS_PLUGIN_INFO (plugin_info), NULL);

	priv = ETHOS_JS_PLUGIN_LOADER (plugin_loader)->priv;

	if (!(filename = ethos_plugin_info_get_filename (plugin_info))) {
		g_set_error (error, ETHOS_ERROR, ETHOS_ERROR_PLUGIN,
		             "Missing filename, cannot locate module");
		return NULL;
	}

	if (!(module = ethos_plugin_info_get_module (plugin_info))) {
		g_set_error (error, ETHOS_ERROR, ETHOS_ERROR_PLUGIN,
		             "Missing module name");
		return NULL;
	}

	dirname = g_path_get_dirname (filename);
	basename = g_strconcat (module, ".js", NULL);
	scriptname = g_build_filename (dirname, basename, NULL);

	if (!gjs_context_eval_file (priv->context,
	                            scriptname,
	                            &exit_status,
	                            error))
	        goto cleanup;

	if (priv->created_plugin) {
		plugin = priv->created_plugin;
		priv->created_plugin = NULL;
	}

cleanup:
	g_free (dirname);
	g_free (basename);
	g_free (scriptname);

	return plugin;
}

static void
register_plugin (EthosPluginLoader *plugin_loader,
                 EthosPlugin       *plugin)
{
	g_return_if_fail (ETHOS_IS_JS_PLUGIN_LOADER (plugin_loader));
	ETHOS_JS_PLUGIN_LOADER (plugin_loader)->priv->created_plugin = g_object_ref (plugin);
}

static void
ethos_js_plugin_loader_base_init (EthosPluginLoaderIface *iface)
{
	iface->get_name   = get_name;
	iface->gc         = gc;
	iface->initialize = initialize;
	iface->unload     = unload;
	iface->load       = ethos_js_plugin_loader_load;
	iface->register_plugin = register_plugin;
}

G_DEFINE_TYPE_EXTENDED (EthosJSPluginLoader,
                        ethos_js_plugin_loader,
                        G_TYPE_OBJECT,
                        0,
                        G_IMPLEMENT_INTERFACE (ETHOS_TYPE_PLUGIN_LOADER,
                                               ethos_js_plugin_loader_base_init));

static void
ethos_js_plugin_loader_finalize (GObject *object)
{
	G_OBJECT_CLASS (ethos_js_plugin_loader_parent_class)->finalize (object);
}

static void
ethos_js_plugin_loader_class_init (EthosJSPluginLoaderClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);
	object_class->finalize = ethos_js_plugin_loader_finalize;
	g_type_class_add_private (object_class, sizeof(EthosJSPluginLoaderPrivate));
}

static void
ethos_js_plugin_loader_init (EthosJSPluginLoader *plugin_loader)
{
	plugin_loader->priv = G_TYPE_INSTANCE_GET_PRIVATE (plugin_loader,
	                                                   ETHOS_TYPE_JS_PLUGIN_LOADER,
	                                                   EthosJSPluginLoaderPrivate);
}

EthosPluginLoader*
ethos_js_plugin_loader_new ()
{
	return g_object_new (ETHOS_TYPE_JS_PLUGIN_LOADER, NULL);
}

G_MODULE_EXPORT EthosPluginLoader*
ethos_plugin_loader_register (void)
{
	return ethos_js_plugin_loader_new ();
}
