# -*- coding: utf-8 -*-

# Copyright (c) 2005 - 2006 Detlev Offenbach <detlev@die-offenbachs.de>
#

"""
Module implementing the VCS project helper for Subversion.
"""

from qt import *

from VCS.ProjectHelper import VcsProjectHelper
from UI.E3Action import E3Action

class SvnProjectHelper(VcsProjectHelper):
    """
    Class implementing the VCS project helper for Subversion.
    """
    def __init__(self, vcsObject, projectObject, parent=None, name=None):
        """
        Constructor
        
        @param vcsObject reference to the vcs object
        @param projectObject reference to the project object
        @param parent parent widget (QWidget)
        @param name name of this object (string or QString)
        """
        VcsProjectHelper.__init__(self, vcsObject, projectObject, parent, name)
        
    def initActions(self):
        """
        Private method to generate the action objects.
        """
        self.vcsNewAct = E3Action(self.trUtf8('New from repository'),
                self.trUtf8('&New from repository...'),0,0,self,'subversion_new')
        self.vcsNewAct.setStatusTip(self.trUtf8(
            'Create a new project from the VCS repository'
        ))
        self.vcsNewAct.setWhatsThis(self.trUtf8(
            """<b>New from repository</b>"""
            """<p>This creates a new local project from the VCS repository.</p>"""
        ))
        self.vcsNewAct.connectIt(SIGNAL('activated()'),self.vcsCheckout)
        self.actions.append(self.vcsNewAct)
        
        self.vcsUpdateAct = E3Action(self.trUtf8('Update from repository'),
                self.trUtf8('&Update from repository'),0,0,self,
                'subversion_update')
        self.vcsUpdateAct.setStatusTip(self.trUtf8(
            'Update the local project from the VCS repository'
        ))
        self.vcsUpdateAct.setWhatsThis(self.trUtf8(
            """<b>Update from repository</b>"""
            """<p>This updates the local project from the VCS repository.</p>"""
        ))
        self.vcsUpdateAct.connectIt(SIGNAL('activated()'),self.vcsUpdate)
        self.actions.append(self.vcsUpdateAct)
        
        self.vcsCommitAct = E3Action(self.trUtf8('Commit changes to repository'),
                self.trUtf8('&Commit changes to repository...'),0,0,self,
                'subversion_commit')
        self.vcsCommitAct.setStatusTip(self.trUtf8(
            'Commit changes to the local project to the VCS repository'
        ))
        self.vcsCommitAct.setWhatsThis(self.trUtf8(
            """<b>Commit changes to repository</b>"""
            """<p>This commits changes to the local project to the VCS repository.</p>"""
        ))
        self.vcsCommitAct.connectIt(SIGNAL('activated()'),self.vcsCommit)
        self.actions.append(self.vcsCommitAct)
        
        self.vcsAddAct = E3Action(self.trUtf8('Add to repository'),
                self.trUtf8('&Add to repository...'),0,0,self,'subversion_add')
        self.vcsAddAct.setStatusTip(self.trUtf8(
            'Add the local project to the VCS repository'
        ))
        self.vcsAddAct.setWhatsThis(self.trUtf8(
            """<b>Add to repository</b>"""
            """<p>This adds (imports) the local project to the VCS repository.</p>"""
        ))
        self.vcsAddAct.connectIt(SIGNAL('activated()'),self.vcsImport)
        self.actions.append(self.vcsAddAct)
        
        self.vcsRemoveAct = E3Action(self.trUtf8(
                'Remove from repository (and disk)'),
                self.trUtf8('&Remove from repository (and disk)'),
                0,0,self,'subversion_remove')
        self.vcsRemoveAct.setStatusTip(self.trUtf8(
            'Remove the local project from the VCS repository (and  disk)'
        ))
        self.vcsRemoveAct.setWhatsThis(self.trUtf8(
            """<b>Remove from repository</b>"""
            """<p>This removes the local project from the VCS repository (and disk).</p>"""
        ))
        self.vcsRemoveAct.connectIt(SIGNAL('activated()'),self.vcsRemove)
        self.actions.append(self.vcsRemoveAct)
        
        self.vcsLogAct = E3Action(self.trUtf8(
                'Show log'),
                self.trUtf8('Show &log'),
                0, 0, self, 'subversion_log')
        self.vcsLogAct.setStatusTip(self.trUtf8(
            'Show the log of the local project'
        ))
        self.vcsLogAct.setWhatsThis(self.trUtf8(
            """<b>Show log</b>"""
            """<p>This shows the log of the local project.</p>"""
        ))
        self.vcsLogAct.connectIt(SIGNAL('activated()'),self.vcsLog)
        self.actions.append(self.vcsLogAct)
        
        self.svnLogLimitedAct = E3Action(self.trUtf8(
                'Show limited log'),
                self.trUtf8('Show limited log'),
                0, 0, self, 'subversion_log_limited')
        self.svnLogLimitedAct.setStatusTip(self.trUtf8(
            'Show a limited log of the local project'
        ))
        self.svnLogLimitedAct.setWhatsThis(self.trUtf8(
            """<b>Show limited log</b>"""
            """<p>This shows the log of the local project limited to a selectable"""
            """ number of entries.</p>"""
        ))
        self.svnLogLimitedAct.connectIt(SIGNAL('activated()'),self.svnLogLimited)
        self.actions.append(self.svnLogLimitedAct)
        
        self.vcsDiffAct = E3Action(self.trUtf8('Show difference'),
                self.trUtf8('Show &difference'),
                0, 0, self, 'subversion_diff')
        self.vcsDiffAct.setStatusTip(self.trUtf8(
            'Show the difference of the local project to the repository'
        ))
        self.vcsDiffAct.setWhatsThis(self.trUtf8(
            """<b>Show difference</b>"""
            """<p>This shows the difference of the local project to the repository.</p>"""
        ))
        self.vcsDiffAct.connectIt(SIGNAL('activated()'),self.vcsDiff)
        self.actions.append(self.vcsDiffAct)
        
        self.svnExtDiffAct = E3Action(self.trUtf8('Show difference (extended)'),
                self.trUtf8('Show difference (extended)'),
                0, 0, self, 'subversion_extendeddiff')
        self.svnExtDiffAct.setStatusTip(self.trUtf8(
            'Show the difference of revisions of the project to the repository'
        ))
        self.svnExtDiffAct.setWhatsThis(self.trUtf8(
            """<b>Show difference (extended)</b>"""
            """<p>This shows the difference of selectable revisions of the project.</p>"""
        ))
        self.svnExtDiffAct.connectIt(SIGNAL('activated()'),self.svnExtendedDiff)
        self.actions.append(self.svnExtDiffAct)
        
        self.vcsStatusAct = E3Action(self.trUtf8('Show status'),
                self.trUtf8('Show &status'),
                0, 0, self, 'subversion_status')
        self.vcsStatusAct.setStatusTip(self.trUtf8(
            'Show the status of the local project'
        ))
        self.vcsStatusAct.setWhatsThis(self.trUtf8(
            """<b>Show status</b>"""
            """<p>This shows the status of the local project.</p>"""
        ))
        self.vcsStatusAct.connectIt(SIGNAL('activated()'),self.vcsStatus)
        self.actions.append(self.vcsStatusAct)
        
        self.vcsTagAct = E3Action(self.trUtf8('Tag in repository'), 
                self.trUtf8('&Tag in repository...'),
                0, 0, self, 'subversion_tag')
        self.vcsTagAct.setStatusTip(self.trUtf8(
            'Tag the local project in the repository'
        ))
        self.vcsTagAct.setWhatsThis(self.trUtf8(
            """<b>Tag in repository</b>"""
            """<p>This tags the local project in the repository.</p>"""
        ))
        self.vcsTagAct.connectIt(SIGNAL('activated()'),self.vcsTag)
        self.actions.append(self.vcsTagAct)
        
        self.vcsExportAct = E3Action(self.trUtf8('Export from repository'), 
                self.trUtf8('&Export from repository...'),
                0, 0, self, 'subversion_export')
        self.vcsExportAct.setStatusTip(self.trUtf8(
            'Export a project from the repository'
        ))
        self.vcsExportAct.setWhatsThis(self.trUtf8(
            """<b>Export from repository</b>"""
            """<p>This exports a project from the repository.</p>"""
        ))
        self.vcsExportAct.connectIt(SIGNAL('activated()'),self.vcsExport)
        self.actions.append(self.vcsExportAct)
        
        self.vcsPropsAct = E3Action(self.trUtf8('Command options'),
                self.trUtf8('Command &options...'),0,0,self,
                'subversion_options')
        self.vcsPropsAct.setStatusTip(self.trUtf8('Show the VCS command options'))
        self.vcsPropsAct.setWhatsThis(self.trUtf8(
            """<b>Command options...</b>"""
            """<p>This shows a dialog to edit the VCS command options.</p>"""
        ))
        self.vcsPropsAct.connectIt(SIGNAL('activated()'),self.vcsCommandOptions)
        self.actions.append(self.vcsPropsAct)
        
        self.vcsRevertAct = E3Action(self.trUtf8('Revert changes'),
                self.trUtf8('Re&vert changes'),
                0, 0, self, 'subversion_revert')
        self.vcsRevertAct.setStatusTip(self.trUtf8(
            'Revert all changes made to the local project'
        ))
        self.vcsRevertAct.setWhatsThis(self.trUtf8(
            """<b>Revert changes</b>"""
            """<p>This reverts all changes made to the local project.</p>"""
        ))
        self.vcsRevertAct.connectIt(SIGNAL('activated()'),self.vcsRevert)
        self.actions.append(self.vcsRevertAct)
        
        self.vcsMergeAct = E3Action(self.trUtf8('Merge'),
                self.trUtf8('Mer&ge changes...'),
                0, 0, self, 'subversion_merge')
        self.vcsMergeAct.setStatusTip(self.trUtf8(
            'Merge changes of a tag/revision into the local project'
        ))
        self.vcsMergeAct.setWhatsThis(self.trUtf8(
            """<b>Merge</b>"""
            """<p>This merges changes of a tag/revision into the local project.</p>"""
        ))
        self.vcsMergeAct.connectIt(SIGNAL('activated()'),self.vcsMerge)
        self.actions.append(self.vcsMergeAct)
        
        self.vcsSwitchAct = E3Action(self.trUtf8('Switch'),
                self.trUtf8('S&witch...'),
                0, 0, self, 'subversion_switch')
        self.vcsSwitchAct.setStatusTip(self.trUtf8(
            'Switch the local copy to another tag/branch'
        ))
        self.vcsSwitchAct.setWhatsThis(self.trUtf8(
            """<b>Switch</b>"""
            """<p>This switches the local copy to another tag/branch.</p>"""
        ))
        self.vcsSwitchAct.connectIt(SIGNAL('activated()'),self.vcsSwitch)
        self.actions.append(self.vcsSwitchAct)
        
        self.vcsResolveAct = E3Action(self.trUtf8('Resolve conflicts'),
                self.trUtf8('Resolve con&flicts'),
                0, 0, self, 'subversion_resolve')
        self.vcsResolveAct.setStatusTip(self.trUtf8(
            'Resolve all conflicts of the local project'
        ))
        self.vcsResolveAct.setWhatsThis(self.trUtf8(
            """<b>Resolve conflicts</b>"""
            """<p>This resolves all conflicts of the local project.</p>"""
        ))
        self.vcsResolveAct.connectIt(SIGNAL('activated()'),self.svnResolve)
        self.actions.append(self.vcsResolveAct)
        
        self.vcsCleanupAct = E3Action(self.trUtf8('Cleanup'),
                self.trUtf8('Cleanu&p'),
                0, 0, self, 'subversion_cleanup')
        self.vcsCleanupAct.setStatusTip(self.trUtf8(
            'Cleanup the local project'
        ))
        self.vcsCleanupAct.setWhatsThis(self.trUtf8(
            """<b>Cleanup</b>"""
            """<p>This performs a cleanup of the local project.</p>"""
        ))
        self.vcsCleanupAct.connectIt(SIGNAL('activated()'),self.vcsCleanup)
        self.actions.append(self.vcsCleanupAct)
        
        self.vcsCommandAct = E3Action(self.trUtf8('Execute command'),
                self.trUtf8('E&xecute command...'),
                0, 0, self, 'subversion_command')
        self.vcsCommandAct.setStatusTip(self.trUtf8(
            'Execute an arbitrary VCS command'
        ))
        self.vcsCommandAct.setWhatsThis(self.trUtf8(
            """<b>Execute command</b>"""
            """<p>This opens a dialog to enter an arbitrary VCS command.</p>"""
        ))
        self.vcsCommandAct.connectIt(SIGNAL('activated()'),self.vcsCommand)
        self.actions.append(self.vcsCommandAct)
        
        self.svnTagListAct = E3Action(self.trUtf8('List tags'), 
                self.trUtf8('List tags...'),
                0, 0, self, 'subversion_list_tags')
        self.svnTagListAct.setStatusTip(self.trUtf8(
            'List tags of the project'
        ))
        self.svnTagListAct.setWhatsThis(self.trUtf8(
            """<b>List tags</b>"""
            """<p>This lists the tags of the project.</p>"""
        ))
        self.svnTagListAct.connectIt(SIGNAL('activated()'),self.svnTagList)
        self.actions.append(self.svnTagListAct)
        
        self.svnBranchListAct = E3Action(self.trUtf8('List branches'), 
                self.trUtf8('List branches...'),
                0, 0, self, 'subversion_list_branches')
        self.svnBranchListAct.setStatusTip(self.trUtf8(
            'List branches of the project'
        ))
        self.svnBranchListAct.setWhatsThis(self.trUtf8(
            """<b>List branches</b>"""
            """<p>This lists the branches of the project.</p>"""
        ))
        self.svnBranchListAct.connectIt(SIGNAL('activated()'),self.svnBranchList)
        self.actions.append(self.svnBranchListAct)
        
        self.svnListAct = E3Action(self.trUtf8('List repository contents'), 
                self.trUtf8('List repository contents...'),
                0, 0, self, 'subversion_contents')
        self.svnListAct.setStatusTip(self.trUtf8(
            'Lists the contents of the repository'
        ))
        self.svnListAct.setWhatsThis(self.trUtf8(
            """<b>List repository contents</b>"""
            """<p>This lists the contents of the repository.</p>"""
        ))
        self.svnListAct.connectIt(SIGNAL('activated()'),self.svnTagList)
        self.actions.append(self.svnListAct)
        
        self.svnPropSetAct = E3Action(self.trUtf8('Set Property'),
                self.trUtf8('Set Property...'),
                0, 0, self, 'subversion_property_set')
        self.svnPropSetAct.setStatusTip(self.trUtf8(
            'Set a property for the project files'
        ))
        self.svnPropSetAct.setWhatsThis(self.trUtf8(
            """<b>Set Property</b>"""
            """<p>This sets a property for the project files.</p>"""
        ))
        self.svnPropSetAct.connectIt(SIGNAL('activated()'),self.svnPropSet)
        self.actions.append(self.svnPropSetAct)
        
        self.svnPropListAct = E3Action(self.trUtf8('List Properties'),
                self.trUtf8('List Properties...'),
                0, 0, self, 'subversion_property_list')
        self.svnPropListAct.setStatusTip(self.trUtf8(
            'List properties of the project files'
        ))
        self.svnPropListAct.setWhatsThis(self.trUtf8(
            """<b>List Properties</b>"""
            """<p>This lists the properties of the project files.</p>"""
        ))
        self.svnPropListAct.connectIt(SIGNAL('activated()'),self.svnPropList)
        self.actions.append(self.svnPropListAct)
        
        self.svnPropDelAct = E3Action(self.trUtf8('Delete Property'),
                self.trUtf8('Delete Property...'),
                0, 0, self, 'subversion_property_delete')
        self.svnPropDelAct.setStatusTip(self.trUtf8(
            'Delete a property for the project files'
        ))
        self.svnPropDelAct.setWhatsThis(self.trUtf8(
            """<b>Delete Property</b>"""
            """<p>This deletes a property for the project files.</p>"""
        ))
        self.svnPropDelAct.connectIt(SIGNAL('activated()'),self.svnPropDel)
        self.actions.append(self.svnPropDelAct)
    
    def initMenu(self):
        """
        Private method to generate the VCS menu.
        """
        try:
            menu = self.project.vcsMenu
        except:
            return
        menu.clear()

        lbl = QLabel(self.vcs.vcsName(), menu)
        lbl.setFrameStyle( QFrame.Panel | QFrame.Sunken )
        lbl.setAlignment(Qt.AlignHCenter)
        font = lbl.font()
        font.setBold(1)
        lbl.setFont(font)
        menu.insertItem(lbl)
        
        menu.insertTearOffHandle()

        self.vcsUpdateAct.addTo(menu)
        self.vcsCommitAct.addTo(menu)
        menu.insertSeparator()
        self.vcsNewAct.addTo(menu)
        self.vcsExportAct.addTo(menu)
        menu.insertSeparator()
        self.vcsAddAct.addTo(menu)
        self.vcsRemoveAct.addTo(menu)
        menu.insertSeparator()
        self.vcsTagAct.addTo(menu)
        if self.vcs.otherData["standardLayout"]:
            self.svnTagListAct.addTo(menu)
            self.svnBranchListAct.addTo(menu)
        else:
            self.svnListAct.addTo(menu)
        menu.insertSeparator()
        self.vcsLogAct.addTo(menu)
        if self.vcs.versionStr >= '1.2.0':
            self.svnLogLimitedAct.addTo(menu)
        self.vcsStatusAct.addTo(menu)
        self.vcsDiffAct.addTo(menu)
        self.svnExtDiffAct.addTo(menu)
        menu.insertSeparator()
        self.vcsRevertAct.addTo(menu)
        self.vcsMergeAct.addTo(menu)
        self.vcsResolveAct.addTo(menu)
        menu.insertSeparator()
        self.vcsSwitchAct.addTo(menu)
        menu.insertSeparator()
        self.svnPropSetAct.addTo(menu)
        self.svnPropListAct.addTo(menu)
        self.svnPropDelAct.addTo(menu)
        menu.insertSeparator()
        self.vcsCleanupAct.addTo(menu)
        menu.insertSeparator()
        self.vcsCommandAct.addTo(menu)
        menu.insertSeparator()
        self.vcsPropsAct.addTo(menu)
    
    def svnResolve(self):
        """
        Private slot used to resolve conflicts of the local project.
        """
        self.vcs.svnResolve(self.project.ppath)
        
    def svnPropList(self):
        """
        Private slot used to list the properties of the project files.
        """
        self.vcs.svnListProps(self.project.ppath, 1)
        
    def svnPropSet(self):
        """
        Private slot used to set a property for the project files.
        """
        self.vcs.svnSetProp(self.project.ppath, 1)
        
    def svnPropDel(self):
        """
        Private slot used to delete a property for the project files.
        """
        self.vcs.svnDelProp(self.project.ppath, 1)
        
    def svnTagList(self):
        """
        Private slot used to list the tags of the project.
        """
        self.vcs.svnListTagBranch(self.project.ppath, 1)
        
    def svnBranchList(self):
        """
        Private slot used to list the branches of the project.
        """
        self.vcs.svnListTagBranch(self.project.ppath, 0)
        
    def svnExtendedDiff(self):
        """
        Private slot used to perform a svn diff with the selection of revisions.
        """
        self.vcs.svnExtendedDiff(self.project.ppath)
        
    def svnLogLimited(self):
        """
        Private slot used to perform a svn log --limit.
        """
        self.vcs.svnLogLimited(self.project.ppath)
