# -*- coding: utf-8 -*-

# Copyright (c) 2002 - 2005 Detlev Offenbach <detlev@die-offenbachs.de>
#

"""
Module implementing the Exception Logger widget.
"""

from qt import *


class ExceptionLogger(QListView):
    """
    Class implementing the Exception Logger widget.
    
    This class displays a log of all exceptions having occured during
    a debugging session.
    """
    def __init__(self,parent=None):
        """
        Constructor
        
        @param parent the parent widget of this widget
        """
        QListView.__init__(self, parent)
        
        self.setRootIsDecorated(1)
        self.setCaption(self.trUtf8("Exceptions"))
        self.addColumn(self.trUtf8("Exception"))
        self.setSorting(-1)     # disable sorting

        self.connect(self,SIGNAL('contextMenuRequested(QListViewItem *, const QPoint &, int)'),
                     self.handleContextMenu)

        QWhatsThis.add(self, self.trUtf8(
            """<b>Exceptions Logger</b>"""
            """<p>This windows shows a trace of all exceptions, that have"""
            """ occured during the last debugging session. Initially only the"""
            """ exception type and exception message are shown. After"""
            """ the expansion of this entry, the complete call stack as reported"""
            """ by the client is show with the most recent call first.</p>"""
        ))
        
        self.menu = QPopupMenu(self)
        self.menu.insertItem(self.trUtf8("Show source"), self.handleOpen)
        self.menu.insertItem(self.trUtf8("Clear"), self.clear)
        
        self.backMenu = QPopupMenu(self)
        self.backMenu.insertItem(self.trUtf8("Clear"), self.clear)
        
    def contentsMouseDoubleClickEvent(self, mouseEvent):
        """
        Protected method of QListView. 
        
        Reimplemented to disable expanding/collapsing
        of items when double-clicking. Instead the double-clicked entry is opened.
        
        @param mouseEvent the mouse event (QMouseEvent), ignored
        """
        self.handleOpen()

    def handleContextMenu(self,itm,coord,col):
        """
        Private slot to show the context menu of the listview.
        
        @param itm the item, that was right clicked (QListViewItem)
        @param coord the global coordinates of the mouse pointer (QPoint)
        @param col the column number of the mouse pointer (int)
        """
        if itm is None:
            self.backMenu.popup(coord)
        else:
            self.menu.popup(coord)
            
    def addException(self, status):
        """
        Public slot to handle the arrival of a new exception.
        
        @param status exception information
        """
        try:
            exclist = eval(status)
        except:
            QListViewItem(self,
                self.trUtf8('An unhandled exception occured. See the shell window for details.'))
            return
            
        exctype, msg = exclist[:2]
        if msg == '':
            itm = QListViewItem(self, "%s" % exctype)
        else:
            itm = QListViewItem(self, "%s, %s" % (exctype, msg))
        
        # now add the call stack, most recent call first
        for fn, ln in exclist[2:]:
            QListViewItem(itm, "%s, %d" % (fn, ln))
            
    def debuggingStarted(self):
        """
        Public slot to clear the listview upon starting a new debugging session.
        """
        self.clear()
        
    def handleOpen(self):
        """
        Private slot to handle a double click on an entry.
        """
        itm = self.currentItem()
        
        if itm.parent() is None:
            return
            
        entry = str(itm.text(0))
        entryList = entry.split(",")
        try:
            self.emit(PYSIGNAL('pythonFile'),(entryList[0], int(entryList[1])))
        except:
            pass

