/*
 *	Prints a list of all MIME Types or matches a MIME Type with a
 *	specified object.
 */

#include <glib.h>
#include <sys/stat.h>
#include <endeavour2.h>


void match_mimetype(edv_context_struct *ctx, const gchar *path)
{
	gint i;
	const gchar *s;
	struct stat lstat_buf;
	const edv_mimetype_struct *m;

	/* Check if object exists and get its stats */
	if(lstat(path, &lstat_buf))
	{
	    g_printerr(
"%s: No such object.\n",
		path
	    );
	    return;
	}

	/* Now that we have the object's path and its stats we can
	 * try and match a MIME Type to it
	 */
	m = EDVMimeTypeMatch(
	    ctx, path, &lstat_buf
	);
	if(m == NULL)
	{
	    /* Failed to find a MIME Type for this object, this should
	     * rarely happen since even if not MIME Type of class
	     * application, format, or unique object is matched for this
	     * object a MIME Type of class system should be returned
	     * for this object
	     */
	    g_printerr(
"%s: No MIME Type associated with object.\n",
		path
	    );
	    return;
	}

	/* Print matched MIME Type */
	g_print("%s (", m->type);
	switch(m->mt_class)
	{
	  case EDV_MIMETYPE_CLASS_SYSTEM:
	    g_print("system object");
	    break;
	  case EDV_MIMETYPE_CLASS_FORMAT:
	    g_print("file format");
	    break;
	  case EDV_MIMETYPE_CLASS_PROGRAM:
	    g_print("program object");
	    break;
	  case EDV_MIMETYPE_CLASS_UNIQUE:
	    g_print("unique object");
	    break;
	}
	g_print(")\n");


	/* Print icon files */
	g_print("Standard Icons:\n");
	i = EDV_MIMETYPE_ICON_STATE_STANDARD;
	s = m->small_icon_file[i];
	g_print("\tSmall: \"%s\"\n", (s != NULL) ? s : "");
	s = m->medium_icon_file[i];
	g_print("\tMedium: \"%s\"\n", (s != NULL) ? s : "");
	s = m->large_icon_file[i];
	g_print("\tLarge: \"%s\"\n", (s != NULL) ? s : "");

	g_print("Selected Icons:\n");
	i = EDV_MIMETYPE_ICON_STATE_SELECTED;
	s = m->small_icon_file[i];
	g_print("\tSmall: \"%s\"\n", (s != NULL) ? s : "");
	s = m->medium_icon_file[i];
	g_print("\tMedium: \"%s\"\n", (s != NULL) ? s : "");
	s = m->large_icon_file[i];
	g_print("\tLarge: \"%s\"\n", (s != NULL) ? s : "");

	g_print("Extended Icons:\n");
	i = EDV_MIMETYPE_ICON_STATE_EXTENDED;
	s = m->small_icon_file[i];
	g_print("\tSmall: \"%s\"\n", (s != NULL) ? s : "");
	s = m->medium_icon_file[i];
	g_print("\tMedium: \"%s\"\n", (s != NULL) ? s : "");
	s = m->large_icon_file[i];
	g_print("\tLarge: \"%s\"\n", (s != NULL) ? s : "");


	/* Print commands */
	if(m->total_commands > 0)
	{
	    g_print("Command%s:\n", (m->total_commands == 1) ? "" : "s");
	    for(i = 0; i < m->total_commands; i++)
	    {
		s = m->command[i];
		if(s == NULL)
		    continue;

		g_print(
		    "\t\"%s\" = \"%s\"",
		    m->command_name[i],
		    s
		);

		/* Does this command reffer to another MIME Type? */
		if(*s != '/')
		{
		    /* Get reffering MIME Type by matching its type
		     * with this command
		     *
		     * Commands that reffer to another MIME Type always
		     * contain that MIME Type's type.
		     */
		    const edv_mimetype_struct *m2 = EDVMimeTypeMatchType(
			ctx, s
		    );
		    if(m2 != NULL)
		    {
			if(m2->total_commands > 0)
			    g_print(" -> \"%s\"", m2->command[0]);
		    }

		}

		g_print("\n");
	    }
	}



}

void list_mimetypes(edv_context_struct *ctx)
{
	gint i, total;
	edv_mimetype_struct *m, **list = EDVMimeTypeList(ctx, &total);

	for(i = 0; i < total; i++)
	{
	    m = list[i];
	    if(m == NULL)
		continue;

	    g_print("%s (", m->type);
	    switch(m->mt_class)
	    {
	      case EDV_MIMETYPE_CLASS_SYSTEM:
		g_print("system object");
		break;
	      case EDV_MIMETYPE_CLASS_FORMAT:
		g_print("file format");
		break;
	      case EDV_MIMETYPE_CLASS_PROGRAM:
		g_print("program object");
		break;
	      case EDV_MIMETYPE_CLASS_UNIQUE:
		g_print("unique object");
		break;
	    }
	    g_print(")\n");
	}
}

int main(int argc, char *argv[])
{
	edv_context_struct *ctx = EDVContextNew();
	EDVContextLoadConfigurationFile(ctx, NULL);

	if(argc > 1)
	{
	    /* Match object */
	    match_mimetype(ctx, argv[1]);
	}
	else
	{
	    const gchar *prog = argv[0];

	    /* Since no arguments given then print list of MIME Types */
	    list_mimetypes(ctx);

	    g_print(
"\n\
Usage: %s [path]\n",
		prog
	    );
	}

	EDVContextSync(ctx);
	EDVContextDelete(ctx);

	return(0);
}
