/* BTP library - Banana Tree Protocol
 * Copyright (C) 1999-2001  The Regents of the University of Michigan
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the 
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */

#ifndef _BTP_H
#define _BTP_H


#include <glib.h>
#include <gnet/gnet.h>
/* No other dependencies */


/*

  Upper-level API for BTP.

  The Banana Tree Protocol is an end-to-end multicast tree creation
  protocol.  It tries to create a low-cost tree, not a low-delay tree.
  BTP is built ontop of TCP, but it could use UDP (with some
  additional tuning).

  It does:
    joining, switching, and leaving parents
    information (degree limit, neighbors)
    pinging and ponging

  It does not:
    transfer packets reliably
    request/retransmit lost packets
    eliminate duplicate packets

*/



/* **************************************** */

#define BTP_PORT   	5656


typedef struct _Btp Btp;

typedef void (*BtpPacketFunc)(Btp* btp, const void* buffer, guint length, gpointer user_data);
typedef void (*BtpErrorFunc) (Btp* btp, gpointer user_data);


struct _Btp
{
  /* Packet read callback and data (programmer sets) */
  BtpPacketFunc   packet_func;
  gpointer 	  packet_user_data;

  /* Error callback and data (programmer sets) */
  BtpErrorFunc    error_func;
  gpointer 	  error_user_data;

  /* ******************** */
  /* Private */

  struct _BtpTree* 	tree;

};

#define BTP_MAX_DEGREE	5

#include "b_peer.h"

Btp*  	    btp_create (BPeer* peer, const gchar* name);
Btp*  	    btp_join (BPeer* peer, const GURL* url);
void  	    btp_leave (Btp* btp);
gboolean    btp_has (BPeer* peer, const gchar* name); 

guint	    btp_get_max_degree (Btp* btp);
void        btp_set_max_degree (Btp* btp, guint max_degree);
gboolean    btp_get_follow_nodes (Btp* btp);
void        btp_set_follow_nodes (Btp* btp, gboolean follow_nodes);
gboolean    btp_get_use_shortcuts (Btp* btp);
void	    btp_set_use_shortcuts (Btp* btp, gboolean use_shortcuts);

GURL*       btp_get_url (Btp* btp);	/* callee owns */
gboolean    btp_is_up (const Btp* btp);

void        btp_send (Btp* btp, const guint8* buffer, guint16 length);

void	    btp_print (FILE* file, BPeer* peer);

/* Connect to the root.  Used for climbing. */
Btp*        btp_join_passive (BPeer* peer, const GURL* url);


#endif /* _BTP_H */
