# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

"""
Input events providers Component base classes
"""


__maintainer__ = 'Philippe Normand <philippe@fluendo.com>'
__maintainer2__ = 'Florian Boucault <florian@fluendo.com>'


from elisa.core.component import Component
from elisa.core.frontend import Frontend
from elisa.core.backend import Backend


class InputProvider(Component):
    """ Input events provider class. Plugins can't subclass it
    directly, they have to use either L{PollInputProvider} or
    L{PushInputProvider} classes.

    @ivar viewport: DOCME
    @type viewport:
    @ivar origin: source of the events; can be a frontend, a backend or None
    @type origin: L{Backend} or L{Frontend} or None
    """
    
    def create_input_event(self, data):
        """ Translate external input data to an InputEvent object,
        understandable by the InputManager. This method has to be
        overriden by InputProvider implementations. This method is
        called by InputProvider implementations.

        @param data: the data to translate, can be of any type
        @returns:    L{elisa.core.input_event.InputEvent}
        """
    
class PollInputProvider(InputProvider):
    """ Non-blocking way to retrieve user input events. Plugins can
    provide InputProviders to support new input devices (remote
    controls, wiimote, gyro mouse, joystick, ...)
    """

    def get_input_events(self):
        """ Retrieve events from an input device and translate them in
        L{elisa.core.input_event.InputEvent} objects understandable by
        the InputManager.
        The returned list can be built with the create_input_event() method,
        for each data retrieved from the input device.

        @returns: Input events from a user-input device
        @rtype:   L{elisa.core.input_event.InputEvent} list
        """
        return []

class PushInputProvider(InputProvider):
    """ InputProvider that will be able to push InputEvents directly
    to the InputManager by method calls
    (input_manager.process_event). This class should be used to
    support input devices that can't be polled for events.

    @ivar input_manager:  the InputManager the Provider is registered with
    @type input_manager:  L{elisa.core.input_manager.InputManager}
    """

    def clean(self):
        """ Additional cleanups to default Component's cleanup:
        disconnect our handlers from external component signals.
        """
        self.unbind()
        InputProvider.clean(self)

    def bind(self):
        """ Subscribe to the input device

        This can be done by connecting external component signals to
        our handlers or register a local callback that will be called
        by the external component when it receives an input event.

        This method must be overriden by implementation because it's
        totally dependent on the infrastructure used by the input
        device the component supports.
        """

    def unbind(self):
        """ Unsubscribe from the input device

        This can be done by disconnecting our handlers from the
        external component signals or unregistering the local callback
        previously registered by the register() method.

        This method must be overriden by implementation because it's
        totally dependent on the infrastructure used by the input
        device the component supports.
        """
