/* et_core.c - 2001/10/21 */
/*
 *  EasyTAG - Tag editor for MP3 and OGG files
 *  Copyright (C) 2000  Jerome Couderc <j.couderc@ifrance.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include <gtk/gtk.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <unistd.h>
#include "errno.h"

#include "easytag.h"
#include "et_core.h"
#include "mpeg_header.h"
#include "id3tag.h"
#ifdef ENABLE_OGG
#    include "ogg_tag.h"
#endif
#include "bar.h"
#include "browser.h"
#include "misc.h"
#include "setting.h"
#include "mylocale.h"



/***************
 * Declaration *
 ***************/



/**************
 * Prototypes *
 **************/

//gboolean ET_File_Is_Supported (gchar *filename);
gchar               *ET_Get_File_Extension                  (gchar *filename);
ET_File_Description *ET_Get_File_Description                (gchar *filename);
ET_File_Description *ET_Get_File_Description_From_Extension (gchar *extension);

//gboolean ET_Free_File_List (void);
gboolean ET_Free_File_List_Item (GList *list_item);
gboolean ET_Free_File_Name_Item (File_Name *FileName);
gboolean ET_Free_File_Tag_Item  (File_Tag *FileTag);
gboolean ET_Free_File_Name_List (ET_File *ETFile);
gboolean ET_Free_File_Tag_List  (ET_File *ETFile);

void     ET_Initialize_File_Item      (ET_File *ETFile);
void     ET_Initialize_File_Tag_Item  (File_Tag *FileTag);
void     ET_Initialize_File_Name_Item (File_Name *FileName);

gulong   ET_File_Key_New (void);
gulong   ET_Undo_Key_New (void);

void     ET_Display_File_And_List_Status_To_UI (gchar *filename);
void     ET_Display_Filename_To_UI             (gchar *filename);
gboolean ET_Display_File_Tag_To_UI             (ET_File *ETFile);
gboolean ET_Display_File_Info_To_UI            (ET_File_Info *ETFileInfo);

gboolean ET_Save_File_Name_From_UI (ET_File *ETFile, File_Name *FileName);
gboolean ET_Save_File_Tag_From_UI  (File_Tag *FileTag);

void     ET_Mark_File_Tag_As_Saved  (ET_File *ETFile);
void     ET_Mark_File_Name_As_Saved (ET_File *ETFile);

gboolean ET_Detect_Changes_Of_File_Name (File_Name *FileName1, File_Name *FileName2);
gboolean ET_Detect_Changes_Of_File_Tag  (File_Tag  *FileTag1,  File_Tag  *FileTag2);
gboolean ET_Add_File_Name_To_List       (ET_File *ETFile, File_Name *FileName);
gboolean ET_Add_File_Tag_To_List        (ET_File *ETFile, File_Tag  *FileTag);
gboolean ET_Add_File_To_History_List    (ET_File *ETFile, GList *FileNameLast, GList *FileNameNew, 
                                         GList *FileTagLast, GList *FileTagNew);

gboolean ET_Read_File_Info (gchar *filename, ET_File_Info *ETFileInfo);


/*******************
 * Basic functions *
 *******************/

/*
 * Returns the extension of the file
 */
gchar *ET_Get_File_Extension (gchar *filename)
{
    if (filename)
        return (gchar *)strrchr(filename,'.');
    else
        return NULL;
}
    

/*
 * Determine description of file using his extension.
 * If extension is NULL or not found into the tab, it returns the last entry for UNKNOWN_FILE.
 */
ET_File_Description *ET_Get_File_Description_From_Extension (gchar *extension)
{
    gint i;
    
    if (!extension)
        return &ETFileDescription[ET_FILE_DESCRIPTION_SIZE];
    
    for (i=0; i<ET_FILE_DESCRIPTION_SIZE; i++)
        if ( strcasecmp(extension,ETFileDescription[i].Extension)==0 )
            return &ETFileDescription[i];

    /* If not found in the list */
    return &ETFileDescription[ET_FILE_DESCRIPTION_SIZE];
}


/*
 * Determines description of file.
 * Determines first the extension. If extension is NULL or not found into the tab,
 * it returns the last entry for UNKNOWN_FILE.
 */
ET_File_Description *ET_Get_File_Description (gchar *filename)
{
    return ET_Get_File_Description_From_Extension(ET_Get_File_Extension(filename));
}


/*
 * Returns TRUE if the file is supported, else returns FALSE
 */
gboolean ET_File_Is_Supported (gchar *filename)
{
    if (ET_Get_File_Description(filename)->FileType != UNKNOWN_FILE)
        return TRUE;
    else
        return FALSE;
}




/**************************
 * Initializing functions *
 **************************/

void ET_Initialize_File_Item (ET_File *ETFile)
{
    if (ETFile)
    {
        ETFile->ETFileKey         = 0;
        ETFile->ETFileDescription = NULL;    
        ETFile->ETFileInfo        = NULL;
        ETFile->FileNameCur       = NULL;    
        ETFile->FileNameNew       = NULL;
        ETFile->FileNameList      = NULL;
        ETFile->FileTag           = NULL;
        ETFile->FileTagList       = NULL;
    }
}


void ET_Initialize_File_Tag_Item (File_Tag *FileTag)
{
    if (FileTag)
    {
        FileTag->key         = 0;
        FileTag->saved       = FALSE;    
        FileTag->title       = NULL;
        FileTag->artist      = NULL;
        FileTag->album       = NULL;
        FileTag->track       = NULL;
        FileTag->track_total = NULL;
        FileTag->year        = NULL;
        FileTag->genre       = NULL;
        FileTag->comment     = NULL;
    }
}


void ET_Initialize_File_Name_Item (File_Name *FileName)
{
    if (FileName)
    {
        FileName->key    = 0;
        FileName->saved  = FALSE;    
        FileName->value  = NULL;
    }
}

/* Key for each item of ETFileList */
gulong ET_File_Key_New (void)
{
    static gulong ETFileKey = 0;
    return ++ETFileKey;
}

/* Key for Undo */
gulong ET_Undo_Key_New (void)
{
    static gulong ETUndoKey = 0;
    return ++ETUndoKey;
}




/**********************************
 * File adding/removing functions *
 **********************************/

/*
 * ET_Add_File_To_File_List: Add a file to the "main" list. And get all informations of the file.
 */
GList *ET_Add_File_To_File_List (gchar *filename)
{
    ET_File_Description *ETFileDescription;
    ET_File      *ETFile;
    File_Name    *FileName;
    File_Tag     *FileTag;
    ET_File_Info *ETFileInfo;
    gulong        ETFileKey;
    
    
    if (!filename)
        return ETFileList;

    /* Primary Key for this file */
    ETFileKey = ET_File_Key_New ();

    /* Get description of the file */
    ETFileDescription = ET_Get_File_Description(filename);

    /* Fill the File_Name structure for FileNameList */
    FileName = g_malloc0(sizeof(File_Name));
    ET_Initialize_File_Name_Item(FileName);
    FileName->key   = 0;
    FileName->saved = TRUE;    /* The file haven't been changed, so it's saved */
    FileName->value = g_strdup(filename);

    /* Fill the File_Tag structure for FileTagList */
    FileTag = g_malloc0(sizeof(File_Tag));
    ET_Initialize_File_Tag_Item(FileTag);
    FileTag->key   = 0;
    FileTag->saved = TRUE;    /* The file haven't been changed, so it's saved */
    
    switch (ETFileDescription->TagType)
    {
        case ID3_TAG:
            Id3tag_Read_File_Tag(filename,FileTag);
            break;
#ifdef ENABLE_OGG
        case OGG_TAG:
            Ogg_Tag_Read_File_Tag(filename,FileTag);
            break;
#endif
        case UNKNOWN_TAG:
        default:
            g_print("FileTag: Undefined tag type %d for file %s.\n",ETFileDescription->TagType,filename);
            break;
    }

    /* Fill the ET_File_Info structure */
    ETFileInfo = g_malloc0(sizeof(ET_File_Info));

    switch (ETFileDescription->FileType)
    {
        case MP3_FILE:
        case MP2_FILE:
            Mpeg_Header_Read_File_Info(filename,ETFileInfo);
            break;
#ifdef ENABLE_OGG
        case OGG_FILE:
            Ogg_Header_Read_File_Info(filename,ETFileInfo);
            break;
#endif
        case FLAC_FILE:
            ET_Read_File_Info(filename,ETFileInfo);
            break;
        case UNKNOWN_FILE:
        default:
            g_print("ETFileInfo: Undefined file type %d for file %s.\n",ETFileDescription->FileType,filename);
            break;
    }

    /* Attach all data defined above to this ETFile item */
    ETFile = g_malloc0(sizeof(ET_File));
    ET_Initialize_File_Item(ETFile);
    ETFile->ETFileKey         = ETFileKey;
    ETFile->ETFileDescription = ETFileDescription;
    ETFile->FileNameList      = g_list_append(NULL,FileName);
    ETFile->FileNameCur       = ETFile->FileNameList;
    ETFile->FileNameNew       = ETFile->FileNameList;
    ETFile->FileTagList       = g_list_append(NULL,FileTag);
    ETFile->FileTag           = ETFile->FileTagList;
    ETFile->ETFileInfo        = ETFileInfo;
    
    /* Add the item to the "main list" */
    ETFileList = g_list_append(ETFileList,ETFile);
    /* Sort the list ... */
    switch (SORTING_FILE_MODE)
    {
        case SORTING_BY_ASCENDING_FILENAME:
            ETFileList = g_list_sort(ETFileList,(GCompareFunc)ET_Comp_Func_Sort_File_By_Ascending_Filename);
            break;
        case SORTING_BY_DESCENDING_FILENAME:
            ETFileList = g_list_sort(ETFileList,(GCompareFunc)ET_Comp_Func_Sort_File_By_Descending_Filename);
            break;
        case SORTING_BY_ASCENDING_TRACK_NUMBER:
            ETFileList = g_list_sort(ETFileList,(GCompareFunc)ET_Comp_Func_Sort_File_By_Ascending_Track_Number);
            break;
        case SORTING_BY_DESCENDING_TRACK_NUMBER:
            ETFileList = g_list_sort(ETFileList,(GCompareFunc)ET_Comp_Func_Sort_File_By_Descending_Track_Number);
            break;
        case SORTING_BY_ASCENDING_CREATION_DATE:
            ETFileList = g_list_sort(ETFileList,(GCompareFunc)ET_Comp_Func_Sort_File_By_Ascending_Creation_Date);
            break;
        case SORTING_BY_DESCENDING_CREATION_DATE:
            ETFileList = g_list_sort(ETFileList,(GCompareFunc)ET_Comp_Func_Sort_File_By_Descending_Creation_Date);
            break;
    }

    return ETFileList;
}




/**************************
 * File sorting functions *
 **************************/

/*
 * Comparison function for sorting by ascending file name.
 */
gint ET_Comp_Func_Sort_File_By_Ascending_Filename (ET_File *ETFile1, ET_File *ETFile2)
{
    gchar *filename1  = ((File_Name *)((GList *)ETFile1->FileNameCur)->data)->value;
    gchar *filename2  = ((File_Name *)((GList *)ETFile2->FileNameCur)->data)->value;
    gchar *dirname1   = g_dirname(filename1);
    gchar *dirname2   = g_dirname(filename2);
    gint cmp_dirname  = strcmp(dirname1,dirname2);
    gint cmp_filename = strcmp(filename1,filename2);
    
    g_free(dirname1);
    g_free(dirname2);
    if ( cmp_dirname==0 )    /* Files are in the same directory */
        return cmp_filename;
    else 
        return cmp_dirname;
}

/*
 * Comparison function for sorting by descending file name.
 */
gint ET_Comp_Func_Sort_File_By_Descending_Filename (ET_File *ETFile1, ET_File *ETFile2)
{
    return ET_Comp_Func_Sort_File_By_Ascending_Filename(ETFile2,ETFile1);
}

/*
 * Comparison function for sorting by ascending track number.
 */
gint ET_Comp_Func_Sort_File_By_Ascending_Track_Number (ET_File *ETFile1, ET_File *ETFile2)
{
    gint track1, track2;
    
    if ( !ETFile1->FileTag->data || !((File_Tag *)ETFile1->FileTag->data)->track )
        track1 = 0;
    else
        track1 = atoi( ((File_Tag *)ETFile1->FileTag->data)->track );
    if ( !ETFile2->FileTag->data || !((File_Tag *)ETFile2->FileTag->data)->track )
        track2 = 0;
    else
        track2 = atoi( ((File_Tag *)ETFile2->FileTag->data)->track );

    return (track1 - track2);
}

/*
 * Comparison function for sorting by descending track number.
 */
gint ET_Comp_Func_Sort_File_By_Descending_Track_Number (ET_File *ETFile1, ET_File *ETFile2)
{
    return ET_Comp_Func_Sort_File_By_Ascending_Track_Number(ETFile2,ETFile1);
}

/*
 * Comparison function for sorting by ascending creation date.
 */
gint ET_Comp_Func_Sort_File_By_Ascending_Creation_Date (ET_File *ETFile1, ET_File *ETFile2)
{
    struct stat statbuf1;
    struct stat statbuf2;
    gchar *filename1 = ((File_Name *)ETFile1->FileNameCur->data)->value;
    gchar *filename2 = ((File_Name *)ETFile2->FileNameCur->data)->value;

    lstat(filename1, &statbuf1);
    lstat(filename2, &statbuf2);

    return (statbuf1.st_ctime - statbuf2.st_ctime); // Creation date
    //return (statbuf1.st_mtime - statbuf2.st_mtime); // Modification date
}

/*
 * Comparison function for sorting by descending creation date.
 */
gint ET_Comp_Func_Sort_File_By_Descending_Creation_Date (ET_File *ETFile1, ET_File *ETFile2)
{
    return ET_Comp_Func_Sort_File_By_Ascending_Creation_Date(ETFile2,ETFile1);
}




/***************************
 * List handling functions *
 ***************************/

/*
 * Returns the first item of the "main list"
 */
GList *ET_File_List_First (void)
{
    ETFileList = g_list_first(ETFileList);
    ETFileList_Index = 1;
    return ETFileList;
}

/*
 * Returns the previous item of the "main list". When no more item, it returns NULL.
 */
GList *ET_File_List_Previous (void)
{
    if (ETFileList && ETFileList->prev)
    {
        ETFileList_Index--;
        return ETFileList = ETFileList->prev;
    }else
    {
        ETFileList_Index = 1;
        return NULL;
    }
}

/*
 * Returns the next item of the "main list".
 * When no more item, it returns NULL to don't "overwrite" the list.
 */
GList *ET_File_List_Next (void)
{
    if (ETFileList && ETFileList->next)
    {
        ETFileList_Index++;
        return ETFileList = ETFileList->next;
    }else
    {
        ETFileList_Index = ETFileList_Length;
        return NULL;
    }
}

/*
 * Returns the last item of the "main list"
 */
GList *ET_File_List_Last (void)
{
    ETFileList = g_list_last(ETFileList);
    ETFileList_Index = ETFileList_Length;
    return ETFileList;
}

/*
 * Remove an item from the "main list"
 */
GList *ET_File_List_Remove (GList *item_to_remove)
{
    guint current_position;
    
    if (!item_to_remove)
        return ETFileList;

    current_position = ETFileList_Index;
    ETFileList = g_list_remove_link(ETFileList,item_to_remove);
    ET_Free_File_List_Item(item_to_remove);
    
    /* Recalculate length of list */
    ET_File_List_Get_Length();

    /* Set position on the previous item */
    // Warning! may cause problem -> new line must be displayed before saving
    //ET_File_List_First();
    //while (ETFileList && ETFileList_Index<current_position-1 && ET_File_List_Next()) {}
    
    return ETFileList;
}

/*
 * Returns the item of the "main list" which correspond to the given 'key' (was used into browser list).
 */
GList *ET_File_List_Nth_By_Key (gulong key)
{
    ET_File_List_First();
    while (ETFileList)
    {
        if ( key == ((ET_File *)ETFileList->data)->ETFileKey )
            break;
        if (ET_File_List_Next() == NULL)
            return NULL;    /* Not found */
    }
    return ETFileList;
}

/*
 * Returns the length of the "main list"
 */
guint ET_File_List_Get_Length (void)
{
    GList *list = NULL;
    
    list = g_list_first(ETFileList);
    ETFileList_Length = g_list_length(list);
    return ETFileList_Length;
}




/*********************
 * Freeing functions *
 *********************/

/*
 * Frees the full main list of files: GList *ETFileList and reinitialize it.
 */
gboolean ET_Free_File_List (void)
{
    if (!ETFileList) return FALSE;
    
    ETFileList = g_list_last(ETFileList);
    while (ETFileList)
    {
        ET_Free_File_List_Item(ETFileList);
        if (!ETFileList->prev) break;
        ETFileList = ETFileList->prev;
    }

    g_list_free(ETFileList);
    ETFileList = (GList *)NULL;
    return TRUE;
}


/*
 * Frees one item of the full main list of files.
 */
gboolean ET_Free_File_List_Item (GList *list_item)
{
    if ( (ET_File *)list_item->data )
    {
        /* Frees: GList *FileNameList */
        ET_Free_File_Name_List((ET_File *)list_item->data);
        /* Frees: GList *FileTagList */
        ET_Free_File_Tag_List((ET_File *)list_item->data);
        /* Frees: ET_File_Info *ETFileInfo */
        if ( (ET_File_Info *)((ET_File *)list_item->data)->ETFileInfo )
            g_free( (ET_File_Info *)((ET_File *)list_item->data)->ETFileInfo );
        g_free( (ET_File *)list_item->data );
        (ET_File *)list_item->data = NULL;
    }
    return TRUE;
}


/*
 * Frees the full list: GList *FileNameList.
 */
gboolean ET_Free_File_Name_List (ET_File *ETFile)
{
    GList *list;
    
    if (!ETFile || !(GList *)ETFile->FileNameList) return FALSE;
    
    list = (GList *)ETFile->FileNameList;
    list = g_list_last(list);
    while (list)
    {
        if ( (File_Name *)list->data )
            ET_Free_File_Name_Item( (File_Name *)list->data );

        if (!list->prev) break;
        list = list->prev;
    }
    g_list_free(list);
    ETFile->FileNameList = (GList *)NULL;
    return TRUE;
}


/*
 * Frees a File_Name item.
 */
gboolean ET_Free_File_Name_Item (File_Name *FileName)
{
    if (!FileName) return FALSE;
    
    if (FileName->value)
        g_free(FileName->value);
    g_free(FileName);
    FileName = (File_Name *)NULL;
    return TRUE;
}

    
/*
 * Frees the full list: GList *TagList.
 */
gboolean ET_Free_File_Tag_List (ET_File *ETFile)
{
    GList *list;

    if (!ETFile || !(GList *)ETFile->FileTagList) return FALSE;

    list = (GList *)ETFile->FileTagList;
    list = g_list_last(list);
    while (list)
    {
        if ( (File_Tag *)list->data )
            ET_Free_File_Tag_Item( (File_Tag *)list->data );

        if (!list->prev) break;
        list = list->prev;
    }
    g_list_free(list);
    ETFile->FileTagList = (GList *)NULL;
    return TRUE;
}


/*
 * Frees a File_Tag item.
 */
gboolean ET_Free_File_Tag_Item (File_Tag *FileTag)
{
    if (!FileTag) return FALSE;

    if (FileTag->title)       g_free(FileTag->title);
    if (FileTag->artist)      g_free(FileTag->artist);
    if (FileTag->album)       g_free(FileTag->album);
    if (FileTag->year)        g_free(FileTag->year);
    if (FileTag->track)       g_free(FileTag->track);
    if (FileTag->track_total) g_free(FileTag->track_total);
    if (FileTag->genre)       g_free(FileTag->genre);
    if (FileTag->comment)     g_free(FileTag->comment);

    g_free(FileTag);
    FileTag = (File_Tag *)NULL;
    return TRUE;
}


gboolean ET_Free_History_File_List (void)
{
    GList *list;

    if (!ETHistoryFileList) return FALSE;

    ETHistoryFileList = g_list_first(ETHistoryFileList);
    list = ETHistoryFileList;
    while (list)
    {
        if ( (ET_History_File *)list->data )
            g_free( (ET_History_File *)list->data );
        list = list->next;
    }
    g_list_free(ETHistoryFileList);
    ETHistoryFileList = (GList *)NULL;
    return TRUE;
}




/************************
 * Displaying functions *
 ************************/

/*
 * Display informations of the file (Position + Header + Tag) to the user interface.
 */
void ET_Display_File_Data_To_UI (ET_File *ETFile)
{
    ET_File_Description *ETFileDescription;
    gchar *cur_filename;
    gchar *new_filename;
    gchar *msg;


    if (!ETFile) return;

    new_filename = ((File_Name *)((GList *)ETFile->FileNameNew)->data)->value;
    cur_filename = ((File_Name *)((GList *)ETFile->FileNameCur)->data)->value;
    ETFileDescription = ETFile->ETFileDescription;
    
    /* Display position in list + show/hide icon if file writable/read_only */
    ET_Display_File_And_List_Status_To_UI(cur_filename);
    
    /* Display filename (and his path) */
    ET_Display_Filename_To_UI(new_filename);
    
    /* Display tag data */
    switch (ETFileDescription->TagType)
    {
        case ID3_TAG:
            gtk_frame_set_label(GTK_FRAME(TagFrame),_("ID3 Tag"));
            ET_Display_File_Tag_To_UI(ETFile);
            break;
#ifdef ENABLE_OGG
        case OGG_TAG:
            gtk_frame_set_label(GTK_FRAME(TagFrame),_("OGG Vorbis Tag"));
            ET_Display_File_Tag_To_UI(ETFile);
            break;
#endif
        case UNKNOWN_TAG:
        default:
            gtk_frame_set_label(GTK_FRAME(TagFrame),_("Tag"));
            ET_Display_File_Tag_To_UI(ETFile); // To reinit screen
            g_print("FileTag: Undefined tag type %d for file %s.\n",ETFileDescription->TagType,cur_filename);
            break;
    }

    /* Display file data, header data and file type */
    switch (ETFileDescription->FileType)
    {
        case MP3_FILE:
            gtk_frame_set_label(GTK_FRAME(FileFrame),_("MP3 File"));
            Mpeg_Header_Display_File_Info_To_UI(cur_filename,ETFile->ETFileInfo);
            break;
        case MP2_FILE:
            gtk_frame_set_label(GTK_FRAME(FileFrame),_("MP2 File"));
            Mpeg_Header_Display_File_Info_To_UI(cur_filename,ETFile->ETFileInfo);
            break;
#ifdef ENABLE_OGG
        case OGG_FILE:
            gtk_frame_set_label(GTK_FRAME(FileFrame),_("OGG File"));
            Ogg_Header_Display_File_Info_To_UI(cur_filename,ETFile->ETFileInfo);
            break;
#endif
        case FLAC_FILE:
            gtk_frame_set_label(GTK_FRAME(FileFrame),_("FLAC File"));
            // No specific function yet to display it
            ET_Display_File_Info_To_UI(ETFile->ETFileInfo);
            break;
        case UNKNOWN_FILE:
        default:
            gtk_frame_set_label(GTK_FRAME(FileFrame),_("File"));
            ET_Display_File_Info_To_UI(ETFile->ETFileInfo);
            g_print("ETFileInfo: Undefined file type %d for file %s.\n",ETFileDescription->FileType,cur_filename);
            break;
    }
    
    msg = g_strdup_printf(_("File: '%s'"),cur_filename);
    Statusbar_Message(msg,FALSE);
    g_free(msg);
}


/*
 * Show the position in the list by using the index and list length.
 * Toggle visibility of the small icon if filename is not null.
 */
void ET_Display_File_And_List_Status_To_UI (gchar *filename)
{
    FILE *file;
    gchar *text;

    /* Show/hide 'AccessStatusIcon' */
    if ( (file=fopen(filename,"r+b"))!=NULL )
    {
        gtk_widget_hide(ReadOnlyStatusIconBox);
        gtk_widget_hide(BrokenStatusIconBox);
        fclose(file);
    }else
    {
        switch(errno)
        {
            case EACCES:    /* Permission denied */
            case EROFS:     /* Read-only file system */
                /* Read only file */
                gtk_widget_show_all(ReadOnlyStatusIconBox);
                gtk_widget_hide(BrokenStatusIconBox);
                break;
            case ENOENT:    /* No such file or directory */
            default:
                /* File not found */
                gtk_widget_show_all(BrokenStatusIconBox);
                gtk_widget_hide(ReadOnlyStatusIconBox);
        }
    }

    /* Show position of current file in list */
    text = g_strdup_printf("%d/%d:",ETFileList_Index,ETFileList_Length);
    gtk_label_set_text(GTK_LABEL(FileIndex),text);
    g_free(text);
}


void ET_Display_Filename_To_UI (gchar *filename)
{
    gchar *text;

    if (!filename) return;

    /*
     * Set filename into FileEntry
     */
    text = g_strdup(g_basename(filename));
    // Remove the extension
    *strrchr(text,'.') = 0;
    gtk_entry_set_text(GTK_ENTRY(FileEntry),text);
    g_free(text);
    // Justify to the left text into FileEntry
    gtk_editable_set_position(GTK_EDITABLE(FileEntry),0);

    /*
     * Set the path to the file into BrowserEntry (dirbrowser)
     */
    text = g_dirname(filename);
    Browser_Entry_Set_Text(text);
    g_free(text);
}


/*
 * Display all tag infos (tags fields) into entry boxes of the user interface.
 * These data have the same structure for all files.
 */
gboolean ET_Display_File_Tag_To_UI (ET_File *ETFile)
{
    File_Tag *FileTag = NULL;


    if (!ETFile || !ETFile->FileTag)
    {
        gtk_entry_set_text(GTK_ENTRY(TitleEntry),                  "");
        gtk_entry_set_text(GTK_ENTRY(ArtistEntry),                 "");
        gtk_entry_set_text(GTK_ENTRY(AlbumEntry),                  "");
        gtk_entry_set_text(GTK_ENTRY(YearEntry),                   "");
        gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(TrackEntry)->entry),"");
        gtk_entry_set_text(GTK_ENTRY(TrackTotalEntry),             "");
        gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(GenreEntry)->entry),"");
        gtk_entry_set_text(GTK_ENTRY(CommentEntry),                "");
        //Tag_Area_Set_Sensitive(FALSE);
        return FALSE;
    }

    //Tag_Area_Set_Sensitive(TRUE); // Causes displaying problem when saving files

    FileTag = (File_Tag *)(ETFile->FileTag->data);

    /* Show title */
    if (FileTag && FileTag->title)
        gtk_entry_set_text(GTK_ENTRY(TitleEntry),FileTag->title);
    else
        gtk_entry_set_text(GTK_ENTRY(TitleEntry),"");

    /* Show artist */
    if (FileTag && FileTag->artist)
        gtk_entry_set_text(GTK_ENTRY(ArtistEntry),FileTag->artist);
    else
        gtk_entry_set_text(GTK_ENTRY(ArtistEntry),"");

    /* Show album */
    if (FileTag && FileTag->album)
        gtk_entry_set_text(GTK_ENTRY(AlbumEntry),FileTag->album);
    else
        gtk_entry_set_text(GTK_ENTRY(AlbumEntry),"");

    /* Show year */
    if (FileTag && FileTag->year)
        gtk_entry_set_text(GTK_ENTRY(YearEntry),FileTag->year);
    else
        gtk_entry_set_text(GTK_ENTRY(YearEntry),"");

    /* Show track */
    if (FileTag && FileTag->track)
        gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(TrackEntry)->entry),FileTag->track);
    else
        gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(TrackEntry)->entry),"");

    /* Show number of tracks on the album */
    if (FileTag && FileTag->track_total)
        gtk_entry_set_text(GTK_ENTRY(TrackTotalEntry),FileTag->track_total);
    else
        gtk_entry_set_text(GTK_ENTRY(TrackTotalEntry),"");

    /* Show genre */
    if (FileTag && FileTag->genre)
        gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(GenreEntry)->entry),FileTag->genre);
    else
        gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(GenreEntry)->entry),"");

    /* Show comment */
    if (FileTag && FileTag->comment)
        gtk_entry_set_text(GTK_ENTRY(CommentEntry),FileTag->comment);
    else
        gtk_entry_set_text(GTK_ENTRY(CommentEntry),"");

    return TRUE;
}


/*
 * "Default" way to display File Info to the user interface.
 */
gboolean ET_Display_File_Info_To_UI(ET_File_Info *ETFileInfo)
{
    gchar *text;
    gchar *time  = NULL;
    gchar *time1 = NULL;
    gchar *size  = NULL;
    gchar *size1 = NULL;

    /* MPEG, Layer versions */
    text = g_strdup_printf("%d, Layer %d",ETFileInfo->version,ETFileInfo->layer);
    gtk_label_set_text(GTK_LABEL(VersionValueLabel),text);
    g_free(text);

    /* Bitrate */
    text = g_strdup_printf(_("%d kb/s"),ETFileInfo->bitrate);
    gtk_label_set_text(GTK_LABEL(BitrateValueLabel),text);
    g_free(text);

    /* Samplerate */
    text = g_strdup_printf(_("%d Hz"),ETFileInfo->samplerate);
    gtk_label_set_text(GTK_LABEL(SampleRateValueLabel),text);
    g_free(text);

    /* Mode */
    text = g_strdup_printf("%d",ETFileInfo->mode);
    gtk_label_set_text(GTK_LABEL(ModeValueLabel),text);
    g_free(text);

    /* Size */
    size  = Convert_Size(ETFileInfo->size);
    size1 = Convert_Size(ETFileList_TotalSize);
    text  = g_strdup_printf("%s (%s)",size,size1);
    gtk_label_set_text(GTK_LABEL(SizeValueLabel),text);
    if (size)  g_free(size);
    if (size1) g_free(size1);
    g_free(text);

    /* Duration */
    time  = Convert_Duration(ETFileInfo->duration);
    time1 = Convert_Duration(ETFileList_TotalDuration);
    text  = g_strdup_printf("%s (%s)",time,time1);
    gtk_label_set_text(GTK_LABEL(DurationValueLabel),text);
    if (time)  g_free(time);
    if (time1) g_free(time1);
    g_free(text);

    return TRUE;
}




/********************
 * Saving functions *
 ********************/

/*
 * Save informations of the file, contained into the entries of the user interface, in the list.
 * An undo key is generated to be used for filename and tag if there are changed is the same time.
 * Filename and Tag.
 */
void ET_Save_File_Data_From_UI (ET_File *ETFile)
{
    ET_File_Description *ETFileDescription;
    File_Name *FileName;
    File_Tag  *FileTag;
    void *FileNameNew_BeforeChanges = NULL;
    void *FileTag_BeforeChanges = NULL;
    gulong undo_key;
    gchar *cur_filename;
    gchar *msg;


    if (!ETFile) return;

    cur_filename = ((File_Name *)((GList *)ETFile->FileNameCur)->data)->value;
    ETFileDescription = ETFile->ETFileDescription;
    undo_key = ET_Undo_Key_New();

    /*
     * Save filename and generate undo for filename
     */
    FileName = g_malloc0(sizeof(File_Name));
    ET_Initialize_File_Name_Item(FileName);
    FileName->key = undo_key;

    ET_Save_File_Name_From_UI(ETFile,FileName);    // Used for all files!
    // Detect changes of filename and generate undo list
    if ( ETFile->FileNameNew && ET_Detect_Changes_Of_File_Name( (File_Name *)(ETFile->FileNameNew)->data,FileName )==TRUE )
    {
        FileNameNew_BeforeChanges = ETFile->FileNameNew;
        ET_Add_File_Name_To_List(ETFile,FileName);
    }else
    {
        ET_Free_File_Name_Item(FileName);
    }

    /*
     * Save tag data and generate undo for tag
     */
    // Fill a new File_Tag structure, detect changes, add it to undo list or delete it
    FileTag = g_malloc0(sizeof(File_Tag));
    ET_Initialize_File_Tag_Item(FileTag);
    FileTag->key = undo_key;

    switch (ETFileDescription->TagType)
    {
        case ID3_TAG:
#ifdef ENABLE_OGG
        case OGG_TAG:
#endif
            ET_Save_File_Tag_From_UI(FileTag);
            break;
        case UNKNOWN_TAG:
        default:
            g_print("FileTag: Undefined tag type %d for file %s.\n",ETFileDescription->TagType,cur_filename);
            break;
    }
    // Detect changes in tag data and generate undo list
    if ( ETFile->FileTag && ET_Detect_Changes_Of_File_Tag( (File_Tag *)(ETFile->FileTag)->data,FileTag )==TRUE )
    {
        FileTag_BeforeChanges = ETFile->FileTag;
        ET_Add_File_Tag_To_List(ETFile,FileTag);
    }else
    {
        ET_Free_File_Tag_Item(FileTag);
    }

    /*
     * Generate main undo (file history of modifications)
     */
    ET_Add_File_To_History_List(ETFile,
                                FileNameNew_BeforeChanges,ETFile->FileNameNew,
                                FileTag_BeforeChanges,    ETFile->FileTag);

    /* Refresh file into browser list */
    Browser_List_Refresh_File_In_List(ETFile);

    msg = g_strdup_printf(_("File: '%s'"),cur_filename);
    Statusbar_Message(msg,FALSE);
    g_free(msg);
}


/*
 * Save displayed filename into list if it had been changed. Generates also an history list for undo/redo.
 */
gboolean ET_Save_File_Name_From_UI (ET_File *ETFile, File_Name *FileName)
{
    gchar *filename = NULL;
    gchar *dirname = NULL;
    gchar *character;


    if (!ETFile || !FileName)
        return FALSE;

    /* Get the current path to the file */
    dirname = g_dirname( ((File_Name *)ETFile->FileNameNew->data)->value );

    /* Regenerate the new filename (without path) */
    filename = g_strconcat(gtk_entry_get_text_1(FileEntry),
                           ETFile->ETFileDescription->Extension,
                           NULL);
    /* Check if new filename seem to be correct */
    if ( !filename || strlen(filename) <= strlen(ETFile->ETFileDescription->Extension) )
    {
        if (filename)  g_free(filename);
        if (dirname)   g_free(dirname);
        return FALSE;
    }else
    {
        // Convert automatically '/' to '-'.
        while ( (character=strchr(filename,'/'))!=NULL )
            *character = '-';
        // Convert other illegal characters on windows filesystems
        if (REPLACE_ILLEGAL_CHARACTERS_IN_FILENAME)
        {
            while ( (character=strchr(filename,'\\'))!=NULL )
                *character = '-';
            while ( (character=strchr(filename,':'))!=NULL )
                *character = ';';
            while ( (character=strchr(filename,'*'))!=NULL )
                *character = '+';
            while ( (character=strchr(filename,'?'))!=NULL )
                *character = '_';
            while ( (character=strchr(filename,'\"'))!=NULL )
                *character = '\'';
            while ( (character=strchr(filename,'<'))!=NULL )
                *character = '(';
            while ( (character=strchr(filename,'>'))!=NULL )
                *character = ')';
            while ( (character=strchr(filename,'|'))!=NULL )
                *character = '-';
        }

        FileName->value  = g_strconcat(dirname,"/",filename,NULL);

        if (filename) g_free(filename);
        if (dirname)  g_free(dirname);
        return TRUE;
    }
}


/*
 * Load values, entered into entries of the UI, into a File_Tag structure which had been newly created.
 */
gboolean ET_Save_File_Tag_From_UI (File_Tag *FileTag)
{
    gchar *buffer = NULL;


    if (!FileTag)
        return FALSE;

    /* Title */
    buffer = gtk_entry_get_text_1(TitleEntry);
    Strip_String(buffer);

    if ( strlen(buffer) > 0 )
    {
        FileTag->title = g_strdup(buffer);
    } else
    {
        FileTag->title = NULL;
    }


    /* Artist */
    buffer = gtk_entry_get_text_1(ArtistEntry);
    Strip_String(buffer);

    if ( strlen(buffer) > 0 )
    {
        FileTag->artist = g_strdup(buffer);
    } else
    {
        FileTag->artist = NULL;
    }


    /* Album */
    buffer = gtk_entry_get_text_1(AlbumEntry);
    Strip_String(buffer);

    if ( strlen(buffer) > 0 )
    {
        FileTag->album = g_strdup(buffer);
    } else
    {
        FileTag->album = NULL;
    }


    /* Year */
    buffer = gtk_entry_get_text_1(YearEntry);
    Strip_String(buffer);

    if ( strlen(buffer) > 0 )
    {
        FileTag->year = g_strdup(buffer);
    } else
    {
        FileTag->year = NULL;
    }


    /* Track */
    buffer = gtk_entry_get_text_1(TrackEntry);
    Strip_String(buffer);

    if ( strlen(buffer) > 0  )
    {
        if (NUMBER_TRACK_FORMATED)
            FileTag->track = g_strdup_printf("%.2d",atoi(buffer));
        else
            FileTag->track = g_strdup(buffer);
    } else
    {
        FileTag->track = NULL;
    }


    /* Track Total */
    buffer = gtk_entry_get_text_1(TrackTotalEntry);
    Strip_String(buffer);

    if ( strlen(buffer) > 0  )
    {
        if (NUMBER_TRACK_FORMATED)
            FileTag->track_total = g_strdup_printf("%.2d",atoi(buffer));
        else
            FileTag->track_total = g_strdup(buffer);
    } else
    {
        FileTag->track_total = NULL;
    }


    /* Genre */
    buffer = gtk_entry_get_text_1(GenreEntry);
    Strip_String(buffer);

    if ( strlen(buffer) > 0 )
    {
        FileTag->genre = g_strdup(buffer);
    } else
    {
        FileTag->genre = NULL;
    }


    /* Comment */
    buffer = gtk_entry_get_text_1(CommentEntry);
    Strip_String(buffer);

    if ( strlen(buffer) > 0 )
    {
        FileTag->comment = g_strdup(buffer);
    } else
    {
        FileTag->comment = NULL;
    }

    return TRUE;
}


/*
 * Save data contained into File_Tag structure to the file on hard disk.
 */
gboolean ET_Save_File_Tag_To_HD (ET_File *ETFile)
{
    ET_File_Description *ETFileDescription;
    gchar *cur_filename;
    gboolean state;


    if (!ETFile) return FALSE;

    cur_filename = ((File_Name *)(ETFile->FileNameCur)->data)->value;
    ETFileDescription = ETFile->ETFileDescription;

    switch (ETFileDescription->TagType)
    {
        case ID3_TAG:
            state = Id3tag_Write_File_Tag(ETFile);
            break;
#ifdef ENABLE_OGG
        case OGG_TAG:
            state = Ogg_Tag_Write_File_Tag(ETFile);
            break;
#endif
        case UNKNOWN_TAG:
        default:
            g_print("Saving to HD: Undefined function for tag type '%d' (file %s).\n",
                ETFileDescription->TagType,cur_filename);
            state = FALSE;
            break;
    }
    if (state==TRUE)
    {
        ET_Mark_File_Tag_As_Saved(ETFile);
        return TRUE;
    }else
    {
        return FALSE;
    }
}

/*
 * Function used to update path of filenames into list after renaming a parent directory
 * (for ex: "/mp3/old_path/file.mp3" to "/mp3/new_path/file.mp3"
 */
void ET_Update_Directory_Name_Into_File_List(gchar* last_path, gchar *new_path)
{
    GList *filelist;
    ET_File *file;
    GList *filenamelist;
    gchar *filename;
    
    if (!ETFileList || !last_path || !new_path || strlen(last_path)<1 || strlen(new_path)<1 )
        return;
    
    filelist = g_list_first(ETFileList);
    while (filelist)
    {
        if ( (file=filelist->data) && (filenamelist=file->FileNameList) )
        {
            while (filenamelist)
            {
                if ( filenamelist->data && (filename=((File_Name *)filenamelist->data)->value) )
                {
                    gchar *last_path_tmp;
                    gchar *filename_tmp;
                    // Add '/' to end of path to avoid ambiguity between a directory and a filename...
                    if (last_path[strlen(last_path)-1]=='/') last_path_tmp = g_strdup(last_path);
                    else                                     last_path_tmp = g_strconcat(last_path,"/",NULL);
                    // Replace path of filename
                    if ( strncmp(filename,last_path_tmp,strlen(last_path_tmp))==0 )
                    {
                        // Create the new filename
                        filename_tmp = g_strconcat(new_path,
                                                   (new_path[strlen(new_path)-1]=='/')?"":"/",
                                                   &filename[strlen(last_path_tmp)],NULL);
                        g_free(filename);
                        ((File_Name *)filenamelist->data)->value = g_strdup(filename_tmp);
                        g_free(filename_tmp);
                    }
                }
                filenamelist = g_list_next(filenamelist);
             }
        }
        filelist = g_list_next(filelist);
    }
}



/***********************
 * Undo/Redo functions *
 ***********************/

/*
 * Compares two File_Name items and returns TRUE if there aren't the same.
 */
gboolean ET_Detect_Changes_Of_File_Name (File_Name *FileName1, File_Name *FileName2)
{
    if ( !FileName1 && !FileName2 )
        return FALSE;

    if ( ( FileName1 && !FileName2)
      || (!FileName1 &&  FileName2)
      || ( FileName1->value && !FileName2->value)
      || (!FileName1->value &&  FileName2->value) )
        return TRUE;

    /* Filename changed ? */
    if ( strcmp(g_basename(FileName1->value),g_basename(FileName2->value))!=0 )
        return TRUE;
    else
        return FALSE; /* No changes */
}

/*
 * Compares two File_Tag items and returns TRUE if there aren't the same.
 * Notes:
 *  - if field is '' or NULL => will be removed
 */
gboolean ET_Detect_Changes_Of_File_Tag (File_Tag *FileTag1, File_Tag *FileTag2)
{
    if ( !FileTag1 && !FileTag2 )
        return FALSE;

    if ( ( FileTag1 && !FileTag2)
      || (!FileTag1 &&  FileTag2) )
        return TRUE;

    /* Title */
    if ( FileTag1->title && !FileTag2->title && strlen(FileTag1->title)>0 ) return TRUE;
    if (!FileTag1->title &&  FileTag2->title && strlen(FileTag2->title)>0 ) return TRUE;
    if ( FileTag1->title &&  FileTag2->title && strcmp(FileTag1->title,FileTag2->title)!=0 ) return TRUE;

    /* Artist */
    if ( FileTag1->artist && !FileTag2->artist && strlen(FileTag1->artist)>0 ) return TRUE;
    if (!FileTag1->artist &&  FileTag2->artist && strlen(FileTag2->artist)>0 ) return TRUE;
    if ( FileTag1->artist &&  FileTag2->artist && strcmp(FileTag1->artist,FileTag2->artist)!=0 ) return TRUE;

    /* Album */
    if ( FileTag1->album && !FileTag2->album && strlen(FileTag1->album)>0 ) return TRUE;
    if (!FileTag1->album &&  FileTag2->album && strlen(FileTag2->album)>0 ) return TRUE;
    if ( FileTag1->album &&  FileTag2->album && strcmp(FileTag1->album,FileTag2->album)!=0 ) return TRUE;

    /* Year */
    if ( FileTag1->year && !FileTag2->year && strlen(FileTag1->year)>0 ) return TRUE;
    if (!FileTag1->year &&  FileTag2->year && strlen(FileTag2->year)>0 ) return TRUE;
    if ( FileTag1->year &&  FileTag2->year && strcmp(FileTag1->year,FileTag2->year)!=0 ) return TRUE;

    /* Track */
    if ( FileTag1->track && !FileTag2->track && strlen(FileTag1->track)>0 ) return TRUE;
    if (!FileTag1->track &&  FileTag2->track && strlen(FileTag2->track)>0 ) return TRUE;
    if ( FileTag1->track &&  FileTag2->track && strcmp(FileTag1->track,FileTag2->track)!=0 ) return TRUE;

    /* Track Total */
    if ( FileTag1->track_total && !FileTag2->track_total && strlen(FileTag1->track_total)>0 ) return TRUE;
    if (!FileTag1->track_total &&  FileTag2->track_total && strlen(FileTag2->track_total)>0 ) return TRUE;
    if ( FileTag1->track_total &&  FileTag2->track_total && strcmp(FileTag1->track_total,FileTag2->track_total)!=0 ) return TRUE;

    /* Genre */
    if ( FileTag1->genre && !FileTag2->genre && strlen(FileTag1->genre)>0 ) return TRUE;
    if (!FileTag1->genre &&  FileTag2->genre && strlen(FileTag2->genre)>0 ) return TRUE;
    if ( FileTag1->genre &&  FileTag2->genre && strcmp(FileTag1->genre,FileTag2->genre)!=0 ) return TRUE;

    /* Comment */
    if ( FileTag1->comment && !FileTag2->comment && strlen(FileTag1->comment)>0 ) return TRUE;
    if (!FileTag1->comment &&  FileTag2->comment && strlen(FileTag2->comment)>0 ) return TRUE;
    if ( FileTag1->comment &&  FileTag2->comment && strcmp(FileTag1->comment,FileTag2->comment)!=0 ) return TRUE;

    return FALSE; /* No changes */
}


/*
 * Add a FileName item to the history list of ETFile
 */
gboolean ET_Add_File_Name_To_List (ET_File *ETFile, File_Name *FileName)
{
    if (!ETFile || !FileName)
        return FALSE;

    /* Add the item to the list */
    ETFile->FileNameList = g_list_append(ETFile->FileNameList, FileName);
    ETFile->FileNameNew  = g_list_last(ETFile->FileNameList);

    return TRUE;
}

/*
 * Add a FileTag item to the history list of ETFile
 */
gboolean ET_Add_File_Tag_To_List (ET_File *ETFile, File_Tag *FileTag)
{
    if (!ETFile || !FileTag)
        return FALSE;
    
    /* Add the item to the list */
    ETFile->FileTagList = g_list_append(ETFile->FileTagList, FileTag);
    ETFile->FileTag     = g_list_last(ETFile->FileTagList);

    return TRUE;
}

/*
 * Add a ETFile item to the main undo list of files
 */
gboolean ET_Add_File_To_History_List (ET_File *ETFile, GList *FileNameLast, GList *FileNameNew, GList *FileTagLast, GList *FileTagNew)
{
    ET_History_File *ETHistoryFile;

    if (!ETFile || (!FileNameLast && !FileTagLast)) return FALSE;

    ETHistoryFile = g_malloc0(sizeof(ET_History_File));
    ETHistoryFile->ETFile       = ETFile;
    ETHistoryFile->FileNameLast = FileNameLast?FileNameLast:FileNameNew;
    ETHistoryFile->FileNameNew  = FileNameNew;
    ETHistoryFile->FileTagLast  = FileTagLast?FileTagLast:FileTagNew;
    ETHistoryFile->FileTagNew   = FileTagNew;

    /* The undo list must contains one item before the 'first undo' data */
    if (!ETHistoryFileList)
        ETHistoryFileList = g_list_append(ETHistoryFileList,g_malloc0(sizeof(ET_History_File)));

    /* Add the item to the list (cut end of list from the current element) */
    ETHistoryFileList = g_list_append(ETHistoryFileList,ETHistoryFile);
    ETHistoryFileList = g_list_last(ETHistoryFileList);

    return TRUE;
}


/*
 * Applies one undo to the ETFile data (returns previous data).
 * Returns TRUE if an undo had been applied.
 */
gboolean ET_Undo_File_Data (ET_File *ETFile)
{
    gboolean has_filename_undo_data = FALSE;
    gboolean has_filetag_undo_data  = FALSE;
    gulong   filename_key, filetag_key, undo_key;
    
    if (!ETFile)
        return FALSE;

    /* Find the valid key */
    if (ETFile->FileNameNew->prev && ETFile->FileNameNew->data)
        filename_key = ((File_Name *)ETFile->FileNameNew->data)->key;
    else
        filename_key = 0;
    if (ETFile->FileTag->prev && ETFile->FileTag->data)
        filetag_key = ((File_Tag *)ETFile->FileTag->data)->key;
    else
        filetag_key = 0;
    // The key to use
    undo_key = MAX(filename_key,filetag_key);

    /* Undo filename */
    if (ETFile->FileNameNew->prev && ETFile->FileNameNew->data
    && (undo_key==((File_Name *)ETFile->FileNameNew->data)->key))
    {
        ETFile->FileNameNew = ETFile->FileNameNew->prev;
        has_filename_undo_data = TRUE; // To indicate that an undo has been applied
    }

    /* Undo tag data */
    if (ETFile->FileTag->prev && ETFile->FileTag->data
    && (undo_key==((File_Tag *)ETFile->FileTag->data)->key))
    {
        ETFile->FileTag = ETFile->FileTag->prev;
        has_filetag_undo_data  = TRUE;
    }

    return has_filename_undo_data | has_filetag_undo_data;
}


/*
 * Returns TRUE if file contains undo data (filename or tag)
 */
gboolean ET_File_Data_Has_Undo_Data (ET_File *ETFile)
{
    gboolean has_filename_undo_data = FALSE;
    gboolean has_filetag_undo_data  = FALSE;

    if (!ETFile) return FALSE;

    if (ETFile->FileNameNew && ETFile->FileNameNew->prev) has_filename_undo_data = TRUE;
    if (ETFile->FileTag && ETFile->FileTag->prev)         has_filetag_undo_data  = TRUE;

    return has_filename_undo_data | has_filetag_undo_data;
}


/*
 * Applies one redo to the ETFile data. Returns TRUE if a redo had been applied.
 */
gboolean ET_Redo_File_Data (ET_File *ETFile)
{
    gboolean has_filename_redo_data = FALSE;
    gboolean has_filetag_redo_data  = FALSE;
    gulong   filename_key, filetag_key, undo_key;
    
    if (!ETFile)
        return FALSE;

    /* Find the valid key */
    if (ETFile->FileNameNew->next && ETFile->FileNameNew->next->data)
        filename_key = ((File_Name *)ETFile->FileNameNew->next->data)->key;
    else
        filename_key = (gulong)~0; // To have the max value for gulong
    if (ETFile->FileTag->next && ETFile->FileTag->next->data)
        filetag_key = ((File_Tag *)ETFile->FileTag->next->data)->key;
    else
        filetag_key = (gulong)~0; // To have the max value for gulong
    // The key to use
    undo_key = MIN(filename_key,filetag_key);

    /* Redo filename */
    if (ETFile->FileNameNew->next && ETFile->FileNameNew->next->data
    && (undo_key==((File_Name *)ETFile->FileNameNew->next->data)->key))
    {
        ETFile->FileNameNew = ETFile->FileNameNew->next;
        has_filename_redo_data = TRUE; // To indicate that a redo has been applied
    }

    /* Redo tag data */
    if (ETFile->FileTag->next && ETFile->FileTag->next->data
    && (undo_key==((File_Tag *)ETFile->FileTag->next->data)->key))
    {
        ETFile->FileTag = ETFile->FileTag->next;
        has_filetag_redo_data  = TRUE;
    }

    return has_filename_redo_data | has_filetag_redo_data;
}


/*
 * Returns TRUE if file contains redo data (filename or tag)
 */
gboolean ET_File_Data_Has_Redo_Data (ET_File *ETFile)
{
    gboolean has_filename_redo_data = FALSE;
    gboolean has_filetag_redo_data  = FALSE;

    if (!ETFile) return FALSE;

    if (ETFile->FileNameNew && ETFile->FileNameNew->next) has_filename_redo_data = TRUE;
    if (ETFile->FileTag && ETFile->FileTag->next)         has_filetag_redo_data  = TRUE;

    return has_filename_redo_data | has_filetag_redo_data;
}


gboolean ET_Undo_History_File_Data (void)
{
    ET_File *ETFile;
    ET_History_File *ETHistoryFile;
    
    if (!ETHistoryFileList || !ET_History_File_List_Has_Undo_Data()) return FALSE;

    ETHistoryFile = (ET_History_File *)ETHistoryFileList->data;
    ET_File_List_Nth_By_Key(ETHistoryFile->ETFile->ETFileKey);
    ETFile              = (ET_File *)ETFileList->data;
    ETFile->FileNameNew = ETHistoryFile->FileNameLast;
    ETFile->FileTag     = ETHistoryFile->FileTagLast;
    
    if (ETHistoryFileList->prev)
        ETHistoryFileList = ETHistoryFileList->prev;
    return TRUE;
}


/*
 * Returns TRUE if undo file list contains undo data
 */
gboolean ET_History_File_List_Has_Undo_Data (void)
{
    if (ETHistoryFileList && ETHistoryFileList->prev)
        return TRUE;
    else
        return FALSE;
}


gboolean ET_Redo_History_File_Data (void)
{
    ET_File *ETFile;
    ET_History_File *ETHistoryFile;

    if (!ETHistoryFileList || !ET_History_File_List_Has_Redo_Data()) return FALSE;

    ETHistoryFile = (ET_History_File *)ETHistoryFileList->next->data;
    ET_File_List_Nth_By_Key(ETHistoryFile->ETFile->ETFileKey);
    ETFile              = (ET_File *)ETFileList->data;
    ETFile->FileNameNew = ETHistoryFile->FileNameNew;
    ETFile->FileTag     = ETHistoryFile->FileTagNew;
 
    if (ETHistoryFileList->next)
        ETHistoryFileList = ETHistoryFileList->next;
    return TRUE;
}


/*
 * Returns TRUE if undo file list contains redo data
 */
gboolean ET_History_File_List_Has_Redo_Data (void)
{
    if (ETHistoryFileList && ETHistoryFileList->next)
        return TRUE;
    else
        return FALSE;
}




/**********************
 * Checking functions *
 **********************/


/*
 * Ckecks if the current files had been changed but not saved.
 * Returns TRUE if the file has been saved.
 * Returns FALSE if some changes haven't been saved.
 */
gboolean ET_Check_If_File_Is_Saved (ET_File *ETFile)
{
    File_Tag  *FileTag     = NULL;
    File_Name *FileNameNew = NULL;

    if (!ETFile) return TRUE;
    
    if (ETFile->FileTag)     FileTag = ETFile->FileTag->data;
    if (ETFile->FileNameNew) FileNameNew = ETFile->FileNameNew->data;
    
    // Check if the tag has been changed
    if ( FileTag && FileTag->saved != TRUE )
        return FALSE;

    // Check if name of file has been changed
    if ( FileNameNew && FileNameNew->saved != TRUE )
        return FALSE;

    // No changes
    return TRUE;
}


/*
 * Ckecks if some files, in the list, had been changed but not saved.
 * Returns TRUE if all files have been saved.
 * Returns FALSE if some changes haven't been saved.
 */
gboolean ET_Check_If_All_Files_Are_Saved (void)
{
    /* Check if some file haven't been saved, if didn't nochange=0 */
    if (!ETFileList)
    {
        return TRUE;
    }else
    {
        GList *tmplist = g_list_first(ETFileList);
        while (tmplist)
        {
            if ( ET_Check_If_File_Is_Saved((ET_File *)tmplist->data) == FALSE )
                return FALSE;

            if (!tmplist->next) break;
            tmplist = g_list_next(tmplist);
        }
        return TRUE;
    }
}




/*******************
 * Extra functions *
 *******************/

/*
 * Set to TRUE the value of 'FileTag->saved' for the File_Tag item passed in parameter.
 * And set ALL other values of the list to FALSE.
 */
void Set_Saved_Value_Of_File_Tag (File_Tag *FileTag, gboolean saved)
{
    if (FileTag) FileTag->saved = saved;
}
void ET_Mark_File_Tag_As_Saved (ET_File *ETFile)
{
    File_Tag *FileTag;
    GList *FileTagList;

    FileTag     = (File_Tag *)ETFile->FileTag->data;    // The current FileTag, to set to TRUE
    FileTagList = ((ET_File *)ETFileList->data)->FileTagList;
    g_list_foreach(FileTagList,(GFunc)Set_Saved_Value_Of_File_Tag,FALSE); // All other FileTag set to FALSE
    FileTag->saved = TRUE; // The current FileTag set to TRUE
}


/*
 * Set to TRUE the value of 'FileName->saved' for the File_Name item passed in parameter.
 * And set ALL other values of the list to FALSE.
 */
void Set_Saved_Value_Of_File_Name (File_Name *FileName, gboolean saved)
{
    if (FileName) FileName->saved = saved;
}
void ET_Mark_File_Name_As_Saved (ET_File *ETFile)
{
    File_Name *FileNameNew;
    GList *FileNameList;

    FileNameNew  = (File_Name *)ETFile->FileNameNew->data;    // The current FileName, to set to TRUE
    FileNameList = ((ET_File *)ETFileList->data)->FileNameList;
    g_list_foreach(FileNameList,(GFunc)Set_Saved_Value_Of_File_Tag,FALSE);
    FileNameNew->saved = TRUE;
}


/*
 * Currently, it's a way by default to fill file size into ET_File_Info structure
 */
gboolean ET_Read_File_Info (gchar *filename, ET_File_Info *ETFileInfo)
{
    FILE *file;

    if (!filename || !ETFileInfo)
        return FALSE;

    if ( (file=fopen(filename,"r"))==NULL )
    {
        g_print(_("ERROR while opening file: '%s' (%s)\n\a"),filename,g_strerror(errno));
        return FALSE;
    }
    fclose(file);

    ETFileInfo->version    = 0;
    ETFileInfo->bitrate    = 0;
    ETFileInfo->samplerate = 0;
    ETFileInfo->mode       = 0;
    ETFileInfo->size       = Get_File_Size(filename);
    ETFileInfo->duration   = 0;

    ETFileList_TotalSize     += ETFileInfo->size;
    ETFileList_TotalDuration += ETFileInfo->duration;

    return TRUE;
}


/*
 * Delete the corresponding file. Return TRUE if deleted.
 */
gboolean ET_Delete_File (ET_File *ETFile)
{
    gchar *cur_filename;
    
    if (!ETFile) return FALSE;

    cur_filename = ((File_Name *)(ETFile->FileNameCur)->data)->value;
    if (Delete_File(cur_filename)==TRUE)
    {
        GList *list = g_list_first(ETFileList);
        while(list)
        {
            if ( (ET_File *)list->data == ETFile )
                break;
            list = list->next;
        }
        ET_File_List_Remove(list);
        return TRUE;
    }else
    {
        return FALSE;
    }
}




/***********************
 * Debugging functions *
 ***********************/

/*
 * Functions for DEBUGGING ONLY
 * ET_Print_File_List => show list of filename
 * Parameters: file = __FILE__
 *             line = __LINE__
 */
void ET_Debug_Print_File_List (gchar *file ,gint line)
{
    gint item = 1;
    GList *savelist = NULL;

    g_print("\n#### File list from %s:%d - start ####\n",file,line);
    /* Get the current position in list */
    savelist = ETFileList;
    ETFileList = g_list_first(ETFileList);
    g_print("### length   : '%d'\n",g_list_length(ETFileList));
    while (ETFileList)
    {
        g_print("### item %d\n",item);
        g_print("# file_cur : '%s'\n",((File_Name *)((ET_File *)ETFileList->data)->FileNameCur->data)->value);
        g_print("# file_new : '%s'\n",((File_Name *)((ET_File *)ETFileList->data)->FileNameNew->data)->value);

        if (!ETFileList->next) break;
        ETFileList = g_list_next(ETFileList);
        item++;
    }
    /* Restore the current position in list */
    ETFileList = savelist;
    g_print("#### Undo list from %s:%d - end   ####\n",file,line);
}


