/*  $Header: /cvsroot/dvipdfmx/src/tpic.c,v 1.9 2004/03/03 13:19:02 hirata Exp $

    This is dvipdfmx, an eXtended version of dvipdfm by Mark A. Wicks.

    Copyright (C) 2002 by Jin-Hwan Cho and Shunsaku Hirata,
    the dvipdfmx project team <dvipdfmx@project.ktug.or.kr>
    
    Copyright (C) 1998, 1999 by Mark A. Wicks <mwicks@kettering.edu>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
    
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
    
    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
*/

#if HAVE_CONFIG_H
#include "config.h"
#endif

#include <string.h>

#include "system.h"
#include "mem.h"
#include "mfileio.h"

#include "pdfparse.h"
#include "pdfdoc.h"
#include "pdfdev.h"
#include "dvi.h"

#include "tpic.h"

/*
 * Following "constant" converts milli-inches to
 * device (in this case PDF stream) coordinates.
 */

#define MI2DEV (0.072/pdf_dev_scale())

struct path
{
  double x, y;
};

static struct
{
  double pen_size;
  int    fill_shape;
  double fill_color;
  struct path *path;
  long   path_length;
  long   max_path_length;
} tp_gstate = {
  1.0, 0, 0.0, NULL, 0, 0
};

static double default_fill_color = 0.5;

static void
tpic_clear_state (void) 
{
  if (tp_gstate.path) {
    RELEASE(tp_gstate.path);
    tp_gstate.path = NULL;
  }
  tp_gstate.path_length     = 0;
  tp_gstate.max_path_length = 0;
  tp_gstate.fill_shape = 0;
  tp_gstate.fill_color = 0.0;
}

static void
set_pen_size (char **buffer, char *end)
{
  char *number;

  skip_white (buffer, end);
  if ((number = parse_number(buffer, end))) {
    tp_gstate.pen_size = atof (number) * MI2DEV;
    RELEASE(number);
  } else {
    dump(*buffer, end);
    WARN("TPIC: Invalid pen size.");
  }
}

static void
set_fill_color (char **buffer, char *end)
{
  char *number;

  tp_gstate.fill_shape = 1;
  tp_gstate.fill_color = default_fill_color; 
  skip_white(buffer, end);
  if ((number = parse_number(buffer, end))) {
    tp_gstate.fill_color = 1.0 - atof(number);
    if (tp_gstate.fill_color > 1.0)
      tp_gstate.fill_color = 1.0;
    if (tp_gstate.fill_color < 0.0)
      tp_gstate.fill_color = 0.0;
    RELEASE(number);
  }
}

static void
add_point (char **buffer, char *end) 
{
  char *x= NULL, *y= NULL;

  skip_white(buffer, end);
  if (*buffer < end)
    x = parse_number(buffer, end);
  skip_white(buffer, end);
  if (*buffer < end)
    y = parse_number(buffer, end);
  if ((x) && (y)) {
    if (tp_gstate.path_length >= tp_gstate.max_path_length) {
      tp_gstate.max_path_length += 256;
      tp_gstate.path = RENEW(tp_gstate.path, tp_gstate.max_path_length, struct path);
    }
    tp_gstate.path[tp_gstate.path_length].x = atof(x)*MI2DEV;
    tp_gstate.path[tp_gstate.path_length].y = atof(y)*MI2DEV;
    tp_gstate.path_length += 1;
  } else {
    dump(*buffer, end);
    WARN("TPIC: Missing coordinate for \"pa\".\n");
  }
  if (x) RELEASE(x);
  if (y) RELEASE(y);

  return;
}

static void
show_path (int hidden) 
{
  int len;

  /*
   * The semantics of a fill_color of 0.0 or 0.5 will be to
   * use current painting color known to dvipdfm.
   */
#if 0
  /* It may destroy content of work_buffer. */
  if (tp_gstate.fill_shape && tp_gstate.fill_color != 0.0) {
    len = sprintf(work_buffer, " %.2f g", tp_gstate.fill_color);
    pdf_doc_add_to_page(work_buffer, len);
  }
#endif
  if (tp_gstate.fill_shape && tp_gstate.fill_color != 0.0) {
    char wbuf[32];
    len = sprintf(wbuf, " %.2f g", tp_gstate.fill_color);
    pdf_doc_add_to_page(wbuf, len);
  }

  if (!hidden && tp_gstate.fill_shape) {
    pdf_doc_add_to_page(" b", 2);
  } else if (hidden && tp_gstate.fill_shape) {
    pdf_doc_add_to_page(" f", 2);
  } else if (!hidden && !tp_gstate.fill_shape) {
    pdf_doc_add_to_page(" S", 2);
  } else if (hidden && !tp_gstate.fill_shape) {
    /*
     * Acrobat claims 'Q' as illegal operation when there are unfinished
     * path (a path without path-painting operator applied)?
     */
    pdf_doc_add_to_page(" n", 2);
  }
  tp_gstate.fill_color = 0.0;
  tp_gstate.fill_shape = 0;
}


static void
flush_path (double x_user, double y_user, int hidden, double dash_dot)
{
  int len;

  /* Make pen_size == 0 equivalent to hidden */
  if (tp_gstate.pen_size == 0.0)
    hidden = 1;

  if (hidden && !tp_gstate.fill_shape) {
    /* Prevent writing unpainted paths. */
  } else if (tp_gstate.path_length > 1) {
    int i;

    len = sprintf(work_buffer, " q");
    pdf_doc_add_to_page(work_buffer, len);
    if (tp_gstate.pen_size != 0.0) {
      len = sprintf(work_buffer, " %.2f w", tp_gstate.pen_size);
      pdf_doc_add_to_page(work_buffer, len);
    }
    len = sprintf(work_buffer, " 1 J 1 j");
    pdf_doc_add_to_page(work_buffer, len);
    if (dash_dot != 0.0) {
      if (dash_dot > 0.0) {
	len = sprintf(work_buffer, " [%.1f %.1f] 0 d",
		      dash_dot*72.0, dash_dot*36.0);
      } else {
	len = sprintf(work_buffer, " [%.1f %.1f] 0 d",
		      tp_gstate.pen_size, -dash_dot*72.0);
      }
      pdf_doc_add_to_page(work_buffer, len);
    }

    len = sprintf(work_buffer, " %.2f %.2f m",
		  x_user + tp_gstate.path[0].x, y_user - tp_gstate.path[0].y);
    pdf_doc_add_to_page(work_buffer, len);
    for (i = 0; i < tp_gstate.path_length; i++) {
      len = sprintf(work_buffer, " %.2f %.2f l",
		    x_user + tp_gstate.path[i].x, y_user - tp_gstate.path[i].y);
      pdf_doc_add_to_page(work_buffer, len);
    } 
    show_path(hidden);
    pdf_doc_add_to_page(" Q", 2);
  } else {
    WARN("TPIC: Not enough points for \"fp\"!\n");
  }
  tpic_clear_state();

  return;
}

static void
spline_path (double x_user, double y_user, double dash_dot)
{
  int len;

  /* Spline is meaningless for path length of less than 3 */
  if (tp_gstate.path_length > 2) {
    int i;
    len = sprintf(work_buffer, " q 1.4 M %.2f w", tp_gstate.pen_size);
    pdf_doc_add_to_page(work_buffer, len);
    if (dash_dot != 0.0) {
      if (dash_dot > 0.0) {
	len = sprintf(work_buffer, " [%.1f %.1f] 0 d",
		      dash_dot*72.0, dash_dot*36.0);
      } else if (dash_dot < 0.0) {
	len = sprintf(work_buffer, " [%.1f %.1f] 0 d 1 J",
		      tp_gstate.pen_size, -dash_dot*72.0);
      }
      pdf_doc_add_to_page(work_buffer, len);
    }
    len = sprintf(work_buffer, " %.2f %.2f m",
		  x_user + tp_gstate.path[0].x, y_user - tp_gstate.path[0].y);
    pdf_doc_add_to_page(work_buffer, len);
    len = sprintf(work_buffer, " %.2f %.2f l",
		  x_user + 0.5*(tp_gstate.path[0].x + tp_gstate.path[1].x), 
		  y_user - 0.5*(tp_gstate.path[0].y + tp_gstate.path[1].y));
    pdf_doc_add_to_page(work_buffer, len);
    for (i = 1; i < tp_gstate.path_length - 1; i++) {
      len = sprintf(work_buffer, " %.2f %.2f %.2f %.2f y",
		    x_user + tp_gstate.path[i].x,
		    y_user - tp_gstate.path[i].y,
		    x_user + 0.5*(tp_gstate.path[i].x + tp_gstate.path[i+1].x),
		    y_user - 0.5*(tp_gstate.path[i].y + tp_gstate.path[i+1].y));
      pdf_doc_add_to_page(work_buffer, len);
    } 
    len = sprintf(work_buffer, " %.2f %.2f l",
		  x_user + tp_gstate.path[i].x,
		  y_user - tp_gstate.path[i].y);
    pdf_doc_add_to_page(work_buffer, len);
    show_path(0);
    pdf_doc_add_to_page(" Q", 2);
  } else {
    WARN("TPIC: Not enough points for \"sp\"!\n");
  }
  tpic_clear_state();

  return;
}

static void
arc (char **buffer, char *end, double x_user, double y_user, int hidden) 
{
  char *xcs, *ycs, *xrs, *yrs, *sas, *eas;
  double xc, yc, xr, yr, sa, ea;
  char *save;

  if (tp_gstate.pen_size == 0.0)
    hidden = 1;

  save = *buffer;
  xcs = ycs = xrs = yrs = sas = eas = NULL;
  if ((xcs = parse_number(buffer, end)) &&
      (ycs = parse_number(buffer, end)) &&
      (xrs = parse_number(buffer, end)) &&
      (yrs = parse_number(buffer, end)) &&
      (sas = parse_number(buffer, end)) &&
      (eas = parse_number(buffer, end))) {
    if (!hidden || tp_gstate.fill_shape) {
      double c, s, cur_x, cur_y, inc_ang;
      double cp1_x, cp1_y, cp2_x, cp2_y;
      int    len, i, nsteps;

      xc = atof(xcs)*MI2DEV; yc = atof(ycs)*MI2DEV;
      xr = atof(xrs)*MI2DEV; yr = atof(yrs)*MI2DEV;
      sa = atof(sas); ea = atof(eas);
      while (ea - sa < 0.0)
	sa -= 2 * 3.14159265358;
#define ROTATE(x,y,c,s) {\
                          double tmp_x, tmp_y;\
                          tmp_x = (c)*(x) - (s)*(y);\
                          tmp_y = (s)*(x) + (c)*(y);\
                          x = tmp_x; y = tmp_y;\
                        }
#define MAX_ANG_STEP 1.0
      nsteps  = (int) ((ea-sa)/MAX_ANG_STEP) + 1;
      inc_ang = (ea-sa)/nsteps;
      c = cos(inc_ang); s = sin(inc_ang);
      cur_x = cos(sa); cur_y = sin(sa);
      cp1_x = cur_x - inc_ang/3.0*cur_y;
      cp1_y = cur_y + inc_ang/3.0*cur_x;
      cp2_x = cur_x + inc_ang/3.0*cur_y;
      cp2_y = cur_y - inc_ang/3.0*cur_x;

      len = sprintf(work_buffer, " q");
      pdf_doc_add_to_page(work_buffer, len);
      if (tp_gstate.pen_size != 0.0) {
	len = sprintf(work_buffer, " %.2f w", tp_gstate.pen_size);
	pdf_doc_add_to_page(work_buffer, len);
      }
      len = sprintf(work_buffer, " 1 J");
      pdf_doc_add_to_page(work_buffer, len);
      len = sprintf(work_buffer, " %.2f %.2f m",
		    x_user + xr*cur_x + xc, y_user - yr*cur_y - yc);
      pdf_doc_add_to_page(work_buffer, len);

      ROTATE(cp2_x, cp2_y, c, s);
      ROTATE(cur_x, cur_y, c, s);
      for (i = 0; i < nsteps; i++) {
	len = sprintf(work_buffer,
		      " %.2f %.2f %.2f %.2f %.2f %.2f c",
		      x_user + xr*cp1_x + xc, y_user - yr*cp1_y - yc,
		      x_user + xr*cp2_x + xc, y_user - yr*cp2_y - yc,
		      x_user + xr*cur_x + xc, y_user - yr*cur_y - yc);
	pdf_doc_add_to_page(work_buffer, len);
	ROTATE(cur_x, cur_y, c, s);
	ROTATE(cp1_x, cp1_y, c, s);
	ROTATE(cp2_x, cp2_y, c, s);
      }
      show_path(hidden);
      pdf_doc_add_to_page(" Q", 2);
    }
  } else {
    dump(save, end);
    WARN("TPIC: Error in parameters for ar/ir.");
  }
  if (xcs) RELEASE(xcs);
  if (ycs) RELEASE(ycs);
  if (xrs) RELEASE(xrs);
  if (yrs) RELEASE(yrs);
  if (sas) RELEASE(sas);
  if (eas) RELEASE(eas);

  tpic_clear_state();

  return;
}

#define TPIC_PN 1
#define TPIC_PA 2
#define TPIC_FP 3
#define TPIC_IP 4
#define TPIC_DA 5
#define TPIC_DT 6
#define TPIC_SP 7
#define TPIC_AR 8
#define TPIC_IA 9
#define TPIC_SH 10
#define TPIC_WH 11
#define TPIC_BK 12
#define TPIC_TX 13

#define TPIC_UNKNOWN -1

static struct {
  const char *s;
  int tpic_command;
} tpic_specials[] = {
  {"pn", TPIC_PN},
  {"pa", TPIC_PA},
  {"fp", TPIC_FP},
  {"ip", TPIC_IP},
  {"da", TPIC_DA},
  {"dt", TPIC_DT},
  {"sp", TPIC_SP},
  {"ar", TPIC_AR},
  {"ia", TPIC_IA},
  {"sh", TPIC_SH},
  {"wh", TPIC_WH},
  {"bk", TPIC_BK},
  {"tx", TPIC_TX},
  {NULL, -1}
};

int
tpic_parse_special (char *buffer, UNSIGNED_QUAD size,
		    double x_user, double y_user)
{
  int   result;
  int   tpic_command;
  char *end = buffer + size;
  char *token;

  result = 0;
  tpic_command = TPIC_UNKNOWN;
  skip_white(&buffer, end);
  token = parse_ident(&buffer, end);
  if (token) {
    int i;
    for (i = 0; tpic_specials[i].s != NULL; i++) {
      if (!strcmp(tpic_specials[i].s, token)) {
	tpic_command = tpic_specials[i].tpic_command;
	break;
      }
    }
    RELEASE(token);
  }
  if (!token || tpic_command == TPIC_UNKNOWN)
    return 0;

  result = 1;
  skip_white(&buffer, end);
  switch (tpic_command) {
  case TPIC_PN:
    set_pen_size(&buffer, end);
    break;
  case TPIC_PA:
    add_point(&buffer, end);
    break;
  case TPIC_FP:
    flush_path(x_user, y_user, 0, 0.0);
    break;
  case TPIC_IP: 
    flush_path(x_user, y_user, 1, 0.0);
    break;
  case TPIC_DA:
    token = parse_number(&buffer, end);
    if (token) {
      flush_path(x_user, y_user, 0, atof(token));
      RELEASE(token);
    }
    break;
  case TPIC_DT:
    token = parse_number(&buffer, end);
    if (token) {
      flush_path(x_user, y_user, 0, -atof(token));
      RELEASE(token);
    }
    break;
  case TPIC_SP:
    token = parse_number(&buffer, end);
    if (token) {
      spline_path(x_user, y_user, atof(token));
      RELEASE(token);
    } else {
      spline_path(x_user, y_user, 0.0);
    }
    break;
  case TPIC_AR:
    arc(&buffer, end, x_user, y_user, 0);
    break;
  case TPIC_IA:
    arc(&buffer, end, x_user, y_user, 1);
    break;
  case TPIC_SH:
    set_fill_color(&buffer, end);
    break;
  case TPIC_WH: 
    tp_gstate.fill_shape = 1;
    tp_gstate.fill_color = 1.0;
    break;
  case TPIC_BK:
    tp_gstate.fill_shape = 1;
    tp_gstate.fill_color = 0.0;
    break;
  case TPIC_TX: 
    tp_gstate.fill_shape = 1;
    {
      long num = 0, den = 0;
      while (buffer++ < end) {
	switch (*(buffer++)) {
	case '0':
	  num += 0;
	case '1': case '2': case '4': case '8':
	  num += 1;
	  break;
	case '3': case '5': case '6': case '9':
	case 'a': case 'A': case 'c': case 'C':
	  num += 2;
	  break;
	case '7': case 'b': case 'B': case 'd':
	case 'D':
	  num += 3;
	  break;
	case 'f': case 'F':
	  num += 4;
	  break;
	default:
	  break;
	}
	den += 16;
      }
      if (den != 0) {
	default_fill_color = 1.0 - (float) (num)/(den);
      }
      else {
	default_fill_color = 0.5;
      }
    }
    break;
  default:
    ERROR("TPIC: Fix me, I'm broke. This should never happen");
  }

  return result;
}
