/*  $Header: /cvsroot/dvipdfmx/src/mpost.c,v 1.21 2004/03/16 09:33:51 hirata Exp $
    
    This is dvipdfmx, an eXtended version of dvipdfm by Mark A. Wicks.

    Copyright (C) 2002 by Jin-Hwan Cho and Shunsaku Hirata,
    the dvipdfmx project team <dvipdfmx@project.ktug.or.kr>
    
    Copyright (C) 1998, 1999 by Mark A. Wicks <mwicks@kettering.edu>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
    
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
    
    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
*/

#if HAVE_CONFIG_H
#include "config.h"
#endif

#include <ctype.h>
#include <string.h>
#include <math.h>

#include "system.h"
#include "mem.h"
#include "error.h"
#include "mfileio.h"

#include "dvi.h"

#include "pdfobj.h"
#include "pdfspecial.h"
#include "pdfparse.h"
#include "pdflimits.h"
#include "pdfdev.h"
#include "pdfdoc.h"

#include "fontmap.h"
#include "subfont.h"

#include "pdfximage.h"

#include "mpost.h"

#define MP_OPTIMIZE_PATH 0

static struct mp_fonts 
{
  char * tex_name;
  int    font_id;
  int    subfont_id;
  double pt_size;
} *mp_fonts = NULL;

static int num_mp_fonts = 0, max_mp_fonts = 0;

static void
need_more_mp_fonts (unsigned n)
{
  if (num_mp_fonts + n > max_mp_fonts) {
    max_mp_fonts += MAX_FONTS;
    mp_fonts = RENEW(mp_fonts, max_mp_fonts, struct mp_fonts);
  }
}

static int
mp_locate_font (char *tfm_name, double pt_size)
{
  int i, map_id, subfont_id;
  char *tex_name;

  for (i = 0; i < num_mp_fonts; i++) {
    if (!strcmp(tfm_name, mp_fonts[i].tex_name) &&
	mp_fonts[i].pt_size == pt_size)
      return i;
  }

  need_more_mp_fonts(1);
  num_mp_fonts++;

  map_id = get_map_record(tfm_name, &subfont_id);

  mp_fonts[i].tex_name = NEW (strlen(tfm_name)+1, char);
  strcpy(mp_fonts[i].tex_name, tfm_name);
  mp_fonts[i].subfont_id = subfont_id;
  mp_fonts[i].pt_size = pt_size;

  /* The following line is a bit of a kludge.
   * MetaPost inclusion was an afterthought. */
  tex_name = (map_id < 0 ? tfm_name : fontmap_tex_name(map_id));
#if 0
  mp_fonts[i].font_id = dev_locate_font(tex_name, (spt_t)(pt_size/dvi_unit_size()), map_id);
#endif
  mp_fonts[i].font_id = dev_locate_font(tex_name, (spt_t)(pt_size*dev_unit_dviunit()), map_id);

  return i;
}

static void
release_fonts (void)
{
  int i;

  for (i = 0; i < num_mp_fonts; i++) {
    RELEASE (mp_fonts[i].tex_name);
  }
  if (mp_fonts) {
    RELEASE (mp_fonts);
    mp_fonts = NULL;
  }
  num_mp_fonts = 0;
  max_mp_fonts = 0;
}

static int
mp_is_font_name (const char *token)
{
  int i;

  for (i = 0; i < num_mp_fonts; i++) {
    if (!strcmp(token, mp_fonts[i].tex_name))
      break;
  }
  if (i < num_mp_fonts)
    return 1;
  else
    return 0;
}

#if 0
static int mp_fontid (char *tex_name, double pt_size)
{
  int i;
  for (i=0; i < num_mp_fonts; i++) {
    if (!strcmp (tex_name, mp_fonts[i].tex_name) &&
	(mp_fonts[i].pt_size == pt_size))
      break;
  }
  if (i < num_mp_fonts) {
    return i;
  }
  else {
    return -1;
  }
}
#endif

static int
scan_bbox (char **start, char *end, pdf_rect *bbox)
{
  char *llx = NULL, *lly = NULL, *urx = NULL, *ury = NULL;
  int error = 1;

  /* Scan for bounding box record */
  while (**start == '%') {
    if (!strncmp(*start, "%%BoundingBox:", 14)) {
      *start += 14;
      skip_white(start, end);
      /* Expect 4 numbers or fail */
      if ((llx = parse_number(start, end)) &&
	  (lly = parse_number(start, end)) &&
	  (urx = parse_number(start, end)) &&
	  (ury = parse_number(start, end))) {
        /* Set the crop box to the true bounding box specified in the file */
        bbox->llx = atof(llx);
        bbox->lly = atof(lly);
        bbox->urx = atof(urx);
        bbox->ury = atof(ury);
        error = 0;
      } else {
        WARN("Missing expected four numbers in BoundingBox.");
        dump(*start, end);
	error = 1;
      }
    }
    skip_line(start, end);
  }

  if (llx) RELEASE(llx);
  if (lly) RELEASE(lly);
  if (urx) RELEASE(urx);
  if (ury) RELEASE(ury);

  if (error) /* Didn't find Bounding box */
    WARN("Failed to find an expected BoundingBox record.");

  return error;
}

static void
skip_prolog (char **start, char *end)
{
  int   found_prolog = 0;
  char *save;

  save = *start;
  while (*start < end) {
    if (!strncmp(*start, "%%EndProlog", 11)) {
      found_prolog = 1;
      skip_line(start, end);
      break;
    } else if (!strncmp(*start, "%%Page:", 7)) {
      break;
    }
    skip_line(start, end);
  }
  if (!found_prolog)
    *start = save;
}

#define FLAG_PATH_CLIP  (1 << 0)
#define FLAG_PATH_CLOSE (1 << 1)

typedef struct pdf_point
{
  double x, y;
} pdf_point;

struct path_element
{
  char type;
  pdf_point p[3];
};

static struct mp_path
{
  int num_paths;
  int max_paths;

  int flags;
  struct path_element *path; /* Sub path */
} currentpath;

static pdf_tmatrix currentmatrix = {
  1.0, 0.0, 0.0, 1.0, 0.0, 0.0
};

static struct pdf_point currentpoint = {0, 0};

#if 0
static struct {
  struct mp_path   path;
  struct pdf_point point;
  pdf_tmatrix      matrix;
  
}
#endif

#define MORE_PATH_POINTS 64

static void
mp_set_clip (void)
{
  currentpath.flags |= FLAG_PATH_CLIP;
}

static void
mp_closepath (void)
{
  if (currentpath.num_paths > 0) {
    struct path_element *pe;

    pe = &(currentpath.path[0]);
    currentpoint.x = pe->p[0].x;
    currentpoint.y = pe->p[0].y;
    currentpath.flags |= FLAG_PATH_CLOSE;
  }
}

#define LASTPATH() (&(currentpath.path[currentpath.num_paths-1]))

#define MP_PATH_MOVETO   1
#define MP_PATH_RMOVETO  2
#define MP_PATH_LINETO   3
#define MP_PATH_RLINETO  4
#define MP_PATH_CURVETO  5
#define MP_PATH_RCURVETO 6

static void
mp_append_path (int type, double *points)
{
  int last_point = 0;
  struct path_element *pe;

  if (currentpath.num_paths >= currentpath.max_paths) {
    currentpath.max_paths += MORE_PATH_POINTS;
    currentpath.path = RENEW(currentpath.path,
			     currentpath.max_paths, struct path_element);
  }

  pe = &(currentpath.path[currentpath.num_paths]);
  switch (type) {
  case MP_PATH_MOVETO:
    pe->type = 'm';
    pe->p[0].x = points[0];
    pe->p[0].y = points[1];
    last_point = 0;
    break;
  case MP_PATH_RMOVETO:
    pe->type = 'm';
    pe->p[0].x = points[0] + currentpoint.x;
    pe->p[0].y = points[1] + currentpoint.y;
    last_point = 0;
    break;
  case MP_PATH_LINETO:
    pe->type = 'l';
    pe->p[0].x = points[0];
    pe->p[0].y = points[1];
    last_point = 0;
    break;
  case MP_PATH_RLINETO:
    pe->type = 'l';
    pe->p[0].x = points[0] + currentpoint.x;
    pe->p[0].y = points[1] + currentpoint.y;
    last_point = 0;
    break;
  case MP_PATH_CURVETO:
#ifdef    MP_OPTIMIZE_PATH
    if (points[0] == currentpoint.x &&
	points[1] == currentpoint.y) {
      pe->type = 'v';
      pe->p[0].x = points[2];
      pe->p[0].y = points[3];
      pe->p[1].x = points[4];
      pe->p[1].y = points[5];
      last_point = 1;
    } else if (points[2] == points[4] &&
	       points[3] == points[5]) {
      pe->type = 'y';
      pe->p[0].x = points[0];
      pe->p[0].y = points[1];
      pe->p[1].x = points[2];
      pe->p[1].y = points[3];
      last_point = 1;
    } else {
#endif /* MP_OPTIMIZE_PATH */
      pe->type = 'c';
      pe->p[0].x = points[0];
      pe->p[0].y = points[1];
      pe->p[1].x = points[2];
      pe->p[1].y = points[3];
      pe->p[2].x = points[4];
      pe->p[2].y = points[5];
      last_point = 2;
#ifdef    MP_OPTIMIZE_PATH
    }
#endif /* MP_OPTIMIZE_PATH */
    break;
  case MP_PATH_RCURVETO:
#ifdef    MP_OPTIMIZE_PATH
    if (points[0] == 0.0 && points[1] == 0.0) {
      pe->type = 'v';
      pe->p[0].x = points[2] + currentpoint.x;
      pe->p[0].y = points[3] + currentpoint.y;
      pe->p[1].x = points[4] + currentpoint.x;
      pe->p[1].y = points[5] + currentpoint.y;
      last_point = 1;
    } else if (points[2] == points[4] &&
	       points[3] == points[5]) {
      pe->type = 'y';
      pe->p[0].x = points[0] + currentpoint.x;
      pe->p[0].y = points[1] + currentpoint.y;
      pe->p[1].x = points[2] + currentpoint.x;
      pe->p[1].y = points[3] + currentpoint.y;
      last_point = 1;
    } else {
#endif /* MP_OPTIMIZE_PATH */
      pe->type = 'c';
      pe->p[0].x = points[0] + currentpoint.x;
      pe->p[0].y = points[1] + currentpoint.y;
      pe->p[1].x = points[2] + currentpoint.x;
      pe->p[1].y = points[3] + currentpoint.y;
      pe->p[2].x = points[4] + currentpoint.x;
      pe->p[2].y = points[5] + currentpoint.y;
      last_point = 2;
#ifdef    MP_OPTIMIZE_PATH
    }
#endif /* MP_OPTIMIZE_PATH */
    break;
  default:
    ERROR("Unknown path type: %d", type);
  }
  currentpoint.x = pe->p[last_point].x;
  currentpoint.y = pe->p[last_point].y;
  currentpath.num_paths++;
}

static void
clear_currentpath (void)
{
  currentpath.flags = 0;
  currentpath.num_paths = 0;
}


static void
release_paths (void)
{
  if (currentpath.path)
    RELEASE(currentpath.path);
  currentpath.path = NULL;
  currentpath.flags = 0;
  currentpath.num_paths = 0;
  currentpath.max_paths = 0;
}

static void
mp_flush_path (void)
{
  int i, len = 0;

  for (i = 0; i < currentpath.num_paths; i++) {
    struct path_element *pe;

    pe = &(currentpath.path[i]);
    switch (pe->type) {
    case 'm':
      /* Moveto must start a new path */
      /* clear_currentpath(); */
      len = sprintf(work_buffer, " %g %g m",
		    ROUND(pe->p[0].x, 0.01), ROUND(pe->p[0].y, 0.01));
      break;
    case 'l':
      len = sprintf(work_buffer, " %g %g l",
		    ROUND(pe->p[0].x, 0.01), ROUND(pe->p[0].y, 0.01));
      break;
    case 'c':
      len = sprintf(work_buffer, " %g %g %g %g %g %g c",
		    ROUND(pe->p[0].x, 0.01), ROUND(pe->p[0].y, 0.01),
		    ROUND(pe->p[1].x, 0.01), ROUND(pe->p[1].y, 0.01),
		    ROUND(pe->p[2].x, 0.01), ROUND(pe->p[2].y, 0.01));
      break;
    case 'v':
      /* First control point coincide with path's start point. */
      len = sprintf(work_buffer, " %g %g %g %g v",
		    ROUND(pe->p[0].x, 0.01), ROUND(pe->p[0].y, 0.01),
		    ROUND(pe->p[1].x, 0.01), ROUND(pe->p[1].y, 0.01));
      break;
    case 'y':
      /* Second control point coincide with path's end point. */
      len = sprintf(work_buffer, " %g %g %g %g y",
		    ROUND(pe->p[0].x, 0.01), ROUND(pe->p[0].y, 0.01),
		    ROUND(pe->p[1].x, 0.01), ROUND(pe->p[1].y, 0.01));
      break;
    default:
      ERROR("Unknown path type: %d", pe->type);
    }
    pdf_doc_add_to_page(work_buffer, len);
  }
  if (currentpath.flags & FLAG_PATH_CLOSE)
    pdf_doc_add_to_page(" h", 2);
  if (currentpath.flags & FLAG_PATH_CLIP)
    pdf_doc_add_to_page(" W", 2);

  currentpath.num_paths = 0;
  currentpath.flags = 0;
}

#define __copymatrix(m,n) do {\
  (m)->a = (n)->a; (m)->b = (n)->b;\
  (m)->c = (n)->c; (m)->d = (n)->d;\
  (m)->e = (n)->e; (m)->f = (n)->f;\
} while (0)

/* Modify M itself */
static void
__invertmatrix (pdf_tmatrix *M)
{
  pdf_tmatrix M_i;  
  double det;

  det = (M->a) * (M->d) - (M->b) * (M->c);
  if (det == 0.0)
    ERROR("Determinant exactly zero in transform_path()");

  M_i.a =  (M->d) / det; M_i.b = -(M->b) / det;
  M_i.c = -(M->c) / det; M_i.d =  (M->a) / det;
  M_i.e = ((M->c) * (M->f) - (M->e) * (M->d)) / det;
  M_i.f = ((M->b) * (M->e) - (M->a) * (M->f)) / det;

  __copymatrix(M, &M_i);
}

static void
__dtransform (pdf_point *p, const pdf_tmatrix *M)
{
  double x, y;

  x = p->x; y = p->y;
  if (M) {
    p->x = x * M->a + y * M->c;
    p->y = x * M->b + y * M->d;
  } else {
    p->x = x * currentmatrix.a + y * currentmatrix.c;
    p->y = x * currentmatrix.b + y * currentmatrix.d;
  }
}

static void
__idtransform (pdf_point *p, const pdf_tmatrix *M)
{
  pdf_tmatrix M_i;

  if (M) {
    __copymatrix(&M_i, M);
  } else {
    __copymatrix(&M_i, &currentmatrix);
  }
  __invertmatrix(&M_i);
  __dtransform(p, &M_i);
}

static void
__transform (pdf_point *p, const pdf_tmatrix *M)
{
  double x, y;

  x = p->x; y = p->y;
  if (M) {
    p->x = x * M->a + y * M->c + M->e;
    p->y = x * M->b + y * M->d + M->f;
  } else {
    p->x = x * currentmatrix.a + y * currentmatrix.c + currentmatrix.e;
    p->y = x * currentmatrix.b + y * currentmatrix.d + currentmatrix.f;
  }
}

static void
__itransform (pdf_point *p, const pdf_tmatrix *M)
{
  pdf_tmatrix M_i;

  if (M) {
    __copymatrix(&M_i, M);
  } else {
    __copymatrix(&M_i, &currentmatrix);
  }
  __invertmatrix(&M_i);
  __transform(p, &M_i);
}


static void
__concat (pdf_tmatrix *M)
{
  double a, b, c, d;

  a = currentmatrix.a; b = currentmatrix.b;
  c = currentmatrix.c; d = currentmatrix.d;

  currentmatrix.a  = (M->a) * a + (M->b) * c;
  currentmatrix.b  = (M->a) * b + (M->b) * d;
  currentmatrix.c  = (M->c) * a + (M->d) * c;
  currentmatrix.d  = (M->c) * b + (M->d) * d;

  currentmatrix.e += (M->e) * a + (M->f) * c;
  currentmatrix.f += (M->e) * b + (M->f) * d;
}

static void
itransform_currentpath (pdf_tmatrix *M)
{
  pdf_tmatrix M_i;
  struct path_element *pe;
  int n = 0, i;

  __copymatrix(&M_i, M);
  __invertmatrix(&M_i);
  for (i = 0; i < currentpath.num_paths; i++) {
    pe = &(currentpath.path[i]);
    switch (pe->type) {
    case 'm': n = 1; break;
    case 'l': n = 1; break;
    case 'c': n = 3; break;
    case 'v': n = 2; break;
    case 'y': n = 2; break;
    default:
      ERROR("Unknown path type: %d", pe->type);
    }
    while (n-- > 0) {
      __transform(&(pe->p[n]), &M_i);
    }
  }

  __transform(&currentpoint, &M_i);
}

/* PostScript Operators */

#define ADD          	1
#define SUB		2
#define MUL		3
#define DIV		4
#define NEG    	        5
#define TRUNCATE	6

#define CLEAR		10
#define EXCH		11
#define POP		12

#define CLIP         	30
#define NEWPATH		31
#define CLOSEPATH    	32
#define MOVETO		33
#define RMOVETO         34
#define CURVETO   	35
#define RCURVETO        36
#define LINETO		37
#define RLINETO		38

#define FILL		40
#define STROKE		41
#define SHOW		42
#define SHOWPAGE	43

#define GSAVE		50
#define GRESTORE	51

#define CONCAT       	52
#define SCALE		53
#define TRANSLATE	54
#define ROTATE          55

#define SETLINEWIDTH	60
#define SETDASH		61
#define SETLINECAP 	62
#define SETLINEJOIN	63
#define SETMITERLIMIT	64

#define SETGRAY		70
#define SETRGBCOLOR	71
#define SETCMYKCOLOR	72

#define CURRENTPOINT    80
#define IDTRANSFORM	81
#define DTRANSFORM	82

#define FONTNAME	1000
#define FSHOW		1001
#define STEXFIG         1002
#define ETEXFIG         1003

static struct operators 
{
  const char *token;
  int         opcode;
} ps_operators[] = {
  {"add",          ADD},
  {"mul",          MUL},
  {"div",          DIV},
  {"neg",          NEG},
  {"sub",          SUB},  
  {"truncate",     TRUNCATE},

  {"clear",        CLEAR},
  {"exch",         EXCH},
  {"pop",          POP},

  {"clip",         CLIP},
  {"closepath",    CLOSEPATH},
  {"concat",       CONCAT},

  {"newpath",      NEWPATH},
  {"moveto",       MOVETO},
  {"rmoveto",      RMOVETO},
  {"lineto",       LINETO},
  {"rlineto",      RLINETO},
  {"curveto",      CURVETO},
  {"rcurveto",     RCURVETO},

  {"stroke",       STROKE},  
  {"fill",         FILL},
  {"show",         SHOW},
  {"showpage",     SHOWPAGE},

  {"gsave",        GSAVE},
  {"grestore",     GRESTORE},
  {"translate",    TRANSLATE},
  {"rotate",       ROTATE},
  {"scale",        SCALE},

  {"setlinecap",    SETLINECAP},
  {"setlinejoin",   SETLINEJOIN},
  {"setlinewidth",  SETLINEWIDTH},
  {"setmiterlimit", SETMITERLIMIT},
  {"setdash",       SETDASH},

  {"setgray",      SETGRAY},
  {"setrgbcolor",  SETRGBCOLOR},
  {"setcmykcolor", SETCMYKCOLOR},

  {"currentpoint", CURRENTPOINT}, /* This is here for rotate support
				     in graphics package-not MP support */
  {"dtransform",   DTRANSFORM},
  {"idtransform",  IDTRANSFORM}
};

static struct operators mps_operators[] = {
  {"fshow",       FSHOW},
  {"startTexFig", STEXFIG},
  {"endTexFig",   ETEXFIG}
};

#define NUM_PS_OPERATORS  (sizeof(ps_operators)/sizeof(ps_operators[0]))
#define NUM_MPS_OPERATORS (sizeof(mps_operators)/sizeof(mps_operators[0]))
static int
get_opcode (const char *token)
{
  int i;

  for (i = 0; i < NUM_PS_OPERATORS; i++) {
    if (!strcmp(token, ps_operators[i].token)) {
      return ps_operators[i].opcode;
    }
  }

  for (i = 0; i < NUM_MPS_OPERATORS; i++) {
    if (!strcmp(token, mps_operators[i].token)) {
      return mps_operators[i].opcode;
    }
  }

  if (mp_is_font_name(token))
    return FONTNAME;

  return -1;
}

#define PS_STACK_SIZE 1024

static pdf_obj *stack[PS_STACK_SIZE];
static unsigned top_stack = 0;

static int state     = 0;
static int num_saves = 0;

#define PUSH(o) { \
  if (top_stack < PS_STACK_SIZE) { \
    stack[top_stack++] = (o); \
  } else { \
    WARN("PS stack overflow including MetaPost file or inline PS code"); \
    error=1; \
    break; \
  } \
}

#define POP_STACK() ((top_stack > 0) ? stack[--top_stack] : NULL)

static int
do_exch (void)
{
  pdf_obj *tmp;

  if (top_stack < 2)
    return -1;

  tmp = stack[top_stack-1];
  stack[top_stack-1] = stack[top_stack-2];
  stack[top_stack-2] = tmp;

  return 0;
}

static int
do_clear (void)
{
  pdf_obj *tmp;

  while (top_stack > 0) {
    tmp = POP_STACK();
    if (tmp)
      pdf_release_obj(tmp);
  }

  return 0;
}

static int
pop_get_numbers (double *values, int count)
{
  pdf_obj *tmp;

  while (count-- > 0) {
    tmp = POP_STACK();
    if (!tmp) {
      WARN("mpost: Stack underflow.");
      break;
    } else if (!PDF_OBJ_NUMBERTYPE(tmp)) {
      WARN("mpost: Not a number!");
      pdf_release_obj(tmp);
      break;
    }
    values[count] = pdf_number_value(tmp);
    pdf_release_obj(tmp);
  }

  return (count + 1);
}

static int
cvr_array (pdf_obj *array, double *values, int count)
{
  if (!PDF_OBJ_ARRAYTYPE(array)) {
    WARN("mpost: Not an array!");
  } else {
    pdf_obj *tmp;

    while (count-- > 0) {
      tmp = pdf_get_array(array, count);
      if (!PDF_OBJ_NUMBERTYPE(tmp)) {
	WARN("mpost: Not a number!");
	break;
      }
      values[count] = pdf_number_value(tmp);
    }
  }
  if (array)
    pdf_release_obj(array);

  return (count + 1);
}

static int
do_mpost_fshow (int opcode)
{
  int      font_id;
  double   font_scale;
  pdf_obj *font_name, *text_str;
  int      length;
  unsigned char *strptr;

  if (pop_get_numbers(&font_scale, 1) != 0)
    return -1;

  font_name = POP_STACK();
  if (!PDF_OBJ_STRINGTYPE(font_name)) {
    if (font_name)
      pdf_release_obj(font_name);
    return -1;
  }

  text_str = POP_STACK();
  if (!PDF_OBJ_STRINGTYPE(text_str)) {
    pdf_release_obj(font_name);
    if (text_str)
      pdf_release_obj(text_str);
    return -1;
  }
  strptr = pdf_string_value (text_str);
  length = pdf_string_length(text_str);

  font_id = mp_locate_font(pdf_string_value(font_name), font_scale);
  pdf_release_obj(font_name);
  if (mp_fonts[font_id].subfont_id >= 0) {
    unsigned short uch;
    unsigned char  wch[2];

    while (length-- > 0) {
      uch = lookup_sfd_record(mp_fonts[font_id].subfont_id, *strptr);
      wch[0] = uch >> 8;
      wch[1] = uch & 0xff;
      dev_set_string((spt_t)(currentpoint.x*dev_unit_dviunit()),
		     (spt_t)(currentpoint.y*dev_unit_dviunit()),
		     wch, 2, 0,
		     mp_fonts[font_id].font_id, 0);
#if 0
      dev_set_string((spt_t)(currentpoint.x/dvi_unit_size()),
		     (spt_t)(currentpoint.y/dvi_unit_size()),
		     wch, 2, 0,
		     mp_fonts[font_id].font_id, 0);
#endif
      strptr++;
    }
  } else {
#if 0
    dev_set_string((spt_t)(currentpoint.x/dvi_unit_size()),
		   (spt_t)(currentpoint.y/dvi_unit_size()),
		   strptr, length, 0,
		   mp_fonts[font_id].font_id, 0);
#endif
    dev_set_string((spt_t)(currentpoint.x*dev_unit_dviunit()),
		   (spt_t)(currentpoint.y*dev_unit_dviunit()),
		   strptr, length, 0,
		   mp_fonts[font_id].font_id, 0);
  }
  graphics_mode();

  pdf_release_obj(text_str);

  return 0;
}

static int
do_texfig_operator (int opcode, double x_user, double y_user)
{
  static transform_info fig_p;
  static int in_tfig = 0;
  double values[6];
  int error = 0;

  switch (opcode) {
  case STEXFIG:
    error = pop_get_numbers(values, 6);
    if (!error) {
      double   dvi2pts;
      pdf_rect cropbox;

      transform_info_clear(&fig_p);
      dvi2pts = 1.0/dev_unit_dviunit();
#if 0
      dvi2pts = dvi_unit_size();
#endif

      fig_p.width  =  values[0] * dvi2pts;
      fig_p.height =  values[1] * dvi2pts;
      fig_p.u_llx  =  values[2] * dvi2pts;
      fig_p.u_lly  = -values[3] * dvi2pts;
      fig_p.u_urx  =  values[4] * dvi2pts;
      fig_p.u_ury  = -values[5] * dvi2pts;
      fig_p.user_bbox = 1;

      cropbox.llx = fig_p.u_llx;
      cropbox.lly = fig_p.u_lly;
      cropbox.urx = fig_p.u_urx;
      cropbox.ury = fig_p.u_ury;

      pdf_doc_start_grabbing(NULL, fig_p.u_llx, fig_p.u_ury, &cropbox);
      
      in_tfig = 1;
    }
    break;
  case ETEXFIG:
    {
      int      xobj_id;
      pdf_obj *xform;
      xform_info  info;

      if (!in_tfig)
	ERROR("endTexFig without valid startTexFig!.");

      pdf_ximage_init_form_info(&info);
      pdf_doc_current_form_info(NULL, &(info.bbox));

      xform   = pdf_doc_end_grabbing();

      xobj_id = pdf_ximage_defineresource("__TexFig",
					  PDF_XOBJECT_TYPE_FORM, &info, xform);
      pdf_ximage_put_image(xobj_id, &fig_p, x_user, y_user);

      in_tfig = 0;
    }
    break;
  default:
    error = 1;
  }

  return error;
}


/*
 * buggy...
 */

/*
 * Again, the only piece that needs x_user and y_user is
 * that piece dealing with texfig.
 */
static int
do_operator (const char *token, double x_user, double y_user)
{
  int      opcode;
  double   values[12];
  pdf_obj *tmp = NULL;
  pdf_tmatrix matrix;
  int len, error = 0;

#define print_transform(M) if (M) {\
  len = 0;\
  work_buffer[len++] = ' ';\
  len += pdf_sprint_matrix(work_buffer+len,(M));\
  work_buffer[len++] = ' ';\
  work_buffer[len++] = 'c';\
  work_buffer[len++] = 'm';\
  pdf_doc_add_to_page(work_buffer, len);\
  len = 0;\
}


  /* PS to PDF conversion is not so simple.  We maintain
     state so we can change "gsave fill grestore stroke" to "B".
     We need to keep track of what we have seen.   This code is not
     too smart and could be easily fooled by real postscript. 
     It makes some assumptions since it is only looking at MetaPost

     States are as follows:
     0: Nothing special
     1: Started a path
     2: Saw gsave in path
     3: Saw a painting operator in state 2(fill, stroke, or newpath)
     4: Saw a grestore in state 3
  */
  opcode = get_opcode(token);

  switch (opcode) {
    /* Arithmetic operators */
  case ADD:
    error = pop_get_numbers(values, 2);
    if (!error)
      PUSH(pdf_new_number(values[0] + values[1]));
    break;
  case MUL:
    error = pop_get_numbers(values, 2);
    if (!error)
      PUSH(pdf_new_number(values[0]*values[1]));
    break;
  case NEG:
    error = pop_get_numbers(values, 1);
    if (!error)
      PUSH(pdf_new_number(-values[0]));
    break;
  case SUB:
    error = pop_get_numbers(values, 2);
    if (!error)
      PUSH(pdf_new_number(values[0] - values[1]));
    break;
  case DIV:
    error = pop_get_numbers(values, 2);
    if (!error)
      PUSH(pdf_new_number(values[0]/values[1]));
    break;
  case TRUNCATE: /* Round toward zero. */
    error = pop_get_numbers(values, 1);
    if (!error)
      PUSH(pdf_new_number(((values[0] > 0) ? floor(values[0]) : ceil(values[0]))));
    break;

    /* Stack operation */
  case CLEAR:
    error = do_clear();
    break;
  case POP:
    tmp = POP_STACK();
    if (tmp)
      pdf_release_obj(tmp);
    break;
  case EXCH:
    error = do_exch();
    break;

    /* Path construction */
  case MOVETO:
    if (state > 1) { /* THIS IS NOT AN ERROR !!! */
      WARN("Unexpected path segment (moveto): state=%d", state);
      error = 1;
    } else {
      error = pop_get_numbers(values, 2);
      if (!error)
	mp_append_path(MP_PATH_MOVETO, values);
      state = 1;
    }
    break;
  case RMOVETO:
    if (state > 1) { /* THIS IS NOT AN ERROR !!! */
      WARN("Unexpected path segment (moveto): state=%d", state);
      error = 1;
    } else {
      error = pop_get_numbers(values, 2);
      if (!error)
	mp_append_path(MP_PATH_RMOVETO, values);
      state = 1;
    }
    break;
  case LINETO: 
    if (state > 1) { /* THIS IS NOT AN ERROR !!! */
      WARN("Unexpected path segment (lineto): state=%d", state);
      error = 1;
    } else {
      error = pop_get_numbers(values, 2);
      if (!error)
	mp_append_path(MP_PATH_LINETO, values);
      state = 1;
    }
    break;
  case RLINETO: 
    if (state > 1) { /* THIS IS NOT AN ERROR !!! */
      WARN("No current point in rlineto?");
      error = 1;
    } else {
      error = pop_get_numbers(values, 2);
      if (!error)
	mp_append_path(MP_PATH_RLINETO, values);
      state = 1;
    }
    break;
  case CURVETO:
    if (state > 1) { /* THIS IS NOT AN ERROR !!! */
      WARN("Unexpected path segment (curveto): state=%d", state);
      error = 1;
    } else {
      error = pop_get_numbers(values, 6);
      if (!error)
	mp_append_path(MP_PATH_CURVETO, values);
      state = 1;
    }
    break;
  case RCURVETO:
    if (state > 1) { /* THIS IS NOT AN ERROR !!! */
      WARN("Unexpected path segment (curveto): state=%d", state);
      error = 1;
    } else {
      error = pop_get_numbers(values, 6);
      if (!error)
	mp_append_path(MP_PATH_RCURVETO, values);
      state = 1;
    }
    break;
  case NEWPATH:
    mp_flush_path();
    pdf_doc_add_to_page(" n", 2);
    break;
  case CLOSEPATH:
    mp_closepath();
    break;

    /* Paint operators:
     * Wrong implementation.
     * gsave should save currentpath.
     */
  case STROKE:
    switch (state) {
    case 0:
      state = 0;
      break;
    case 1:
      mp_flush_path();
      pdf_doc_add_to_page(" S", 2);
      state = 0;
      break;
    case 2:
      state = 3;
      break;
    case 3:
      WARN("Unexpected fill.");
      break;
    case 4: /* gsave fill grestore stroke? */
      mp_flush_path();
      pdf_doc_add_to_page(" B", 2);
      state = 0;
      break;
    }
    break;
  case FILL:
    switch (state) {
    case 0:
      state = 0;
      break;
    case 1:
      mp_flush_path();
      pdf_doc_add_to_page (" f", 2);
      state = 0;
      break;
    case 2:
      state = 3;
      break;
    case 3:
      WARN("Unexpected fill.");
      break;
    case 4: /* gsave stroke grestore fill ? */
      mp_flush_path();
      pdf_doc_add_to_page (" B", 2);
      state = 0;
      break;
    }
    break;
  case CLIP:
    mp_set_clip();
    break;

    /*
     * Graphics state operators:
     * Not implemented properly.
     */
  case GSAVE:
    switch (state) {
    case 0:
      pdf_doc_add_to_page (" q", 2);
      num_saves += 1;
      break;
    case 1:
      state = 2;
      break;
    case 4:
      state = 2;
      break;
    default:
      WARN("Unexpected gsave.");
      break;
    }
    break;
  case GRESTORE:
    switch (state) {
    case 0:
      if (num_saves > 0) {
	num_saves -= 1;
	pdf_doc_add_to_page (" Q", 2);
	/* Unfortunately, the following two lines are necessary in case of a font or color
	   change inside of the save/restore pair.  Anything that was done
	   there must be redone, so in effect, we make no assumptions about
	   what fonts are active.  We act like we are starting a new page */
	dev_reselect_font();
	/* The following line was causing trouble - 1/30/01 */
	/*	dev_do_color(); */
      }
      else {
	WARN("PS special: \"grestore\" ignored. More restores than saves on a page.");
      }
      break;
    case 2:
      state = 1;
      break;
    case 3:
      state = 4;
      break;
    default:
      WARN("Unexpected grestore.");
      break;
    }
    break;
    /*
     * Path is not flushed immediately. We print transformation matrix
     * immediately for nasty raw PS codes. We must (inverse) transform
     * pending path to compensate additional transformations.
     *
     * This might output wasteful PDF.
     */
  case CONCAT:
    tmp = POP_STACK();
    error = cvr_array(tmp, values, 6); /* This does pdf_release_obj() */
    if (error)
      WARN("Missing array before \"concat\".");
    else {
      matrix.a = values[0]; matrix.b = values[1];
      matrix.c = values[2]; matrix.d = values[3];
      matrix.e = values[4]; matrix.f = values[5];

      __concat(&matrix);
      print_transform(&matrix);        /* This macro use work_buffer and len. */
      itransform_currentpath(&matrix); /* Transform pending path, if any */
    }
    tmp = NULL;
    break;
  case SCALE:
    error = pop_get_numbers(values, 2);
    if (!error) {
      matrix.a = values[0]; matrix.b = 0.0;
      matrix.c = 0.0;       matrix.d = values[1];
      matrix.e = 0.0;       matrix.f = 0.0;

      __concat(&matrix);
      print_transform(&matrix);        /* This macro use work_buffer and len. */
      itransform_currentpath(&matrix); /* Transform pending path, if any */
    }
    break;
  case ROTATE:
    error = pop_get_numbers(values, 1);
    if (!error) {
      values[0] = values[0] * M_PI / 180;

      matrix.a =  cos(values[0]); matrix.b = -sin(values[0]);
      matrix.c =  sin(values[0]); matrix.d =  cos(values[0]);
      matrix.e =  0.0; matrix.f = 0.0;

      __concat(&matrix);
      print_transform(&matrix);        /* This macro use work_buffer and len. */
      itransform_currentpath(&matrix); /* I think we should have this. */
    }
    break;
  case TRANSLATE:
    error = pop_get_numbers(values, 2);
    if (!error) {
      matrix.a = 1.0; matrix.b = 0.0;
      matrix.c = 0.0; matrix.d = 1.0;
      matrix.e = values[0];
      matrix.f = values[1];

      __concat(&matrix);
      print_transform(&matrix);        /* This macro use work_buffer and len. */
      itransform_currentpath(&matrix); /* Transform pending path, if any */
    }
    break;

  case SETDASH:
    {
      double   offset;
      pdf_obj *dash;

      error = pop_get_numbers(&offset, 1);
      if (!error && (dash = POP_STACK()) != NULL) {
	pdf_doc_add_to_page(" [", 2);
	if (!PDF_OBJ_ARRAYTYPE(dash))
	  error = 1;
	else {
	  int i, size;

	  size = pdf_array_length(dash);
	  for (i = 0; i < size && !error; i++) {
	    /* Should be all non-negative. At least one non-zero. */
	    tmp = pdf_get_array(dash, i);
	    if (!PDF_OBJ_NUMBERTYPE(tmp))
	      error = 1;
	    else {
	      len = sprintf(work_buffer, " %.3f", pdf_number_value(tmp));
	      pdf_doc_add_to_page(work_buffer, len);
	    }
	  }
	}
	pdf_doc_add_to_page (" ]", 2);

	len = sprintf(work_buffer, " %g d", ROUND(offset, 0.01)); /* integer ? */
	pdf_doc_add_to_page(work_buffer, len);
	
	pdf_release_obj(dash);
      }
    }
    break;
  case SETLINECAP:
    error = pop_get_numbers(values, 1);
    if (!error) {
      len = sprintf(work_buffer, " %d J", (int) values[0]);
      pdf_doc_add_to_page (work_buffer, len);
    }
    break;
  case SETLINEJOIN:
    error = pop_get_numbers(values, 1);
    if (!error) {
      len = sprintf(work_buffer, " %d j", (int) values[0]);
      pdf_doc_add_to_page(work_buffer, len);
    }
    break;
  case SETLINEWIDTH:
    error = pop_get_numbers(values, 1);
    if (!error) {
      len = sprintf(work_buffer, " %.3f w", values[0]);
      pdf_doc_add_to_page(work_buffer, len);
    }
    break;
  case SETMITERLIMIT:
    error = pop_get_numbers(values, 1);
    if (!error) {
      len = sprintf(work_buffer, " %.3f M", values[0]);
      pdf_doc_add_to_page(work_buffer, len);
    }
    break;

  case SETCMYKCOLOR:
    error = pop_get_numbers(values, 4);
    if (!error) {
      len = sprintf(work_buffer,
		    " %.2f %.2f %.2f %.2f k %.2f %.2f %.2f %.2f K",
		    values[0], values[1], values[2], values[3],
		    values[0], values[1], values[2], values[3]);
      pdf_doc_add_to_page(work_buffer, len);
    }
    break;
  case SETGRAY:
    error = pop_get_numbers(values, 1);
    if (!error) {
      len = sprintf(work_buffer,
		    " %.2f g %.2f G", values[0], values[0]);
      pdf_doc_add_to_page(work_buffer, len);
    }
    break;
  case SETRGBCOLOR:
    error = pop_get_numbers(values, 3);
    if (!error) {
      len = sprintf(work_buffer,
		    " %.2f %.2f %.2f rg %.2f %.2f %.2f RG",
		    values[0], values[1], values[2],
		    values[0], values[1], values[2]);
      pdf_doc_add_to_page(work_buffer, len);
    }
    break;

  case SHOWPAGE: /* Let's ignore this for now */
    clear_currentpath();
    break;

    /* Wrong implementation :-( */
  case CURRENTPOINT:
    /* THIS IS NOT REAL CURRENTPOINT OPERATOR!!! */
    state = 0;
    PUSH(pdf_new_number(x_user)); /* Remember that x_user and y_user */
    PUSH(pdf_new_number(y_user)); /* are off by 0.02 % */
    break;
  case DTRANSFORM:
    /* ???? */
    error = pop_get_numbers(values, 2);
    if (!error) {
      PUSH(pdf_new_number(values[0]*100.0));
      PUSH(pdf_new_number(values[1]*100.0));
    }
    break;

  case IDTRANSFORM:
    /* ???? */
    error = pop_get_numbers(values, 2);
    if (!error) {
      PUSH(pdf_new_number(values[0]/100.0));
      PUSH(pdf_new_number(values[1]/100.0));
    }
    break;

    /* Extensions */
  case FSHOW: 
    do_mpost_fshow(opcode);
    /* Treat fshow as a path terminator of sorts */
    state = 0;
    break;
  case STEXFIG:
  case ETEXFIG:
    error = do_texfig_operator(opcode, x_user, y_user);
    break;
  case FONTNAME:
    /* ... */
  default:
    PUSH(pdf_new_string(token, strlen(token)));
    break;
  }

  return error;
}

/*
 * the only sections that need to know x_user and y _user are those
 * dealing with texfig.
 */
static int
mp_parse_body (char **start, char *end, double x_user, double y_user)
{
  char    *token;
  pdf_obj *obj;
  int      error = 0;

  skip_white(start, end);
  while (*start < end && !error) {
    if (isdigit(**start) ||
	(*start < end - 1 &&
	 (**start == '+' || **start == '-' || **start == '.' ))) {
      double value;
      char  *next;

      value = strtod(*start, &next);
      if (next < end && !strchr("<([{/%", *next) && !isspace(*next)) {
	WARN("Unkown PostScript operator.");
	dump(*start, next);
	error = 1;
      } else {
	PUSH(pdf_new_number(value));
	*start = next;
      }
      /*
       * PDF parser can't handle PS operator inside arrays.
       * This shouldn't use parse_pdf_array().
       */
    } else if (**start == '[' &&
	       (obj = parse_pdf_array (start, end))) {
      PUSH(obj);
      /* This cannot handle ASCII85 string. */
    } else if (*start < end - 1 &&
	       (**start == '<' && *(*start+1) == '<') &&
	       (obj = parse_pdf_dict(start, end))) {
      PUSH(obj);
    } else if ((**start == '(' || **start == '<') &&
	       (obj = parse_pdf_string (start, end))) {
      PUSH(obj);
    } else if (**start == '/') {
      WARN("Unable to handle names in raw PS code.");
      dump(*start, end);
      error = 1;
    } else {
      token = parse_ident(start, end);
      if (!token)
	error = 1;
      else {
	error = do_operator(token, x_user, y_user);
	RELEASE(token);
      }
    }
    skip_white(start, end);
  }
  if (error || *start < end) {
    WARN("Remainder of line unparsed.");
    dump(*start, end);
  }

  return error;
}

int
do_raw_ps_special (char **start, char* end,
		   int cleanup, double x_user, double y_user)
{
  int error;

  state = 0;
  error = mp_parse_body(start, end, x_user, y_user);

  if (cleanup)
    mp_cleanup(1);

  return !error;
}

void
mp_cleanup (int sloppy_ok)
{
  release_fonts();

  if (state != 0) {
    if (!sloppy_ok)
      WARN("mp_cleanup(): State not zero.");
    state = 0;
  }

  if (top_stack != 0) {
    if (!sloppy_ok)
      WARN("PS/MetaPost: PS stack not empty at end of figure!");
  }
  do_clear();

  /* Cleanup paths */
  if (currentpath.num_paths > 0) {
    if (!sloppy_ok)
      WARN("PS/MetaPost: Pending path at end of figure!");
  }

  release_paths();
}

void
mp_eop_cleanup (void)
{
  mp_cleanup(1);
  num_saves = 0;
}

static void
clear_state (void)
{
  top_stack = 0;
  currentpoint.x = 0.0;
  currentpoint.y = 0.0;
  state     = 0;
}

/* mp inclusion is a bit of a hack.  The routine
 * starts a form at the lower left corner of
 * the page and then calls begin_form_xobj telling
 * it to record the image drawn there and bundle it
 * up in an xojbect.  This allows us to use the coordinates
 * in the MP file directly.  This appears to be the
 * easiest way to be able to use the dev_set_string()
 * command (with its scaled and extended fonts) without
 * getting all confused about the coordinate system.
 * After the xobject is created, the whole thing can
 * be scaled any way the user wants */
 
int
mps_include_page (pdf_ximage *ximage, FILE *image_file)
{
  int error = 1;
  pdf_obj *xobject;
  xform_info info;
  char *buffer, *start, *end;
  long  size;

  pdf_ximage_init_form_info(&info);

  rewind(image_file);
  if ((size = file_size(image_file)) == 0) {
    WARN("Can't read any byte in the MPS file.");
    return NULL;
  }

  buffer = NEW(size+1, char);
  fread(buffer, sizeof(char), size, image_file);
  buffer[size] = 0;
  start = buffer;
  end   = buffer + size;

  error = scan_bbox(&start, end, &(info.bbox));
  if (error) {
    WARN("Errors occured while scanning MetaPost file headers.");
    RELEASE(buffer);
    return -1;
  }

  dev_start_mp_mode();

  pdf_doc_start_grabbing(NULL, info.bbox.llx, info.bbox.lly, &(info.bbox));
  clear_state();

  skip_prolog(&start, end);
  error = mp_parse_body(&start, end, 0.0, 0.0);
  if (error) {
    WARN("Errors occured while interpreting MetaPost file.");
    xobject = pdf_doc_end_grabbing();
    if (xobject)
      pdf_release_obj(xobject);
  } else {
    xobject = pdf_doc_end_grabbing();
    pdf_ximage_set_form(ximage, &info, xobject);
  }

  dev_end_mp_mode();

  RELEASE(buffer);
  mp_cleanup(0);

  /*
   * The reason why we don't return XObject itself is
   * PDF inclusion may not be made so.
   */
  return (error ? -1 : 0);
}

int
mps_do_page (FILE *image_file)
{
  int   error = 0;
  pdf_rect bbox;
  char *buffer, *start, *end;
  long  size;

  rewind(image_file);
  if ((size = file_size(image_file)) == 0) {
    WARN("Can't read any byte in the MPS file.");
    return NULL;
  }

  buffer = NEW(size+1, char);
  fread(buffer, sizeof(char), size, image_file);
  buffer[size] = 0;
  start = buffer;
  end   = buffer + size;

  error = scan_bbox(&start, end, &bbox);
#if 0
  dev_set_page_size(bbox.urx - bbox.llx, bbox.ury - bbox.lly);
#endif
  if (error) {
    WARN("Errors occured while scanning MetaPost file headers.");
    RELEASE(buffer);
    return -1;
  }


  dev_bop();
  pdf_doc_set_mediabox(pdf_doc_current_page_no(), &bbox);

  dev_start_mp_mode();
  clear_state();

  skip_prolog(&start, end);
  error = mp_parse_body(&start, end, 0.0, 0.0);
  if (error) {
    WARN("Errors occured while interpreting MetaPost file.");
  }
  dev_end_mp_mode();
  dev_eop();

  RELEASE(buffer);
  mp_cleanup(0);

  /*
   * The reason why we don't return XObject itself is
   * PDF inclusion may not be made so.
   */
  return (error ? -1 : 0);
}

int
check_for_mp (FILE *image_file) 
{
  int try_count = 10;

  rewind (image_file);
  mfgets(work_buffer, WORK_BUFFER_SIZE, image_file);
  if (strncmp(work_buffer, "%!PS", 4))
    return 0;

  while (try_count > 0) {
    mfgets(work_buffer, WORK_BUFFER_SIZE, image_file);
    if (!strncmp(work_buffer, "%%Creator:", 10)) {
      if (strlen(work_buffer+10) >= 8 &&
	  strstr(work_buffer+10, "MetaPost"))
	break;
    }
    try_count--;
  }

  return ((try_count > 0) ? 1 : 0);
}
