/* 
    L64781  -  DVB-T COFDM demodulator used in Technotrend DVB-T cards

    Copyright (C) 2001 Holger Waechtler <holger@convergence.de>
                       for Convergence Integrated Media GmbH
                       Marko Kohtala <marko.kohtala@nokia.com>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/    

#include <linux/module.h>
#include <linux/init.h>
#include <linux/delay.h>
#include <linux/slab.h>
#include <linux/poll.h>
#include <linux/i2c.h>
#include <linux/smp_lock.h>

#include "dvb_frontend.h"

#ifdef MODULE
MODULE_DESCRIPTION("L64781 based DVB-T Demodulator");
MODULE_AUTHOR("Holger Waechtler, Marko Kohtala");
#ifdef MODULE_LICENSE
MODULE_LICENSE("GPL");
#endif
#endif

static struct i2c_driver dvbt_driver;
static struct i2c_client client_template;

MODULE_PARM(debug,"i");
static int debug = 0;
#define dprintk	if (debug) printk

static
int write_reg(struct i2c_client *client, u8 reg, u8 data)
{
	int ret;
	u8 msg[2] = { reg, data };
#if 0
	if (reg < 0x02 || (reg > 0x31 && reg != 0x3e)) {
		printk ("L64781: try to write readonly register 0x%02x !\n",
				reg);
		return -1;
	}
#endif

	if ((ret = i2c_master_send (client, msg, 2)) != 2)
		printk ("L64781: write_reg error (addr == %02x) = %02x!\n",
			client->addr, ret);

	mdelay(10);

	return ret;
}


static
u8 read_reg(struct i2c_client *client, u8 reg)
{
	struct i2c_adapter *adap=client->adapter;
	struct i2c_msg msgs[2];
	u8 data;

	msgs[0].addr = client->addr;
	msgs[0].len = 1;
	msgs[0].flags = 0;
	msgs[0].buf = &reg;
	msgs[1].addr = client->addr;
	msgs[1].len = 1;
	msgs[1].flags = I2C_M_RD;
	msgs[1].buf = &data;

	i2c_transfer(adap, msgs, 2);

	return data;
}


static
void apply_tps (struct i2c_client *client)
{
	write_reg (client, 0x2a, 0x00);
	write_reg (client, 0x2a, 0x01);

	/* This here is a little bit questionable because it enables
	   the automatic update of TPS registers. I think we'd need to
	   handle the IRQ from FE to update some other registers as
	   well, or at least implement some magic to tuning to correct
	   to the TPS received from transmission. */
	write_reg (client, 0x2a, 0x02);
}


static
void reset_afc (struct i2c_client *client)
{
	/* Set AFC stall for the AFC_INIT_FRQ setting, TIM_STALL for
	   timing offset */
	write_reg (client, 0x07, 0x9e); /* stall AFC */
	write_reg (client, 0x08, 0);    /* AFC INIT FREQ */
	write_reg (client, 0x09, 0);
	write_reg (client, 0x0a, 0);
	write_reg (client, 0x07, 0x8e);
	write_reg (client, 0x0e, 0);    /* AGC gain to zero in beginning */
	write_reg (client, 0x11, 0x80); /* stall TIM */
	write_reg (client, 0x10, 0);    /* TIM_OFFSET_LSB */
	write_reg (client, 0x12, 0);
	write_reg (client, 0x13, 0);
	write_reg (client, 0x11, 0x00);
}


static
int apply_frontend_param (struct i2c_client *client, FrontendParameters *param)
{
	/* The coderates for FEC_AUTO and FEC_NONE are arbitrary */
	static const u8 vcr_tab[] = { 7, 1, 2, 3, 5, 7, 8 };
	/* QPSK, QAM_16, QAM_64 */
	static const u8 qam_tab [] = { 2, 4, 0, 6 };
	static const u8 bw_tab [] = { 8, 7, 6 };  /* 8Mhz, 7MHz, 6MHz */
	static const u8 guard_tab [] = { 1, 2, 4, 8 };
	/* The Grundig 29504-401.04 Tuner comes with 18.432MHz crystal. */
	static const u32 ppm = 8000;
	OFDMParameters *p = &param->u.ofdm;
	u32 ddfs_offset_fixed;
/*	u32 ddfs_offset_variable = 0x6000-((1000000UL+ppm)/ */
/*			bw_tab[p->bandWidth]<<10)/15625; */
	u32 init_freq;
	u32 spi_bias;
	u8 val0x04;
	u8 val0x05;
	u8 val0x06;

	if (param->Inversion != INVERSION_ON &&
	    param->Inversion != INVERSION_OFF)
		return -EINVAL;

	if (p->bandWidth < BANDWIDTH_8_MHZ || p->bandWidth > BANDWIDTH_6_MHZ)
		return -EINVAL;
	
	if (p->HP_CodeRate < FEC_1_2 || p->HP_CodeRate > FEC_7_8)
		return -EINVAL;

	if (p->HierarchyInformation != HIERARCHY_NONE &&
	    (p->LP_CodeRate < FEC_1_2 || p->LP_CodeRate > FEC_7_8))
		return -EINVAL;

	if (p->HierarchyInformation != HIERARCHY_NONE &&
	    (p->LP_CodeRate < FEC_1_2 || p->LP_CodeRate > FEC_7_8))
		return -EINVAL;

	if (p->Constellation != QPSK && p->Constellation != QAM_16 &&
	    p->Constellation != QAM_64)
		return -EINVAL;

	if (p->TransmissionMode != TRANSMISSION_MODE_2K &&
	    p->TransmissionMode != TRANSMISSION_MODE_8K)
		return -EINVAL;

	if (p->guardInterval < GUARD_INTERVAL_1_32 ||
	    p->guardInterval > GUARD_INTERVAL_1_4)
		return -EINVAL;

	if (p->HierarchyInformation < HIERARCHY_NONE ||
	    p->HierarchyInformation > HIERARCHY_4)
		return -EINVAL;

	ddfs_offset_fixed = 0x4000-(ppm<<16)/bw_tab[p->bandWidth]/1000000;

	/* This works up to 20000 ppm, it overflows if too large ppm! */
	init_freq = (((8UL<<25) + (8UL<<19) / 25*ppm / (15625/25)) /
			bw_tab[p->bandWidth] & 0xFFFFFF);

	/* SPI bias calculation is slightly modified to fit in 32bit */
	/* will work for high ppm only... */
	spi_bias = 378 * (1 << 10);
	spi_bias *= 16;
	spi_bias *= bw_tab[p->bandWidth];
	spi_bias *= qam_tab[p->Constellation];
	spi_bias /= (vcr_tab[p->HP_CodeRate] + 1);
	spi_bias /= (guard_tab[p->guardInterval] + 32);
	spi_bias *= 1000ULL;
	spi_bias /= 1000ULL + ppm/1000;
	spi_bias *= vcr_tab[p->HP_CodeRate];

	val0x04 = (p->TransmissionMode << 2) | p->guardInterval;
	val0x05 = p->HP_CodeRate - FEC_1_2;
	if (p->HierarchyInformation != HIERARCHY_NONE)
		val0x05 |= (p->LP_CodeRate - FEC_1_2) << 3;

	val0x06 = (p->HierarchyInformation << 2) | p->Constellation;

	write_reg (client, 0x04, val0x04);
	write_reg (client, 0x05, val0x05);
	write_reg (client, 0x06, val0x06);

	reset_afc (client);

	/* Technical manual section 2.6.1, TIM_IIR_GAIN optimal values */
	write_reg (client, 0x15, p->TransmissionMode == TRANSMISSION_MODE_2K ?
			1 : 3);
	write_reg (client, 0x16, init_freq & 0xFF);
	write_reg (client, 0x17, (init_freq >> 8) & 0xFF);
	write_reg (client, 0x18, (init_freq >> 16) & 0xFF);

	write_reg (client, 0x1B, spi_bias & 0xFF);
	write_reg (client, 0x1C, (spi_bias >> 8) & 0xFF);
	write_reg (client, 0x1D, ((spi_bias >> 16) & 0x7F) |
		(param->Inversion == INVERSION_ON ? 0x80 : 0x00));

	write_reg (client, 0x22, ddfs_offset_fixed & 0xff);
	write_reg (client, 0x23, (ddfs_offset_fixed >> 8) & 0x3f);

	read_reg (client, 0x00);  /*  clear interrupt registers... */
	read_reg (client, 0x01);  /*  dto. */

	apply_tps (client);

	return 0;
}


static
void reset_and_configure (struct i2c_client *client)
{
	u8 tmp_addr = client->addr;
	u8 msg = 0x06;

	client->addr = 0x0;
	i2c_master_send (client, &msg, 1);
	client->addr = tmp_addr;
}



static
int init (struct i2c_client *client)
{
        reset_and_configure (client);

	/* Power up */
	write_reg (client, 0x3e, 0xa5);

	/* Reset hard */
	write_reg (client, 0x2a, 0x04);
	write_reg (client, 0x2a, 0x00);

	/* Set tuner specific things */
	/* AFC_POL, set also in reset_afc */
	write_reg (client, 0x07, 0x8e);

	/* Use internal ADC */
	write_reg (client, 0x0b, 0x81);

	/* AGC loop gain, and polarity is positive */
	write_reg (client, 0x0c, 0x84);

	/* Internal ADC outputs two's complement */
	write_reg (client, 0x0d, 0x8c);

	/* With ppm=8000, it seems the DTR_SENSITIVITY will result in
           value of 2 with all possible bandwidths and guard
           intervals, which is the initial value anyway. */
        /*write_reg (client, 0x19, 0x92);*/

	/* Everything is two's complement, soft bit and CSI_OUT too */
	write_reg (client, 0x1e, 0x49);

	return 0;
}


typedef struct {
	struct task_struct *thread;
	int exit;
} StatusThreadData;


static
int L64781_check_status (void *data)
{
	struct i2c_client *client = data;
	StatusThreadData *d = client->data;

	lock_kernel ();
	daemonize ();
	sigfillset (&current->blocked);
	strcpy (current->comm, "L64781_check\0");
	d->thread = current;
	unlock_kernel ();

	do {
		read_reg (client, 0x00);
		read_reg (client, 0x01);
		current->state = TASK_INTERRUPTIBLE;
		schedule_timeout (HZ/5);
	} while (!d->exit);

	d->thread = NULL;
	return 0;
}




static
int attach_adapter(struct i2c_adapter *adap)
{
	struct i2c_client *client;

	client_template.adapter=adap;

	if (i2c_master_send(&client_template,NULL,0) == 0) {
		dprintk("L64781: probably an EEPROM @ 0x%02x, skipping ...\n",
			client_template.addr);
                return -1;
	}

        reset_and_configure (&client_template);

	if (i2c_master_send(&client_template,NULL,0)) {
		dprintk("L64781: no L64781 found ...\n");
                return -1;
	}

        client_template.adapter=adap;
        
        if (!(client = kmalloc(sizeof(struct i2c_client), GFP_KERNEL)))
                return -ENOMEM;

        memcpy(client, &client_template, sizeof(struct i2c_client));

	if (!(client->data = kmalloc(sizeof(StatusThreadData), GFP_KERNEL))) {
		kfree (client);
		return -ENOMEM;
	}

	((StatusThreadData*) client->data)->thread = (void*) 1;
	((StatusThreadData*) client->data)->exit = 0;
	
        i2c_attach_client(client);

	kernel_thread (L64781_check_status, client, 0);
        
        printk ("L64781: attaching L64781 @ 0x%02x (adap %p)\n",
		client->addr, client->adapter);
        
	MOD_INC_USE_COUNT;

        return 0;
}


static
int detach_client(struct i2c_client *client)
{
	((StatusThreadData*) client->data)->exit = 1;
	while (((StatusThreadData*) client->data)->thread) {
		current->state = TASK_INTERRUPTIBLE;
		schedule_timeout (HZ/10);
	}
	
	/* Power down, results in kernel oopses */
/*	write_reg (client, 0x3e, 0x5a); */
/*	mdelay (100); */
	
        i2c_detach_client(client);

        kfree(client->data);
        kfree(client);

	MOD_DEC_USE_COUNT;
        return 0;
}


static 
int dvb_command(struct i2c_client *client, unsigned int cmd, void *arg)
{
        switch (cmd) {
	case FE_INIT:
	{
        	init(client);
                break;
	}
        case FE_RESET:
        {
		//reset_afc (client);
		apply_tps (client);
		read_reg (client, 0x00);  /*  clear interrupt registers... */
		read_reg (client, 0x01);  /*  dto. */
                break;
	}
	case FE_SET_POWER_STATE:
	{
		switch (*(long*) arg) {
		case FE_POWER_ON:
			init (client);
                	break;
		case FE_POWER_SUSPEND:
		case FE_POWER_STANDBY:
		case FE_POWER_OFF:
			write_reg (client, 0x3e, 0x5a);
			break;
		default:
			return -EINVAL;
		};
		break;
	}
	case FE_READ_STATUS:
	{
		FrontendStatus *status = (FrontendStatus *) arg;
		int sync, gain;

		*status = FE_HAS_POWER;

                gain = read_reg (client, 0x0e);
                sync = read_reg (client, 0x32);

		if (gain < 0xff)
			*status |= FE_HAS_SIGNAL;
		if (sync & 0x02) /* VCXO locked, this criteria should be ok */
			*status |= FE_HAS_CARRIER;
		if (sync & 0x20)
			*status |= FE_HAS_VITERBI;
		if (sync & 0x40)
			*status |= FE_HAS_SYNC;
		if (sync == 0x7f)
			*status |= FE_HAS_LOCK;
		break;
	}
        case FE_READ_BER:
	{
		u32 *ber=(u32 *) arg;
		*ber = read_reg (client, 0x39) | (read_reg (client, 0x3a) << 8);
		break;
	}
        case FE_READ_SIGNAL_STRENGTH:
	{
		s32 *signal=(s32 *) arg;
		*signal = 0;
		break;
	}
        case FE_READ_SNR:
	{
		s32 *snr=(s32 *) arg;
		*snr=0; /* average quality: read_reg(client, 0x33); */
		break;
	}
	case FE_READ_UNCORRECTED_BLOCKS: 
	{
		u32 *ub=(u32 *) arg;
		*ub = read_reg(client, 0x37) | (read_reg(client, 0x38) << 8);
		break;
	}
        case FE_READ_AFC:
	{
		s32 *afc=(s32 *) arg;
		*afc=0;
		break;
	}
        case FE_SET_FRONTEND:
        {
		FrontendParameters *param = (FrontendParameters *) arg;

		return apply_frontend_param (client, param);
        }
        case FE_WRITEREG:
        {
                u8 *msg = (u8 *) arg;
                write_reg(client, msg[0], msg[1]);
                break;
        }
        case FE_READREG:
        {
                u8 *msg = (u8 *) arg;
                msg[1]=read_reg(client, msg[0]);
                break;
        }
        default:
		dprintk ("%s: unknown command !!!\n", __FUNCTION__);
                return -1;
        };
        
        return 0;
} 

void inc_use (struct i2c_client *client)
{
#ifdef MODULE
        MOD_INC_USE_COUNT;
#endif
}

void dec_use (struct i2c_client *client)
{
#ifdef MODULE
        MOD_DEC_USE_COUNT;
#endif
}

static
struct i2c_driver dvbt_driver = {
	name: 		"L64781 DVB demodulator",
	id:		I2C_DRIVERID_L64781,
	flags:		I2C_DF_NOTIFY,
	attach_adapter:	attach_adapter,
	detach_client:	detach_client,
	command:	dvb_command,
	inc_use:	inc_use,
	dec_use:	dec_use
};

static
struct i2c_client client_template = {
        name:    "L64781",
        id:      I2C_DRIVERID_L64781,
        flags:   0,
        addr:    0x55,
        adapter: NULL,
        driver:  &dvbt_driver
};


static __init
int init_l64781 (void) {
        int res;
        
        if ((res = i2c_add_driver(&dvbt_driver))) 
        {
                printk("L64781: i2c driver registration failed\n");
                return res;
        }

        return 0;
}

static __exit
void exit_l64781 (void)
{
        int res;

        if ((res = i2c_del_driver(&dvbt_driver))) 
        {
                printk("L64781: Driver deregistration failed, "
                       "module not removed.\n");
        }
}

module_init(init_l64781);
module_exit(exit_l64781);

