
// Software scaling and colorspace conversion routines for MPlayer

// Orginal C implementation by A'rpi/ESP-team <arpi@thot.banki.hu>
// current version mostly by Michael Niedermayer (michaelni@gmx.at)
// the parts written by michael are under GNU GPL
#include <glib.h>
#include "../src/drip.h"
#include "fast_memcpy.hh"

#undef MOVNTQ
#undef PAVGB
#undef PREFETCH
#undef PREFETCHW
#undef EMMS
#undef SFENCE

#ifdef HAVE_3DNOW
/* On K6 femms is faster of emms. On K7 femms is directly mapped on emms. */
#define EMMS     "femms"
#else
#define EMMS     "emms"
#endif

#ifdef HAVE_3DNOW
#define PREFETCH  "prefetch"
#define PREFETCHW "prefetchw"
#elif defined ( HAVE_MMX2 )
#define PREFETCH "prefetchnta"
#define PREFETCHW "prefetcht0"
#else
#define PREFETCH "/nop"
#define PREFETCHW "/nop"
#endif

#ifdef HAVE_MMX2
#define SFENCE "sfence"
#else
#define SFENCE "/nop"
#endif

#ifdef HAVE_MMX2
#define PAVGB(a,b) "pavgb " #a ", " #b " \n\t"
#elif defined (HAVE_3DNOW)
#define PAVGB(a,b) "pavgusb " #a ", " #b " \n\t"
#endif

#ifdef HAVE_MMX2
#define MOVNTQ(a,b) "movntq " #a ", " #b " \n\t"
#else
#define MOVNTQ(a,b) "movq " #a ", " #b " \n\t"
#endif


#define YSCALEYUV2YV12X(x) \
			"xorl %%eax, %%eax		\n\t"\
			"pxor %%mm3, %%mm3		\n\t"\
			"pxor %%mm4, %%mm4		\n\t"\
			"movl %0, %%edx			\n\t"\
			".balign 16			\n\t" /* FIXME Unroll? */\
			"1:				\n\t"\
			"movl (%1, %%edx, 4), %%esi	\n\t"\
			"movq (%2, %%edx, 8), %%mm0	\n\t" /* filterCoeff */\
			"movq " #x "(%%esi, %%eax, 2), %%mm2	\n\t" /* srcData */\
			"movq 8+" #x "(%%esi, %%eax, 2), %%mm5	\n\t" /* srcData */\
			"pmulhw %%mm0, %%mm2		\n\t"\
			"pmulhw %%mm0, %%mm5		\n\t"\
			"paddw %%mm2, %%mm3		\n\t"\
			"paddw %%mm5, %%mm4		\n\t"\
			"addl $1, %%edx			\n\t"\
			" jnz 1b			\n\t"\
			"psraw $3, %%mm3		\n\t"\
			"psraw $3, %%mm4		\n\t"\
			"packuswb %%mm4, %%mm3		\n\t"\
			MOVNTQ(%%mm3, (%3, %%eax))\
			"addl $8, %%eax			\n\t"\
			"cmpl %4, %%eax			\n\t"\
			"pxor %%mm3, %%mm3		\n\t"\
			"pxor %%mm4, %%mm4		\n\t"\
			"movl %0, %%edx			\n\t"\
			"jb 1b				\n\t"

#define YSCALEYUV2YV121 \
			"movl %2, %%eax			\n\t"\
			".balign 16			\n\t" /* FIXME Unroll? */\
			"1:				\n\t"\
			"movq (%0, %%eax, 2), %%mm0	\n\t"\
			"movq 8(%0, %%eax, 2), %%mm1	\n\t"\
			"psraw $7, %%mm0		\n\t"\
			"psraw $7, %%mm1		\n\t"\
			"packuswb %%mm1, %%mm0		\n\t"\
			MOVNTQ(%%mm0, (%1, %%eax))\
			"addl $8, %%eax			\n\t"\
			"jnc 1b				\n\t"


static inline void RENAME(yuv2yuvX)(int16_t *lumFilter, int16_t **lumSrc, int lumFilterSize,
				    int16_t *chrFilter, int16_t **chrSrc, int chrFilterSize,
				    uint8_t *dest, uint8_t *uDest, uint8_t *vDest, int dstW,
				    int16_t * lumMmxFilter, int16_t * chrMmxFilter)
{
#ifdef HAVE_MMX
	if (uDest != NULL) {
		asm volatile(
				YSCALEYUV2YV12X(0)
				: : "m" (-chrFilterSize), "r" (chrSrc+chrFilterSize),
				"r" (chrMmxFilter+chrFilterSize*4), "r" (uDest), "m" (dstW>>1)
				: "%eax", "%edx", "%esi"
			);

		asm volatile(
				YSCALEYUV2YV12X(4096)
				: : "m" (-chrFilterSize), "r" (chrSrc+chrFilterSize),
				"r" (chrMmxFilter+chrFilterSize*4), "r" (vDest), "m" (dstW>>1)
				: "%eax", "%edx", "%esi"
			);
	}

	asm volatile(
			YSCALEYUV2YV12X(0)
			: : "m" (-lumFilterSize), "r" (lumSrc+lumFilterSize),
			   "r" (lumMmxFilter+lumFilterSize*4), "r" (dest), "m" (dstW)
			: "%eax", "%edx", "%esi"
		);
#else
yuv2yuvXinC(lumFilter, lumSrc, lumFilterSize,
	    chrFilter, chrSrc, chrFilterSize,
	    dest, uDest, vDest, dstW);
#endif
}

static inline void RENAME(yuv2yuv1)(int16_t *lumSrc, int16_t *chrSrc,
				    uint8_t *dest, uint8_t *uDest, uint8_t *vDest, int dstW)
{
#ifdef HAVE_MMX
	if(uDest != NULL)
	{
		asm volatile(
				YSCALEYUV2YV121
				: : "r" (chrSrc + (dstW>>1)), "r" (uDest + (dstW>>1)),
				"g" (-(dstW>>1))
				: "%eax"
			);

		asm volatile(
				YSCALEYUV2YV121
				: : "r" (chrSrc + 2048 + (dstW>>1)), "r" (vDest + (dstW>>1)),
				"g" (-(dstW>>1))
				: "%eax"
			);
	}

	asm volatile(
		YSCALEYUV2YV121
		: : "r" (lumSrc + dstW), "r" (dest + dstW),
		"g" (-dstW)
		: "%eax"
	);
#else
	//FIXME Optimize (just quickly writen not opti..)
	//FIXME replace MINMAX with LUTs
	int i;
	for(i=0; i<dstW; i++)
	{
		int val= lumSrc[i]>>7;

		dest[i]= MIN(MAX(val>>19, 0), 255);
	}

	if(uDest != NULL)
		for(i=0; i<(dstW>>1); i++)
		{
			int u=chrSrc[i]>>7;
			int v=chrSrc[i + 2048]>>7;

			uDest[i]= MIN(MAX(u>>19, 0), 255);
			vDest[i]= MIN(MAX(v>>19, 0), 255);
		}
#endif
}

// Bilinear / Bicubic scaling
static inline void RENAME(hScale)(int16_t *dst, int dstW, uint8_t *src, int srcW, int xInc,
				  int16_t *filter, int16_t *filterPos, int filterSize)
{
#ifdef HAVE_MMX
//printf("dst %p, dstW %i, src %p srcW %i, xInc %i, filtersize %i\n",dst,dstW,src,srcW,xInc,filterSize);fflush(NULL);
	if(filterSize==4) // allways true for upscaling, sometimes for down too
	{
		int counter= -2*dstW;
		filter-= counter*2;
		filterPos-= counter/2;
		dst-= counter/2;
		asm volatile(
			"pxor %%mm7, %%mm7		\n\t"
			"movq w02, %%mm6		\n\t"
			"pushl %%ebp			\n\t" // we use 7 regs here ...
			"movl %%eax, %%ebp		\n\t"
			".balign 16			\n\t"
			"1:				\n\t"
			"movzwl (%2, %%ebp), %%eax	\n\t"
			"movzwl 2(%2, %%ebp), %%ebx	\n\t"
			"movq (%1, %%ebp, 4), %%mm1	\n\t"
			"movq 8(%1, %%ebp, 4), %%mm3	\n\t"
			"movd (%3, %%eax), %%mm0	\n\t"
			"movd (%3, %%ebx), %%mm2	\n\t"
			"punpcklbw %%mm7, %%mm0		\n\t"
			"punpcklbw %%mm7, %%mm2		\n\t"
			"pmaddwd %%mm1, %%mm0		\n\t"
			"pmaddwd %%mm2, %%mm3		\n\t"
			"psrad $8, %%mm0		\n\t"
			"psrad $8, %%mm3		\n\t"
			"packssdw %%mm3, %%mm0		\n\t"
			"pmaddwd %%mm6, %%mm0		\n\t"
			"packssdw %%mm0, %%mm0		\n\t"
			"movd %%mm0, (%4, %%ebp)	\n\t"
			"addl $4, %%ebp			\n\t"
			" jnc 1b			\n\t"

			"popl %%ebp			\n\t"
			: "+a" (counter)
			: "c" (filter), "d" (filterPos), "S" (src), "D" (dst)
			: "%ebx"
		);
	}
	else if(filterSize==8)
	{
		int counter= -2*dstW;
		filter-= counter*4;
		filterPos-= counter/2;
		dst-= counter/2;
		asm volatile(
			"pxor %%mm7, %%mm7		\n\t"
			"movq w02, %%mm6		\n\t"
			"pushl %%ebp			\n\t" // we use 7 regs here ...
			"movl %%eax, %%ebp		\n\t"
			".balign 16			\n\t"
			"1:				\n\t"
			"movzwl (%2, %%ebp), %%eax	\n\t"
			"movzwl 2(%2, %%ebp), %%ebx	\n\t"
			"movq (%1, %%ebp, 8), %%mm1	\n\t"
			"movq 16(%1, %%ebp, 8), %%mm3	\n\t"
			"movd (%3, %%eax), %%mm0	\n\t"
			"movd (%3, %%ebx), %%mm2	\n\t"
			"punpcklbw %%mm7, %%mm0		\n\t"
			"punpcklbw %%mm7, %%mm2		\n\t"
			"pmaddwd %%mm1, %%mm0		\n\t"
			"pmaddwd %%mm2, %%mm3		\n\t"

			"movq 8(%1, %%ebp, 8), %%mm1	\n\t"
			"movq 24(%1, %%ebp, 8), %%mm5	\n\t"
			"movd 4(%3, %%eax), %%mm4	\n\t"
			"movd 4(%3, %%ebx), %%mm2	\n\t"
			"punpcklbw %%mm7, %%mm4		\n\t"
			"punpcklbw %%mm7, %%mm2		\n\t"
			"pmaddwd %%mm1, %%mm4		\n\t"
			"pmaddwd %%mm2, %%mm5		\n\t"
			"paddd %%mm4, %%mm0		\n\t"
			"paddd %%mm5, %%mm3		\n\t"
						
			"psrad $8, %%mm0		\n\t"
			"psrad $8, %%mm3		\n\t"
			"packssdw %%mm3, %%mm0		\n\t"
			"pmaddwd %%mm6, %%mm0		\n\t"
			"packssdw %%mm0, %%mm0		\n\t"
			"movd %%mm0, (%4, %%ebp)	\n\t"
			"addl $4, %%ebp			\n\t"
			" jnc 1b			\n\t"

			"popl %%ebp			\n\t"
			: "+a" (counter)
			: "c" (filter), "d" (filterPos), "S" (src), "D" (dst)
			: "%ebx"
		);
	}
	else
	{
		int counter= -2*dstW;
		filterPos-= counter/2;
		dst-= counter/2;
		asm volatile(
			"pxor %%mm7, %%mm7		\n\t"
			"movq w02, %%mm6		\n\t"
			".balign 16			\n\t"
			"1:				\n\t"
			"movl %2, %%ecx			\n\t"
			"movzwl (%%ecx, %0), %%eax	\n\t"
			"movzwl 2(%%ecx, %0), %%ebx	\n\t"
			"movl %5, %%ecx			\n\t"
			"pxor %%mm4, %%mm4		\n\t"
			"pxor %%mm5, %%mm5		\n\t"
			"2:				\n\t"
			"movq (%1), %%mm1		\n\t"
			"movq (%1, %6), %%mm3		\n\t"
			"movd (%%ecx, %%eax), %%mm0	\n\t"
			"movd (%%ecx, %%ebx), %%mm2	\n\t"
			"punpcklbw %%mm7, %%mm0		\n\t"
			"punpcklbw %%mm7, %%mm2		\n\t"
			"pmaddwd %%mm1, %%mm0		\n\t"
			"pmaddwd %%mm2, %%mm3		\n\t"
			"paddd %%mm3, %%mm5		\n\t"
			"paddd %%mm0, %%mm4		\n\t"
			"addl $8, %1			\n\t"
			"addl $4, %%ecx			\n\t"
			"cmpl %4, %%ecx			\n\t"
			" jb 2b				\n\t"
			"addl %6, %1			\n\t"
			"psrad $8, %%mm4		\n\t"
			"psrad $8, %%mm5		\n\t"
			"packssdw %%mm5, %%mm4		\n\t"
			"pmaddwd %%mm6, %%mm4		\n\t"
			"packssdw %%mm4, %%mm4		\n\t"
			"movl %3, %%eax			\n\t"
			"movd %%mm4, (%%eax, %0)	\n\t"
			"addl $4, %0			\n\t"
			" jnc 1b			\n\t"

			: "+r" (counter), "+r" (filter)
			: "m" (filterPos), "m" (dst), "m"(src+filterSize),
			  "m" (src), "r" (filterSize*2)
			: "%ebx", "%eax", "%ecx"
		);
	}
#else
	int i;
	for(i=0; i<dstW; i++)
	{
		int j;
		int srcPos= filterPos[i];
		int val=0;
		for(j=0; j<filterSize; j++)
		{
			val += ((int)src[srcPos + j])*filter[filterSize*i + j];
		}
		dst[i] = MIN(MAX(0, val>>7), (1<<15)-1); // the cubic equation does overflow ...
	}
#endif
}

      // *** horizontal scale Y line to temp buffer
static inline void RENAME(hyscale)(int16_t *dst, int dstWidth, uint8_t *src, int srcW, int xInc)
{
#ifdef HAVE_MMX
	// use the new MMX scaler if th mmx2 cant be used (its faster than the x86asm one)
    if(sws_flags != SWS_FAST_BILINEAR || (!canMMX2BeUsed))
#else
    if(sws_flags != SWS_FAST_BILINEAR)
#endif
    {
//printf("Calling hScale from hyscale, dst = %p",dst);
    	RENAME(hScale)(dst, dstWidth, src, srcW, xInc, hLumFilter, hLumFilterPos, hLumFilterSize);
    }
    else // Fast Bilinear upscale / crap downscale
    {
#ifdef ARCH_X86
#ifdef HAVE_MMX2
	int i;
	if(canMMX2BeUsed)
	{
		asm volatile(
			"pxor %%mm7, %%mm7		\n\t"
			"pxor %%mm2, %%mm2		\n\t" // 2*xalpha
			"movd %5, %%mm6			\n\t" // xInc&0xFFFF
			"punpcklwd %%mm6, %%mm6		\n\t"
			"punpcklwd %%mm6, %%mm6		\n\t"
			"movq %%mm6, %%mm2		\n\t"
			"psllq $16, %%mm2		\n\t"
			"paddw %%mm6, %%mm2		\n\t"
			"psllq $16, %%mm2		\n\t"
			"paddw %%mm6, %%mm2		\n\t"
			"psllq $16, %%mm2		\n\t" //0,t,2t,3t		t=xInc&0xFF
			"movq %%mm2, temp0		\n\t"
			"movd %4, %%mm6			\n\t" //(xInc*4)&0xFFFF
			"punpcklwd %%mm6, %%mm6		\n\t"
			"punpcklwd %%mm6, %%mm6		\n\t"
			"xorl %%eax, %%eax		\n\t" // i
			"movl %0, %%esi			\n\t" // src
			"movl %1, %%edi			\n\t" // buf1
			"movl %3, %%edx			\n\t" // (xInc*4)>>16
			"xorl %%ecx, %%ecx		\n\t"
			"xorl %%ebx, %%ebx		\n\t"
			"movw %4, %%bx			\n\t" // (xInc*4)&0xFFFF

#define FUNNY_Y_CODE \
			PREFETCH" 1024(%%esi)		\n\t"\
			PREFETCH" 1056(%%esi)		\n\t"\
			PREFETCH" 1088(%%esi)		\n\t"\
			"call funnyYCode		\n\t"\
			"movq temp0, %%mm2		\n\t"\
			"xorl %%ecx, %%ecx		\n\t"

FUNNY_Y_CODE
FUNNY_Y_CODE
FUNNY_Y_CODE
FUNNY_Y_CODE
FUNNY_Y_CODE
FUNNY_Y_CODE
FUNNY_Y_CODE
FUNNY_Y_CODE

			: : "m" (src), "m" (dst), "m" (dstWidth), "m" ((xInc*4)>>16),
			"m" ((xInc*4)&0xFFFF), "m" (xInc&0xFFFF)
			: "%eax", "%ebx", "%ecx", "%edx", "%esi", "%edi"
		);
		for(i=dstWidth-1; (i*xInc)>>16 >=srcW-1; i--) dst[i] = src[srcW-1]*128;
	}
	else
	{
#endif
	//NO MMX just normal asm ...
	asm volatile(
		"xorl %%eax, %%eax		\n\t" // i
		"xorl %%ebx, %%ebx		\n\t" // xx
		"xorl %%ecx, %%ecx		\n\t" // 2*xalpha
		".balign 16			\n\t"
		"1:				\n\t"
		"movzbl  (%0, %%ebx), %%edi	\n\t" //src[xx]
		"movzbl 1(%0, %%ebx), %%esi	\n\t" //src[xx+1]
		"subl %%edi, %%esi		\n\t" //src[xx+1] - src[xx]
		"imull %%ecx, %%esi		\n\t" //(src[xx+1] - src[xx])*2*xalpha
		"shll $16, %%edi		\n\t"
		"addl %%edi, %%esi		\n\t" //src[xx+1]*2*xalpha + src[xx]*(1-2*xalpha)
		"movl %1, %%edi			\n\t"
		"shrl $9, %%esi			\n\t"
		"movw %%si, (%%edi, %%eax, 2)	\n\t"
		"addw %4, %%cx			\n\t" //2*xalpha += xInc&0xFF
		"adcl %3, %%ebx			\n\t" //xx+= xInc>>8 + carry

		"movzbl (%0, %%ebx), %%edi	\n\t" //src[xx]
		"movzbl 1(%0, %%ebx), %%esi	\n\t" //src[xx+1]
		"subl %%edi, %%esi		\n\t" //src[xx+1] - src[xx]
		"imull %%ecx, %%esi		\n\t" //(src[xx+1] - src[xx])*2*xalpha
		"shll $16, %%edi		\n\t"
		"addl %%edi, %%esi		\n\t" //src[xx+1]*2*xalpha + src[xx]*(1-2*xalpha)
		"movl %1, %%edi			\n\t"
		"shrl $9, %%esi			\n\t"
		"movw %%si, 2(%%edi, %%eax, 2)	\n\t"
		"addw %4, %%cx			\n\t" //2*xalpha += xInc&0xFF
		"adcl %3, %%ebx			\n\t" //xx+= xInc>>8 + carry


		"addl $2, %%eax			\n\t"
		"cmpl %2, %%eax			\n\t"
		" jb 1b				\n\t"


		: : "r" (src), "m" (dst), "m" (dstWidth), "m" (xInc>>16), "m" (xInc&0xFFFF)
		: "%eax", "%ebx", "%ecx", "%edi", "%esi"
		);
#ifdef HAVE_MMX2
	} //if MMX2 cant be used
#endif
#else
	int i;
	unsigned int xpos=0;
	for(i=0;i<dstWidth;i++)
	{
		register unsigned int xx=xpos>>16;
		register unsigned int xalpha=(xpos&0xFFFF)>>9;
		dst[i]= (src[xx]<<7) + (src[xx+1] - src[xx])*xalpha;
		xpos+=xInc;
	}
#endif
    }
}

inline static void RENAME(hcscale)(int16_t *dst, int dstWidth,
				uint8_t *src1, uint8_t *src2, int srcW, int xInc)
{
#ifdef HAVE_MMX
	// use the new MMX scaler if th mmx2 cant be used (its faster than the x86asm one)
    if(sws_flags != SWS_FAST_BILINEAR || (!canMMX2BeUsed))
#else
    if(sws_flags != SWS_FAST_BILINEAR)
#endif
    {
    	RENAME(hScale)(dst     , dstWidth, src1, srcW, xInc, hChrFilter, hChrFilterPos, hChrFilterSize);
    	RENAME(hScale)(dst+2048, dstWidth, src2, srcW, xInc, hChrFilter, hChrFilterPos, hChrFilterSize);
    }
    else // Fast Bilinear upscale / crap downscale
    {
#ifdef ARCH_X86
#ifdef HAVE_MMX2
	int i;
	if(canMMX2BeUsed)
	{
		asm volatile(
		"pxor %%mm7, %%mm7		\n\t"
		"pxor %%mm2, %%mm2		\n\t" // 2*xalpha
		"movd %5, %%mm6			\n\t" // xInc&0xFFFF
		"punpcklwd %%mm6, %%mm6		\n\t"
		"punpcklwd %%mm6, %%mm6		\n\t"
		"movq %%mm6, %%mm2		\n\t"
		"psllq $16, %%mm2		\n\t"
		"paddw %%mm6, %%mm2		\n\t"
		"psllq $16, %%mm2		\n\t"
		"paddw %%mm6, %%mm2		\n\t"
		"psllq $16, %%mm2		\n\t" //0,t,2t,3t		t=xInc&0xFFFF
		"movq %%mm2, temp0		\n\t"
		"movd %4, %%mm6			\n\t" //(xInc*4)&0xFFFF
		"punpcklwd %%mm6, %%mm6		\n\t"
		"punpcklwd %%mm6, %%mm6		\n\t"
		"xorl %%eax, %%eax		\n\t" // i
		"movl %0, %%esi			\n\t" // src
		"movl %1, %%edi			\n\t" // buf1
		"movl %3, %%edx			\n\t" // (xInc*4)>>16
		"xorl %%ecx, %%ecx		\n\t"
		"xorl %%ebx, %%ebx		\n\t"
		"movw %4, %%bx			\n\t" // (xInc*4)&0xFFFF

#define FUNNYUVCODE \
			PREFETCH" 1024(%%esi)		\n\t"\
			PREFETCH" 1056(%%esi)		\n\t"\
			PREFETCH" 1088(%%esi)		\n\t"\
			"call funnyUVCode		\n\t"\
			"movq temp0, %%mm2		\n\t"\
			"xorl %%ecx, %%ecx		\n\t"

FUNNYUVCODE
FUNNYUVCODE
FUNNYUVCODE
FUNNYUVCODE

FUNNYUVCODE
FUNNYUVCODE
FUNNYUVCODE
FUNNYUVCODE
		"xorl %%eax, %%eax		\n\t" // i
		"movl %6, %%esi			\n\t" // src
		"movl %1, %%edi			\n\t" // buf1
		"addl $4096, %%edi		\n\t"

FUNNYUVCODE
FUNNYUVCODE
FUNNYUVCODE
FUNNYUVCODE

FUNNYUVCODE
FUNNYUVCODE
FUNNYUVCODE
FUNNYUVCODE

		: : "m" (src1), "m" (dst), "m" (dstWidth), "m" ((xInc*4)>>16),
		  "m" ((xInc*4)&0xFFFF), "m" (xInc&0xFFFF), "m" (src2)
		: "%eax", "%ebx", "%ecx", "%edx", "%esi", "%edi"
	);
		for(i=dstWidth-1; (i*xInc)>>16 >=srcW-1; i--)
		{
			dst[i] = src1[srcW-1]*128;
			dst[i+2048] = src2[srcW-1]*128;
		}
	}
	else
	{
#endif
	asm volatile(
		"xorl %%eax, %%eax		\n\t" // i
		"xorl %%ebx, %%ebx		\n\t" // xx
		"xorl %%ecx, %%ecx		\n\t" // 2*xalpha
		".balign 16			\n\t"
		"1:				\n\t"
		"movl %0, %%esi			\n\t"
		"movzbl  (%%esi, %%ebx), %%edi	\n\t" //src[xx]
		"movzbl 1(%%esi, %%ebx), %%esi	\n\t" //src[xx+1]
		"subl %%edi, %%esi		\n\t" //src[xx+1] - src[xx]
		"imull %%ecx, %%esi		\n\t" //(src[xx+1] - src[xx])*2*xalpha
		"shll $16, %%edi		\n\t"
		"addl %%edi, %%esi		\n\t" //src[xx+1]*2*xalpha + src[xx]*(1-2*xalpha)
		"movl %1, %%edi			\n\t"
		"shrl $9, %%esi			\n\t"
		"movw %%si, (%%edi, %%eax, 2)	\n\t"

		"movzbl  (%5, %%ebx), %%edi	\n\t" //src[xx]
		"movzbl 1(%5, %%ebx), %%esi	\n\t" //src[xx+1]
		"subl %%edi, %%esi		\n\t" //src[xx+1] - src[xx]
		"imull %%ecx, %%esi		\n\t" //(src[xx+1] - src[xx])*2*xalpha
		"shll $16, %%edi		\n\t"
		"addl %%edi, %%esi		\n\t" //src[xx+1]*2*xalpha + src[xx]*(1-2*xalpha)
		"movl %1, %%edi			\n\t"
		"shrl $9, %%esi			\n\t"
		"movw %%si, 4096(%%edi, %%eax, 2)\n\t"

		"addw %4, %%cx			\n\t" //2*xalpha += xInc&0xFF
		"adcl %3, %%ebx			\n\t" //xx+= xInc>>8 + carry
		"addl $1, %%eax			\n\t"
		"cmpl %2, %%eax			\n\t"
		" jb 1b				\n\t"

		: : "m" (src1), "m" (dst), "m" (dstWidth), "m" (xInc>>16), "m" (xInc&0xFFFF),
		"r" (src2)
		: "%eax", "%ebx", "%ecx", "%edi", "%esi"
		);
#ifdef HAVE_MMX2
	} //if MMX2 cant be used
#endif
#else
	int i;
	unsigned int xpos=0;
	for(i=0;i<dstWidth;i++)
	{
		register unsigned int xx=xpos>>16;
		register unsigned int xalpha=(xpos&0xFFFF)>>9;
		dst[i]=(src1[xx]*(xalpha^127)+src1[xx+1]*xalpha);
		dst[i+2048]=(src2[xx]*(xalpha^127)+src2[xx+1]*xalpha);
		xpos+=xInc;
	}
#endif
   }
}

static inline void RENAME(initFilter)(int16_t *dstFilter, int16_t *filterPos, int *filterSize, int xInc,
				      int srcW, int dstW, int filterAlign, int one)
{
	int i;
	double filter[8000];
#ifdef HAVE_MMX
	asm volatile("emms\n\t": : : "memory"); //FIXME this shouldnt be required but it IS (even for non mmx versions)
#endif

	if(ABS(xInc - 0x10000) <10) // unscaled
	{
		int i;
		*filterSize= (1 +(filterAlign-1)) & (~(filterAlign-1)); // 1 or 4 normaly
		for(i=0; i<dstW*(*filterSize); i++) filter[i]=0;

		for(i=0; i<dstW; i++)
		{
			filter[i*(*filterSize)]=1;
			filterPos[i]=i;
		}

	}
	else if(xInc <= (1<<16) || sws_flags==SWS_FAST_BILINEAR) // upscale
	{
		int i;
		int xDstInSrc;
		if(sws_flags==SWS_BICUBIC) *filterSize= 4;
		else			   *filterSize= 2;
		*filterSize= (*filterSize +(filterAlign-1)) & (~(filterAlign-1));

		xDstInSrc= xInc/2 - 0x8000;
		for(i=0; i<dstW; i++)
		{
			int xx= (xDstInSrc>>16) - (*filterSize>>1) + 1;
			int j;

			filterPos[i]= xx;
			if(sws_flags == SWS_BICUBIC)
			{
				double d= ABS(((xx+1)<<16) - xDstInSrc)/(double)(1<<16);
				double y1,y2,y3,y4;
				double A= -0.75;
					// Equation is from VirtualDub
				y1 = (        +     A*d -       2.0*A*d*d +       A*d*d*d);
				y2 = (+ 1.0             -     (A+3.0)*d*d + (A+2.0)*d*d*d);
				y3 = (        -     A*d + (2.0*A+3.0)*d*d - (A+2.0)*d*d*d);
				y4 = (                  +           A*d*d -       A*d*d*d);

				filter[i*(*filterSize) + 0]= y1;
				filter[i*(*filterSize) + 1]= y2;
				filter[i*(*filterSize) + 2]= y3;
				filter[i*(*filterSize) + 3]= y4;
			}
			else
			{
				for(j=0; j<*filterSize; j++)
				{
					double d= ABS((xx<<16) - xDstInSrc)/(double)(1<<16);
					double coeff= 1.0 - d;
					if(coeff<0) coeff=0;
					filter[i*(*filterSize) + j]= coeff;
					xx++;
				}
			}
			xDstInSrc+= xInc;
		}
	}
	else // downscale
	{
		int xDstInSrc;
		if(sws_flags==SWS_BICUBIC) *filterSize= (int)ceil(1 + 4.0*srcW / (double)dstW);
		else			   *filterSize= (int)ceil(1 + 2.0*srcW / (double)dstW);
		*filterSize= (*filterSize +(filterAlign-1)) & (~(filterAlign-1));

		xDstInSrc= xInc/2 - 0x8000;
		for(i=0; i<dstW; i++)
		{
			int xx= (int)((double)xDstInSrc/(double)(1<<16) - ((*filterSize)-1)*0.5 + 0.5);
			int j;

			filterPos[i]= xx;
			for(j=0; j<*filterSize; j++)
			{
				double d= ABS((xx<<16) - xDstInSrc)/(double)xInc;
				double coeff;
				if(sws_flags == SWS_BICUBIC)
				{
					double A= -0.75;
//					d*=2;
					// Equation is from VirtualDub
					if(d<1.0)
						coeff = (1.0 - (A+3.0)*d*d + (A+2.0)*d*d*d);
					else if(d<2.0)
						coeff = (-4.0*A + 8.0*A*d - 5.0*A*d*d + A*d*d*d);
					else
						coeff=0.0;
				}
				else
				{
					coeff= 1.0 - d;
					if(coeff<0) coeff=0;
				}
				filter[i*(*filterSize) + j]= coeff;
				xx++;
			}
			xDstInSrc+= xInc;
		}
	}

	//fix borders
	for(i=0; i<dstW; i++)
	{
		int j;
		if(filterPos[i] < 0)
		{
			// Move filter coeffs left to compensate for filterPos
			for(j=1; j<*filterSize; j++)
			{
				int left= MAX(j + filterPos[i], 0);
				filter[i*(*filterSize) + left] += filter[i*(*filterSize) + j];
				filter[i*(*filterSize) + j]=0;
			}
			filterPos[i]= 0;
		}

		if(filterPos[i] + (*filterSize) > srcW)
		{
			int shift= filterPos[i] + (*filterSize) - srcW;
			// Move filter coeffs right to compensate for filterPos
			for(j=(*filterSize)-2; j>=0; j--)
			{
				int right= MIN(j + shift, (*filterSize)-1);
				filter[i*(*filterSize) +right] += filter[i*(*filterSize) +j];
				filter[i*(*filterSize) +j]=0;
			}
			filterPos[i]= srcW - (*filterSize);
		}
	}

	//FIXME try to align filterpos if possible / try to shift filterpos to put zeros at the end
	// and skip these than later

	//Normalize
	for(i=0; i<dstW; i++)
	{
		int j;
		double sum=0;
		double scale= one;
		for(j=0; j<*filterSize; j++)
		{
			sum+= filter[i*(*filterSize) + j];
		}
		scale/= sum;
		for(j=0; j<*filterSize; j++)
		{
			dstFilter[i*(*filterSize) + j]= (int)(filter[i*(*filterSize) + j]*scale);
		}
	}
}

#ifdef HAVE_MMX2
static void initMMX2HScaler(int dstW, int xInc, uint8_t *funnyCode)
{
	uint8_t *fragment;
	int imm8OfPShufW1;
	int imm8OfPShufW2;
	int fragmentLength;

	int xpos, i;

	// create an optimized horizontal scaling routine

	//code fragment

	asm volatile(
		"jmp 9f				\n\t"
	// Begin
		"0:				\n\t"
		"movq (%%esi), %%mm0		\n\t" //FIXME Alignment
		"movq %%mm0, %%mm1		\n\t"
		"psrlq $8, %%mm0		\n\t"
		"punpcklbw %%mm7, %%mm1	\n\t"
		"movq %%mm2, %%mm3		\n\t"
		"punpcklbw %%mm7, %%mm0	\n\t"
		"addw %%bx, %%cx		\n\t" //2*xalpha += (4*lumXInc)&0xFFFF
		"pshufw $0xFF, %%mm1, %%mm1	\n\t"
		"1:				\n\t"
		"adcl %%edx, %%esi		\n\t" //xx+= (4*lumXInc)>>16 + carry
		"pshufw $0xFF, %%mm0, %%mm0	\n\t"
		"2:				\n\t"
		"psrlw $9, %%mm3		\n\t"
		"psubw %%mm1, %%mm0		\n\t"
		"pmullw %%mm3, %%mm0		\n\t"
		"paddw %%mm6, %%mm2		\n\t" // 2*alpha += xpos&0xFFFF
		"psllw $7, %%mm1		\n\t"
		"paddw %%mm1, %%mm0		\n\t"

		"movq %%mm0, (%%edi, %%eax)	\n\t"

		"addl $8, %%eax			\n\t"
	// End
		"9:				\n\t"
		"leal 0b, %0			\n\t"
		"leal 1b, %1			\n\t"
		"leal 2b, %2			\n\t"
		"decl %1			\n\t"
		"decl %2			\n\t"
		"subl %0, %1			\n\t"
		"subl %0, %2			\n\t"
		"leal 9b, %3			\n\t"
		"subl %0, %3			\n\t"
		:"=r" (fragment), "=r" (imm8OfPShufW1), "=r" (imm8OfPShufW2),
		"=r" (fragmentLength)
	);

	xpos= 0; //lumXInc/2 - 0x8000; // difference between pixel centers

	for(i=0; i<dstW/8; i++)
	{
		int xx=xpos>>16;

		if((i&3) == 0)
		{
			int a=0;
			int b=((xpos+xInc)>>16) - xx;
			int c=((xpos+xInc*2)>>16) - xx;
			int d=((xpos+xInc*3)>>16) - xx;

			fast_memcpy(funnyCode + fragmentLength*i/4, fragment, fragmentLength);

			funnyCode[fragmentLength*i/4 + imm8OfPShufW1]=
			funnyCode[fragmentLength*i/4 + imm8OfPShufW2]=
				a | (b<<2) | (c<<4) | (d<<6);

			// if we dont need to read 8 bytes than dont :), reduces the chance of
			// crossing a cache line
			if(d<3) funnyCode[fragmentLength*i/4 + 1]= 0x6E;

			funnyCode[fragmentLength*(i+4)/4]= RET;
		}
		xpos+=xInc;
	}
}
#endif // HAVE_MMX2

static void RENAME(SwScale_YV12slice)(unsigned char* srcptr[],int stride[], int srcSliceY ,
			     int srcSliceH, uint8_t* dstptr[], int dststride, int dstbpp,
			     int srcW, int srcH, int dstW, int dstH){


unsigned int lumXInc= (srcW << 16) / dstW;
unsigned int lumYInc= (srcH << 16) / dstH;
unsigned int chrXInc;
unsigned int chrYInc;

static int dstY;

static int verbose = FALSE;
// used to detect a size change
static int oldDstW= -1;
static int oldSrcW= -1;
static int oldDstH= -1;
static int oldSrcH= -1;
static int oldFlags=-1;

static glong lastInLumBuf;
static glong lastInChrBuf;

int chrDstW, chrDstH;

static int lumBufIndex=0;
static int chrBufIndex=0;

static int firstTime=1;

const int widthAlign= dstbpp==12 ? 16 : 8;
const int bytespp= (dstbpp+1)/8; //(12->1, 15&16->2, 24->3, 32->4)
const int over= dstbpp==12 ? 	  (((dstW+15)&(~15))) - dststride
				: (((dstW+7)&(~7)))*bytespp - dststride;

//printf("SwScale_YV12slice: dstptr[0] = %p, \n");
if(dststride%widthAlign !=0 ) {
	if(firstTime)
		g_log(DRIP_LD,G_LOG_LEVEL_WARNING,"SwScaler: Warning: dstStride is not a multiple of %d!  -> cannot do aligned memory acesses anymore",widthAlign);
}

if(over>0 && verbose)
{
	if(firstTime)
		g_log(DRIP_LD,G_LOG_LEVEL_WARNING,"SwScaler: Warning: output width is not a multiple of 8 (16 for YV12) and dststride is not large enough to handle %d extra bytes >using unoptimized C version for last line(s)",over);
}



#ifdef HAVE_MMX2
canMMX2BeUsed= (lumXInc <= 0x10000 && (dstW&31)==0 && (srcW&15)==0) ? 1 : 0;
if(!canMMX2BeUsed && lumXInc <= 0x10000 && (srcW&15)==0 && sws_flags==SWS_FAST_BILINEAR)
{
	if(firstTime)
		g_log(DRIP_LD,G_LOG_LEVEL_WARNING,"SwScaler: output Width is not a multiple of 32 -> no MMX2 scaler");
}
#else
canMMX2BeUsed=0; // should be 0 anyway but ...
#endif

if(firstTime)
{
        #if defined (DITHER1XBPP) && defined (HAVE_MMX)
	char *dither= " dithered";
        #else
	char *dither= "";
        #endif
	if(sws_flags==SWS_FAST_BILINEAR)
		g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"SwScaler: FAST_BILINEAR scaler ");
	else if(sws_flags==SWS_BILINEAR)
		g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"SwScaler: BILINEAR scaler ");
	else if(sws_flags==SWS_BICUBIC)
		g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"SwScaler: BICUBIC scaler ");
	else
		g_log(DRIP_LD,G_LOG_LEVEL_WARNING,"SwScaler: ehh flags invalid?! ");

	if(dstbpp==12)
		g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"with YV12 output ");
	else
		g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"without output ");

                #ifdef HAVE_MMX2
		g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"using MMX2");
                #elif defined (HAVE_3DNOW)
		g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"using 3DNOW");
                #elif defined (HAVE_MMX)
		g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"using MMX");
                #elif defined (ARCH_X86)
		g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"using X86 ASM");
                #else
		g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"using C");
                #endif
}


// match pixel 0 of the src to pixel 0 of dst and match pixel n-2 of src to pixel n-2 of dst
// n-2 is the last chrominance sample available
// this is not perfect, but noone shuld notice the difference, the more correct variant
// would be like the vertical one, but that would require some special code for the
// first and last pixel
if(sws_flags==SWS_FAST_BILINEAR)
{
	if(canMMX2BeUsed) 	lumXInc+= 20;
#ifndef HAVE_MMX //we dont use the x86asm scaler if mmx is available
	else			lumXInc = ((srcW-2)<<16)/(dstW-2) - 20;
#endif
}

if(fullUVIpol && !(dstbpp==12)) 	chrXInc= lumXInc>>1, chrDstW= dstW;
else					chrXInc= lumXInc,    chrDstW= (dstW+1)>>1;

if(dstbpp==12)	chrYInc= lumYInc,    chrDstH= (dstH+1)>>1;
else		chrYInc= lumYInc>>1, chrDstH= dstH;

  // force calculation of the horizontal interpolation of the first line

  if(srcSliceY ==0){
	lumBufIndex=0;
	chrBufIndex=0;
	dstY=0;

	//precalculate horizontal scaler filter coefficients
	if(oldDstW!=dstW || oldSrcW!=srcW || oldFlags!=sws_flags)
	{
#ifdef HAVE_MMX
		const int filterAlign=4;
#else
		const int filterAlign=1;
#endif
		oldDstW= dstW; oldSrcW= srcW; oldFlags= sws_flags;

		RENAME(initFilter)(hLumFilter, hLumFilterPos, &hLumFilterSize, lumXInc,
				srcW   , dstW   , filterAlign, 1<<14);
		RENAME(initFilter)(hChrFilter, hChrFilterPos, &hChrFilterSize, chrXInc,
				(srcW+1)>>1, chrDstW, filterAlign, 1<<14);

#ifdef HAVE_MMX2
// cant downscale !!!
		if(canMMX2BeUsed && sws_flags == SWS_FAST_BILINEAR)
		{
			initMMX2HScaler(dstW   , lumXInc, funnyYCode);
			initMMX2HScaler(chrDstW, chrXInc, funnyUVCode);
		}
#endif
	} // Init Horizontal stuff

	if(oldDstH!=dstH || oldSrcH!=srcH || oldFlags!=sws_flags)
	{
		int i;
		oldDstH= dstH; oldSrcH= srcH; oldFlags= sws_flags; //FIXME swsflags conflict with x check

		// deallocate pixbufs
		for(i=0; i<vLumBufSize; i++) {
                    free(lumPixBuf[i]);
                }
		for(i=0; i<vChrBufSize; i++) free(chrPixBuf[i]);

		RENAME(initFilter)(vLumFilter, vLumFilterPos, &vLumFilterSize, lumYInc,
				srcH   , dstH,    1, (1<<12)-4);
		RENAME(initFilter)(vChrFilter, vChrFilterPos, &vChrFilterSize, chrYInc,
				(srcH+1)>>1, chrDstH, 1, (1<<12)-4);

		// Calculate Buffer Sizes so that they wont run out while handling these damn slices
		vLumBufSize= vLumFilterSize; vChrBufSize= vChrFilterSize;
		for(i=0; i<dstH; i++)
		{
			int chrI= i*chrDstH / dstH;
			int nextSlice= MAX(vLumFilterPos[i   ] + vLumFilterSize - 1,
					 ((vChrFilterPos[chrI] + vChrFilterSize - 1)<<1));
			nextSlice&= ~1; // Slices start at even boundaries
			if(vLumFilterPos[i   ] + vLumBufSize < nextSlice)
				vLumBufSize= nextSlice - vLumFilterPos[i   ];
			if(vChrFilterPos[chrI] + vChrBufSize < (nextSlice>>1))
				vChrBufSize= (nextSlice>>1) - vChrFilterPos[chrI];
		}

		// allocate pixbufs (we use dynamic allocation because otherwise we would need to
		// allocate several megabytes to handle all possible cases)
		for(i=0; i<vLumBufSize; i++) {
			lumPixBuf[i]= lumPixBuf[i+vLumBufSize]= (int16_t*)malloc(4000);
                        //printf("Malloced lumPixBuf[%i]\n",i);
                }
		for(i=0; i<vChrBufSize; i++)
			chrPixBuf[i]= chrPixBuf[i+vChrBufSize]= (int16_t*)malloc(8000);

		//try to avoid drawing green stuff between the right end and the stride end
		for(i=0; i<vLumBufSize; i++) memset(lumPixBuf[i], 0, 4000);
		for(i=0; i<vChrBufSize; i++) memset(chrPixBuf[i], 64, 8000);

		ASSERT(chrDstH<=dstH)
		ASSERT(vLumFilterSize*dstH*4<16000)
		ASSERT(vChrFilterSize*chrDstH*4<16000)
#ifdef HAVE_MMX
		// pack filter data for mmx code
		for(i=0; i<vLumFilterSize*dstH; i++)
			lumMmxFilter[4*i]=lumMmxFilter[4*i+1]=lumMmxFilter[4*i+2]=lumMmxFilter[4*i+3]=
				vLumFilter[i];
		for(i=0; i<vChrFilterSize*chrDstH; i++)
			chrMmxFilter[4*i]=chrMmxFilter[4*i+1]=chrMmxFilter[4*i+2]=chrMmxFilter[4*i+3]=
				vChrFilter[i];
#endif
	}

	if(firstTime && verbose)
	{
#ifdef HAVE_MMX2
		int mmx2=1;
#else
		int mmx2=0;
#endif
#ifdef HAVE_MMX
		int mmx=1;
#else
		int mmx=0;
#endif

#ifdef HAVE_MMX
		if(canMMX2BeUsed && sws_flags==SWS_FAST_BILINEAR)
			g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"SwScaler: using FAST_BILINEAR MMX2 scaler for horizontal scaling");
		else
		{
			if(hLumFilterSize==4)
				g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"SwScaler: using 4-tap MMX scaler for horizontal luminance scaling");
			else if(hLumFilterSize==8)
				g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"SwScaler: using 8-tap MMX scaler for horizontal luminance scaling");
			else
				g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"SwScaler: using n-tap MMX scaler for horizontal luminance scaling");

			if(hChrFilterSize==4)
				g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"SwScaler: using 4-tap MMX scaler for horizontal chrominance scaling");
			else if(hChrFilterSize==8)
				g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"SwScaler: using 8-tap MMX scaler for horizontal chrominance scaling");
			else
				g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"SwScaler: using n-tap MMX scaler for horizontal chrominance scaling");
		}
#elif defined (ARCH_X86)
		g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"SwScaler: using X86-Asm scaler for horizontal scaling");
#else
		if(sws_flags==SWS_FAST_BILINEAR)
			g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"SwScaler: using FAST_BILINEAR C scaler for horizontal scaling");
		else
			g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"SwScaler: using C scaler for horizontal scaling");
#endif

		if(dstbpp==12)
		{
			if(vLumFilterSize==1)
				g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"SwScaler: using 1-tap %s \"scaler\" for vertical scaling (YV12)", mmx ? "MMX" : "C");
			else
				g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"SwScaler: using n-tap %s scaler for vertical scaling (YV12)", mmx ? "MMX" : "C");
		}

		g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"SwScaler: %dx%d -> %dx%d", srcW, srcH, dstW, dstH);
	}

	lastInLumBuf= -1;
	lastInChrBuf= -1;
  } // if(firstLine)

	for(;dstY < dstH; dstY++){
		unsigned char *dest =dstptr[0]+dststride*dstY;
		unsigned char *uDest=dstptr[1]+(dststride>>1)*(dstY>>1);
		unsigned char *vDest=dstptr[2]+(dststride>>1)*(dstY>>1);
		const int chrDstY= dstbpp==12 ? (dstY>>1) : dstY;

		const int firstLumSrcY= vLumFilterPos[dstY]; //First line needed as input
		const int firstChrSrcY= vChrFilterPos[chrDstY]; //First line needed as input
		const int lastLumSrcY= firstLumSrcY + vLumFilterSize -1; // Last line needed as input
		const int lastChrSrcY= firstChrSrcY + vChrFilterSize -1; // Last line needed as input

		if(sws_flags == SWS_FAST_BILINEAR)
		{
			//handle holes
			if(firstLumSrcY > lastInLumBuf) lastInLumBuf= firstLumSrcY-1;
			if(firstChrSrcY > lastInChrBuf) lastInChrBuf= firstChrSrcY-1;
		}

		ASSERT(firstLumSrcY >= lastInLumBuf - vLumBufSize + 1)
		ASSERT(firstChrSrcY >= lastInChrBuf - vChrBufSize + 1)

		// Do we have enough lines in this slice to output the dstY line
		if(lastLumSrcY < srcSliceY + srcSliceH && lastChrSrcY < ((srcSliceY + srcSliceH)>>1))
		{
			//Do horizontal scaling
			while(lastInLumBuf < lastLumSrcY)
			{
				uint8_t *src= srcptr[0]+(lastInLumBuf + 1 - srcSliceY)*stride[0];
				lumBufIndex++;
				ASSERT(lumBufIndex < 2*vLumBufSize)
				ASSERT(lastInLumBuf + 1 - srcSliceY < srcSliceH)
				ASSERT(lastInLumBuf + 1 - srcSliceY >= 0)
//printf("1) lumPixBuf[ %i ] = %p\n", lumBufIndex,lumPixBuf[ lumBufIndex ]);
				RENAME(hyscale)(lumPixBuf[ lumBufIndex ], dstW, src, srcW, lumXInc);
				lastInLumBuf++;
			}
			while(lastInChrBuf < lastChrSrcY)
			{
				uint8_t *src1= srcptr[1]+(lastInChrBuf + 1 - (srcSliceY>>1))*stride[1];
				uint8_t *src2= srcptr[2]+(lastInChrBuf + 1 - (srcSliceY>>1))*stride[2];
				chrBufIndex++;
				ASSERT(chrBufIndex < 2*vChrBufSize)
				ASSERT(lastInChrBuf + 1 - (srcSliceY>>1) < (srcSliceH>>1))
				ASSERT(lastInChrBuf + 1 - (srcSliceY>>1) >= 0)
//printf("chrPixBuf[ chrBufIndex ] = %p\n",chrPixBuf[ chrBufIndex ]);
				RENAME(hcscale)(chrPixBuf[ chrBufIndex ], chrDstW, src1, src2, (srcW+1)>>1, chrXInc);
				lastInChrBuf++;
			}
			//wrap buf index around to stay inside the ring buffer
			if(lumBufIndex >= vLumBufSize ) lumBufIndex-= vLumBufSize;
			if(chrBufIndex >= vChrBufSize ) chrBufIndex-= vChrBufSize;
		}
		else // not enough lines left in this slice -> load the rest in the buffer
		{
			//Do horizontal scaling
			while(lastInLumBuf+1 < srcSliceY + srcSliceH)
			{
				uint8_t *src= srcptr[0]+(lastInLumBuf + 1 - srcSliceY)*stride[0];
				lumBufIndex++;
				ASSERT(lumBufIndex < 2*vLumBufSize)
				ASSERT(lastInLumBuf + 1 - srcSliceY < srcSliceH)
				ASSERT(lastInLumBuf + 1 - srcSliceY >= 0)
//printf("2) lumPixBuf[ lumBufIndex ] = %p\n");
				RENAME(hyscale)(lumPixBuf[ lumBufIndex ], dstW, src, srcW, lumXInc);
				lastInLumBuf++;
			}
			while(lastInChrBuf+1 < ((srcSliceY + srcSliceH)>>1))
			{
				uint8_t *src1= srcptr[1]+(lastInChrBuf + 1 - (srcSliceY>>1))*stride[1];
				uint8_t *src2= srcptr[2]+(lastInChrBuf + 1 - (srcSliceY>>1))*stride[2];
				chrBufIndex++;
				ASSERT(chrBufIndex < 2*vChrBufSize)
				ASSERT(lastInChrBuf + 1 - (srcSliceY>>1) < (srcSliceH>>1))
				ASSERT(lastInChrBuf + 1 - (srcSliceY>>1) >= 0)
				RENAME(hcscale)(chrPixBuf[ chrBufIndex ], chrDstW, src1, src2, (srcW+1)>>1, chrXInc);
				lastInChrBuf++;
			}
			//wrap buf index around to stay inside the ring buffer
			if(lumBufIndex >= vLumBufSize ) lumBufIndex-= vLumBufSize;
			if(chrBufIndex >= vChrBufSize ) chrBufIndex-= vChrBufSize;
			break; //we cant output a dstY line so lets try with the next slice
		}

#ifdef HAVE_MMX
		b5Dither= dither8[dstY&1];
		g6Dither= dither4[dstY&1];
		g5Dither= dither8[dstY&1];
		r5Dither= dither8[(dstY+1)&1];
#endif
	    if(dstY < dstH-2 || over<=0)
	    {
		if(dstbpp==12) //YV12
		{
			if(dstY&1) uDest=vDest= NULL; //FIXME split functions in lumi / chromi
			if(vLumFilterSize == 1 && vChrFilterSize == 1) // Unscaled YV12
			{
				int16_t *lumBuf = lumPixBuf[0];
				int16_t *chrBuf= chrPixBuf[0];
				RENAME(yuv2yuv1)(lumBuf, chrBuf, dest, uDest, vDest, dstW);
			}
			else //General YV12
			{
				int16_t **lumSrcPtr= lumPixBuf + lumBufIndex + firstLumSrcY - lastInLumBuf + vLumBufSize;
				int16_t **chrSrcPtr= chrPixBuf + chrBufIndex + firstChrSrcY - lastInChrBuf + vChrBufSize;
				RENAME(yuv2yuvX)(
					vLumFilter+dstY*vLumFilterSize     , lumSrcPtr, vLumFilterSize,
					vChrFilter+(dstY>>1)*vChrFilterSize, chrSrcPtr, vChrFilterSize,
					dest, uDest, vDest, dstW,
					lumMmxFilter+dstY*vLumFilterSize*4, chrMmxFilter+(dstY>>1)*vChrFilterSize*4);
			}
		}
		else
		{
			uint16_t **lumSrcPtr= (uint16_t**)(lumPixBuf + lumBufIndex + firstLumSrcY - lastInLumBuf + vLumBufSize);
			uint16_t **chrSrcPtr= (uint16_t**)(chrPixBuf + chrBufIndex + firstChrSrcY - lastInChrBuf + vChrBufSize);

			ASSERT(lumSrcPtr + vLumFilterSize - 1 < lumPixBuf + vLumBufSize*2);
			ASSERT(chrSrcPtr + vChrFilterSize - 1 < chrPixBuf + vChrBufSize*2);
		}
            }
	    else // hmm looks like we cant use MMX here without overwriting this arrays tail
	    {
		int16_t **lumSrcPtr= lumPixBuf + lumBufIndex + firstLumSrcY - lastInLumBuf + vLumBufSize;
		int16_t **chrSrcPtr= chrPixBuf + chrBufIndex + firstChrSrcY - lastInChrBuf + vChrBufSize;
		if(dstbpp==12) //YV12
		{
			if(dstY&1) uDest=vDest= NULL; //FIXME split functions in lumi / chromi
			yuv2yuvXinC(
				vLumFilter+dstY*vLumFilterSize     , lumSrcPtr, vLumFilterSize,
				vChrFilter+(dstY>>1)*vChrFilterSize, chrSrcPtr, vChrFilterSize,
				dest, uDest, vDest, dstW);
		}
	    }
	}

#ifdef HAVE_MMX
	__asm __volatile(SFENCE: : :"memory");
	__asm __volatile(EMMS: : :"memory");
#endif
	firstTime=0;
}
