/* Drip - a transcoder for Unix
 * Copyright (C) 2001-2003 Jarl van Katwijk
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "main.hh"
#include "encoder.hh"
#ifndef STANDALONE
#include "../src/external.h"
#endif
#include "plugin-loader.hh"
#include <signal.h>
#include <setjmp.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#include <ostream.h>
#include <fstream>
#include <strstream>
#include <cstdlib>
#include <cstdio>
#include <string.h>
#include <dirent.h>
#include <dlfcn.h>
#include <time.h>
#include <pthread.h>


using std::cout;
using std::cerr;
using std::endl;

gboolean encoding = FALSE;
gboolean stopped = FALSE;
gboolean encoding_paused = FALSE;
pthread_t serverthread;
gint serverthread_id;
pthread_t encoderthread;
gint encoderthread_id;
pthread_t guiclientthread;
gint guiclientthread_id = -1;
gint linenr = 0;
gint log_handler_encoder_id;
gboolean forking = TRUE;
sigjmp_buf environment; /* used for SIGSEGV handling */
gint log_file;


/* Functions */

#ifdef STANDALONE
#ifdef __cplusplus
extern "C" {
void log_handler(const gchar *log_domain,GLogLevelFlags log_level,const gchar *message,gpointer user_data) {
    time_t Time = time(NULL);
    GString *date;
    GString *level;
    GString *logstring;
    GString *displaystring;
    gboolean quit,display;
    gint bytes = 0;
    gchar *last_char;

    /* Lock if not logging from a CallBack function */
    if (strcmp(log_domain,DRIP_LD)==0) {
        #ifndef STANDALONE
        drip_lock("encoder_log_handler1");
        #endif
    }

    /* Handle log_level, strip flags */
    log_level = (GLogLevelFlags)(log_level & 252);
    quit = FALSE;
    display = FALSE;
    switch (log_level) {
        case G_LOG_LEVEL_ERROR:      level = g_string_new("Error");
                                     quit = TRUE;
                                     display = TRUE;
                                     break;
        case G_LOG_LEVEL_CRITICAL:   level = g_string_new("Critical");
                                     quit = TRUE;
                                     display = TRUE;
                                     break;
        case G_LOG_LEVEL_WARNING:    level = g_string_new("Warning");
                                     display = TRUE;
                                     break;
        case G_LOG_LEVEL_MESSAGE:    level = g_string_new("Message");
                                     display = TRUE;
                                     break;
        case G_LOG_LEVEL_INFO:       level = g_string_new("Info");
                                     break;
        case G_LOG_LEVEL_DEBUG:      level = g_string_new("Debug");
                                     break;
        default:                     level = g_string_new("Unknown");
                                     break;
    }

    /* Text formatting */
    while (level->len < 8) {
        level =  g_string_append(level," ");
    }

    /* Format date string */
    date = g_string_new(ctime(&Time));
    date = g_string_truncate(date,(date->len)-1);

    /* Store log entry */
    logstring = g_string_new("");
    g_string_sprintf(logstring,"%s\t| %i\t| %s | %s\n",date->str,linenr,level->str,message);
    displaystring = g_string_new("");
    g_string_sprintf(displaystring,"%s | %s\n",date->str,message);
    last_char = (gchar*)((glong)logstring->str + logstring->len - 2); /* we got 2x \n? */
    if (*last_char == 10) {
        logstring = g_string_truncate(logstring,logstring->len -1);
    }

    /* Print on STDOUT */
    if (display) fprintf(stderr,"%s",displaystring->str);

    /* Write to log file */
    bytes = write(log_file,logstring->str,logstring->len); /* write log entry to log file */

    /* Should we exit? */
    if (quit) {
       g_print("sleeping...");
       sleep(10);
       g_print(" done\n");
       exit(-1);
    }

    /* Clean & Exit */
    linenr++;
    g_string_free(date,TRUE);
    g_string_free(level,TRUE);
    g_string_free(logstring,TRUE);
    g_string_free(displaystring,TRUE);
    /* Lock if not logging from a CallBack function */
    if (strcmp(log_domain,DRIP_LD)==0) {
        #ifndef STANDALONE
        drip_unlock("encoder_log_handler1");
        #endif
    }
    return;
}
}
#endif
#endif

/* Handles signals */
void signal_handler(int sig) {

#ifndef STANDALONE
    gui_errorsignal(sig);
#else
    g_print("Got an error (%i), please report details of this bug",sig);
#endif

    if (sig == SIGSEGV) {
        /* Restore situation as it was before the segvault signal */
        siglongjmp(environment,1);
        printf("After SIGSEGV catching longjmp? Something is very wrong here...\n");
        fflush(0);
        signal(SIGSEGV,SIG_DFL); /* Reset segv signal handler, something has gone very wrong */
    }

    fflush(0);
    exit(0);
    return;
}

/* Hooks signals to internal handler */
void signal_hook(void) {

    /* Save environment */
    sigsetjmp(environment,1);

    if (sigsetjmp(environment,1) == 0) {
        /* Hook signals */
        signal(SIGTERM,signal_handler);
        signal(SIGHUP,signal_handler);
        signal(SIGABRT,signal_handler);
        signal(SIGTERM,signal_handler);
        //signal(,signal_handler);
        //signal(,signal_handler);
        //signal(,signal_handler);
        //signal(,signal_handler);
        //signal(SIGSEGV,signal_handler);
        //signal(SIGINT,signal_handler);
    } else {
        /* Got a SIGSEGV inside handler? */
        signal(SIGSEGV,SIG_DFL);
    }

    return;
}

int main(int argc, char** argv) {
    arg_parmS params;
    struct stat file_stats;
    GString *logfile = g_string_new("");
    GString *logfile_old = g_string_new("");
    GString *command = g_string_new("");
    GString *HOME = g_string_new("");
    GString *configfile;
    GString *status_action;

#ifdef STANDALONE
    HOME = g_string_new(getenv("HOME"));
    /* Initialize all drip defaults */
    g_string_sprintf(command,"%s/.drip",HOME->str);
    mkdir (command->str, S_IRUSR | S_IWUSR | S_IXUSR);
    g_log(DRIP_LD,G_LOG_LEVEL_INFO,"Created %s",command->str);
    g_string_sprintf(command,"%s/.drip/logs_old",HOME->str);
    mkdir (command->str, S_IRUSR | S_IWUSR | S_IXUSR);
    g_string_sprintf(command,"%s/.drip/plugins",HOME->str);
    mkdir (command->str, S_IRUSR | S_IWUSR | S_IXUSR);

    status_action = g_string_new("");
    configfile = g_string_new("");
    g_string_sprintf(configfile,"%s/.drip/drip.cfg",getenv("HOME"));

    /* Init logging */
    g_string_sprintf(logfile,"%s/.drip/drip.log",getenv("HOME"));
    if (stat(logfile->str,&file_stats)!=-1) {
        g_string_sprintf(logfile_old,"%s/.drip/logs_old/drip.%li.log",getenv("HOME"),file_stats.st_ctime);
        rename(logfile->str,logfile_old->str);
    } else {
        g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Could not stat log file, removed it");
        unlink(logfile->str); /* delete old log file, TODO: log file rotation */
    }
    log_file = open(logfile->str,O_WRONLY|O_CREAT,S_IREAD|S_IWRITE); /* create log file */
    gint log_handler_drip_id = g_log_set_handler (DRIP_LD, (GLogLevelFlags)(G_LOG_LEVEL_WARNING |G_LOG_LEVEL_MESSAGE | G_LOG_LEVEL_INFO | G_LOG_LEVEL_DEBUG | G_LOG_LEVEL_CRITICAL | G_LOG_LEVEL_ERROR | G_LOG_FLAG_FATAL | G_LOG_FLAG_RECURSION),log_handler, NULL);
    gint log_handler_dripcb_id = g_log_set_handler (DRIP_CB_LD, (GLogLevelFlags)(G_LOG_LEVEL_WARNING |G_LOG_LEVEL_MESSAGE | G_LOG_LEVEL_INFO | G_LOG_LEVEL_DEBUG | G_LOG_LEVEL_CRITICAL | G_LOG_LEVEL_ERROR | G_LOG_FLAG_FATAL | G_LOG_FLAG_RECURSION),log_handler, NULL);
#endif

    g_log(DRIP_LD,G_LOG_LEVEL_MESSAGE,APPNAME " " VERSION " by Jarl van Katwijk <jarl@xs4all.nl>");
    g_log(DRIP_LD,G_LOG_LEVEL_INFO,"Based on work by many others, see AUTHORS file");

    /* Check libavi version */
    if (GetAvifileVersion()!=AVIFILE_VERSION) {
        g_log(DRIP_LD,G_LOG_LEVEL_ERROR,"This binary was compiled for Avifile ver. %s, but the library is ver. %s. Aborting.",AVIFILE_VERSION,GetAvifileVersion());
    }
    g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Started dripencoder, pid %i",getpid());

    /* Init threading */
    g_thread_init(NULL);

    /* Load plugins */
    #ifdef STANDALONE
    if (init_plugins() == FALSE) {
        //g_log(DRIP_LD,G_LOG_LEVEL_ERROR,"Plugins not loaded");
    }
    #endif

    /* Pass params */
    params.argc = argc;
    params.argv = argv;
    encode((gpointer)&params);

    /* Clean & exit */
    g_log_remove_handler(DRIP_LD,log_handler_drip_id);
    #ifndef STANDALONE
    drip_lock("encoder_main1");
    #endif
    close(log_file);
    g_string_free(logfile,TRUE);
    g_string_free(logfile_old,TRUE);
    g_string_free(command,TRUE);
    g_string_free(HOME,TRUE);
    g_string_free(configfile,TRUE);
    g_string_free(status_action,TRUE);
    #ifndef STANDALONE
    drip_unlock("encoder_main1");
    #endif
    exit(0); // ?
    return(0);
}

