
import java.io.*;
import java.util.ResourceBundle;
import java.text.MessageFormat;

/** 
 * Class responsible for logging.
 * <p>This is responsible for storing the log of index generation. 
 * When the system is being used with a gui, it will also be responsible for
 * sending progress reports to the gui.
 * <br>
 * Meaning of priorities:
 * <ul>
 * 	<li>1	Fatal error
 * 	<li>2	Serious error, e.g. file access error.
 * 	<li>3	less serious error, e.g. data not found in file
 * 	<li>4	start / end of major section
 * 	<li>5	progress reports
 * 	<li>6	Start processing file
 * 	<li>7	Results from various processing sections
 * 	<li>8	Debugging code
 * 	<li>9	Start of procedures. 
 * </ul>
 * 
 * <p>Each file should have it's own log level setting for debugging purposes. 
 * For production releases, this should be set at 5 in all files.
 * 
 * <BR><BR>$Id: IGLog.java,v 1.23 2002/09/24 14:32:49 howama Exp $
 */
class IGLog{
	/**
	 * This is the version of the current release. It
	 * will be changed every time a public release is made. Any files requiring
	 * the version string should look here (e.g. for the version placeholder). 
	 *
	 * TODO: Remember to change this *before* making a release. Everybody -
	 * please send lots of emails at release time. (I forgot for 0.1)
	 */
	public static final String INDEXGEN_VERSION = "0.2cvs";

	public static UserInterface ui;
	
	/** An error */
	public static final int ERROR_NEW = -1;
	/** A warning */
	public static final int WARNING_NEW = -2;

	
	/** Gives important information which will be useful for debugging. 
	 * e.g. which method is being used to do something
	 */
	public static final int INFO = 6;

    /** Indicate start/end of a major section */
    public static final int SECTION = 4;

    /** Give a progress report */
    public static final int PROGRESS = 5;

    /** Indicate a file access */
    public static final int FILE = 6;

    /** Provide results from processing sections  */
    public static final int RESULT = 7;

    /** Show debugging code  */
    public static final int DEBUG = 8;

    /** Indicate the start of a procedure */
    public static final int PROCEDURE = 9;
	
	/** Version strings, useful for debugging.  */
	public static final int VERSION = 5;

	private static final int LOGLEVEL = 9;

	/** */
	private BufferedWriter buffer;
	/** */
	private FileWriter outputStream;
	/** */
	private static ResourceBundle rb;
	/** Log filename (including path) */
	private static String logFile;
	
	/** 
	 * Returns the name of the log file being used.
	 */
	public static String getLogName(){
		return logFile;
	}

	/** 
	 * Creates the output stream, ready for writing the log to file
	 * @param filename the output file, including full path.
	 */
	IGLog(String filename, ResourceBundle rb, UserInterface ui) throws IOException{
		logFile = filename;
		this.ui = ui;
		this.rb = rb;
		outputStream = new FileWriter(filename);
		buffer = new BufferedWriter( outputStream );
		
		add(IGLog.VERSION, "Java specs version: " + System.getProperty("java.specification.version") );
		add(IGLog.VERSION, "Java specs vendor: " + System.getProperty("java.specification.vendor") );
		add(IGLog.VERSION, "Java specs name: " + System.getProperty("java.specification.name") );
		add(IGLog.VERSION, "Java version: " + System.getProperty("java.version") );
		add(IGLog.VERSION, "Java vendor: " + System.getProperty("java.vendor") );
		add(IGLog.VERSION, "Java class path: " + System.getProperty("java.class.path") );
		add(IGLog.VERSION, "OS name: " + System.getProperty("os.name") );
		add(IGLog.VERSION, "OS version: " + System.getProperty("os.version") );
		add(IGLog.VERSION, "OS architecture: " + System.getProperty("os.arch") );
	}

	/**
	 * Adds an error message to the log. Errors should occur very infrequently.
	 * they indicate that something has gone wrong with the system and so the
	 * output may not be as expected. In most cases, you will want to use 
	 * {@link #addWarning(int, String, Object[])} instead..
	 * @param errorNo Error number. This will be used for cross reference with
	 * help documentation. Each call to addError should have a different
	 * errorNo.
	 * @param resourceID ID of the resource in MainStrings.java 
	 * @param params Parameters for merging into the resource ID. Additional
	 * parameters may also be added to be used by the userInterface.
	 */
	protected void addError(int errorNo, String resourceID, Object[] params){
		String msg = formatResource( resourceID, params);
		add(ERROR_NEW, errorNo + " " + msg);
		ui.error(msg, errorNo, resourceID, params);
	}
	
	/**
	 * Adds a warning message to the log. 
	 * @param errorNo Error number. This will be used for cross reference with
	 * help documentation. Each call to addError should have a different
	 * errorNo.
	 * @param resourceID ID of the resource in MainStrings.java 
	 * @param params Parameters for merging into the resource ID. Additional
	 * parameters may also be added to be used by the userInterface.
	 */
	protected void addWarning(int errorNo, String resourceID, Object[] params){
		String msg = formatResource( resourceID, params);
		add(WARNING_NEW, errorNo + " " + msg);
		ui.warning(msg, errorNo, resourceID, params);
	}
			
	
	/**
	 * Add an item to the log
	 * @param priority	Priorities are defined in the header for this class.	
	 */
	protected void add(int priority, String message){
		try{
			buffer.write( System.currentTimeMillis() + ":" + priority+':'+message);		
			buffer.newLine();
		}catch(IOException e){
			System.out.println("*** ERROR: LogAdd IOException: "+e);
			System.out.println("\tAttempted log line: "+priority+':'+message);
		}
	}

	protected static String formatResource(String resourceID, Object[] params){
		String msg = rb.getString(resourceID);
		if (params == null || params.length <= 0){
			return msg;
		}else{
			MessageFormat formatter = new MessageFormat("");
			formatter.setLocale( rb.getLocale() );
			formatter.applyPattern( msg );
			return formatter.format(params);
		}

	}
	
	/**
	 * Adds an item to the log based on l10n resource, without replacement
	 * patterns
	 */
	protected void addResource(int priority, String resourceID){
		addResource(priority, resourceID, null);
	}

	/**
	 * Add an item to the log based on l10n resource
	 */
	protected void addResource(int priority, String resourceID, Object[] params){
		add(priority, formatResource(resourceID, params));
	}

	/**
	 * Saves the file and closes the output stream. Useful to put in try/finally
	 * clauses.
	 */
	protected void close() throws IOException{
		buffer.flush();
		buffer.close();
		outputStream.close();
	}


    /**
     * Retrieve a localized string from the MainStrings resource bundle
     * @param resourceID The ID for the desired string
     * @return The localized string from the log's resource bundle
     */
    protected String getString(String resourceID) {
	return rb.getString(resourceID);
    }
}

/*
 * $Log: IGLog.java,v $
 * Revision 1.23  2002/09/24 14:32:49  howama
 * various changes, bugfixes, etc
 *
 * Revision 1.22  2002/09/23 10:37:27  howama
 * various changes
 *
 * Revision 1.21  2002/09/20 18:11:09  howama
 * bug fixes, more gui work, random changes...
 *
 * Revision 1.20  2002/09/11 13:22:25  howama
 * bug fixes
 *
 * Revision 1.19  2002/09/06 11:44:04  howama
 * merged command line options by Eugene <team_pro@gala.net>.
 * More ui work
 * Changed log position
 *
 * Revision 1.18  2002/09/05 14:01:40  howama
 * new log methods for erros
 *
 * Revision 1.17  2002/09/04 20:08:48  howama
 * user interface work
 *
 * Revision 1.16  2002/08/26 13:56:26  howama
 * More work & bug fixes
 *
 * Revision 1.15  2002/08/22 13:42:38  howama
 * more work
 *
 * Revision 1.14  2002/08/21 16:35:35  blsecres
 * Converted log messages to method using resource bundles for localization.
 *
 * Revision 1.13  2002/08/21 15:13:12  howama
 * more work
 *
 * Revision 1.12  2002/08/20 13:22:25  howama
 * added l10n add method
 *
 * Revision 1.11  2002/08/20 09:18:20  howama
 * various
 *
 * Revision 1.10  2002/08/09 20:39:26  blsecres
 * *** empty log message ***
 *
 * Revision 1.9  2002/08/09 18:14:33  howama
 * log items
 *
 * Revision 1.8  2002/08/09 17:43:11  howama
 * removed unused RE import
 *
 * Revision 1.7  2002/08/09 00:53:42  blsecres
 * Modified Settings interface so no localized strings are used outside the
 * settings file parser.
 * Added logging facilities to the Settings and Project interfaces.
 * Converted Vectors to String arrays.
 * Fixed bugs encountered when trying to load the settings file's locale.
 *
 * Revision 1.6  2002/08/08 17:24:42  howama
 * updates
 *
 * Revision 1.5  2002/07/24 09:24:21  howama
 * updated javadoc docs
 *
 * Revision 1.4  2002/07/12 14:07:31  howama
 * updates
 *
 * Revision 1.3  2002/07/08 16:20:40  howama
 * updates
 *
 * Revision 1.2  2002/07/05 13:05:23  howama
 * more work
 *
 * Revision 1.1  2002/07/05 09:44:11  howama
 * Initial conversion to java.
 *
 */
