/*
 *  dmachinemon / a distributed machine monitor by dancer.
 *  Copyright (C) 2001 Junichi Uekawa
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * structure tree viewer: gives you a clue on tree structure of dmachinemon
 */
/* 
 * 2001 Sep 12
 * $Id: treeview-3ddotout.c,v 1.11 2002/12/01 13:27:24 dancer Exp $
 *
 */


#define _GNU_SOURCE
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <locale.h>
#include <pthread.h>
#include <math.h>

#define GRAPHICCENTER 50.0
#define PI 3.141592
#include "dmachinemon/dmachinemon.h"
#include "config.h"

#define PROGRAMNAME "treeview-3ddotout"

char * hostname ;
char * portnumber;

static void * 
malloc_with_error (int size)
{
  void * m = malloc (size);
  if (!m )
    {
      fprintf (stderr, PROGRAMNAME ": out of memory\n");
      exit (EXIT_FAILURE);
    }
  return m;
}

typedef struct machinetree
{
  char * myname;
  struct machinetree * nextlink;
  struct machinetree * downlink;
} machinetree;

/* getcar */
static char *
getcar (const char * ch)
{
  char * b = ch?strrchr(ch, ','):NULL;
  if (b) b++;
  return b?strdup(b):(ch?strdup(ch):NULL);
}

/* getcdr */
static char*
getcdr (const char * ch)
{
  char * s = ch?strdup(ch):NULL;
  char * b = s?strrchr (s, ','):NULL;
  if (b) 
    *b = 0;
  else 
    {
      free (s);
      s=NULL;
    }
  return s;
}

static void 
processinternal (machinetree * t, char * data)
{
  machinetree * mydata ;
  int found;
  char * buf = getcar (data);
  if (!data)
    return;

  found = 0;
  mydata = t -> downlink;
  while (mydata)
    {
      if (!strcmp(mydata -> myname, buf))
	{
	  found = 1 ;      
	  break;
	}
      mydata = mydata -> nextlink;
    }
  if (!found)
    {
      mydata = malloc(sizeof (machinetree));
      mydata -> downlink = NULL;
      mydata -> myname = buf;
      mydata -> nextlink = t -> downlink;
      t->downlink = mydata;
    }
  else
    free (buf);
  processinternal(mydata, getcdr(data));
  free (data);
}

static machinetree * 
build_local_route_tree (void)
{
  dm_machinelist * new = NULL;
  machinetree * treeptr = NULL;

  /* creating the tree */
  new = maininfo.machinedb;
  treeptr = malloc_with_error (sizeof(machinetree));
  treeptr -> myname = NULL;
  treeptr -> nextlink = treeptr -> downlink = NULL;

  while(new)
    {
      const char * buf;
      buf = dm_get_value_text(new, "Seen-By");
      if (buf)
	{
	  if (!treeptr->myname)
	    treeptr->myname = getcar (buf);
	  if (strchr (buf, ','))
	    processinternal (treeptr,getcdr(buf));
	}
      new = new -> next;
    }  
  return treeptr;  
}


static void access_master(dm_commandoption * cdat)
{
  int fd = DNAS_connect_to_host(cdat, hostname, atoi(portnumber),1);
  FILE* f=(fd!=-1)?fdopen (fd, "r"):NULL;
  
  if (!f)
    {
      fprintf (stderr, PROGRAMNAME ": cannot open server socket\n");
      exit(1);
    }

  if (dm_process_nodes_input (f, &maininfo, NULL, NULL))
    {
      fprintf (stderr, PROGRAMNAME ": could not handle input...\n");
      exit (1);
    }

  fclose(f);
}


/* elle is the allowed space. */
static void
display_local_route_tree(machinetree* m, 
			 const float x, const float y, const float z, float A, 
			 const int have_parent_flag, 
			 float parentx, float parenty, float parentz)
{
  int numbercount = 0 ;
  const int yoffset = 2.0;
  machinetree * mym = m;
  float r ;
  float R ;
  int i = 0;
  
  while (mym)
    {
      numbercount ++;
      mym=mym->nextlink;
    }
  if (!numbercount)
    {
      fprintf (stderr, "WARNING\n");
      exit (1);
    }
  

  
  R = 0.5 * (A / (sin ( PI / numbercount) + 1.0 )) ;
  if ( numbercount == 1)
    {
      r = R;
    }
  else if (numbercount == 2)
    {
      r = R / 2.0;
    }
  else 
    {
      r = R * cos ( PI / numbercount);
    }
  
  fprintf (stderr, "A=%f R=%f r=%f \n ", A, R, r);

  while (m)
    {
      if (!have_parent_flag)
	{
	  parentx=x;
	  parenty=y;
	}

      printf ("%f %f %f %s %f %f %f\n", 
	      x + R * sin ( PI * 2.0 * ((float)i) / ((float)numbercount)), y, 
	      z +                      R * cos ( PI * 2.0 * ((float)i) / ((float)numbercount)) - 30.0,
	      m->myname, 
	      parentx , parenty, parentz - 30.0);

      if (m->downlink)
	{
	  display_local_route_tree(m->downlink, 
				   x + R * sin ( PI * 2.0 * ((float)i) / ((float)numbercount)) , 
				   y - yoffset, 
				   z + R * cos ( PI * 2.0 * ((float)i) / ((float)numbercount)), 
				   r, 1, 
				   x + R * sin ( PI * 2.0 * ((float)i) / ((float)numbercount)), 
				   y,
				   z + R * cos ( PI * 2.0 * ((float)i) / ((float)numbercount)) );
	}
      i++;
      
      m=m->nextlink;
    }  
}

static void
processing_command (dm_commandoption * cdat)
{
  machinetree * m;
  
  access_master(cdat);
  m = build_local_route_tree();
  display_local_route_tree(m, 0, 0, 0, GRAPHICCENTER, 0, 0, 0, 0);
  return ;  
}

int main(int ac, char ** av)
{ 
  dm_commandoption cdat;
  setlocale(LC_TIME, "");

  dmachinemon_parse_options(ac, av, &cdat);
  if (!cdat.parenthostname)
    {
      fprintf (stderr, 
	       "treeview-3ddotout v. %s \n\n"
	       "Not enough command-line information specified\n\n"
	       ,VERSION
	       );
      dmachinemon_print_commandline_help();
      return 1;
    }

  hostname=cdat.parenthostname;
  portnumber=cdat.port_client;
  processing_command(&cdat);  
  return 0;
}
