/*
 *  dmachinemon / a distributed machine monitor by dancer.
 *  Copyright (C) 2001 Junichi Uekawa
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * command-line parser
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <getopt.h>
#include "dmachinemon/dmachinemon-commandlineparser.h"

/**
   Print some common command-line information
 */
void dmachinemon_print_commandline_help (void)
{
  printf(
	 "-c   --clientport               Specify the client port\n"
	 "-n   --nodeport                 Specify the node port\n"
	 "-p   --parent                   Specify the parent host name\n"
	 "-s   --sleep                    Specify the interval to sleep between actions\n"
	 "-m   --maxreconnect             Number of times it will try to connect to uplink before giving up\n"
	 "-l   --layers                   Number of layers it allows for message to pass (Message TTL)\n"
	 "-d   --dieonload                Die on load average of specified value\n"
	 "-D   --Downlinknum              Number of downlinks to have for each servent\n"
	 "-h   --help                     Give out this message\n"
	 "-L d --static-link d            Use static link for downlink\n"
	 "-L u --static-link u            Use static link for uplink\n"
 );
}

static void 
initialize_cdat (dm_commandoption * cdat)
{
  cdat->port_client = "3007" ;
  cdat->port_nodes = "3008" ;
  cdat->parenthostname = NULL ;
  cdat->sleeptime = 15 ;
  cdat->max_num_reconnect = 3;
  cdat->number_of_hosts_of_seenby = 0; /* default is none, or no checking. */
  cdat->dieonload_flag= 0;
  cdat->dieonload=0.0;
  cdat->downlinknum_threshold=4;
  cdat->downlink_static=0;
  cdat->uplink_static=0;
}

/**
   Parse command-line options for dmachinemon
 */
void
dmachinemon_parse_options (int ac /// argc from main.
			   , char ** av, /// argv from main. 
			   dm_commandoption * cdat /// pointer to the option
			   )
{
  int index_point;  
  int c;			/* option */
  
  static struct option long_options[]=
    {
      {"clientport", required_argument, 0, 'c'}, 
      {"nodeport", required_argument, 0, 'n'}, 
      {"parent", required_argument, 0, 'p'}, 
      {"sleep", required_argument, 0, 's'}, 
      {"layers", required_argument, 0, 'l'}, 
      {"maxreconnect", required_argument, 0, 'm'}, 
      {"dieonload", required_argument, 0, 'd'}, 
      {"Downlinknum", required_argument, 0, 'D'}, 
      {"static-link", required_argument, 0, 'L'}, 
      {"help", no_argument, 0, 'h'}, 
      {0,0,0,0}
    };
  
  initialize_cdat (cdat);
  
  while((c = getopt_long (ac, av, "c:n:p:s:hl:m:d:", long_options, &index_point)) != -1)
    {
      switch (c)
	{
	case 'c':		/* client port */
	  cdat->port_client = strdup(optarg);
	  break;
	case 'n':		/* nodes port */
	  cdat->port_nodes = strdup(optarg);
	  break;
	case 'p':		/* parent hostname */
	  cdat->parenthostname = strdup(optarg);
	  break;
	case 's':		/* sleep time */
	  cdat->sleeptime = atoi(optarg);
	  break;
	case 'l':		/* relaying layer number */
	  cdat->number_of_hosts_of_seenby = atoi(optarg);
	  break;
	case 'm':		/* max number of reconnects */
	  cdat->max_num_reconnect = atoi(optarg);
	  break;
	case 'd':		/* die on load */
	  cdat->dieonload_flag = 1;
	  cdat->dieonload = atof(optarg);
	  break;
	case 'D':		/* Number of downlinks */
	  cdat->downlinknum_threshold = atoi(optarg);
	  break;
	case 'L':		/* static links for */
	  switch (*optarg)
	    {
	    case 'u': case 'U':
	      cdat -> uplink_static = 1;
	      break;
	    case 'd': case 'D':
	      cdat -> downlink_static = 1;
	      break;
	    default:
	      dmachinemon_print_commandline_help();
	      exit (1);
	    }
	  break;
	case 'h':
	  dmachinemon_print_commandline_help();
	  exit(1);
	  break;	  
	default:
	  dmachinemon_print_commandline_help();
	  exit(1);
	  break;	  
	}
    }
}


