#include "mp3.h"
#ifdef HAVE_LIBMAD
#include <unistd.h>
#include <pthread.h>
#include <stdlib.h>
#include <stdio.h>
#include <fcntl.h>
#include <qthread.h>
#include "mainwnd.h"
#include "pump.h"
#ifdef HAVE_LIBID3
#undef HAVE_CONFIG_H
#include <id3/tag.h>
#endif

#define BLOCK_SKIP 1152

static inline signed int scale(mad_fixed_t sample)
{
  /* round */
  sample += (1L << (MAD_F_FRACBITS - 16));

  /* clip */
  if (sample >= MAD_F_ONE)
    sample = MAD_F_ONE - 1;
  else if (sample < -MAD_F_ONE)
    sample = -MAD_F_ONE;

  /* quantize */
  return sample >> (MAD_F_FRACBITS + 1 - 16);
}

void Mp3StreamSource::render_frame(int size)
{
    int i;
	char *buf=pcm_buf;

    for(i=0;i<sy.pcm.length;i++)
    {
        int sample;

        sample=scale((int)sy.pcm.samples[0][i]);
        (*(buf++))=sample & 0xff;
        (*(buf++))=(sample >> 8) & 0xff;
	}
	if(sy.pcm.channels == 2)
	{
		buf=pcm_buf+2*size;
		for(i=0;i<sy.pcm.length;i++)
		{
			int sample;

			sample=scale((int)sy.pcm.samples[1][i]);
			(*(buf++))=sample & 0xff;
			(*(buf++))=(sample >> 8) & 0xff;
		}
	}
}

Mp3StreamSource::Mp3StreamSource(QWidget *dest) : StreamSource(dest)
{
	wave_source=0;
	fd=-1;
	file_pos=0L;
	external_map=0;
	map=new Mp3Map;
	frame_buffer=new unsigned char[8000];
	init=false;
}

Mp3StreamSource::~Mp3StreamSource()
{
	if(fd >= 0)
		close(fd);
	if(!external_map)
		delete map;
	delete frame_buffer;
	if(init)
	{
		mad_synth_finish(&sy);
		mad_frame_finish(&fr);
		mad_stream_finish(&st);
	}
}

int Mp3StreamSource::attach(QString file)
{
	if(access(file, 0) < 0)
		return -1;
	
	filename=file;

	if(!external_map)
	{
		delete map;
		map=0;
	}
	fd=open((char *)(const char *)file, O_RDONLY);
	if(fd < 0)
		return -1;
	if(!map)
	{
		map=new Mp3Map;
		if(map->attach(fd) < 0)
		{
			close(fd);
			return -1;
		}
		m_artist="";
		m_title="";
#ifdef HAVE_LIBID3
		/* This is a primary open (no map), get ID3 tags */
		const char *s_artist=0;
		const char *s_title=0;
		ID3_Tag tag(file);
		ID3_Frame *artist=tag.Find(ID3FID_LEADARTIST);
		ID3_Frame *title=tag.Find(ID3FID_TITLE);
		if(artist)
		{
			ID3_Field *f_artist=0;
			f_artist=artist->GetField(ID3FN_TEXT);
			if(f_artist)
				s_artist=f_artist->GetRawText();
		}
		if(title)
		{
			ID3_Field *f_title=0;
			f_title=title->GetField(ID3FN_TEXT);
			if(f_title)
				s_title=f_title->GetRawText();
		}
		if(s_artist)
			m_artist=s_artist;
		if(s_title)
			m_title=s_title;
#endif
		if(m_artist == "" && m_title == "")
		{
			char fnb[1024];
			strcpy(fnb, file);
			char *fn=strrchr(fnb, '/');
			if(!fn)
				fn=fnb;
			else
				fn++;
			char *dot=strrchr(fn, '.');
			if(dot)
				(*dot)='\0';
			char *sep=strstr(fn, " - ");
			if(sep)
			{
				(*sep)='\0';
				sep+=3;
			}
			if(sep)
			{
				m_artist=fn;
				m_title=sep;
			}
			else
			{
				m_artist="";
				m_title=fn;
			}
		}
	}
	sample_rate=map->samplerate;
	if(map->frames == 0)
	{
		close(fd);
		return -1;
	}
	init=1;
	mad_frame_init(&fr);
	mad_synth_init(&sy);
	mad_stream_init(&st);

	max_frames=map->samples;

	return 0;
}

int Mp3StreamSource::get_buffer(char *buf, int max)
{
	if(fd < 0 || !playing)
		return 0;
	
	pthread_mutex_lock(&control_lock);
	if(file_pos+max > max_frames)
		max=max_frames-file_pos;

	if(!max)
	{
		if(playing)
			QThread::postEvent(this, new QEvent(QEvent::User));
		playing=0;
		pthread_mutex_unlock(&control_lock);
		return 0;
	}

	if(play_samples)
	{
		if((unsigned long)max > play_samples)
			max=(int)play_samples;
	}

	read_synth_data(buf, max);

	file_pos+=max;

	if(play_samples)
	{
		play_samples-=max;
		if(play_samples <= 0)
		{
			if(loop_play)
				play(last_start, last_samples, true);
			else
			{
				playing=0;
				QThread::postEvent(this, new QEvent(QEvent::User));
			}
		}
	}
	
	pthread_mutex_unlock(&control_lock);
	calcAgc(buf, max);
	return max;
}

void Mp3StreamSource::exit_loop(void)
{
	pthread_mutex_lock(&control_lock);
	play_samples=0;
	loop_play=false;
	pthread_mutex_unlock(&control_lock);
}

int Mp3StreamSource::play(unsigned long start_frame, unsigned long samples, bool loop)
{
	if(fd < 0)
	{
		printf("No file\n");
		return -1;
	}
	
	if(!loop_play || !loop)
		stop();

	if(start_frame >= max_frames)
	{
		printf("No more frames\n");
		return -1;
	}

	loop_play=loop;
	last_start=start_frame;

	// Seek
	int frame=start_frame/map->framesize;
	frame-=3;
	if(frame < 0)
		frame=0;
	first_frame=frame;
	frames_buffered=8;
	if(first_frame+frames_buffered >= map->frames)
		frames_buffered=map->frames-first_frame;
	
	lseek(fd, map->frame_to_offset(first_frame), SEEK_SET);
	read(fd, frame_buffer, 8000);
	pcm_pos=map->framesize*first_frame;
	sample_offset=0;

	mad_synth_finish(&sy);
	mad_frame_finish(&fr);
	mad_stream_finish(&st);
	mad_stream_init(&st);
	mad_stream_buffer(&st, frame_buffer, 8000);
	mad_frame_init(&fr);
	mad_synth_init(&sy);
	while(pcm_pos+map->framesize < (int)start_frame)
	{
		if(mad_header_decode(&fr.header, &st) < 0)
		{
			printf("Can't decode header\n");
			return -1;
		}
		if(mad_frame_decode(&fr, &st) < 0)
		{
			mad_synth_finish(&sy);
			mad_synth_init(&sy);
		}
		mad_synth_frame(&sy, &fr);
		pcm_pos+=map->framesize;
	}

	if(mad_header_decode(&fr.header, &st) < 0)
	{
		printf("Can't decode header 2\n");
		return -1;
	}
	if(mad_frame_decode(&fr, &st) < 0)
	{
		printf("Can't decode frame 2\n");
		return -1;
	}
	
	mad_synth_frame(&sy, &fr);
	render_frame(map->framesize);

	sample_offset=start_frame%map->framesize;

	file_pos=start_frame;
	frame_pos=start_frame;
	play_samples=samples;
	last_samples=play_samples;

	playing=1;
	app_window->get_pump()->work();
	return 0;
}

int Mp3StreamSource::stop(void)
{
	if(!player || fd < 0 || !playing)
		return 0;

	loop_play=false;
	StreamSource::stop();
	playing=0;
	return 0;
}

void Mp3StreamSource::read_synth_data(char *buf, int max)
{
	int done_samples=0;

	memset(buf, 0, max*2*sizeof(short));

	while(done_samples < max)
	{
		int copy_samples=max-done_samples;
		if(copy_samples > map->framesize-sample_offset)
			copy_samples=map->framesize-sample_offset;

		if(!copy_samples)
		{
			int frame=pcm_pos/map->framesize+1;
			if(frame >= first_frame+frames_buffered)
			{
				lseek(fd, map->frame_to_offset(frame), SEEK_SET);
				read(fd, frame_buffer, 8000);
				first_frame=frame;
				frames_buffered=8;
				mad_stream_buffer(&st, frame_buffer, 8000);
			}
			mad_header_decode(&fr.header, &st);
			mad_frame_decode(&fr, &st);
			mad_synth_frame(&sy, &fr);
			render_frame(map->framesize);
			pcm_pos+=map->framesize;
			sample_offset=0;
			continue;
		}

		memcpy(buf+sizeof(short)*done_samples, pcm_buf+sample_offset*sizeof(short), copy_samples*sizeof(short));
		memcpy(buf+sizeof(short)*done_samples+sizeof(short)*max, pcm_buf+map->framesize*sizeof(short)+sample_offset*sizeof(short), copy_samples*sizeof(short));

		sample_offset+=copy_samples;
		done_samples+=copy_samples;
	}
}

StreamSource *Mp3StreamSource::get_source(void)
{
	StreamSource *new_source=new Mp3StreamSource(app_window);
	((Mp3StreamSource *)new_source)->setMap(map);
    new_source->attach(filename);

	return new_source;
}

void Mp3StreamSource::setMap(Mp3Map *m)
{
	if(map)
		delete map;
	external_map=true;
	map=m;
}

void Mp3StreamSource::generate_wave(void)
{
    if(!max_frames)
        return;
    if(wave)
        return;
    if(filename == QString::null)
    {
        printf("generate_wave called while filename is NULL\n");
        return;
    }
    wave_source=get_source();
    wave_source->app_window=app_window;
    int frames;

    wave=new Wavelet*[(frames=(wave_source->total_frames()+32767)/32768)];
    memset((char *)wave, 0, sizeof(Wavelet *)*frames);
    wavelets=frames;
}

#endif /* HAVE_LIBMAD */
