/*
 *   This file is part of Dianara
 *   Copyright 2012-2014  JanKusanagi <janjabber@gmail.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .
 */

#include "audienceselector.h"

AudienceSelector::AudienceSelector(PumpController *pumpController,
                                   QString selectorType,
                                   QWidget *parent) : QFrame(parent)
{
    this->selectorType = selectorType;

    QString titlePart;
    if (this->selectorType == "to")
    {
        titlePart = tr("'To' List");
    }
    else
    {
        titlePart = tr("'CC' List");
    }

    this->setWindowTitle(titlePart + " - Dianara");
    this->setWindowIcon(QIcon::fromTheme("system-users"));
    this->setWindowFlags(Qt::Window);
    this->setWindowModality(Qt::WindowModal);
    this->setMinimumSize(300, 300);



    // Left side, all contacts, with filter
    this->peopleWidget = new PeopleWidget(tr("&Add to Selected") + " >>",
                                          PeopleWidget::EmbeddedWidget,
                                          pumpController);
    connect(peopleWidget, SIGNAL(contactSelected(QIcon,QString)),
            this, SLOT(copyToSelected(QIcon,QString)));
    connect(peopleWidget, SIGNAL(addButtonPressed(QIcon,QString)),
            this, SLOT(copyToSelected(QIcon,QString)));



    this->allGroupboxLayout = new QVBoxLayout();
    allGroupboxLayout->addWidget(peopleWidget);

    allContactsGroupbox = new QGroupBox(tr("All Contacts"));
    allContactsGroupbox->setLayout(allGroupboxLayout);


    // Right side, selected contacts
    explanationLabel = new QLabel(tr("Select people from the list on the left.\n"
                                     "You can drag them with the mouse, click or "
                                     "double-click on them, or select them and "
                                     "use the button below."));
    explanationLabel->setWordWrap(true);

    selectedListWidget = new QListWidget();
    selectedListWidget->setDragDropMode(QListView::DragDrop);
    selectedListWidget->setDefaultDropAction(Qt::MoveAction);
    selectedListWidget->setSelectionMode(QListView::ExtendedSelection);

    this->clearSelectedListButton = new QPushButton(QIcon::fromTheme("edit-clear-list"),
                                                    tr("Clear &List"));
    connect(clearSelectedListButton, SIGNAL(clicked()),
            selectedListWidget, SLOT(clear()));


    doneButton = new QPushButton(QIcon::fromTheme("dialog-ok"),
                                 tr("&Done"));
    connect(doneButton, SIGNAL(clicked()),
            this, SLOT(setAudience()));
    cancelButton = new QPushButton(QIcon::fromTheme("dialog-cancel",
                                                    QIcon(":/images/button-cancel.png")),
                                   tr("&Cancel"));
    connect(cancelButton, SIGNAL(clicked()),
            this, SLOT(close()));
    buttonsLayout = new QHBoxLayout();
    buttonsLayout->setAlignment(Qt::AlignRight);
    buttonsLayout->addWidget(doneButton);
    buttonsLayout->addWidget(cancelButton);


    selectedGroupboxLayout = new QVBoxLayout();
    selectedGroupboxLayout->addWidget(explanationLabel);
    selectedGroupboxLayout->addSpacing(12);
    selectedGroupboxLayout->addWidget(selectedListWidget);
    selectedGroupboxLayout->addWidget(clearSelectedListButton);
    selectedGroupboxLayout->addSpacing(8);
    selectedGroupboxLayout->addLayout(buttonsLayout);

    this->selectedListGroupbox = new QGroupBox(tr("Selected People"));
    selectedListGroupbox->setLayout(selectedGroupboxLayout);

    this->mainLayout = new QHBoxLayout();
    mainLayout->addWidget(allContactsGroupbox,  3);
    mainLayout->addWidget(selectedListGroupbox, 4);
    this->setLayout(mainLayout);

    qDebug() << "AudienceSelector created" << titlePart;
}


AudienceSelector::~AudienceSelector()
{
    qDebug() << "AudienceSelector destroyed";
}



/*
 * Reset lists and widgets to default status
 *
 */
void AudienceSelector::resetLists()
{
    this->peopleWidget->resetWidget();

    this->selectedListWidget->clear();
    restoreSelected();
}

void AudienceSelector::deletePrevious()
{
    foreach (QListWidgetItem *item, previousItems)
    {
        delete item;
    }
    previousItems.clear();
}


void AudienceSelector::saveSelected()
{
    qDebug() << "AudienceSelector::saveSelected()";
    // Clear and delete all first
    this->deletePrevious();

    int totalItems = this->selectedListWidget->count();
    for (int counter = 0; counter < totalItems; ++counter)
    {
        this->previousItems.append(selectedListWidget->item(counter)->clone());
    }
}


void AudienceSelector::restoreSelected()
{
    qDebug() << "AudienceSelector::restoreSelected()";
    foreach (QListWidgetItem *item, this->previousItems)
    {
        this->selectedListWidget->addItem(item->clone());
    }
}



//////////////////////////////////////////////////////////////////////////////
///////////////////////////////// SLOTS //////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////



/*
 * Copy a contact to the list of Selected
 *
 * The contact string comes in a SIGNAL from PeopleWidget
 *
 */
void AudienceSelector::copyToSelected(QIcon contactIcon,
                                      QString contactString)
{
    if (!contactString.isEmpty())
    {
        QListWidgetItem *item = new QListWidgetItem(contactIcon,
                                                    contactString);
        this->selectedListWidget->addItem(item);
    }
}



/*
 * The "Done" button: emit signal with list of selected people
 *
 */
void AudienceSelector::setAudience()
{
    QStringList addressList;

    int addressCount = this->selectedListWidget->count();

    for (int counter=0; counter < addressCount; ++counter)
    {
        addressList.append(selectedListWidget->item(counter)->text());
    }


    emit audienceSelected(selectorType, addressList);

    saveSelected();  // To restore the list later, if the dialog is shown again

    this->hide();  // Don't close(), because that resets the lists =)
}



//////////////////////////////////////////////////////////////////////////////
/////////////////////////////// PROTECTED ////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////



void AudienceSelector::keyPressEvent(QKeyEvent *event)
{
    if (event->key() == Qt::Key_Escape)
    {
        this->close();
    }
    else
    {
        event->accept();
    }
}

void AudienceSelector::closeEvent(QCloseEvent *event)
{
    this->resetLists();

    event->accept();
}
