/*
 * Diagnostics - a unified framework for code annotation, logging,
 * program monitoring, and unit-testing.
 *
 * Copyright (C) 2002-2005 Christian Schallhart
 *               2006-2007 model.in.tum.de group
 *  
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


/**
 * @file diagnostics/unittest/test_system/test_list_result.hpp
 *
 * @brief [LEVEL: beta] @ref diagnostics::unittest::Test_List_Result class
 *
 * $Id: test_list_result.hpp,v 1.13 2005/06/23 09:54:26 esdentem Exp $
 * 
 * @author Christian Schallhart
 *
 * @test diagnostics/unittest/test_system/test_list_result.t.cpp
 */

#ifndef DIAGNOSTICS__UNITTEST__TEST_SYSTEM__TEST_LIST_RESULT_HPP__INCLUDE_GUARD
#define DIAGNOSTICS__UNITTEST__TEST_SYSTEM__TEST_LIST_RESULT_HPP__INCLUDE_GUARD

#include <diagnostics/unittest/namespace.hpp>

// used in the interface by value
#include <diagnostics/frame/level.hpp>

// used in the interface by reference
#include <string>

// used in the interface by reference
#include <iostream>

DIAGNOSTICS_NAMESPACE_BEGIN;
UNITTEST_NAMESPACE_BEGIN;

/**
 * @class Test_List_Result diagnostics/unittest/test_system/test_list_result.hpp
 *
 * @brief a Test_List_Result represents the data on a single Test_Case
 * generated by a @ref List_Test_Suite_Traversal.
 *
 * @nosubgrouping
 */
class Test_List_Result
{
    ////////////////////////////////////////////////////////////////////////////////
    /**
     * @name Types
     * @{
     */
private:
    typedef Test_List_Result Self;

    // @}

    ////////////////////////////////////////////////////////////////////////////////
    /**
     * @name Disabled Creation
     * @{
     */
private:
    Test_List_Result();
    // @}

    ////////////////////////////////////////////////////////////////////////////////
    /**
     * @name Creation
     * @{
     */
public:
    /**
     * @brief assembly-cstr
     *
     * @pre @ref path_adheres_convention(@a path)==true
     * @pre timeout>=0
     *
     * @throw Test_System_Exception the preconditions are not met
     */
    Test_List_Result(::std::string const & path,
		     int const timeout,
		     bool const is_rerunable,
		     bool const is_testable_at_prod,
		     bool const is_testable_at_debug,
		     bool const is_testable_at_audit
		     );
    
    /**
     * @throw never
     */
    Test_List_Result(Self const & other);
    /**
     * @throw never
     */
    Self & operator=(Self const & other);
    // @}

    ////////////////////////////////////////////////////////////////////////////////
    /**
     * @name Comparision
     * @{
     */
public:
    /**
     * @throw never
     */
    bool operator==(Self const & r) const;

    /**
     * @throw never
     */
    inline bool operator!=(Self const & r) const { return !operator==(r); }
    // @}

    ////////////////////////////////////////////////////////////////////////////////
    /**
     * @name Accessors
     * @{
     */
public:
    /**
     * @brief a path to the @ref Test_Case (if generated by @ref
     * List_Test_Suite_Traversal, then it is the complete path wrt.
     * the @ref Test_Suite used as argument in the @ref
     * List_Test_Suite_Traversal::traverse call).
     *
     * @throw never
     */
    inline ::std::string const & test_case_path() const { return m_test_case_path; }
    /**
     * @brief the timeout of the @ref Test_Case (HYBRID: 0 means no timeout at all)
     *
     * @throw never
     */
    inline int timeout() const { return m_timeout; }
    /**
     * @brief whether the @ref Test_Case can be run twice within the same process
     *
     * @throw never
     */
    inline bool is_rerunable() const { return m_is_rerunable; }
    /**
     * @brief whether the @ref Test_Case is supposed to succeed at build level @a level.
     *
     * @pre @a level != @ref LEVEL_TEST
     *
     * @throw @ref Test_System_Exception if the precondition is not met
     */
    bool is_testable_at(Level_t const level) const;

    // @}

    ////////////////////////////////////////////////////////////////////////////////
    /**
     * @name Encapsulated State 
     * @{
     */
private:
    ::std::string m_test_case_path;
    int m_timeout;
    bool m_is_rerunable;
    int m_levels;
    // @}
};

UNITTEST_NAMESPACE_END;
DIAGNOSTICS_NAMESPACE_END;




/**
 * @brief prints a @ref diagnostics::unittest::Test_List_Result in one line (no new line)
 *
 * @throw never
 */
::std::ostream & operator<<(::std::ostream & stream, ::diagnostics::unittest::Test_List_Result const & r);


#endif
// vim:ts=4:sw=4
