/*
 * Diagnostics - a unified framework for code annotation, logging,
 * program monitoring, and unit-testing.
 *
 * Copyright (C) 2002-2005 Christian Schallhart
 *               2006-2007 model.in.tum.de group
 *  
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


/**
 * @file diagnostics/unittest/test_system/stream_test_system.cpp
 *
 * @brief [LEVEL: beta] implementation of @ref diagnostics::unittest::Stream_Test_System Class
 *
 * $Id: stream_test_system.cpp,v 1.12 2005/06/23 09:56:44 esdentem Exp $
 * 
 * @author Christian Schallhart
 */

#include <diagnostics/unittest/test_system/stream_test_system.hpp>

#include <diagnostics/frame/logging_facility.hpp>

#include <diagnostics/unittest/name_separators.hpp>
#include <diagnostics/unittest/test_suite.hpp>

#include <diagnostics/unittest/test_system/file_test_data_source.hpp>
#include <diagnostics/unittest/test_system/stream_test_data_adaptor.hpp>

#include <diagnostics/unittest/test_system/test_run_result.hpp>
#include <diagnostics/unittest/test_system/run_test_suite_traversal.hpp>
#include <diagnostics/unittest/test_system/test_list_result.hpp>
#include <diagnostics/unittest/test_system/list_test_suite_traversal.hpp>


#include <diagnostics/unittest/test_exception.hpp>
#include <diagnostics/unittest/test_system_exception.hpp>

#include <diagnostics/util/from_string.hpp>
#include <diagnostics/util/to_string.hpp>

#include <iostream>
#include <fstream>
#include <sstream>
#include <iomanip>
#include <cstdlib>
#include <cstring>

DIAGNOSTICS_NAMESPACE_BEGIN;
UNITTEST_NAMESPACE_BEGIN;

class Stream_Test_System_Implementation 
{
public:
    Stream_Test_System_Implementation(Level_t const build_level,
				      int const argc,
				      char const * const * argv,
				      ::std::string const & default_test_data_file_name,
				      ::std::string const & default_result_log_file_name,
				      ::std::istream & in,
				      ::std::ostream & out);
    
    
    ~Stream_Test_System_Implementation();

    void add(Test_Suite * const test_suite);

    void add(Test_Case * const test_case);
    
	int run();

    void interactive_run();

private:
    int p_run(::std::string const & mask,
			  Level_t const level);
	
    void p_list(::std::string const & mask,
		Level_t const level);


    ::std::string p_fraction_to_string(int const value, int const total);


    ::std::istream & m_in;
    ::std::ostream & m_out;

    Level_t m_build_level;
    int const m_argc;
    char const * const * const m_argv;
    ::std::string m_test_data_file_name;
    ::std::string m_result_log_file_name;
	
    Test_Suite m_test_suite;

    File_Test_Data_Source m_test_data_source;
    Stream_Test_Data_Adaptor m_interactive_test_data;

    ::std::ofstream m_log;
};

Stream_Test_System_Implementation::
Stream_Test_System_Implementation(Level_t const build_level,
				  int const argc,
				  char const * const * argv,
				  ::std::string const & default_test_data_file_name,
				  ::std::string const & default_result_log_file_name,
				  ::std::istream & in,
				  ::std::ostream & out)
    : m_in(in),
      m_out(out),
      m_build_level(build_level),
      m_argc(argc),
      m_argv(argv),
      m_test_data_file_name(default_test_data_file_name),
      m_result_log_file_name(default_result_log_file_name),
      m_test_suite(""),
      m_interactive_test_data(m_in,m_out)
{
    // checking build level /////////////////////////////////////////////
    if(build_level>=LEVEL_TEST) 
	throw Test_System_Exception(::std::string("Build_Level '") + 
				    level_to_string(build_level) + 
				    "' failed to one of PRO,DEB, or AUD");

    // checking arguments ///////////////////////////////////////////////
    if(m_argc>1) {
	int cmd_offset(0);
	if(::std::strcmp(m_argv[1],"-h"))
    {
        if(!::std::strcmp(m_argv[1],"-i") 
                || !::std::strcmp(m_argv[1],"-a") 
                || !::std::strcmp(m_argv[1],"-r")){
            cmd_offset=1;
            if(m_argc!=5)
                throw Test_System_Exception(::std::string(m_argv[1])+" requires list|run <mask> <level>");
        }
        if(!::std::strcmp(m_argv[1+cmd_offset],"list") 
           || !::std::strcmp(m_argv[1+cmd_offset],"run")) {
            using ::diagnostics::internal::from_string;
            using ::diagnostics::internal::to_string;
            if(m_argc!=4+cmd_offset)
            throw Test_System_Exception(::std::string(m_argv[1+cmd_offset])+" requires <mask> <level>");
            if(!mask_adheres_convention(m_argv[2+cmd_offset]))
            throw Test_System_Exception(::std::string("Invalid mask '") + argv[2+cmd_offset] + "'");
                
            int const level(from_string<int>(m_argv[3+cmd_offset]));
            if(level<0 || level>LEVEL_TEST)
            throw Test_System_Exception
                (::std::string("Level '") 
                 + level_to_string(static_cast<Level_t>(level)) + "' (numerically: " 
                 + to_string(level) + ") failed to one of PRO,DEB,AUD, or TES");
        }
        else throw Test_System_Exception(::std::string("unkown command: '")+m_argv[1+cmd_offset]+"'");
    }
    }

    // opening the result log /////////////////////////////////////////////////////////////
    m_log.open(default_result_log_file_name.c_str(),::std::ios::app);
    if(!m_log.good()) 
	throw Test_System_Exception(::std::string("Could not open ") + m_result_log_file_name);
	


    // loading test_data //////////////////////////////////////////////
    try {
	m_test_data_source.load(m_test_data_file_name);
    }
    catch(Test_System_Exception & e){
	::std::ifstream s(m_test_data_file_name.c_str());
	if(s.good()) 
	    // the file exists but is not properly formated
	    throw;
	m_test_data_source.init(m_test_data_file_name);
	// we save to check, whether we can write the file
	m_test_data_source.save(m_test_data_file_name,true);
    }
    m_interactive_test_data.attach(&m_test_data_source);

}

Stream_Test_System_Implementation::~Stream_Test_System_Implementation()
{    
    try {
	m_test_data_source.save(m_test_data_file_name);
    }
    catch(Test_System_Exception & e){
	DIAGNOSTICS_PANIC_LOG("Exception while saving  '" 
			      + m_test_data_file_name
			      + "' failed: EXCEPTION=\"Test_System_Exception\" WHAT=\""
			      + e.what() + "\"");
    }
    catch(...){
	DIAGNOSTICS_PANIC_LOG("Unknown exception while saving test data '" 
			      + m_test_data_file_name 
			      + "'. CHANGES MIGHT BE LOST");
    }
}

void Stream_Test_System_Implementation::add(Test_Suite * const test_suite)
{
    m_test_suite.add(test_suite);
}

void Stream_Test_System_Implementation::add(Test_Case * const test_case)
{
    m_test_suite.add(test_case);
}


int Stream_Test_System_Implementation::p_run(::std::string const & mask,
											 Level_t const level)
{
    if(!mask_adheres_convention(mask))
		throw Test_System_Exception("Invalid mask '" + mask + "'");
    if(level>LEVEL_TEST)
		throw Test_System_Exception(::std::string("Level '") + 
									level_to_string(level) + "' failed to one of PRO,DEB,AUD, or TES");
	
    {
		::std::ostringstream buffer;
		buffer << "\n********************************************************************************\n" 
			   << "Started RUN with mask " << mask 
			   << "\n            and level " << level_to_string(level) 
			   << "\n     with build Level " << level_to_string(m_build_level) 
			   << "\n  in test application " << m_argv[0]
			   << "\n--------------------------------------------------------------------------------\n";
		m_out << buffer.str();
		m_log << buffer.str();
    }
	
	
    int failed(0);
    int invalid(0);
    int success(0);
	
    // uses the List_Test_Suite_Traversal to iterate through the test
    // cases AND the Run_Test_Suite_Traversal to execute these Test
    // Cases by case by case (Run_Test_Suite_Traversal could execute
    // them all in one traversal).
    //
    // However, if we would use Run_Test_Suite_Traversal to execute
    // all test cases in a single traversal, we would get the
    // Test_Run_Results when ALL test cases have been executed. To
    // give a more immediate feedback (precisely: after the execution
    // of each single test case), we use construction below.
    //
    // Performance at this level is not an issue. 

    typedef List_Test_Suite_Traversal::Test_List_Results_t List_Results_t;
    List_Results_t list_results;
    typedef Run_Test_Suite_Traversal::Test_Run_Results_t Run_Results_t;
    Run_Results_t run_results;


    List_Test_Suite_Traversal list_traversal(list_results);
    Run_Test_Suite_Traversal run_traversal(m_build_level,m_interactive_test_data,run_results);
    list_traversal.traverse(m_test_suite,mask,level);

    List_Results_t::const_iterator current(list_results.begin());
    List_Results_t::const_iterator const end(list_results.end());
    for(;current!=end;++current){
		run_traversal.traverse(m_test_suite,(*current)->test_case_path(),level);
		m_out << *run_results[0] << '\n';
		m_log << *run_results[0] << '\n';
		if(run_results[0]->invalidation_count()!=0) ++invalid;
		else if(run_results[0]->failure_count()!=0) ++failed;
		else ++success;
		delete run_results[0];
		delete *current;
		run_results.clear();
    } 


    int const total(failed+invalid+success);
	
    {
		::std::ostringstream buffer;
		buffer << "--------------------------------------------------------------------------------\n"
			   << "Finished RUN with mask " << mask 
			   << "\n             and level " << level_to_string(level) 
			   << "\n      with build Level " << level_to_string(m_build_level) 
			   << "\n   in test application " << m_argv[0]
			   << "\nSUCCESS       " << p_fraction_to_string(success,total)
			   << "\nFAILURES      " << p_fraction_to_string(failed,total) 
			   << "\nINVALIDATIONS " << p_fraction_to_string(invalid,total)
			   << "\nTOTAL         " << ::std::setw(3) << total
			   << "\n********************************************************************************\n";
	
		m_out << buffer.str() << ::std::endl;
		m_log << buffer.str() << ::std::endl;
    }
	
	return (failed!=0 ? Test_System::TEST_SYSTEM_RESULT_FAILURE : 0) 
		+ (invalid!=0 ? Test_System::TEST_SYSTEM_RESULT_INVALID : 0);
}

::std::string Stream_Test_System_Implementation::p_fraction_to_string(int const value, int const total)
{
    ::std::ostringstream buffer;
	
    buffer << ::std::setw(3) << value << " (" << ::std::setw(3)
	   << static_cast<int>((total!=0 ? ((100*(value))/total) : 100)) << "%)";
    return buffer.str();
}
		   
			  

void Stream_Test_System_Implementation::p_list(::std::string const & mask,
					       Level_t const level)
{
    if(!mask_adheres_convention(mask))
	throw Test_System_Exception("Invalid mask '" + mask + "'");
    if(level>LEVEL_TEST)
	throw Test_System_Exception(::std::string("Level '") + 
				    level_to_string(level) + "' failed to one of PRO,DEB,AUD, or TES");


    {
	::std::ostringstream buffer;

	buffer << "\n********************************************************************************\n" 
	       << "Started LIST with mask " << mask << " and level " << level_to_string(level) 
	       << "\n--------------------------------------------------------------------------------";
		
	m_out << buffer.str() << ::std::endl;
	m_log << buffer.str() << ::std::endl;
    }
	

    typedef List_Test_Suite_Traversal::Test_List_Results_t Results_t;
    Results_t results;

    List_Test_Suite_Traversal traversal(results);
    traversal.traverse(m_test_suite,mask,level);
    
    Results_t::const_iterator current(results.begin());
    Results_t::const_iterator const end(results.end());
    for(;current!=end;++current) {
	m_out << **current << '\n';
	m_log << **current << '\n';
	delete *current;
    }

    {
	::std::ostringstream buffer;

	buffer << "--------------------------------------------------------------------------------\n" 
	       << "Finished LIST with mask " << mask << " and level " << level_to_string(level)
	       << "\nTOTAL         " << results.size()
	       << "\n********************************************************************************" << ::std::endl;

	m_out << buffer.str() << ::std::endl;
	m_log << buffer.str() << ::std::endl;
    }

}


int Stream_Test_System_Implementation::run()
{
    using ::diagnostics::internal::from_string;

    // checks on the arguments are already done by the constructor

    if(m_argc==1) {
		interactive_run();
		return 0;
	}
    else {
		m_interactive_test_data.mode(Stream_Test_Data_Adaptor::MODE_REJECT);

		if(!::std::strcmp(m_argv[1],"-h")) {
            m_out
                << "Diagnostics test suite" << ::std::endl << ::std::endl
                << "    -h                      Print this help message" << ::std::endl
                << "    -i <command spec>       Ask in case of new data" << ::std::endl
                << "    -r <command spec>       Default to reject new data" << ::std::endl
                << "    -a <command spec>       Default to accept new data" << ::std::endl
                << "  command spec:" << ::std::endl
                << "    run <mask> <level>      Run the tests <mask> of <level> non-interactively" << ::std::endl
                << "    list <mask> <level>     List the available tests matching <mask> from <level>" << ::std::endl;
            return 0;
		}
		
        int cmd_offset(0);
		if(!::std::strcmp(m_argv[1],"-i")) {
			m_interactive_test_data.mode(Stream_Test_Data_Adaptor::MODE_INTERACTIVE);
			cmd_offset=1;
		}
		if(!::std::strcmp(m_argv[1],"-r")) {
			m_interactive_test_data.mode(Stream_Test_Data_Adaptor::MODE_REJECT);
			cmd_offset=1;
		}
		if(!::std::strcmp(m_argv[1],"-a")) {
			m_interactive_test_data.mode(Stream_Test_Data_Adaptor::MODE_ACCEPT);
			cmd_offset=1;
		}

		if(!::std::strcmp(m_argv[1+cmd_offset],"list")){
			p_list(m_argv[2+cmd_offset],static_cast<Level_t>(from_string<int>(m_argv[3+cmd_offset])));
			return 0;
		}
		else // if(!::std::strcmp(m_argv[1+cmd_offset],"run"))
			return p_run(m_argv[2+cmd_offset],static_cast<Level_t>(from_string<int>(m_argv[3+cmd_offset])));
    }
}

void Stream_Test_System_Implementation::interactive_run()
{
    m_out << "Test Data is taken from: " << m_test_data_file_name  
	  << "\nResult Log is writen to: " << m_result_log_file_name  
	  << "\nBuild Level            : " << level_to_string(m_build_level)
	  << ::std::endl;

    for(;;) try {
	::std::string command;
	::std::string mask;
	int level;
	m_out << "> ";
	m_in >> command;
	if(m_in.eof())
	    break;
	
	if(command=="interactive") {
	    int value;
	    m_in >> value;
	    m_interactive_test_data.mode(static_cast<Stream_Test_Data_Adaptor::Mode_t>(value));
	}
	else if(command=="run") {
	    m_in >> mask;
	    m_in >> level;
	    p_run(mask,static_cast<Level_t>(level));
	}
	else if(command=="list") {
	    m_in >> mask;
	    m_in >> level;
	    p_list(mask,static_cast<Level_t>(level));
	}
	else if(command=="quit")
	    break;
	else {
	    m_out << "Commands: " << ::std::endl 
		  << "run <mask> <level>" << ::std::endl
		  << "list <mask> <level>" << ::std::endl
		  << "interactive <mode>" << ::std::endl
		  << "quit" << ::std::endl;
	}
    }
    catch(Test_System_Exception & e) {
	m_out << "An Test_System_Exception occured: " << e.what() << '\n' << ::std::endl;
    }
    /// @todo clarify
}


////////////////////////////////////////////////////////////////////////////////

Stream_Test_System::Stream_Test_System(Level_t const build_level,
				       int const argc,
				       char const * const * argv,
				       ::std::string const & default_test_data_file_name,
				       ::std::string const & default_result_log_file_name,
				       ::std::istream & in,
				       ::std::ostream & out)
    : m_impl(new Stream_Test_System_Implementation(build_level,
						   argc,
						   argv,
						   default_test_data_file_name,
						   default_result_log_file_name,
						   in,
						   out))
{
}

    
Stream_Test_System::~Stream_Test_System() 
{
    delete m_impl;
}

    
void Stream_Test_System::add(Test_Suite * const test_suite) 
{
    m_impl->add(test_suite);
}

void Stream_Test_System::add(Test_Case * const test_case) 
{
    m_impl->add(test_case);
}


int Stream_Test_System::run() 
{
    return m_impl->run();
}


UNITTEST_NAMESPACE_END;
DIAGNOSTICS_NAMESPACE_END;

// vim:ts=4:sw=4
