/*
 * Diagnostics - a unified framework for code annotation, logging,
 * program monitoring, and unit-testing.
 *
 * Copyright (C) 2002-2005 Christian Schallhart
 *               2006-2007 model.in.tum.de group
 *  
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


/**
 * @file diagnostics/unittest/test_suite.hpp
 *
 * @brief [LEVEL: beta] @ref diagnostics::unittest::Test_Suite class
 *
 * $Id: test_suite.hpp,v 1.14 2005/06/23 09:54:24 esdentem Exp $
 * 
 * @author Christian Schallhart
 *
 * @test diagnostics/unittest/test_suite.t.cpp
 */

#ifndef DIAGNOSTICS__UNITTEST__TEST_SUITE_HPP__INCLUDE_GUARD
#define DIAGNOSTICS__UNITTEST__TEST_SUITE_HPP__INCLUDE_GUARD

#include <diagnostics/frame/namespace.hpp>
#include <diagnostics/unittest/namespace.hpp>

// used in the interface by reference
// used in the implementation by value
#include <string>

// used in the interface by reference
// used in the implementation by value
#include <vector>

DIAGNOSTICS_NAMESPACE_BEGIN;
UNITTEST_NAMESPACE_BEGIN;

// used in the interface by pointer
// <diagnostics/unittest/test_case.hpp>
class Test_Case;

/**
 * @class Test_Suite diagnostics/unittest/test_suite.hpp
 *
 * @brief a Test_Suite is a container for Test_Cases and other Test_Suites
 *
 * The unittest system allows to arrange Test_Cases in a hierarchical
 * manner, i.e., they can be grouped together into Test_Suites. A
 * number of Test_Suites can be grouped together in another
 * Test_Suite. In other words, Test_Suites can be arranged in a tree
 * with nodes of unbounded degree where each leave if a @ref Test_Case
 * or an empty Test_Suite.
 *
 * @nosubgrouping
 */
class Test_Suite
{
    ////////////////////////////////////////////////////////////////////////////////
    /**
     * @name Types
     * @{
     */
private:
    typedef Test_Suite Self;

    // @}

    ////////////////////////////////////////////////////////////////////////////////
    /**
     * @name Disabled Creation
     * @{
     */
private:
    /**
     * @brief no default cstr
     */
    Test_Suite();
    /**
     * @brief no copy cstr
     */
    Test_Suite(Self const & other);

    /**
     * @brief no assignement
     */
    Self & operator=(Self const & other);
    // @}

    ////////////////////////////////////////////////////////////////////////////////
    /**
     * @name Creation
     * @{
     */
public:
    /**
     * @brief destroys all sub-Test_Suites and Test_Cases
     *
     * @throw never
     */
    ~Test_Suite();

    /**
     * @brief the only way to construct a Test_Suite
     *
     * @throw Test_System_Exception if @a name does not match the
     * naming convention, i.e., it contains a @ref 
     * DIAGNOSTICS_UNITTEST_TEST_CASE_NAME_SEPARATOR or a @ref
     * DIAGNOSTICS_UNITTEST_TEST_DATA_NAME_SEPARATOR
     */
    explicit Test_Suite(::std::string const & name);
    // @}

    ////////////////////////////////////////////////////////////////////////////////
    /**
     * @name Modifiers
     * @{
     */
public:
    /**
     * @brief add a Test_Case to the Test_Suite (transfers ownership)
     *
     * @arg @a test_case has an empty string as name
     * @arg @a test_case->name().size()!=0
     * @arg there is a already a Test_Case which has the same name as @a test_case 
     *
     * @throw Test_System_Exception
     */
    void add(Test_Case* const test_case);
    
    /**
     * @brief add a Test_Suite to the Test_Suite (transfers ownership)
     *
     * @arg @a test_suite has an empty string as name
     * @arg @a test_suite->name().size()!=0
     * @arg there is a already a Test_Suite which has the same name as @a test_suite
     *
     * @throw Test_System_Exception
     */
    void add(Test_Suite* const test_suite);
    // @}


    ////////////////////////////////////////////////////////////////////////////////
    /**
     * @name Accessors
     * @{
     */
public:
    typedef ::std::vector<Test_Case*> Test_Cases_t;
    typedef ::std::vector<Test_Suite*> Test_Suites_t;

    /**
     * @brief the Test_Suites directly owned by the Test_Suite (i.e., excluding
     * the Test_Suites which are owned by sub-Test_Suites).
     * 
     * @throw never
     */
    Test_Suites_t const & test_suites() const { return m_test_suites; }

    /**
     * @brief the Test_Cases directly owned by the Test_Suite (i.e., excluding
     * the Test_Cases which are owned by sub-Test_Suites).
     * 
     * @throw never
     */
    Test_Cases_t  const & test_cases()  const { return m_test_cases; }

    /**
     * @brief the name of the Test_Suite
     * 
     * @throw never
     */
    ::std::string const & name() const { return m_name; }
    // @}

    ////////////////////////////////////////////////////////////////////////////////
    /** 
     * @name Encapsulated State
     * @{
     */
private:
    ::std::string const m_name;
    Test_Cases_t m_test_cases;
    Test_Suites_t m_test_suites;
    // @}
};

UNITTEST_NAMESPACE_END;
DIAGNOSTICS_NAMESPACE_END;


#endif
// vim:ts=4:sw=4
