/*
 * Diagnostics - a unified framework for code annotation, logging,
 * program monitoring, and unit-testing.
 *
 * Copyright (C) 2002-2005 Christian Schallhart
 *               2006-2007 model.in.tum.de group
 *  
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


/**
 * @file diagnostics/frame/logging_implementation.cpp
 *
 * $Id: logging_implementation.cpp,v 1.11 2005/06/23 09:54:28 esdentem Exp $
 *
 * @author Christian Schallhart
 *
 * @brief [LEVEL: beta] Implementation of @ref
 * diagnostics::internal::Logging_Implementation
 */

#include <diagnostics/frame/logging_implementation.hpp>


#include <diagnostics/frame/set_initial_loggers.hpp>
#include <diagnostics/frame/logging_config_exception.hpp>

#include <diagnostics/frame/platform.hpp>

#include <cstdlib> // for atexit

#include <sstream>
#include <vector>
#include <iostream>
#include <exception>



DIAGNOSTICS_NAMESPACE_BEGIN;
INTERNAL_NAMESPACE_BEGIN;


Logging_Implementation::Logging_Implementation()
#if DIAGNOSTICS_SWITCH_SYSTEM_CALLS_ENABLED == 1
    : m_pid(Platform::get_pid())
#endif
{
#if DIAGNOSTICS_SWITCH_SYSTEM_CALLS_ENABLED == 1
    Platform::get_time(m_startup_sec,m_startup_usec);

    if(Platform::get_hostname(m_hostname,m_HOSTNAME_MAX_LEN)) {
		DIAGNOSTICS_PANIC_LOG_INTERNAL("Error getting hostname -- aborting");
		::std::abort();
    }
#endif

    // setting the initial loggers
    try {
		set_initial_loggers(m_loggers);
    }
    catch(::std::exception & e){
		DIAGNOSTICS_PANIC_LOG_INTERNAL
			(::std::string("Unexpected ::std::exception while calling set_initial_loggers -- aborting: ")+
			 e.what());
		::std::abort();
    }
    catch(...){
		DIAGNOSTICS_PANIC_LOG_INTERNAL("Unexpected Exception while calling set_initial_loggers -- aborting");
		::std::abort();
    }

    // sending initial message
    ::std::ostringstream what;
#if DIAGNOSTICS_SWITCH_SYSTEM_CALLS_ENABLED == 1
    what << "diagnostics startup time " << m_startup_sec << "-" << m_startup_usec;
#endif
    try {
		log(LEVEL_SYSTEM,TYPE_LOG_OPEN,0,"Starting Logging Framework: "+what.str(),"","",1);
    }
    catch(::std::exception & e){
		DIAGNOSTICS_PANIC_LOG_INTERNAL
			(::std::string("Unexpected ::std::exception while sending log_open -- aborting: ")+
			 e.what());
		::std::abort();
    }
    catch(...){
		DIAGNOSTICS_PANIC_LOG_INTERNAL("Unexpected Exception while sending log_open -- aborting");
		::std::abort();
    }
}

Logging_Implementation::~Logging_Implementation()
{
    // sending final message
    try {
		log(LEVEL_SYSTEM,TYPE_LOG_CLOSE,0,"Stopping Logging Framework","","",1);
    }
    catch(::std::exception & e){
		DIAGNOSTICS_PANIC_LOG_INTERNAL
			(::std::string("Unexpected ::std::exception while sending log_close: ")+
			 e.what());
    }
    catch(...){
		DIAGNOSTICS_PANIC_LOG_INTERNAL("Unexpected Exception while sending log_close");
    }
	
    // closing loggers
    Loggers_t::const_iterator current(m_loggers.begin());
    Loggers_t::const_iterator const end(m_loggers.end());
    for(;current!=end;++current) {
		try {
			(*current)->close();
		}
		catch(::std::exception & e){
			DIAGNOSTICS_PANIC_LOG_INTERNAL
				(::std::string("Unexpected ::std::exception while closing logger: ")+
				 e.what());
		}
		catch(...){
			DIAGNOSTICS_PANIC_LOG_INTERNAL("Unexpected Exception while closing logger");
		}
    }

    // unregistering them all
    m_loggers.resize(0);
}

////////////////////////////////////////////////////////////////////////////////

Logging_Implementation* Logging_Implementation::m_instance=NULL;

Logging_Implementation* Logging_Implementation::setup_instance()
{
    // registering for deletion at exit -- not clean
    ::std::atexit(&Logging_Implementation::atexit_handler);
    return m_instance=new Self;
}

void Logging_Implementation::atexit_handler()
{
    delete Logging_Implementation::m_instance;
}


////////////////////////////////////////////////////////////////////////////////

void Logging_Implementation::register_logger(Logger * const logger) 
{
    if(logger==NULL) throw Logging_Config_Exception("logger==NULL");

    Loggers_t::iterator current(m_loggers.begin());
    Loggers_t::iterator const end(m_loggers.end());

    for(;current!=end;++current) 
		if(*current==logger) throw Logging_Config_Exception("logger already registered");
    
    // sending initial message
    ::std::ostringstream what;
#if DIAGNOSTICS_SWITCH_SYSTEM_CALLS_ENABLED == 1
    what << "diagnostics startup time " << m_startup_sec << "-" << m_startup_usec;
#endif
    DIAGNOSTICS_RECORD(LEVEL_SYSTEM,TYPE_LOG_OPEN,0,"Starting Logger "+what.str(),"","",0);

    // if the logger throws for whatever reason, it is not handled here.
    logger->log(record);

    m_loggers.push_back(logger);
}


void Logging_Implementation::unregister_logger(Logger * const logger) 
{
    if(logger==NULL) throw Logging_Config_Exception("logger==NULL");

    Loggers_t::iterator current(m_loggers.begin());
    Loggers_t::iterator const end(m_loggers.end());

    for(;current!=end;++current) {
		if(*current==logger) {
			m_loggers.erase(current);
			// logger is unregistered -- afterwards it might throw...
			DIAGNOSTICS_RECORD(LEVEL_SYSTEM,TYPE_LOG_CLOSE,0,"Stopping Logger","","",0);
			logger->log(record);
			return;
		}
    }
    throw Logging_Config_Exception("logger not registered");
}

void Logging_Implementation::panic_log(::std::string const & what,
									   char const * const base_file_name,
									   char const * const file_name,
									   int const line) 
{
    // if this does not work -- we're done.
    ::std::cerr << "PANIC LOG: " 
				<< what << " [ " 
				<< base_file_name << " "
				<< file_name << " "
				<< line << " ]" 
				<< ::std::endl;
}


INTERNAL_NAMESPACE_END;
DIAGNOSTICS_NAMESPACE_END;
// vim:ts=4:sw=4
