/*
 * Diagnostics - a unified framework for code annotation, logging,
 * program monitoring, and unit-testing.
 *
 * Copyright (C) 2002-2005 Christian Schallhart
 *               2006-2007 model.in.tum.de group
 *  
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


#ifndef DIAGNOSTICS__EXTENSIONS__MEMORY__ALLOCATION_DATABASE__INCLUDE_GUARD
#define DIAGNOSTICS__EXTENSIONS__MEMORY__ALLOCATION_DATABASE__INCLUDE_GUARD

// used by value in the interface
#include <diagnostics/extensions/memory/tick.hpp>

DIAGNOSTICS_NAMESPACE_BEGIN;
MEMORY_NAMESAPCE_BEGIN;

class Allocation_Database
{
public:
    /**
     * @brief register an allocation-call
     */
    static void register_allocation(void* const p, ::std::size_t const len);

    /**
     * @brief register a deallocation-call. Returns true if the block
     * should be deallocated. At the moment, it returns false, if it
     * does not know the block, thus it could be a double-deallocation.
     */
    static bool register_deallocation(void* const p);

    /**
     * @brief reset the complete database
     */
    static void reset();

    /**
     * @brief switch on analysis -- the register calls are ignored
     * until the analysis is switched on. Afterwards, all calls are
     * registered. 
	 *
	 * @attention When using multiple threads: If other threads are
	 * working in the meantime, some allocation will, some other will
	 * not be registered. THUS, LET ALL OTHER THREADS WAIT WHILE
	 * SWITCHING ON/OFF THE ANALYSIS.
     *
     * @a logging indicates whether the allocations/deallocations
     * should be reflected in the log. Multiple calls to analysis_on
     * are allowed, a corresponding counter is increased.
     */
    static void analysis_on(bool const logging);

    /**
     * @brief switch off analysis -- the converse to analysis_on. The
     * calls must be symetric to analysis_on.
	 *
	 * @attention be aware of problems with multi-threading! see 
	 * @ref analysis_on
     */
    static void analysis_off(bool const logging);

    /**
     * @brief each registered allocation is associated with a
     * creation_tick. This methods return the current creation_tick
     * which allows to filter the database on currently allocated
     * blocks.
     */
    static Tick_t creation_tick();

    /**
     * @brief sets @a size to the number of bytes allocated after the
     * creation_tick @a start_from. @a number is corresponding number
     * of blocks.
     */
    static void balance(Tick_t const start_from,
						::std::size_t & size,
						Tick_t & number);
	
    /**
     * @brief sets @a size to the total number of bytes allocated
     * (precisely: returns the sum over all registered blocks). @a
     * number is corresponding number of blocks.
     */
    static void balance(::std::size_t & size,
						Tick_t & number);

    static Tick_t* unallocated_blocks(Tick_t const start_from,
									  Tick_t * const b,
									  Tick_t * const e);
};


MEMORY_NAMESAPCE_END;
DIAGNOSTICS_NAMESPACE_END;

#endif

// vim:ts=4:sw=4
