/* dia-default-tool.c
 * Copyright (C) 2004  Arjan Molenaar
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "dia-default-tool.h"
#include "dia-canvas-i18n.h"

#include "dia-handle-tool.h"
#include "dia-selection-tool.h"
#include "dia-item-tool.h"

enum {
	PROP_HANDLE_TOOL = 1,
	PROP_ITEM_TOOL,
	PROP_SELECTION_TOOL
};

static void dia_default_tool_class_init		(DiaToolClass *class);
static void dia_default_tool_init		(DiaDefaultTool *tool);
static void dia_default_tool_dispose		(GObject *object);
static void dia_default_tool_set_property	(GObject	*gobject,
						 guint		 property_id,
		                                 const GValue	*value,
						 GParamSpec	*pspec);
static void dia_default_tool_get_property	(GObject	*gobject,
						 guint		 property_id,
		                                 GValue		*value,
						 GParamSpec	*pspec);

static gboolean dia_default_tool_button_press	(DiaTool *tool,
						 DiaCanvasView *view,
						 GdkEventButton *event);
static gboolean dia_default_tool_button_release (DiaTool *tool,
						 DiaCanvasView *view,
						 GdkEventButton *event);
static gboolean dia_default_tool_motion_notify	(DiaTool *tool,
						 DiaCanvasView *view,
						 GdkEventMotion *event);
static gboolean dia_default_tool_key_press	(DiaTool *tool,
						 DiaCanvasView *view,
						 GdkEventKey *event);
static gboolean dia_default_tool_key_release	(DiaTool *tool,
						 DiaCanvasView *view,
						 GdkEventKey *event);

static DiaToolClass *parent_class = NULL;

GType
dia_default_tool_get_type (void)
{
	static GtkType object_type = 0;

	if (!object_type) {
		static const GTypeInfo object_info = {
			sizeof (DiaDefaultToolClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) dia_default_tool_class_init,
			(GClassFinalizeFunc) NULL,
			(gconstpointer) NULL, /* class_data */
			sizeof (DiaDefaultTool),
			(guint16) 0, /* n_preallocs */
			(GInstanceInitFunc) dia_default_tool_init,
		};

		object_type = g_type_register_static (DIA_TYPE_TOOL,
						      "DiaDefaultTool",
						      &object_info, 0);
	}

	return object_type;
}

static void
dia_default_tool_class_init (DiaToolClass *klass)
{
	GObjectClass *object_class = (GObjectClass*) klass;

	parent_class = g_type_class_peek_parent (klass);

	object_class->dispose = dia_default_tool_dispose;
	object_class->get_property = dia_default_tool_get_property;
	object_class->set_property = dia_default_tool_set_property;

	klass->button_press_event = dia_default_tool_button_press;
	klass->button_release_event = dia_default_tool_button_release;
	klass->motion_notify_event = dia_default_tool_motion_notify;
	klass->key_press_event = dia_default_tool_key_press;
	klass->key_release_event = dia_default_tool_key_release;

	g_object_class_install_property (object_class,
					 PROP_HANDLE_TOOL,
					 g_param_spec_object ("handle-tool",
						_("Tool"),
						_("Set a tool"),
						DIA_TYPE_TOOL,
						G_PARAM_READWRITE));
	g_object_class_install_property (object_class,
					 PROP_ITEM_TOOL,
					 g_param_spec_object ("item-tool",
						_("Tool"),
						_("Set a tool"),
						DIA_TYPE_TOOL,
						G_PARAM_READWRITE));
	g_object_class_install_property (object_class,
					 PROP_SELECTION_TOOL,
					 g_param_spec_object ("selection-tool",
						_("Tool"),
						_("Set a tool"),
						DIA_TYPE_TOOL,
						G_PARAM_READWRITE));
}

static void
dia_default_tool_init (DiaDefaultTool *tool)
{
	tool->handle_tool = dia_handle_tool_new ();
	tool->selection_tool = dia_selection_tool_new ();
	tool->item_tool = dia_item_tool_new ();
	tool->current_tool = NULL;
}

static void
dia_default_tool_dispose (GObject *object)
{
	DiaDefaultTool *tool = (DiaDefaultTool*) object;

	tool->current_tool = NULL;

	if (tool->handle_tool) {
		g_object_unref (tool->handle_tool);
		tool->handle_tool = NULL;
	}

	if (tool->selection_tool) {
		g_object_unref (tool->selection_tool);
		tool->selection_tool = NULL;
	}

	if (tool->item_tool) {
		g_object_unref (tool->item_tool);
		tool->item_tool = NULL;
	}

	G_OBJECT_CLASS (parent_class)->dispose (object);
}

static void
dia_default_tool_set_property (GObject *object, guint property_id,
			       const GValue *value, GParamSpec *pspec)
{
	DiaDefaultTool *tool = (DiaDefaultTool*) object;

	switch (property_id) {
	case PROP_HANDLE_TOOL:
		dia_default_tool_set_handle_tool (tool,
						  g_value_get_object (value));
		break;
	case PROP_ITEM_TOOL:
		dia_default_tool_set_item_tool (tool,
						g_value_get_object (value));
		break;
	case PROP_SELECTION_TOOL:
		dia_default_tool_set_selection_tool (tool,
						     g_value_get_object (value));
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
	}
}

static void
dia_default_tool_get_property (GObject *object, guint property_id,
			       GValue *value, GParamSpec *pspec)
{
	DiaDefaultTool *tool = (DiaDefaultTool*) object;

	switch (property_id) {
	case PROP_HANDLE_TOOL:
		g_value_set_object (value, tool->handle_tool);
		break;
	case PROP_ITEM_TOOL:
		g_value_set_object (value, tool->item_tool);
		break;
	case PROP_SELECTION_TOOL:
		g_value_set_object (value, tool->selection_tool);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
	}
}

static gboolean
dia_default_tool_button_press (DiaTool *tool, DiaCanvasView *view,
			         GdkEventButton *event)
{
	DiaDefaultTool *dtool = (DiaDefaultTool*) tool;

	//dia_canvas_push_undo (view->canvas, NULL);

	if (dia_tool_button_press (dtool->handle_tool, view, event)) {
		dtool->current_tool = dtool->handle_tool;
		return TRUE;
	}

	
	if (dia_tool_button_press (dtool->item_tool, view, event)) {
		dtool->current_tool = dtool->item_tool;
		return TRUE;
	}

	if (dia_tool_button_press (dtool->selection_tool, view, event)) {
		dtool->current_tool = dtool->selection_tool;
		return TRUE;
	}
	return FALSE;
}

static gboolean
dia_default_tool_button_release (DiaTool *tool, DiaCanvasView *view,
				   GdkEventButton *event)
{
	DiaDefaultTool *dtool = DIA_DEFAULT_TOOL (tool);

	if (dtool->current_tool)
		return dia_tool_button_release (dtool->current_tool, view, event);

	return FALSE;
}

static gboolean
dia_default_tool_motion_notify (DiaTool *tool, DiaCanvasView *view,
				  GdkEventMotion *event)
{
	DiaDefaultTool *dtool = DIA_DEFAULT_TOOL (tool);

	if (dtool->current_tool)
		return dia_tool_motion_notify (dtool->current_tool, view, event);

	return FALSE;
}

static gboolean
dia_default_tool_key_press (DiaTool *tool, DiaCanvasView *view,
			      GdkEventKey *event)
{
	DiaDefaultTool *dtool = DIA_DEFAULT_TOOL (tool);

	if (dtool->current_tool)
		return dia_tool_key_press (dtool->current_tool, view, event);

	return FALSE;
}

static gboolean
dia_default_tool_key_release (DiaTool *tool, DiaCanvasView *view,
			        GdkEventKey *event)
{
	DiaDefaultTool *dtool = DIA_DEFAULT_TOOL (tool);

	if (dtool->current_tool)
		return dia_tool_key_release (dtool->current_tool, view, event);

	return FALSE;
}

DiaTool*
dia_default_tool_new (void)
{
	return g_object_new (DIA_TYPE_DEFAULT_TOOL, NULL);
}

void
dia_default_tool_set_handle_tool (DiaDefaultTool *tool, DiaTool *handle_tool)
{
	g_return_if_fail (DIA_IS_DEFAULT_TOOL (tool));
	g_return_if_fail (DIA_IS_TOOL (handle_tool));

	if (tool->handle_tool == tool->current_tool)
		tool->current_tool = handle_tool;

	g_object_unref (tool->handle_tool);
	tool->handle_tool = g_object_ref (handle_tool);
}

DiaTool*
dia_default_tool_get_handle_tool (DiaDefaultTool *tool)
{
	g_return_if_fail (DIA_IS_DEFAULT_TOOL (tool));

	return tool->handle_tool;
}

void
dia_default_tool_set_item_tool (DiaDefaultTool *tool, DiaTool *item_tool)
{
	g_return_if_fail (DIA_IS_DEFAULT_TOOL (tool));
	g_return_if_fail (DIA_IS_TOOL (item_tool));

	if (tool->item_tool == tool->current_tool)
		tool->current_tool = item_tool;

	g_object_unref (tool->item_tool);
	tool->item_tool = g_object_ref (item_tool);
}

DiaTool*
dia_default_tool_get_item_tool (DiaDefaultTool *tool)
{
	g_return_if_fail (DIA_IS_DEFAULT_TOOL (tool));

	return tool->item_tool;
}


void
dia_default_tool_set_selection_tool (DiaDefaultTool *tool, DiaTool *selection_tool)
{
	g_return_if_fail (DIA_IS_DEFAULT_TOOL (tool));
	g_return_if_fail (DIA_IS_TOOL (selection_tool));

	if (tool->selection_tool == tool->current_tool)
		tool->current_tool = selection_tool;

	g_object_unref (tool->selection_tool);
	tool->selection_tool = g_object_ref (selection_tool);
}

DiaTool*
dia_default_tool_get_selection_tool (DiaDefaultTool *tool)
{
	g_return_if_fail (DIA_IS_DEFAULT_TOOL (tool));

	return tool->selection_tool;
}

