/*
 * Detect a Library for hardware detection
 * 
 * Copyright (C) 1998-2000 MandrakeSoft
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */
 
#include <ctype.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/wait.h>

#include "detect.h"
#include "utils.h"


/**********************************************************************/
/* exec_detect_helper:  Run a command with arguments to catch its     */
/*                      stdout and stderr stream                      */
/*                                                                    */
/*   path:    command to be executed.                                 */
/*   args:    arguments to pass to command.                           */
/*   out:     stdout output of the command.                           */
/*   err:     stderr output of the command.                           */
/**********************************************************************/
extern int exec_detect_helper(char *path, char *args[], char **out, 
                                                            char **err){
  int read_errfd, write_errfd;
  int read_outfd, write_outfd;
  int errpipe[2];
  int outpipe[2];
  int i, pid, status;
  int resultSize;
  char *err_result;
  char *out_result;
  char *resultbuf[200];
  
  struct stat sb;
  
  if(out){
    pipe(outpipe);
    read_outfd = outpipe[0];
    write_outfd = outpipe[1];
  }else{
    read_outfd = -1;
    write_outfd = -1;
  }/*endif*/
  if(err){
    pipe(errpipe);
    read_errfd = errpipe[0];
    write_errfd = errpipe[1];
  }else{
    read_errfd = -1;
    write_errfd = -1;
  }/*endif*/

  if(!(pid = fork())){
    if((out) || (err)){
      if(out){
        close(1);
        dup2(write_outfd, 1);
        close(read_outfd);
        close(write_outfd);
      }/*endif*/
      if(err){
        close(2);
        dup2(write_errfd, 2);
        close(read_errfd);
        close(write_errfd);
      }/*endif*/
    }/*endif*/
    if(stat(path, &sb)){
      *err = (char *)my_malloc(strlen(path) + 
                        strlen(_("Can't run following command: ")) + 1);
      sprintf(*err , _("Cant' run following command: %s"), path);
      return 1;
    }/*endif*/
    execv(path, args);
    if(debug)
      printf(_("Something is wrong\n"));
    exit(-1);
  }/*endif*/
  
  if(out){
    close(write_outfd);
    resultSize = 0;
    out_result = (char *) NULL;
    do{
      i = read(read_outfd, resultbuf, sizeof(resultbuf));
      if(!out_result){
        out_result = (char *)my_malloc(i + 1);
      }else{
        out_result = (char *)realloc(out_result, resultSize + i + 1);
      }/*endif*/
      memcpy(out_result + resultSize, resultbuf, i);
      resultSize += i;
      out_result[resultSize] = '\0';
    }while(i > 0);
    close(read_outfd);
    *out = out_result;
  }/*endif*/
  if(err){
    close(write_errfd);
    resultSize = 0;
    err_result = (char *) NULL;
    do{
      i = read(read_errfd, resultbuf, sizeof(resultbuf));
      if(!err_result){
        err_result = (char *)my_malloc(i + 1);
      }else{
        err_result = (char *)realloc(err_result, resultSize + i + 1);
      }/*endif*/
      memcpy(err_result + resultSize, resultbuf, i);
      resultSize += i;
      err_result[resultSize] = '\0';
    }while(i > 0);
    close(read_errfd);
    *err = err_result;
  }/*endif*/
  
  waitpid(pid, &status, 0);
  
  if (WIFEXITED(status)){
    return WEXITSTATUS(status);
  }/*endif*/
  return -1; 
}/*endfunc exec_detect_helper*/


/**********************************************************************/
/* check_helper_version: Try to check if a current detect helper      */
/*                         has a recent version.                      */
/*                                                                    */
/*     path:  path to the detect helper we will check.                */
/*     major: major number of the detect helper.                      */
/*     minor: minor number of the detect helper.                      */
/*     patch: patch number of the detect helper.                      */
/**********************************************************************/
extern int check_helper_version(char *path, int major, int minor, 
                                                             int patch){
  char *err = (char *) NULL;
  char *out = (char *) NULL;
  char *args[] = {
    NULL, 
    "--version", 
    NULL
  };
  int cur_major = 0, cur_minor = 0, cur_patch = 0;

  args[0]= (char *)my_malloc(strlen(path)+1);
  args[0] = path;
  exec_detect_helper(path, args, &out, &err);

  if(!strcmp(path, "/sbin/modprobe")){
    sscanf(out, "modprobe version %d.%d.%d", 
            &cur_major, &cur_minor, &cur_patch);
  }/*endif*/
  if(cur_major < major || (cur_major == major && cur_minor < minor)
                       || (cur_major == major && cur_minor == minor 
                                              && cur_patch < patch)){
    return -1;
  }/*endif*/
  return 0;
}/*endfunc check_helper_version*/


/**********************************************************************/
/* exec_modprobe:  Try to call modprobe (to insert a module).         */
/*                                                                    */
/*    modulename:    name of the kernel module to be probed in.       */
/*    tmpconfmodule: if (tmpconfmodule != NULL) use it instead of     */
/*                   the module configuration file in etc.            */
/**********************************************************************/
extern char *exec_modprobe(char *modulename, char *tmpconfmodule){
  char *err = (char *) NULL;
  char *out = (char *) NULL;
  char *args[] = {
    "modprobe", 
    NULL, 
    "-C", 
    NULL,
    NULL
  };

  args[1] = (char *)my_malloc(strlen(modulename) + 1);
  args[1] = modulename;
  
  /* If we are not in testing mode, we're using the module            */
  /* configuration file in etc.                                       */
  if(tmpconfmodule != (char *)NULL){
    args[3] = (char *)my_malloc(strlen(tmpconfmodule) + 1);
    args[3] = tmpconfmodule;
  }else{
    args[2] = NULL;
  }/*endif*/
  if(exec_detect_helper("/sbin/modprobe", args, &out, &err) != 0){
    return err;
  }/*endif*/
  return (char *)NULL;
}/*endfunc exec_modprobe*/

