/* keysigdialog.c
 * Prompts the user to change the key signature

 * for Denemo, a gtk+ frontend to GNU Lilypond
 * (c) 2000, 2001 Matthew Hiller */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "calculatepositions.h"
#include "contexts.h"
#include "datastructures.h"
#include "dialogs.h"
#include "draw.h"
#include "objops.h"
#include "staffops.h"
#include "utils.h"

#define KEYNAME_ARRAY_OFFSET 7

static gchar *majorkeys[15] =
  { "C flat", "G flat", "D flat", "A flat", "E flat", "B flat", "F",
  "C", "G", "D", "A", "E", "B", "F sharp", "C sharp"
};

static gchar *minorkeys[15] =
  { "A flat", "E flat", "B flat", "F", "C", "G", "D",
  "A", "E", "B", "F sharp", "C sharp", "G sharp", "D sharp", "A sharp"
};

static GList *majorlist = NULL;
static GList *minorlist = NULL;

struct callbackdata
{
  struct scoreinfo *si;
  staff *curstaffstruct;
  GtkWidget *combobox;
  GtkWidget *radiobutton2;
  GtkWidget *checkbutton;
};

/* Returns G_MININT if keyname cannot be found */

static gint
findkey (GtkWidget * combobox, gboolean isminor)
{
  gchar *tokeystring =
    gtk_entry_get_text (GTK_ENTRY (GTK_COMBO (combobox)->entry));
  gint ret;

  if (isminor)
    ret = g_list_position
      (minorlist,
       g_list_find_custom (minorlist, tokeystring, (GCompareFunc) strcmp));
  else
    ret = g_list_position
      (majorlist,
       g_list_find_custom (majorlist, tokeystring, (GCompareFunc) strcmp));
  if (ret != -1)
    return ret - KEYNAME_ARRAY_OFFSET;
  else
    return G_MININT;
}

void
set_keysig (GtkWidget * widget, gpointer data)
{
  struct callbackdata *cbdata = data;
  struct scoreinfo *si = cbdata->si;
  staffnode *curstaff;
  staff *curstaffstruct;
  gboolean isminor =
    gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (cbdata->radiobutton2));
  gint tokey = findkey (cbdata->combobox, isminor);

  if (tokey != G_MININT)
    {
      if (gtk_toggle_button_get_active
	  (GTK_TOGGLE_BUTTON (cbdata->checkbutton)))
	{
	  for (curstaff = si->thescore; curstaff; curstaff = curstaff->next)
	    {
	      curstaffstruct = curstaff->data;
	      curstaffstruct->skey = tokey;
	      curstaffstruct->skey_isminor = isminor;
	      initkeyaccs (curstaffstruct->skeyaccs, tokey);
	      showwhichaccidentalswholestaff (curstaffstruct);
	    }
	  find_leftmost_allcontexts (si);
	}
      else
	{
	  cbdata->curstaffstruct->skey = tokey;
	  cbdata->curstaffstruct->skey_isminor = isminor;
	  initkeyaccs (cbdata->curstaffstruct->skeyaccs, tokey);
	  showwhichaccidentalswholestaff (cbdata->curstaffstruct);
	  find_leftmost_staffcontext (cbdata->curstaffstruct, si);
	}
      find_xes_in_all_measures (si);
      si->haschanged = TRUE;
      gtk_widget_draw (si->scorearea, NULL);
    }
}

void
insert_keysig (GtkWidget * widget, gpointer data)
{
  struct callbackdata *cbdata = data;
  staffnode *curstaff;
  struct scoreinfo *si = cbdata->si;
  measurenode *curmeasure;

  gboolean isminor =
    gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (cbdata->radiobutton2));
  gint tokey = findkey (cbdata->combobox, isminor);

  if (tokey != G_MININT)
    {
      if (gtk_toggle_button_get_active
	  (GTK_TOGGLE_BUTTON (cbdata->checkbutton)))
	{
	  for (curstaff = si->thescore; curstaff; curstaff = curstaff->next)
	    {
	      curmeasure = g_list_nth (firstmeasurenode (curstaff),
				       si->currentmeasurenum - 1);
	      curmeasure->data = g_list_append (curmeasure->data,
						newkeyobj (tokey, isminor));
	      if (curmeasure == si->currentmeasure)
		si->currentobject =
		  g_list_nth (curmeasure->data, si->cursor_x);
	      showwhichaccidentalswholestaff (curstaff->data);
	    }			/* End for */
	}			/* End if */
      else
	{
	  si->currentmeasure->data = g_list_append (si->currentmeasure->data,
						    newkeyobj (tokey,
							       isminor));
	  si->currentobject =
	    g_list_nth (si->currentmeasure->data, si->cursor_x);
	  showwhichaccidentalswholestaff (si->currentstaff->data);
	}
      si->cursor_appending = FALSE;
      find_xes_in_all_measures (si);
      nudgerightward (si);
      si->haschanged = TRUE;
      gtk_widget_draw (si->scorearea, NULL);
    }				/* End if */
}


void
majorcallback (GtkWidget * widget, gpointer data)
{
  gchar *text =
    g_strdup (gtk_entry_get_text (GTK_ENTRY (GTK_COMBO (data)->entry)));

  gtk_combo_set_popdown_strings (GTK_COMBO (data), majorlist);
  gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (data)->entry), text);
  g_free (text);
}

void
minorcallback (GtkWidget * widget, gpointer data)
{
  gchar *text =
    g_strdup (gtk_entry_get_text (GTK_ENTRY (GTK_COMBO (data)->entry)));

  gtk_combo_set_popdown_strings (GTK_COMBO (data), minorlist);
  gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (data)->entry), text);
  g_free (text);
}

void key_change
  (gpointer callback_data, guint callback_action, GtkWidget * widget)
{
  struct scoreinfo *si = callback_data;
  GtkWidget *dialog;
  GtkWidget *label;
  GtkWidget *radiobutton1, *radiobutton2;
  GtkWidget *checkbutton;
  GtkWidget *combobox;
  GtkWidget *okbutton;
  GtkWidget *cancelbutton;
  GtkWidget *hbox;
  static struct callbackdata cbdata;
  gint i;
  static GString *menustring = NULL;
  staff *curstaffstruct = si->currentstaff->data;

  /* Initialization */
  if (!menustring)
    menustring = g_string_new (NULL);

  if (!majorlist)
    for (i = 0; i < 15; i++)
      {
	majorlist = g_list_append (majorlist, majorkeys[i]);
	minorlist = g_list_append (minorlist, minorkeys[i]);
      }

  /* GUI setup */
  dialog = gtk_dialog_new ();

  if (callback_action == CHANGEINITIAL)
    gtk_window_set_title (GTK_WINDOW (dialog),
			  _("Change initial key signature"));
  else
    gtk_window_set_title (GTK_WINDOW (dialog),
			  _("Insert key signature change"));

  label = gtk_label_new (_("Select desired key signature"));
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox),
		      label, TRUE, TRUE, 0);
  gtk_widget_show (label);

  combobox = gtk_combo_new ();

  hbox = gtk_hbox_new (FALSE, 0);
  radiobutton1 = gtk_radio_button_new_with_label (NULL, _("Major"));
  gtk_signal_connect (GTK_OBJECT (radiobutton1), "clicked",
		      GTK_SIGNAL_FUNC (majorcallback), combobox);
  gtk_box_pack_start (GTK_BOX (hbox), radiobutton1, TRUE, TRUE, 0);
  gtk_widget_show (radiobutton1);

  radiobutton2 = gtk_radio_button_new_with_label
    (gtk_radio_button_group (GTK_RADIO_BUTTON (radiobutton1)), _("Minor"));
  gtk_signal_connect (GTK_OBJECT (radiobutton2), "clicked",
		      GTK_SIGNAL_FUNC (minorcallback), combobox);
  gtk_box_pack_start (GTK_BOX (hbox), radiobutton2, TRUE, TRUE, 0);
  gtk_widget_show (radiobutton2);

  if (curstaffstruct->skey_isminor)
    {
      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (radiobutton2), TRUE);
      minorcallback (NULL, combobox);
      gtk_entry_set_text
	(GTK_ENTRY (GTK_COMBO (combobox)->entry),
	 minorkeys[curstaffstruct->skey + KEYNAME_ARRAY_OFFSET]);
    }
  else
    {
      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (radiobutton1), TRUE);
      majorcallback (NULL, combobox);
      gtk_entry_set_text
	(GTK_ENTRY (GTK_COMBO (combobox)->entry),
	 majorkeys[curstaffstruct->skey + KEYNAME_ARRAY_OFFSET]);
    }
  /* This setting-active will also complete the initialization of
   * the combobox */

  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox),
		      combobox, TRUE, TRUE, 0);
  gtk_widget_show (combobox);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox),
		      hbox, TRUE, TRUE, 0);
  gtk_widget_show (hbox);

  checkbutton = gtk_check_button_new_with_label (_("Apply to all staves?"));
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox),
		      checkbutton, TRUE, TRUE, 0);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (checkbutton), TRUE);
  gtk_widget_show (checkbutton);

  okbutton = gtk_button_new_with_label (_("OK"));
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->action_area),
		      okbutton, TRUE, TRUE, 0);
  cbdata.si = si;
  cbdata.curstaffstruct = curstaffstruct;
  cbdata.combobox = combobox;
  cbdata.radiobutton2 = radiobutton2;
  cbdata.checkbutton = checkbutton;
  if (callback_action == CHANGEINITIAL)
    {
      processenter (GTK_COMBO (combobox)->entry, set_keysig, cbdata, dialog);
      gtk_signal_connect (GTK_OBJECT (okbutton), "clicked",
			  GTK_SIGNAL_FUNC (set_keysig), &cbdata);
    }
  else
    {
      processenter (GTK_COMBO (combobox)->entry, insert_keysig, cbdata,
		    dialog);
      gtk_signal_connect (GTK_OBJECT (okbutton), "clicked",
			  GTK_SIGNAL_FUNC (insert_keysig), &cbdata);
    }

  gtk_signal_connect_object (GTK_OBJECT (okbutton), "clicked",
			     gtk_widget_destroy, GTK_OBJECT (dialog));
  gtk_widget_show (okbutton);

  cancelbutton = gtk_button_new_with_label (_("Cancel"));
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->action_area),
		      cancelbutton, TRUE, TRUE, 0);
  gtk_signal_connect_object (GTK_OBJECT (cancelbutton), "clicked",
			     gtk_widget_destroy, GTK_OBJECT (dialog));
  gtk_widget_show (cancelbutton);

  gtk_widget_grab_focus (combobox);
  gtk_window_set_modal (GTK_WINDOW (dialog), TRUE);
  gtk_window_set_position (GTK_WINDOW (dialog), GTK_WIN_POS_MOUSE);
  gtk_widget_show (dialog);
}
