/*
 * This file is licensed under the terms of the GNU General Public License,
 * version 2. See the file COPYING in the main directory for details.
 * 
 *  Copyright (C) 2000-2002  Florian Lohoff <flo@rfc822.org>
 *  Copyright (C) 2002,2003  Thiemo Seufer <seufer@csv.ica.uni-stuttgart.de>
 */

#include "delo.h"
#include "stringops.h"

#define PROGRAM    "delo"
#define VERSION    "0.8"
#define COPYRIGHT1 "Copyright 2000-2002"
#define AUTHOR1    "Florian Lohoff <flo@rfc822.org>"
#define COPYRIGHT2 "Copyright 2003"
#define AUTHOR2    "Thiemo Seufer <seufer@csv.ica.uni-stuttgart.de>"

#define CONFIG_FILE		"/etc/delo.conf"
#define MAX_CONFIG_NAME_LEN	25

/* The PROM callback vector. */
const struct callback *callv = NULL;

/* The DECstation system id. */
int dec_sysid = 0;

static void uint_to_string(char *str, unsigned int val)
{
	char buf[11];
	const char num[] = "0123456789";
	int i;

	memset(buf, 0, 11);
	for (i = 0; i < 10 && val; i++) {
		buf[9 - i] = num[val % 10];
		val /= 10;
	}
	memcpy(str, buf + 10 - i, i + 1);
}

int delomain(int argc, char **argv, int magic, struct callback *cv)
{
	int rex_prom = 0;
	int cargc = argc;
	char **cargv = argv;
	void (*entry)(int argc, char **argv, int magic, void *cv);
	int i;
	char **parray;
	char partition[3];
	char config[MAX_CONFIG_NAME_LEN];
	void *rd_loc = NULL;
	unsigned int rd_size = 0;
	char rd_start_str[25];
	char rd_size_str[25];


	if (magic == DEC_REX_MAGIC) {
		/* Store Call Vector. */
		callv = cv;
		rex_prom = 1;
		clearcache();
	} else {
		/*
		 * We can't even issue a message here because we know
		 * nothing about the cv.
		 */
		EMERGENCY_RESET();
	}

	printf("%s V%s\n", PROGRAM, VERSION);
        printf("%s %s\n", COPYRIGHT1, AUTHOR1);
	printf("%s %s\n", COPYRIGHT2, AUTHOR2);

	dprintf("callv addr %p\n", callv);
	for(i = 0; i < argc; i++)
		dprintf("clo: %d %s\n", i, argv[i]);

	if (rex_prom) {
		dec_sysid = getsysid();
		switch (DEC_FIRMREV(dec_sysid)) {
		case FIRM_TFC0:
			dprintf("REX Firmware revision TFC0\n");
			break;
		case FIRM_TFC1:
			dprintf("REX Firmware revision TFC1\n");
			break;
		default:
			printf("REX Firmware revision unknown (%d)\n",
			       DEC_FIRMREV(dec_sysid));
			break;
		}
	}

	/*
	 * Boot command line will look like this:
	 *
	 * boot 3/rz0 5/vmlinux
	 *
	 * It might contain replacement argv stuff e.g.
	 *
	 * boot 3/rz0 5/vmlinux root=/dev/sda5 console=ttyS2
	 *
	 * Now we don't use the config file args but the prom command line.
	 */

	/* Initialize to empty partition name. */
	memset(partition, 0, sizeof(partition));
	/* Initialize to empty config name. */
	memset(config, 0, sizeof(config));

	/* Get partition name. */
	if (argc >= 3) {
		unsigned int i;

		for(i = 0; isdigit(argv[2][i]) && i < (sizeof(partition) - 1); i++)
			partition[i] = argv[2][i];

		/* Is there a slash? */
		if (argv[2][i] == '/') {
			/* Copy config name. */
			strncpy(config, &argv[2][i + 1], sizeof(config) - 1);
		}
	}

	if (!(parray = getconfig(partition, CONFIG_FILE, config)))
		return 1;

	/* Do we have parms on the prom command line? */
	if (argc <= 3) {
		/* FIXME: This is completely stupid to work around
		 * a kernel stupidity - Probably the kernel should not
		 * skip anything and/or check for existance of "boot"
		 * at the beginning and skip only if this is the case. */
		if (rex_prom) 
			/*
			 * Copy pointer - We take 0 as the kernel
			 * will skip 2 parms anyway for REX so it doesn't
			 * care about our store usage.
			 */
			cargv = &parray[0];
		else
			/*
			 * For non REX proms the kernel skips 1 arg.
 			 */
			cargv = &parray[1];

		/* Count cargv values. */
		for(cargc = 0; cargv[cargc]; cargc++)
			dprintf("configargv: %d %s\n", cargc, cargv[cargc]);
	}

	if (delo_open(partition, parray[1]))
		return 1;

	/* Begin of progress indicator. */
	printf("Loading %s ", parray[1]);

	if ((entry = loadelf()))
		;
	else if ((entry = loadecoff()))
		;
	else {
		printf("\n%s is not a little endian MIPS kernel\n", parray[1]);
		return 1;
	}

	/* End of progress indicator. */
	printf(" ok\n");

	for (i = 0; i < cargc; i++) {
		if (strncasecmp(cargv[i], "initrd=", 7) == 0) {
			char *rd_name = cargv[i] + 7;

			rd_size = readfile(&rd_loc, partition, rd_name, PAGE_SIZE);
			break;
		}
	}
	memset(rd_start_str, 0, sizeof(rd_start_str));
	memset(rd_size_str, 0, sizeof(rd_size_str));
	if (rd_loc && rd_size) {
		if (cargc + 2 >= PARM_LIMIT) {
			printf("Kernel parameter count exceeded\n");
			return 1;
		}

		/* Append ramdisk parameters "rd_start=" and "rd_size=". */
		strcpy(rd_start_str, "rd_start=");
		uint_to_string(rd_start_str + strlen("rd_start="),
			       (unsigned long)rd_loc);
		cargv[cargc++] = rd_start_str;
		strcpy(rd_size_str, "rd_size=");
		uint_to_string(rd_size_str + strlen("rd_size="), rd_size);
		cargv[cargc++] = rd_size_str;
	}

	entry(cargc, cargv, magic, cv);

	/* The kernel start failed */
	return 1;
}
