/*
 
    Copyright (C) 2000 Florian Lohoff <flo@rfc822.org>


    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include <assert.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <stdio.h>
#include <sys/ioctl.h>
#include <linux/fs.h>
#include <linux/hdreg.h>


#define	_FILE_TO_BOOT	"/boot/delo.2nd"
#define _SECTOR_SIZE	512
#define	_TMP_DEV_DIR	"/tmp"
#define _MAX_MAPS	51
#define _LOAD_ADDRESS	0x80440000
#define _JUMP_ADDRESS	0x80440000
#define _DEC_BOOT_MAGIC	0x02757a

#define O_NOACCESS 3
#define MAJOR_SD	8

static	int	debug;

typedef struct	extent {
	int	count, start;
} extent;

typedef struct	dec_bootblock {
	unsigned char	pad[8];
	int		magic;
	int		mode;
	int		loadAddr;
	int		execAddr;
	extent		bootmap[_MAX_MAPS];
} dec_bootblock;	

int opendev(dev_t device) {
	char	devname[sizeof(_TMP_DEV_DIR) + 20];
	int	pfd;

	/* Create temp device name */
	sprintf(devname,"%s/dev.%d",
			_TMP_DEV_DIR,
			getpid());

	/* Create device file */
	assert(mknod(devname,0600 | S_IFBLK, device) == 0);

	/* And open it */
	assert((pfd=open(devname,O_NOACCESS)));

	unlink(devname);

	return(pfd);
}

int getpartoffset(int fd) {
	struct stat		sb;
	struct hd_geometry	geo;
	int			pfd,
				start;

	/* get stat (we want the device) */
	assert(!fstat(fd, &sb));

	assert((pfd=opendev(sb.st_dev)));

	switch(MAJOR(sb.st_dev)) {
		case MAJOR_SD:
			if (ioctl(pfd, HDIO_GETGEO, &geo) != 0) {
				printf("Unable to get disk geometry\n");
				return(-1);
			}
			start=geo.start;
			break;
		default:
			printf("Oops - unsupported device to install bootsector on");
			return(-1);
	}

	close(pfd);

	if (debug)
		printf("Start of partition %d\n",start);

	return(start);
}

struct extent *getextents(int fd) {
	int		block,
			no_blocks,
			i,j;
	int		blocksize,
			poffset;
	struct extent	*extents;
	struct stat	sb;
			
	assert((blocksize=getblocksize(fd)));

	/* get stat (we wont the filesize) */
	assert(!fstat(fd, &sb));

	no_blocks = (sb.st_size + blocksize - 1) / blocksize;

	if (debug)
		printf("size: %d blocks: %d sectorsize: %d blocksize %d\n",
				sb.st_size,
				no_blocks,
				_SECTOR_SIZE,
				blocksize);

	/* Allocate extents like blocks - Aggregation may only shrink it */
	assert(extents=(struct extent *)
		malloc((no_blocks + 1) * sizeof(struct extent)));

	/* Start at extent 0 */
	j=0;

	/* Loop on blocks and get position via FIBMAP */
	for(i=0;i<no_blocks;i++) {
		block=i;

		/* get fibmap block */
		if (ioctl(fd, FIBMAP, &block) != 0) {
			printf("Unable to get FIBMAP\n");
			return(0);
		}

		if (extents[j].start == 0) {
			/* Starting at first extent */
			extents[j].start=block * blocksize / _SECTOR_SIZE;
			extents[j].count=blocksize / _SECTOR_SIZE;
		} else {
			/* Is this block directly behind the last extent ? */
			if (extents[j].start + extents[j].count == 
					block * blocksize / _SECTOR_SIZE) {

				extents[j].count+=(blocksize / _SECTOR_SIZE);

			} else {
				/* We are not behind nor on free extent*/
				j++;
				extents[j].start=block * blocksize / _SECTOR_SIZE;
				extents[j].count=blocksize / _SECTOR_SIZE;
			}
		}
	}

	if (debug)
		printf("We have %d extents\n",j+1);

	if (j+1 > _MAX_MAPS) {
		printf("Oops - to many extents for the DECstation bootblock\n");
		return(0);
	}

	/* Get partition offset */
	if ((poffset=getpartoffset(fd)) < 0)
		return(0);

	/* Add partition offset to extent start */
	for(i=0;i<=j;i++) {

		/* Add partition offset */
		extents[i].start+=poffset;

		if (debug)
			printf("Extent %2d Start %8d Count %6d\n",
				i,
				extents[i].start,
				extents[i].count);
	}

	return(extents);
}

int getblocksize(int fd) {
	int	blocksize;

	assert(ioctl(fd, FIGETBSZ, &blocksize) == 0);

	return(blocksize);
}

int writesector(char *disk, extent *extents) {
	dec_bootblock		*bb;
	int			i,
				fd;

	/* Malloc memory */
	assert(bb=(struct dec_bootblock *) malloc(_SECTOR_SIZE));

	if (debug)
		printf("Opening %s to install bootsector\n", disk);

	/* Open disk */
	if ((fd=open(disk,O_RDWR)) < 0) {
		printf("Unable to open disk %s\n",disk);		
		return(-1);
	}

	/* Read sector - Contains the disk label (MSDOS/ULTRIX) */
	if(read(fd,bb,_SECTOR_SIZE) != _SECTOR_SIZE) {
		printf("Read from disk failed\n");
		close(fd);
		return(-1);
	}

	/* Fill in our values we care on */
	bb->magic=_DEC_BOOT_MAGIC;	/* We are a DEC BootBlock */
	bb->mode=1;			/* Multi extent boot */
	bb->loadAddr=_LOAD_ADDRESS;	/* Load below kernel */
	bb->execAddr=_JUMP_ADDRESS;	/* And exec there */

	/* Copy extents to boot sector */
	i=0;
	while(extents[i].start != 0 && i < _MAX_MAPS-1) {
		bb->bootmap[i].start=extents[i].start;
		bb->bootmap[i].count=extents[i].count;
		i++;
	}
	for(;i<_MAX_MAPS-1;i++) {
		/* Mark extents end .. */
		bb->bootmap[i].start=0x0;
		bb->bootmap[i].count=0x0;
	}

	/* And write back sector */
	assert(lseek(fd, 0, SEEK_SET) != -1);

	if (write(fd, bb, _SECTOR_SIZE) != _SECTOR_SIZE) {
		printf("Write of bootsector failed\n");
		close(fd);
		return(-1);
	}

	close(fd);

	return(0);
}

int writebootsec(char *fname, char *disk) {
	int		fd,
			blocksize,
			poffset;

	struct	extent	*extents;

	if (debug)
		printf("Writing bootsector\n");

	if ((fd=open(fname,O_RDONLY)) < 0) {
 		printf("Unable to open %s\n",fname);
		return(-1);
	}

	if (!(extents=getextents(fd)))
		printf("Failed to get extents\n");
	else
		writesector(disk, extents);

	close(fd);
}

void usage(void ) {
	printf("usage: delo [-d] <device>\n");
	exit(1);
}

int main(int argc, char **argv) {
	int	i;
	char	*device=0;

	if (argc < 2)
		usage();

	for(i=1;i<argc;i++) {
		if (argv[i][0] == '-') {
			switch(argv[i][1]) {
				case 'd':
					debug=1;
					break;
				default:
					usage();
					break;
			}
		} else {
			if (strncmp("/dev/", argv[i], 5) == 0)
				device=argv[i];
			else
				usage();
		}
	}	

	/* If we didnt get and device */
	if (!device) 
		usage();

	writebootsec(_FILE_TO_BOOT,device);	

	return(0);
}
