/* keep.c - Maintain a list of packages to keep for deborphan.
   Copyright (C) 2000 Cris van Pelt

   Distributed under the terms of the Artistic License. 
   
   The general idea was borrowed from Wessel Danker's depfoster program.
*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>

#include <sys/types.h>
#include <sys/stat.h>

#include <deborphan.h>
#include <xalloc.h>

/* If package is a valid package, this function returns 0. On error, it
 * returns -1, if it is not a valid package, it returns the position in
 * the 'args' list +1.
 */
int
pkggrep(const char *sfile, char **pkgnames)
{
    FILE *fp;
    char *s, *t;
    int i = 0;
    s = (char *) xmalloc (200 * sizeof(char));

    if (!(fp = fopen(sfile, "r")))
	return -1;

    while (*pkgnames) {
	t = (char *) xmalloc ((9+strlen(*pkgnames)) * sizeof(char));
	rewind(fp);

	strcpy(t, "ackage:");
	strcat(t, *pkgnames);
	strcat(t, "\n");
    
	while (fgets(s, 200, fp)) {
	    if (upcase(*s) != 'P')
		continue;
	    strstripchr(s, ' ');
	    if (strcmp(s+1, t) == 0)
		return 0;
	}

	free(t);
	i++;
	pkgnames++;
    }

    return i;
}

/* A somewhat slow function to read in the list of packages to keep.
 * The file shouldn't be too large though, so the loss of speed is hardly
 * noticable. Yes, this will probably blow up when you feed it lines > 100
 * characters.
 */
dep *
readkeep(const char *kfile)
{
    FILE *fd;
    dep *d, *ret;
    int size = 10, i = 1;
    char s[100];

    d = ret = xmalloc(10 * sizeof(dep));

    if (!(fd = fopen(kfile, "r")))
	return NULL;

    while (fgets(s, 100, fd)) {
	d->name = xmalloc(strlen(s) * sizeof(char));
	strstripchr(s, '\n');
	strcpy(d->name, s);
	d->namehash = strhash(d->name);

	d++;
	if (++i == size) {
	    size *= 2;
	    ret = xrealloc(ret, size * sizeof(dep));
	    d = ret + i - 1;
	}
    }
    d->name = NULL;
    fclose(fd);

    return ret;
}

int
mustkeep(const dep d)
{
    dep *c;
    if (!keep) return 0;

    for (c = keep; c->name; c++) {
	if (pkgcmp(*c, d))
	    return 1;
    }

    return 0;
}

int
addkeep(const char *kfile, char **add)
{
    FILE *fd;
    int i = 0;

    if (!(fd = fopen(kfile, "a+")))
	return -1;

    while (*add) {
	if (**add && **add != '\n') {
	    fputs(*add, fd);
	    fputc('\n', fd);
	}
	add++;
	i++;
    }

    return i;
}

/* This function is in definite need of rewriting. It uses a temporary file
 * for removal a package. Which is icky.
 */
int
delkeep(const char *kfile, char **del)
{
    FILE *fd, *tmp;
    char s[100];
    int i, n = 0;
    struct stat kinfo;
    off_t before;

    if (!(fd = fopen(kfile, "r+")))
	return -1;
    if (!(tmp = tmpfile()))
	return -1;

    stat(kfile, &kinfo);
    before = kinfo.st_size;

    while (fgets(s, 100, fd)) {
	strstripchr(s, '\n');
	for (i = 0; del[i]; i++) {
	    if (strcmp(del[i], s) == 0)
		n = 1;
	}
	if (n)
	    n = 0;
	else
	    fprintf(tmp, "%s\n", s);
    }

    rewind(tmp);
    rewind(fd);
    ftruncate(fileno(fd), 0);

    while (fgets(s, 100, tmp))
	fputs(s, fd);

    fclose(tmp);
    fclose(fd);
    stat(kfile, &kinfo);

    if (kinfo.st_size == before)
	return 1;

    return 0;
}

/* If something in list is found in keep, this function returns its 
 * position in the list +1, else it returns 0.
 */
unsigned int
hasduplicate(char **list)
{
    unsigned int i;
    dep d;

    for (i = 0; list[i]; i++) {
	d.name = list[i];
	d.namehash = strhash(list[i]);
	if (mustkeep(d))
	    return i+1;
    }

    return 0;
}

dep *
mergekeep(const dep *a, const dep *b)
{
    unsigned int i = 0, j = 0, x = 0, y = 0;
    dep *ret;

    if (!(a || b))
	return NULL;

    if (a)
	while (a[i++].name);
    if (b)
	while (b[j++].name);

    ret = (dep *) xmalloc (sizeof(dep) * (j+i));

    for (i = 0; a && a[i].name; i++)
	ret[i] = a[i];

    for (j = 0; b && b[j].name; j++) {
	for (x = 0, y = 0; ret[x].name; x++) {
	    if (pkgcmp(ret[x], b[j]))
		y = 1;
	}
	if (!y)
	    ret[i++] = b[j];
    }

    return ret;
}

int
listkeep(const char *kfile)
{
    FILE *fp = fopen(kfile, "r");
    char s[100];

    if (!fp)
	return 0;

    while (fgets(s, 100, fp)) {
	if (*s != '\n')
	    printf("%s", s);
    }

    return 1;
}

/* This function handles the arguments to --add-keep and --del-keep.
 */
char **
parseargs(int argind, int argc, char **argv)
{
    char **ret = xcalloc(50, sizeof(char *));
    char t[100];
    int i = 0, s = 50;
    
    while (argind < argc) {
	if (*argv[argind] == '-' && *(argv[argind]+1) == '\0') {
	    while (fgets(t, 100, stdin)) {
		if (i == s) {
		    s *= 2;
		    ret = xrealloc(ret, s * sizeof(char *));
		}
		/* Allocate an extra byte, just to be on the safe side. */
		ret[i] = (char *) xmalloc((strlen(t)+1) * sizeof(char));
		strstripchr(t, ' ');
		strstripchr(t, '\r');
		strstripchr(t, '\n');
		strcpy(ret[i++], t);
	    }
	    argind++;
	    continue;
	}
	ret[i] = (char *) xmalloc((strlen(argv[argind]) + 1) * sizeof(char));
	strcpy(ret[i], argv[argind]);
	argind++;
	i++;
    }

    return ret;
}
