#!/bin/bash

# Script to check on the number of available file handles
#   and increase the number of max open files if necessary
#
# Should be run from a cron job every five minutes or so
#
# Algorithm:
#  If there is less then 10% of file-max available file handler,
#  double file-max.  If there is more the 75% of file-max available
#  file handles, make file-max half it old value.  Never go below 4096.
#
# August 2003
# Ragnar Wislff <ragnar@skolelinux.no>
#
# Licence: GPL
#
#
# Numbers in /proc/sys/fs/file-nr is explained by this url:
#    http://www.cs.tau.ac.il/~didi/file-nr.html
#
# Apparently this problem will no longer be with us once we reach
#   the wild and wonderful world of kernel 2.6 as explained here:
#    http://kniggit.net/wwol26.html
#   look for the section named "Size Matters -- Scalability Improvements" 
#

#debug=1

debug() {
    if [ "$debug" ] ; then
    	echo "log: $@"
    fi
}

log() {
    debug "$@"
    logger -i -t filehandle_ctl "$@"
}

# Get the current situation
max_files=$(cat /proc/sys/fs/file-max)
allocated_files=$(cat /proc/sys/fs/file-nr |awk '{print $1}')
free_files=$(cat /proc/sys/fs/file-nr |awk '{print $2}')

# How close to max open files number do we want to get before ramping up?
hilimit=$[max_files/10]

# If the currently used amount is less then 25 % of the current
# maximum, decrease the maximum
lowlimit=$[3 * max_files/4]

# Do some maths to find available files remaining
available_files=$[($max_files - ($allocated_files - $free_files))]

debug "h: $hilimit l: $lowlimit a: $available_files m: $max_files"

# Find out if we are getting close to trouble area
# And then do something if we are
if [ $available_files -lt $hilimit ]; then
    new_file_max=$[max_files * 2]
    echo $new_file_max > /proc/sys/fs/file-max
    log "Available files is $available_files.  Increased file-max from $max_files to $new_file_max."
fi

if [ $available_files -gt $lowlimit ]; then
    new_file_max=$[max_files/2]
    if [ "$new_file_max" -lt 4096 ] ; then
    	# Never go too low
    	:
    else
#        echo $new_file_max > /proc/sys/fs/file-max
#        log "Available files is $available_files.  Decreased file-max from $max_files to $new_file_max."
	:
    fi
fi
