/* curves.cxx
     $Id: curves.cxx,v 1.42 1998/09/06 21:46:23 elf Exp $

   written by Marc Singer
   20 September 1996

   This file is part of the project CurVeS.  See the file README for
   more information.

   Copyright (C) 1996 Marc Singer

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   in a file called COPYING along with this program; if not, write to
   the Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA
   02139, USA.

   -----------
   DESCRIPTION
   -----------

   Curves class control.  This is the user session level class.


   --------------------
   IMPLEMENTATION NOTES
   --------------------

   -- m_dir sorting WRT file and directory indices

   While there used to be an array to index file and directory entries
   in the m_dir structures, this has changed to a simpler, index-free
   design.  We sort the m_dir data in place putting directories at the
   top, files second, and all other files, such as links, at the end.
   Since we know the count of directories and files, it is easy to
   recover the m_dir index of the first file, m_cDirectories.  This
   has the benefits of less memory allocation and much simpler
   reference code.

*/


#include "std.h"
#include "preferences.h"	// Everyone MUST get into the pref scene

#include <dirent.h>
#include <time.h>
#include "curves.h"
#include "blue6.h"


#define CB_FILE_CURRENT		20 	// Status line chars for file name
#define CB_DIR_CURRENT		20 	// Status line chars for dir name

static const char* g_szSortCriteria;	// Limitation of sort requires globals
static LCVS* g_pCvs;			//  to augment the call parameters


/* LCurves::_classify

   evaluates the filesystem and status information for a file and
   determines the color class and the prefix character.

*/

int LCurves::_classify (FILE_INFO* pInfo)
{
  if (!g_pCvs->is_controlled () ||
      (!LCVS::is_controlled (pInfo) && LCVS::is_ignored (pInfo)))
    return classUncontrolled;

  if (!LCVS::is_controlled (pInfo))
    return classNewAdd;
  
  if (LCVS::is_status_merge (pInfo))
    return classMerge;
  if (LCVS::is_status_patch (pInfo))
    return classPatch;
  if (LCVS::is_status_conflict (pInfo))
    return classConflict;
  if (LCVS::is_status_added (pInfo))
    return classAdded;
  if (LCVS::is_status_removed (pInfo))
    return classRemoved;
  if (LCVS::is_status_checkout (pInfo)) {
    if (LDirectory::time (pInfo) == 0)
      return classLost;
    return classCheckout;
  }
  if (LCVS::is_edited (pInfo))
    return classEdited;

  return classUpToDate;
}


/* LCurves::do_directory_change

   change to currently selected directory.

*/

void LCurves::do_directory_change (void)
{
  char sz[256];
  strcpy (sz, m_dir.path ());
  const char* szDir = _dir (m_iDirCursor);
  if (szDir && strcmp (szDir, "..") == 0) {
    int ichLast = 0;
    while (1) {
      int ich = strcspn (sz + ichLast, "/\\:");
      if (sz[ichLast + ich])
	ichLast += ich + 1;
      else 
	break;
    };
    if (ichLast == 1)
      ++ichLast;
    if (ichLast)
      sz[ichLast - 1] = 0;
  }
  else {
    if (sz[strlen (sz) - 1] != '/')
      strcat (sz, "/");
    strcat (sz, szDir);
  }
  set_path (sz);
}

void LCurves::do_directory_move (int direction)
{
  //  int f = false;
  if (direction < 0)
    direction = -1;
  if (direction > 0)
    direction = 1;
  if (!direction 
      || m_iDirCursor + direction == m_cDirectories
      || m_iDirCursor + direction <  0)
    return;
  show_cursor (false);
  m_iDirCursor += direction;
  if (m_iDirCursor < m_iDirScroll) {
    if (m_pPaneDir->scroll (-1))
      render_dir (--m_iDirScroll);
    else {
      --m_iDirScroll;
      m_fDirtyDir = true;	// Force redraw, boo hoo
    }
    //    f = true;
  }
  else if (m_iDirCursor - m_iDirScroll >= m_pPaneDir->dy ()) {
    if (m_pPaneDir->scroll (1))
      render_dir (++m_iDirScroll + m_pPaneDir->dy () - 1);
    else {
      ++m_iDirScroll;
      m_fDirtyDir = true;	// Force redraw, boo hoo
    }
    //    f = true;
  }

  refresh ();
}


/* LCurves::do_file_binarytoggle

   toggle the binary-ness of the currently selected file.

*/

void LCurves::do_file_binarytoggle (void)
{
  int idx = _idx_file (m_iFileCursor);
  m_cvs.binary (idx, !m_cvs.is_binary (idx));
  m_fDirtyStatus = true;	// Status for this file may have changed
}


void LCurves::do_file_move (int direction)
{
  if (direction < -1)
    direction = -m_pPaneFile->dy ();
  if (direction > 1)
    direction = m_pPaneFile->dy ();
  if (!direction 
      || m_iFileCursor + direction >= m_cFiles
      || m_iFileCursor + direction <  0)
    return;
  show_cursor (false);
  m_iFileCursor += direction;

  while (m_iFileCursor < m_iFileScroll) {
    m_iFileScroll -= m_pPaneFile->dy ();
    m_fDirtyFile = true;
  }
  while (m_iFileCursor
	 >= m_iFileScroll + _columns_of_files ()*m_pPaneFile->dy ()) {
    m_iFileScroll += m_pPaneFile->dy ();
    m_fDirtyFile = true;
  }

  if (m_fDirtyFile)
    m_pPaneFile->erase ();

  m_fDirtyStatus = true;

  refresh ();
}


/* LCurves::do_file_tag

   toggle the tag of the currently selected file.

*/

void LCurves::do_file_tag (void)
{
  show_cursor (false);
  m_cFilesTagged += m_cvs.is_tagged (_idx_file (m_iFileCursor)) ? -1 : 1;
  m_cvs.tag (_idx_file (m_iFileCursor), 
	     !m_cvs.is_tagged (_idx_file (m_iFileCursor)));
  show_cursor (true);
}


/* LCurves::do_search

   changes the selection according to a search string.  This allows us
   to find files and directories quickly.  The search is
   case-insensitive and will wrap at the end of the list.

*/

void LCurves::do_search (const char* szPattern, bool fReverse)
{
  char ch = tolower (*szPattern);
  int cchPattern = strlen (szPattern);
  int iStart  = (m_iPane ? m_cDirectories : 0);
  int iCursor = (m_iPane ? m_iFileCursor  : m_iDirCursor);
  int iMax    = (m_iPane ? m_cFiles       : m_cDirectories);
  for (int i = 1; i < iMax; ++i) {
    int iFile = iCursor + (fReverse ? -i : i);
    if (iFile < 0)
      iFile += iMax;
    iFile %= iMax;
    const char* szPath = m_dir.file (iFile + iStart);
    for (; szPath && *szPath; ++szPath) {
      if (tolower (*szPath) != ch
	  || strncasecmp (szPath, szPattern, cchPattern))
	continue;
      show_cursor (false);
      if (m_iPane) {
	m_iFileCursor = iFile;
	if (m_iFileCursor < m_iFileScroll) {
	  m_iFileScroll = m_iFileCursor - m_iFileCursor%m_pPaneFile->dy ();
	  m_fDirtyFile = true;
	}
	else if (m_iFileCursor
		 >= m_iFileScroll + _columns_of_files ()*m_pPaneFile->dy ()) {
	  m_iFileScroll = m_iFileCursor - m_iFileCursor%m_pPaneFile->dy ()
	    - (_columns_of_files () - 1)*m_pPaneFile->dy ();
	  m_fDirtyFile = true;
	}
	if (m_fDirtyFile)
	  m_pPaneFile->erase ();
	m_fDirtyStatus = true;
	refresh ();
      }
      else {
	m_iDirCursor  = iFile;
	if (m_iDirCursor < m_iDirScroll) {
	  m_pPaneDir->scroll (m_iDirScroll - m_iDirCursor);
	  m_iDirScroll = m_iDirCursor;
	  m_fDirtyDir = true;
	  refresh ();
	}
	else if (m_iDirCursor >= m_iDirScroll + m_pPaneDir->dy ()) {
	  m_pPaneDir->scroll (m_iDirCursor - m_iDirScroll
			     - m_pPaneDir->dy () + 1);
	  m_iDirScroll = m_iDirCursor - m_pPaneDir->dy () + 1;
	  m_fDirtyDir = true;
	  refresh ();
	}
      }
      show_cursor (true);
      return;
    }	
  }
}


/* LCurves::do_update

   updates the status for the named files.  This is used to change the
   displayed status of files after committing them or some other
   operation that may change the files display class.  

*/

void LCurves::do_update (const char* szFiles)
{
  m_cvs.fetch_status (szFiles);
  m_cvs.parse_status ();
  sort ();
  dirty_path ();
}


bool LCurves::_explore (const char* szFile, void*) 
{
  if (!strcmp (szFile, "."))
    return false;
  
  return true;
}

void LCurves::explore (void)
{
  Pane& pane = *m_pPaneCommand;

  pane.erase ();
  pane.draw_text (Position (0, 0), "Requesting status information...");
  update ();
  m_cvs.fetch_status ();	// Must be before enumerating the directory
  pane.erase ();
  pane.draw_text (Position (0, 0), "Reading directory...");
  update ();
  m_dir.enumerate (_explore, NULL);
  pane.erase ();
  pane.draw_text (Position (0, 0), "Analyzing status information...");
  m_cvs.analyze ();		// Must be done before sorting
  pane.erase ();
  pane.draw_text (Position (0, 0), "Sorting...");
  update ();
  sort ();
  update ();
  pane.erase ();

  m_cchFileColumn = m_pPaneFile->dx ()/(m_cchFilenameMax + 2);
  m_cchFileColumn = m_cchFileColumn 
    ? (m_pPaneFile->dx ()/m_cchFileColumn) : m_pPaneFile->dx ();

  dirty_path ();
  //  m_fDirtyFile = m_fDirtyDir = m_fDirtyStatus = true;
}


int LCurves::getch (void)
{
  int x, y;
  if (!m_iPane) {
    cursorpos_dir (x, y);
    return m_pPaneDir->getch (Position (x, y));
  }
  else {
    cursorpos_file (x, y);
    return m_pPaneFile->getch (Position (x, y));
  }
}


void LCurves::init (void)
{
  m_termcap.init ();
  m_pWindow = new Window;
  m_pWindow->init (&m_termcap);
  m_pWindow->inhibit_alt_charset (g_preferences.as_bool ("InhibitAltCharset"));
  m_pWindow->reset ();		// Reset to a known state
  m_pWindow->erase ();
  //  m_pScreen = new LScreen;
  m_fDirtyRoot = m_fDirtyMenu = true;
  //  cbreak ();			// Raw input mode
  //  nonl ();			// Don't translate /r/n

  m_rgClassMark[classEdited]       = '>';
  m_rgClassMark[classNewAdd]	   = '?';
  m_rgClassMark[classAdded]	   = '+';
  m_rgClassMark[classRemoved]	   = '-';
  m_rgClassMark[classMerge]	   = '!';
  m_rgClassMark[classPatch]	   = '<';
  m_rgClassMark[classCheckout]	   = '=';
  m_rgClassMark[classLost]	   = '^';
  m_rgClassMark[classConflict]	   = '*';
  m_rgClassMark[classUpToDate]     = ':';
  m_rgClassMark[classUncontrolled] = ' ';
  m_color_scheme = atoi (g_preferences.fetch ("ColorScheme"));
  init_colors ();

  m_dxDirectories = 15;		// Width of directory pane

  //  LWindow& root  = *m_pScreen->root ();
  Window& win = *m_pWindow;
  
  m_pPaneMenu    = new Pane (win, Rectangle (0, 0, win.dx (), 1));
  m_pPaneMenu->attrBg () = m_rgattrClass[classMenu];
  m_pPaneDir     = new Pane (win, Rectangle (0, 1, m_dxDirectories, 
					      win.dy () - 3));
  m_pPaneFile    = new Pane (win, Rectangle (m_dxDirectories + 1, 1,
					      win.dx () - m_dxDirectories - 1, 
					      win.dy () - 3));
  m_pPaneStatus  = new Pane (win, Rectangle (0, win.dy () - 2, win.dx (), 1));
  m_pPaneStatus->attrBg () = m_rgattrClass[classStatus];
  m_pPaneCommand = new Pane (win, Rectangle (0, win.dy () - 1, win.dx (), 1));
}


void LCurves::init_colors (void)
{
				// Force monochrome scheme
  if (m_color_scheme == 0 && !m_termcap.num (termcapNumColors))
    m_color_scheme = 2;

#define _DEF_COLOR(cl,fg,bg,m) m_rgattrClass[(cl)].set \
  	(Attribute::##fg | Attribute::##bg | Attribute::##m)

  switch (m_color_scheme) {
  default:
  case 0:			// Linux console, white on black display
    _DEF_COLOR (classEdited,       FgYellow,  BgDefault, Bold);
    _DEF_COLOR (classNewAdd,       FgGreen,   BgDefault, Bold);
    _DEF_COLOR (classAdded,        FgMagenta, BgDefault, Plain);
    _DEF_COLOR (classRemoved,      FgMagenta, BgDefault, Plain);
    _DEF_COLOR (classMerge,        FgMagenta, BgDefault, Bold);
    _DEF_COLOR (classPatch,        FgMagenta, BgDefault, Bold);
    _DEF_COLOR (classCheckout,     FgMagenta, BgDefault, Plain);
    _DEF_COLOR (classLost,         FgRed,     BgDefault, Plain);
    _DEF_COLOR (classConflict,     FgRed,     BgDefault, Plain);
    _DEF_COLOR (classUpToDate,     FgCyan,    BgDefault, Plain);
    _DEF_COLOR (classUncontrolled, FgWhite,   BgDefault, Plain);
    _DEF_COLOR (classDir,	   FgWhite,   BgDefault, Plain);

    _DEF_COLOR (classStatus, 	   FgBlack,   BgWhite,   Plain);
    _DEF_COLOR (classMenu, 	   FgBlack,   BgWhite,   Plain);
    _DEF_COLOR (classCursor, 	   FgBlue,    BgDefault, Bold);
    break;

  case 1:			// XTerm, black on white display
    _DEF_COLOR (classEdited,       FgBlue,    BgDefault, Bold);
    _DEF_COLOR (classNewAdd,       FgGreen,   BgDefault, Plain);
    _DEF_COLOR (classAdded,        FgRed,     BgDefault, Plain);
    _DEF_COLOR (classRemoved,      FgMagenta, BgDefault, Bold);
    _DEF_COLOR (classMerge,        FgRed,     BgDefault, Bold);
    _DEF_COLOR (classPatch,        FgRed,     BgDefault, Bold);
    _DEF_COLOR (classCheckout,     FgRed,     BgDefault, Plain);
    _DEF_COLOR (classLost,         FgRed,     BgDefault, Plain);
    _DEF_COLOR (classConflict,	   FgRed,     BgDefault, Plain);
    _DEF_COLOR (classUpToDate,     FgCyan,    BgDefault, Plain);
    _DEF_COLOR (classUncontrolled, FgDefault, BgDefault, Plain);
    _DEF_COLOR (classDir,	   FgDefault, BgDefault, Plain);

    _DEF_COLOR (classStatus, 	   FgBlack,   BgWhite,   Plain);
    _DEF_COLOR (classMenu, 	   FgBlack,   BgWhite,   Plain);
    _DEF_COLOR (classCursor, 	   FgDefault, BgDefault, Bold);
    break;

  case 2:			// Console, no color
    _DEF_COLOR (classEdited,       FgDefault, BgDefault, Inverse);
    _DEF_COLOR (classNewAdd,       FgDefault, BgDefault, Underline);
    _DEF_COLOR (classAdded,        FgDefault, BgDefault, Bold);
    _DEF_COLOR (classRemoved,      FgDefault, BgDefault, Bold);
    _DEF_COLOR (classMerge,        FgDefault, BgDefault, Underline);
    _DEF_COLOR (classPatch,        FgDefault, BgDefault, Underline);
    _DEF_COLOR (classCheckout,     FgDefault, BgDefault, Bold);
    _DEF_COLOR (classLost,         FgDefault, BgDefault, Bold);
    _DEF_COLOR (classConflict,     FgDefault, BgDefault, Bold);
    _DEF_COLOR (classUpToDate,     FgDefault, BgDefault, Plain);
    _DEF_COLOR (classUncontrolled, FgDefault, BgDefault, Plain);
    _DEF_COLOR (classDir,	   FgDefault, BgDefault, Plain);

    _DEF_COLOR (classStatus, 	   FgDefault, BgDefault, Inverse);
    _DEF_COLOR (classMenu, 	   FgDefault, BgDefault, Inverse);
    _DEF_COLOR (classCursor, 	   FgDefault, BgDefault, Bold);
    break;
  }
}


/* LCurves::name_files

   returns a list of the files matching the specified criteria.  In
   general, this call is used to name the tagged files so we make a
   provision for the case where no files are tagged by returning the
   name of the currently selected file.

   By default, it returns full pathnames to each file.  If the
   fFilenameOnly flag is set only the filenames are returned.

   The flagMask and flagSet parameters control the bits we check and
   whether those bits must be set or clear.

   The return value is a pointer to a malloc'd buffer of file names.
   If there were no files that match the selection criteria the
   returned pointer will be NULL.

*/

char* LCurves::name_files (bool fFilenameOnly, int flagMask, int flagSet)
{
				// -- Make space for file list
  int cFiles = ((flagMask & flagSet & flagTag) 
		? (m_cFilesTagged ? m_cFilesTagged : 1)
		: m_cFiles);
  char* szFiles
    = (char*) malloc (cFiles*((fFilenameOnly ? 0 : strlen (m_dir.path ()) + 1)
			      + m_cchFilenameMax + 1));
  *szFiles = 0;

				// -- Special case for selecting for
				//    tagged files and there are none
  if ((flagMask & flagSet & flagTag) && !m_cFilesTagged) {
    flagMask &= ~flagTag;
    flagSet  &= ~flagTag;
    if ((m_cvs.flags (_idx_file (m_iFileCursor)) & flagMask) == flagSet) {
      m_dir.stat (_idx_file (m_iFileCursor));
      if (!fFilenameOnly) {
	strcat (szFiles, m_dir.path ());
	strcat (szFiles, "/");
      }
      strcat (szFiles, _file (m_iFileCursor));
    }
  }
				// -- Normal case for selecting
				//    arbitrary list of files
  else {
    for (int iFile = m_cFiles; iFile--; ) {
      if ((m_cvs.flags (_idx_file (iFile)) & flagMask) != flagSet)
	continue;
      m_dir.stat (_idx_file (iFile));
      if (!fFilenameOnly) {
	strcat (szFiles, m_dir.path ());
	strcat (szFiles, "/");
      }
      strcat (szFiles, _file (iFile));
      strcat (szFiles, " ");
    }
  }
  if (!*szFiles) {
    free (szFiles);
    szFiles = NULL;
  }
  return szFiles;
}


/* LCurves::_rank

   performs sort order ranking.  The method is an ASCII character
   specifying the method of comparison.  'a' for alphanumeric
   ascending, 'c' for file class in descending order of importance,
   'n' for newest first, 'o' for oldest first, 'l' for longest file
   length, 's' for shortest file length, and 't' for tagged files
   first.  We do one other ranking before employing the above method
   rank.  We sort directories to the top of the list, files next, and
   other files, such as links, to the end.  Also, we enforce an
   alphanumeric sort on directories.

*/

int LCurves::_rank (char method, FILE_INFO* pInfo0, FILE_INFO* pInfo1)
{
  int fileRank0 = LDirectory::rank (pInfo0);
  int fileRank1 = LDirectory::rank (pInfo1);
  if (fileRank0 == fileRank1) {
    if (fileRank0 != fileRankFile)
      method = 'a';
  }
  else
    return fileRank0 - fileRank1;

  switch (method) {
  case 'a':			// Alpha
    {
      int result = strcasecmp (LDirectory::file (pInfo0), 
			       LDirectory::file (pInfo1));
      if (!result)
	return strcmp (LDirectory::file (pInfo0), 
		       LDirectory::file (pInfo1));
      else
	return result;
    }
    break;
  case 'A':
    {
      return strcmp (LDirectory::file (pInfo0), 
		     LDirectory::file (pInfo1));
    }
    break;
  case 'c':			// Class
  case 'C':
    return _classify (pInfo0) - _classify (pInfo1);
    break;
  case 'n':			// Newest
  case 'N':
    return LDirectory::time (pInfo1) - LDirectory::time (pInfo0);
    break;
  case 'o':			// Oldest
  case 'O':
    return LDirectory::time (pInfo0) - LDirectory::time (pInfo1);
    break;
  case 'l':			// Longest
  case 'L':
    return LDirectory::size (pInfo1) - LDirectory::size (pInfo0);
    break;
  case 's':			// Smallest
  case 'S':
    return LDirectory::size (pInfo0) - LDirectory::size (pInfo1);
    break;
  case 't':			// Tag
  case 'T':
    return 0;
    break;
  default:
    return 0;
  }
}


void LCurves::refresh (bool fForce)
{
  g_pCvs = &m_cvs;		// Required for classify.  Sorry.

  if (fForce)
    m_fDirtyRoot = true;

  if (m_fDirtyRoot) {
    m_pWindow->erase ();	// Clear screen
    m_pWindow->draw_vertical_line (m_dxDirectories, 1, m_pPaneDir->dy ());
    m_fDirtyRoot   = false;
    m_fDirtyMenu   = true;
    m_fDirtyDir    = true;
    m_fDirtyFile   = true;
    m_fDirtyStatus = true;
  }

  if (m_fDirtyMenu) {
    m_pPaneMenu->draw_text (Position (0, 0), "%-*.*s", 
			    m_pPaneMenu->dx (), m_pPaneMenu->dx (), 
			    m_rgszMenu[m_iMenu]);
    m_fDirtyMenu = false;
  }

  if (m_fDirtyDir) {
    for (int iDir = m_iDirScroll; 
	 iDir < m_iDirScroll + m_pPaneDir->dy ()
	   && iDir < m_cDirectories; ++iDir)
      render_dir (iDir);
    m_fDirtyDir = false;
  }

  if (m_fDirtyFile) {
    int dy = m_pPaneFile->dy ();
    int dx = m_pPaneFile->dx ();
    int y = 0;
    int x = 0;			// Tag column skipped
    for (int iFile = m_iFileScroll; iFile < m_cFiles && x < dx - 1; ++iFile) {

				// -- Draw mark
      bool fTagged = m_cvs.is_tagged (_idx_file (iFile));
      m_pPaneFile->attr () = fTagged
	? m_rgattrClass[classCursor]
	: Attribute::normal;
      m_pPaneFile->draw_text (Position (x, y), "%c", fTagged ? '`' : ' ');

				// -- Draw file and class
      int iClass = _classify (_info_file (iFile));
      m_pPaneFile->attr () = m_rgattrClass[iClass];
      m_pPaneFile->draw_text (Position (x + 1, y++), "%c%-*.*s", 
			      m_rgClassMark[iClass], 
			      m_cchFileColumn - 2, m_cchFileColumn - 2,
			      _file (iFile));
      if (y == dy) {
	y = 0;
	x += m_cchFileColumn;
      }	  
    }
#if 0
    y = 0;
    x = 1;			// Tag column skipped
    for (int iFile = m_iFileScroll; iFile < m_cFiles && x < dx; ++iFile) {
				// -- Draw tag
      m_pPaneFile->attr () = m_rgattrClass[classCursor];
      m_pPaneFile->draw_text (Position (x - 1, y++), "%c",
			      m_cvs.is_tagged (_idx_file (iFile)) ? '`' : ' ');

      if (y == dy) {
	y = 0;
	x += m_cchFileColumn;
      }	  
    }
#endif
    m_fDirtyFile = false;
  }

  if (m_fDirtyStatus) {
    int dx = m_pPaneStatus->dx ();
    char sz[200];
    if (m_iPane == 0 || !is_files ()) {
      if (m_iPane == 0) {
	sprintf (sz, " %-*.*s  ", CB_DIR_CURRENT, CB_DIR_CURRENT,
		 m_dir.abbreviate (CB_DIR_CURRENT));
	if (is_files ()) {
	  sprintf (sz + strlen (sz), "%3d director%s", m_cDirectories,
		   (m_cDirectories == 1) ? "y  " : "ies");
	  sprintf (sz + strlen (sz), "  %4d file%s (%d)   ", m_cFiles, 
		   (m_cFiles == 1) ? " " : "s", m_cFilesTagged);
	}
      }
    }
    else {
      sprintf (sz, " %-*.*s  %8ld byte%s  ",
	       CB_FILE_CURRENT, CB_FILE_CURRENT,
	       _file (m_iFileCursor), 
	       m_dir.size (_idx_file (m_iFileCursor)),
	       (m_dir.size (_idx_file (m_iFileCursor)) == 1) ? " " : "s");
      time_t time = m_dir.time (_idx_file (m_iFileCursor));
      char* pch = ctime (&time);
      strncat (sz, pch, strcspn (pch, "\n"));
      switch (_classify (_info_file (m_iFileCursor))) {
      default:
      case classUncontrolled:
	if (m_cvs.is_binary (_idx_file (m_iFileCursor)))
	  strcat (sz, "  BINARY");
	break;
      case classUpToDate:
	strcat (sz, "  --UpToDate--");
	break;
      case classEdited:
	strcat (sz, "  --Edited--");
	break;
      case classPatch:
	strcat (sz, "  --Patch--");
	break;
      case classMerge:
	strcat (sz, "  --Merge--");
	break;
      case classNewAdd:
	if (m_cvs.is_binary (_idx_file (m_iFileCursor)))
	  strcat (sz, "  --NewAdd (B)--");
	else
	  strcat (sz, "  --NewAdd (T)--");
	break;
      case classAdded:
	strcat (sz, "  --Added--");
#if 0
				// FIXME: In order to make this work,
				// we need to read the flags from the
				// Entries file and set the flagBinary
				// flag according to the status of
				// that flag.
	if (m_cvs.is_binary (_idx_file (m_iFileCursor)))
	  strcat (sz, "  --Added (B)--");
	else
	  strcat (sz, "  --Added (T)--");
#endif
	break;
      case classRemoved:
	strcat (sz, "  --Removed--");
	break;
      case classLost:
	strcat (sz, "  --Lost--");
	break;
      case classConflict:
	strcat (sz, "  --CONFLICT--");
	break;
      case classCheckout:
	strcat (sz, "  --Checkout--");
	break;
      }
    }
    m_pPaneStatus->draw_text (Position (0, 0), "%-*.*s", dx, dx, sz);
    m_fDirtyStatus = false;
  }

  show_cursor (true);

  update ();

}

void LCurves::release_screen (void)
{
  if (m_pPaneMenu) {
    delete m_pPaneMenu;
    m_pPaneMenu = NULL;
  }    
  if (m_pPaneDir) {
    delete m_pPaneDir;
    m_pPaneDir = NULL;
  }
  if (m_pPaneFile) {
    delete m_pPaneFile;
    m_pPaneFile = NULL;
  }
  if (m_pPaneStatus) {
    delete m_pPaneStatus;
    m_pPaneStatus = NULL;
  }
  if (m_pPaneCommand) {
    delete m_pPaneCommand;
    m_pPaneCommand = NULL;
  }
  delete m_pWindow;
  m_pWindow = NULL;
}

void LCurves::release_this (void)
{
  release_screen ();
}

void LCurves::render_dir (int iDir)
{
  const char* szDir = _dir (iDir);
  m_pPaneDir->draw_text (Position (1, iDir - m_iDirScroll), "%-*.*s%c", 
			 m_dxDirectories - 2, m_dxDirectories - 2, szDir,
			 ((int) strlen (szDir ? szDir : "")
			  > m_dxDirectories - 2) 
			 ? '\\' : ' ');
}

void LCurves::restore (bool fPrompt)
{
  Window& win = *m_pWindow;
  win.activate (true);		// Gives us back our control

  if (fPrompt) {
    m_pPaneCommand->draw_text (Position (0, 0), "Press a key to continue.");
    m_pPaneCommand->getch (Position (24, 0));
    m_pPaneCommand->erase ();
  }

  win.erase ();
  win.pos () = Position (0, 0);
  
  //  m_fDirtyRoot = true;		// *** fixme, this is here ONLY
				// because we aren't caching on-screen
				// data.  It happens to cause some
				// redisplay duplication, but there's
				// nothing easy to do about this.
  refresh ();
  win.refresh ();
  update ();
}

void LCurves::save (void)
{
  Window& win = *m_pWindow;
  win.push_attribute ();
  win.erase ();
  win.pos () = Position (0, 0);
  update ();		// Flush all changes before deactivating
  m_fDirtyRoot = true;		// *** FIXME next time, we need to
				// draw everything, but we should be
				// able to use cached screen information
  win.activate (false);
}

void LCurves::select_pane (int iPane)
{
  if (iPane != m_iPane) {
    show_cursor (false);
    m_iPane = iPane;
				// Update status
    m_fDirtyStatus = true;
    refresh ();
  }
}

void LCurves::set_menu (int iMenu)
{
  if (iMenu < 0 || iMenu >= m_cMenuMax)
    return;

  m_iMenu = iMenu;
  m_fDirtyMenu = true;
}


/* LCurves::set_path

   initializes the current directory structures to the specified path.
   If the szPath parameter is NULL, the current directory is
   refreshed.  If the path does not exist, nothing is done.

*/

void LCurves::set_path (const char* szPath)
{ 
  LDirectory dir (szPath);
  if (!dir.exists ())
    return;

  m_iPane = 0;
  m_dir.init (szPath); 
  dirty_path (); 
  m_cFilesTagged = 0;

  m_pPaneFile->erase ();
  m_pPaneDir->erase ();
  m_iDirScroll = m_iDirCursor = m_iFileScroll = m_iFileCursor = 0;
  m_cDirectories = m_cFiles = 0;

  m_cvs.init (&m_dir);
}

void LCurves::show_cursor (bool fShow)
{
  int x, y;
  cursorpos_dir (x, y);
  //  m_pPaneDir->attribute (m_rgClassAttribute[classCursor], TRUE);
  m_pPaneDir->attr () = m_rgattrClass[classCursor];
  m_pPaneDir->draw_text (Position (x, y), "%c", 
			 (fShow && m_iPane == 0 && is_files ()) ? '>' : ' ');
  m_pPaneDir->attr () = m_rgattrClass[classDir];
  char ch = ((m_cFiles && m_cvs.is_tagged (_idx_file (m_iFileCursor)))
	     ? ((m_iPane == 1 && fShow && is_files ()) ? '&' : '`')
	     : ((m_iPane == 1 && fShow && is_files ()) ? '>' : ' '));
  m_pPaneFile->attr () = m_rgattrClass[classCursor];
  cursorpos_file (x, y);
  m_pPaneFile->draw_text (Position (x, y), "%c", ch);
}


/* LCurves::sort

   resorts the file entries.  Used by the host to finalize the new
   sort criteria.  In addition, we compute the number of directories
   and files in the 

*/

void LCurves::sort (void)
{
  g_szSortCriteria = g_preferences.fetch ("Sort");
  assert_ (g_szSortCriteria);
  g_pCvs = &m_cvs;
  m_dir.sort (_sort_compare);
  m_iFileCursor = 0;

  m_cFiles = m_cDirectories = 0;
  m_cchFilenameMax = 0;

  for (int i = m_dir.count_files (); i--; ) {
    if (m_dir.is_file (i)) {
      ++m_cFiles;
      int cch = strlen (m_dir.file (i));
      if (cch > m_cchFilenameMax)
	m_cchFilenameMax = cch;
      continue;
    }
    if (m_dir.is_directory (i)) {
      ++m_cDirectories;
      continue;
    }
  }
}


/* LCurves::_sort_compare

   The criteria are a null terminated string of characters where the
   letter determines the sort method and the order determines th
   hierarchy.  
     a - alpha
     c - class
     n - newest
     o - oldest
     l - longest
     s - smallest
     t - tag

*/

int LCurves::_sort_compare (const void* pv0, const void* pv1)
{
  FILE_INFO* pInfo0 = (FILE_INFO*) pv0;
  FILE_INFO* pInfo1 = (FILE_INFO*) pv1;


  for (const char* pch = g_szSortCriteria; pch && *pch; ++pch) {
    int result = _rank (*pch, pInfo0, pInfo1);
    if (result)
      return result;
  }
  return 0;
}

void LCurves::tag (int iClass, bool fExclusive)
{
  show_cursor (false);

  for (int iFile = 0; iFile < m_cFiles; ++iFile) {
    int iClassThis = _classify (_info_file (iFile));
    if (iClassThis == iClass || fExclusive) {
      m_cFilesTagged +=
	((m_cvs.is_tagged (_idx_file (iFile)) == (iClassThis == iClass)) 
	 ? 0 : ((iClassThis == iClass) ? 1 : -1));
      m_cvs.tag (_idx_file (iFile), iClassThis == iClass);
    }
  }		 
  show_cursor (true);
  m_fDirtyFile = m_fDirtyStatus = true;
}

