# log.rb, copyright (c) 2006 by Vincent Fourmond: 
# The general logging functions for Ctioga
  
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
  
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details (in the COPYING file).

require 'logger'
require 'forwardable'

module CTioga

  Version::register_svn_info('$Revision: 653 $', '$Date: 2007-11-15 21:58:42 +0100 (Thu, 15 Nov 2007) $')

  # This module contains code to be included by PlotMaker for debugging.
  # Even if it doesn't deserve a real module of it's own, it makes sense
  # to have it separated anyway.
  module Log

    extend Forwardable
    def_delegators :@@log, :debug, :warn, :info, :error, :fatal

    def init_logger(stream = STDERR)
      Logger::Formatter::Format.replace("[%4$s] %6$s\n")
      @@log = Logger.new(stream)
      @@log.level = Logger::WARN # Warnings and more only by default
    end

    # Simple accessor for the @@log class variable.
    def logger
      return @@log
    end

    def logger_options(opt)
      opt.on("-v", "--verbose", "Display useful information") do
        @@log.level = Logger::INFO
      end
      opt.on("--debug", "Turn on debugging info") do
        @@log.level = Logger::DEBUG
      end
      opt.on("--quiet", "Display only errors") do
        @@log.level = Logger::ERROR
      end
    end

    # A logging replacement for system
    def spawn(cmd, priority = :info)
      retval = system(cmd)
      self.send(priority, "Running #{cmd} -> " + 
                if retval
                  "success"
                else
                  "failure"
                end
                )
      return retval
    end

    # Returns a string suitable for identification of an object, but
    # without instance variables.
    def identify(obj)
      return "#<#{obj.class} 0x%x>" % obj.object_id
    end

  end

end
