# layout.rb, copyright (c) 2006 by Vincent Fourmond: 
# The core of the layout computation
  
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
  
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details (in the COPYING file).

require 'Tioga/Utils'
require 'CTioga/log'
require 'CTioga/dimension'
require 'CTioga/boundaries'


module CTioga

  Version::register_svn_info('$Revision: 836 $', '$Date: 2008-10-10 01:27:54 +0200 (Fri, 10 Oct 2008) $')

  # This class merely stores various information about the layout,
  # such as the desired mimimum amount of padding on the sides,
  # the desired distance between plots and legends, the size and
  # position of the legend...
  #
  # For now, this class is hardly more than a Struct, but that is
  # a very convenient way to carry across several objects and keep the
  # documentation.
  #
  # Every Container should have a LayoutPreference object in its
  # layout_preferences attribute.
  class LayoutPreferences

    LayoutDefaults = {
      :padding => [Dimension.new(0.05),
                   Dimension.new(0.05),
                   Dimension.new(0.05),
                   Dimension.new(0.05)],
      :legend_size => Dimension.new(0.15),
      :legend_position => :right,
      :legend_glue => Dimension.new("2pt")
    }

    # Padding: (left, right, top, bottom)
    # An array of Dimensions specifiying the desired amount of padding
    # around a given object.
    attr_accessor :padding

    # The legend position: :left, :right, :top, :bottom or :inside
    attr_accessor :legend_position

    # The size of the legend. Not applicable in the case of an
    # :inside legend specification.
    attr_accessor :legend_size

    # The full legend specification, in the case of an inside legend.
    attr_accessor :legend_spec

    # The minimum space between the graph and the legend, when the latter
    # is on one of the sides of the graph.
    attr_accessor :legend_glue

    def initialize
      for key,val in LayoutDefaults
        self.send("#{key}=", val)
      end
    end


    # Converts the padding to values relative to the given
    # frames
    def self.padding_relative_values(p, frames)
      ret_val = []
      4.times do |i|
        if p[i]
          ret_val[i] = p[i].to_relative(frames,
                                        i < 2 ? :horizontal : :vertical)
        else
          ret_val[i] = 0        # safely default to 0 ???
        end
      end
      return ret_val
    end

    def padding_relative_values(frames)
      return LayoutPreferences.padding_relative_values(@padding, frames)
    end
  end

  # Handles the gory details of placement of objects. It can be called
  # to provide any of the specification for a subplot/subfigure, or
  # the hash necessary to be fed to show_plot_with_legend.
  #
  # Every layout object has a root object (object), the base one,
  # the one which
  # limits its capabilities. No layout object should touch anything
  # that is not a descendant of the root object -- or the root object
  # itself.
  #
  # In addition to that, the layout can handle many children of this
  # root object that request it.
  class Layout

    include Log

    # The object the layout is handling. Should never be null.
    attr_accessor :object
    
    # The layouts that would wish to report extensions information
    # to this one.
    attr_accessor :child_layouts

    # The parent layout. Can be nil. Most of the layouts won't
    # reall take something from this.
    attr_accessor :parent

    # Performs basic initialization. All descendants from this class
    # should have at least three arguments, and at most three mandatory
    # ones, always the same. Else, the convert_layout function
    # won't work, and that means trouble.
    def initialize(ro, ch = [], children = [] )
      @object = ro
      @object.layout = self
      @child_layouts = []
      for c in children
        add_child(c)
      end
      @parent = nil
    end

    # Converts self into another layout, whose class is _cls_.
    def convert_layout(cls)
      return cls.new(object, @child_layouts)
    end

    # Ask this layout to manage a child. No check is done to
    # actually ensure that the child really is a child.
    def add_child(child)
      child.parent = self
      @child_layouts << child
    end
    
    # Returns the specs to be fed to show_plot_with_legend
    # for the given _object_.
    #
    # Defaults to full plot.
    def legend_specs(object, t = nil)
      return [0,0,0,0].to_frame("plot_%s_margin")
    end

    # Now, various utility functions that should make the
    # life much easier for derived classes (or at least
    # to the person who'll be writing them).

    # A utility function that adds an extension as reported by
    # Axes#extension (or any other, for what matters)
    # to an already existing array of _extensions_.
    # It will replace the current element unless it is smaller.
    def update_extension_with_object(extensions, object, t, scale = 1)
      new_ext = object.extension(t,scale)
      return update_extensions(extensions, new_ext)
    end

    # Simply updates the given extensions with the new ones, making
    # sure that nall the objects can be crammed inside.
    def update_extensions(extensions, new_ext)
      return Dimension.update_extensions(extensions, new_ext)
    end

    # Returns the extension of the layout
    def extension(t)
      extensions = [0,0,0,0]
      # TODO: maybe this code should move within the PlotStyle object
      for a in [:title, :xlabel, :ylabel]
        update_extension_with_object(extensions, 
                                     @object.plot_style.send(a),
                                     t, @object.rescale || 1)
      end
      update_extension_with_object(extensions, @object.plot_style.edges,t)
      debug "Layout #{"%x" % self.object_id} extensions #{extensions.inspect}"
      return extensions
    end

    # Computes the extension for the legend, according to the
    # position. Does not check if the object actually needs to
    # display any legend whatsoever.
    def legend_extension(t)
      frame = @object.outer_frame(t)
      case @object.layout_preferences.legend_position
      when :left
        return [@object.layout_preferences.
                legend_size.to_absolute(frame, :horizontal),0,0,0]
      when :right
        return [0,@object.layout_preferences.
                legend_size.to_absolute(frame, :horizontal),0,0]
      when :top
        return [0,0,@object.layout_preferences.
                legend_size.to_absolute(frame, :vertical),0]
      when :bottom
        return [0,0,0,
                @object.layout_preferences.
                legend_size.to_absolute(frame, :vertical)]
      when :inside
        return [0,0,0,0]        # Inside doesn't count !
      else
        warn "Invalid legend position specification"
        return [0,0,0,0]        # Does not count.
      end
    end

    # All the things about legends here should eventually be
    # moved to their own file, very much like the axes things. There
    # is no real reason to differentiate them.

    # Converts the legend extension to a full legend specification.
    def legend_only_specs(t, padding)
      frame = @object.outer_frame(t)
      case @object.layout_preferences.legend_position
      when :inside
        # Use the legend_spec without any other modification.
        return @object.layout_preferences.legend_spec
      else
        ext = legend_extension(t)
        mar = Dimension.absolute_to_relative(ext, frame)
      end

      target = [0,0,0,0]
      4.times do |i|
        if mar[i] > 0
          target[(i/2) * 2 + ((i+1) % 2)] = 1 - mar[i]
        end
      end

      # Now, the legend spec is nearly ready; we just need to
      # modify it so that it matches the sides of the plot.
      case @object.layout_preferences.legend_position
      when :left, :right
        target[2] = padding[2]
        target[3] = padding[3]
      when :top, :bottom
        target[0] = padding[0]
        target[1] = padding[1]
      end

      return target.to_frame("legend_%s_margin")
    end

    # Returns the padding for the given object, taking into
    # account the position of legends:
    def compute_padding(object)
      prefs = @object.layout_preferences
      padding = @object.layout_preferences.padding.dup

      # If the object has a legend 
      if @object.display_legend? && Utils::side?(prefs.legend_position)
        padding[Utils::side(prefs.legend_position)] = 
          prefs.legend_glue.dup
      end
      return padding
    end

  end

  # Now come the interesting part: the layouts !!!!


  # This class takes care of simple plots. All children
  # are considered for computing the extension of the plots, but
  # they won't receive anything else than a default full frame.
  #
  # For now, the legend is not taken into consideration -- and the
  # children as well ;-) !!
  #
  # The SimpleLayout class is also meant as a base class for any
  # layout that would need to know its extensions, but that will
  # not get positionned at all in the same way.
  class SimpleLayout < Layout

    # A hash setting the 'bounding box' for the plot. It is relative
    # to the Container#outer_frame. Defaults to 0: the object occupies
    # all the frame.
    attr_accessor :bounding_box

    def initialize(ro, ch = [], children = [] )
      super
      @bounding_box = [0.0, 0.0, 0.0, 0.0]
    end

    # Returns the object's frame.
    def legend_specs(t = nil)
      a = extension(t)
      frames = @object.outer_frame(t)
      padding = Dimension.absolute_to_relative(a, frames)

      padding_prefs = 
        LayoutPreferences.padding_relative_values(compute_padding(@object), 
                                                  frames)

      update_extensions(padding, padding_prefs)
      
      # If we display a legend, add the legend to the side.
      if @object.display_legend?
        a = Dimension.absolute_to_relative(legend_extension(t),frames)
        4.times do |i|
          padding[i] += a[i]
        end
      end
      
      specs = {}
      i = 0
      for side in %w(left right top bottom)
        specs["plot_#{side}_margin"] = padding[i] + @bounding_box[i]
        i += 1
      end
      if @object.display_legend?
        a = legend_only_specs(t, padding)
        return specs.merge(a)
      else
        return specs
      end
    end

    # Returns this layout's extension. By default, it also
    # includes the children's extensions, unless the
    # @ignore_children_extensions variable is set.
    def extension(t)
      # We use the simple extensions computation from Layout
      a = super
      # We update it with the children's informations:
      unless @ignore_children_extensions
        for f in @child_layouts
          update_extensions(a, f.extension(t))
        end
      end
      return a
    end

  end

  # This class implements a layout for objects that only need to report
  # extension information to the parent but don't need to really take
  # this into consideration for themselves. A good example is the
  # 'alternative axes' things.
  class FixedLayout < SimpleLayout

    # The margins of the given object, either an array or a hash
    # with plot_%s_margin
    attr_accessor :position

    # Creates a Fixed layout for the given object.
    def initialize(ro, ch = [], children = [])
      super
      @position = [0, 0, 0, 0]
    end

    # Return default plot specifications: we don't take the
    # legend into consideration, for instance.
    def legend_specs(object, t = nil)

      # First, the position of the layout:
      if @position.is_a? Hash
        specs = @position.dup
      else
        specs = @position.to_frame("plot_%s_margin")
      end


      # Then handling of the legends if applicable:
      # TODO: this will fail for normal legends....
      if @object.display_legend?
        case @object.layout_preferences.legend_position
        when :inside
          a = @object.layout_preferences.legend_spec
          
          legend = Utils::frame_to_array(a, "legend_%s_margin")
          frame = Utils::frame_to_array(specs, "plot_%s_margin")
          
          target = Utils::compose_margins(frame, legend)
          a = target.to_frame("legend_%s_margin")
        else
          warn "Legend position #{@object.layout_preferences.legend_position.inspect} not implemented for FixedLayout"
          a = {}
        end
        return specs.merge(a)
      else
        return specs
      end
    end

  end

  # A layout that will handle objects aligned on a grid. All objects
  # belonging to one row of the grid will share the same vertical padding.
  # All the ones belonging to a column will share the same
  # horizontal padding. All layouts added to this one should
  # be a GridItemLayout or a descendant, or at least something
  # behaving correctly in that respect.
  class GridLayout < SimpleLayout

    def initialize(*a)
      super
      # We don't want the children's extensions messing
      # with this layout's extension
      @ignore_children_extensions = true
    end

    # Some various 'accessors' for rows and columns:

    # Sets the number of rows. The number of columns will be computed
    # automatically.
    def rows=(nb)
      @rows = nb
      @columns = false
    end

    # Returns the number of rows
    def rows
      return @rows if @rows
      return (@child_layouts.size.to_f / @columns).ceil
    end

    # Sets the number of columns. The number of rows will be computed
    # automatically.
    def columns=(nb)
      @columns = nb
      @rows = false
    end

    # Returns the number of rows
    def columns
      return @columns if @columns
      return (@child_layouts.size.to_f / @rows).ceil
    end

    # Returns the position of the given child layout in terms of
    # [row, col]. The layouts are added first in columns
    def child_position(child)
      i = @child_layouts.index(child)
      return [i % rows, i / rows]
    end

    # Returns the child at position (row, col)
    def child(row, col)
      return @child_layouts[col * rows + row ]
    end
    
    # Computes the various extensions for each columns and row
    def compute_extensions(t)
      @rows_extensions = []
      rows.times do |i|
        a = [0,0,0,0]
        columns.times do |j|
          update_extensions(a, child(i,j).extension(t)) if child(i,j)
        end
        @rows_extensions[i] = a
      end
      debug "Grid layout rows #{@rows_extensions.inspect}"
      @columns_extensions = []
      columns.times do |j|
        a = [0,0,0,0]
        columns.times do |i|
          update_extensions(a, child(i,j).extension(t)) if child(i,j)
        end
        @columns_extensions[j] = a
      end
      debug "Grid layout columns #{@columns_extensions.inspect}"
    end

    # Computes the positions of the various columns/rows, so that
    # all the objects in the same line/column share the same size
    # in both directions -- that is, all plots are identical.
    def compute_positions(t)
      compute_extensions(t) unless @rows_extensions
      frame = @object.outer_frame(t)
      @rows_positions = []
      vert_ext = 0
      vert_top = []
      vert_bottom = []
      for f in @rows_extensions
        a = Dimension.absolute_to_relative(f,frame)
        vert_top << a[2]
        vert_bottom << a[3]
        vert_ext += a[2] + a[3]
      end

      # Now, vert_top and vert_bottom contain the necessary
      # spacing above and below each element, in relative size.
      height = (1.0 - vert_ext)/rows
      debug "Grid layout height #{height}"
      current = 0
      rows.times do |i|
        new = current + vert_top[i] + height + vert_bottom[i]
        @rows_positions << [0,0, # ignored
                            current, 1 - new]
        current  = new
      end

      debug "Grid layout rows positions #{@rows_positions.inspect}"

      # Now, rows_positions contains the vertical position of
      # the given row.

      # We can now do the same for the horizontal positions:
      @columns_positions = []
      horiz_ext = 0
      horiz_left = []
      horiz_right = []
      for f in @columns_extensions
        a = Dimension.absolute_to_relative(f,frame)
        horiz_left << a[0]
        horiz_right << a[1]
        horiz_ext += a[0] + a[1]
      end
      debug "Grid horiz left #{ horiz_left.inspect }"
      debug "Grid horiz right #{horiz_right.inspect}"

      # Now, horiz_left and horiz_right contain the necessary
      # spacing above and below each element, in relative size.
      width = (1.0 - horiz_ext)/columns
      debug "Grid layout width #{width}"
      current = 0
      columns.times do |i|
        new = current + horiz_left[i] + width + horiz_right[i]
        @columns_positions << [current, 1 - new, 0, 0]
        current =  new
      end
      debug "Grid layout columns positions #{@columns_positions.inspect}"
      
    end

    # Returns the extension for the given child.
    def child_extension(child,t)
      compute_extensions(t) unless @columns_extensions
      row,col = child_position(child)
      a = @rows_extensions[row].dup
      # We override the horizontal extensions with the
      # @columns_extensions
      a[0] = @columns_extensions[col][0]
      a[1] = @columns_extensions[col][1]
      return a
    end

    # Returns the frame allocated for one item of the layout.
    def child_frame(t, row,col = nil)
      if row.is_a? Layout       # Row is a child, actually...
        return child_frame(t, *child_position(row))
      end

      compute_positions(t) unless @rows_positions
      a = @rows_positions[row].dup
      a[0] = @columns_positions[col][0]
      a[1] = @columns_positions[col][1]
      return a
    end

    
  end

  # The layout that should be used for every single object
  # in a grid. Should.
  class GridItemLayout < SimpleLayout

    # Returns the object's frame.
    def legend_specs(t = nil)
      # To compute the legend specification, 
      a = parent.child_extension(self, t)

      frames = @object.outer_frame(t)
      padding = Dimension.absolute_to_relative(a, frames)

      padding_prefs = @object.layout_preferences.
        padding_relative_values(frames)

      update_extensions(padding, padding_prefs)
      my_frame = parent.child_frame(t, self)
      
      debug "Child frame #{my_frame.inspect}"
      debug "Child padding #{padding.inspect}"
      debug "Child padding prefs #{padding_prefs.inspect}"
      
      specs = {}
      i = 0
      for s in %w(left right top bottom)
        specs["plot_#{s}_margin"] = padding[i] + my_frame[i]
        i += 1
      end
      debug "Child specs #{specs.inspect}"
      return specs
    end
  end

end
