# curves.rb, copyright (c) 2006 by Vincent Fourmond: 
# The class describing a curve to be plotted.
  
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
  
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details (in the COPYING file).

require 'CTioga/elements'
require 'CTioga/utils'
require 'CTioga/boundaries'

module CTioga

  Version::register_svn_info('$Revision: 873 $', '$Date: 2009-01-20 17:07:00 +0100 (Tue, 20 Jan 2009) $')

  # This structure holds the different properties to be transmitted to
  # a Curve object for appropriate drawing. 

  # TODO: create StrokeStyle and a FillStyle objects that would handle every
  # place where there are fills and strokes (background, grids,
  # legend frame and of course curves) !!!

  class CurveStyle
    ELEMENTS = [
                :color, :marker, :marker_color,
                :line_style, :legend, :linewidth,
                :interpolate, :marker_scale, 
                :error_bar_color, :drawing_order,
                :transparency,   # Stroke transparency
                :marker_transparency, 
                :error_bars_transparency,
                # Now, fill information
                :fill_type,     # not false/nil => there will be
                # some filling. See Curve2D#draw_fill for more info
                :fill_color, :fill_transparency,
                :hist_type,      # See below
                :hist_left,     # the position of the left part of the step
                :hist_right,     # the position of the right part of the step
               ]
    attr_writer *ELEMENTS
    
    # Special accessors: they implement the :"=>stuff" redirection. Beware
    # of circular redirections !!!
    for el in ELEMENTS
      eval <<"EOE"
def #{el.to_s}
  if @#{el.to_s}.is_a?(Symbol) && @#{el.to_s}.to_s =~ /^=>(.*)/
    new_sym = $1.to_sym
    if new_sym == :#{el.to_s}
      return nil
    end
    return self[new_sym]
  else
    return @#{el.to_s}
  end
end
EOE
    end

    # The order of the plotting operations. Default is 0
    # (path then markers then error_bars)
    DrawingOrder = [
                    [:path, :markers, :error_bars],
                    [:path, :error_bars, :markers],
                    [:markers, :path, :error_bars],
                    [:markers, :error_bars, :path ],
                    [:error_bars, :path, :markers],
                    [:error_bars,  :markers, :path]
                   ]

    Defaults = {
      :marker_scale => 0.5,
      :error_bar_color => :'=>color', # Defaults to the same as color
      :drawing_order => 0,
      :transparency => false,
      :fill_type => false,      # No fill by default.
      :fill_color => :'=>color', # Defaults to the same as color
      :hist_type => false, # old style by default...
      :hist_left => 0.0,       # joint lines by default
      :hist_right => 1.0,       # joint lines by default
    }

    # A hash to deal with fill types, to be used as an argument for
    # the Utils::interpret_arg function.
    FillTypeArguments = {
      /no(ne)?/ => false,
      /y[_-]ax(is)?/ => :to_y_axis,
      /bottom/ => :to_bottom,
      /top/ => :to_top,
      /old-styke/ => :old_style, # essentially for histograms
    }
    
    # Creates a CurveStyle element. There are several ways to
    # initialize:
    # * no arguments (_args_ empty): a CurveStyle object is created
    #   with all its values set to nil
    # * at least one argument: arguments are taken as values in the
    #   order given by the ELEMENTS array. In this case, arguments
    #   not present default to the value given in the Defaults
    #   hash. This behaviour is mainly to keep old things working.
    # * The new and best way to do it is to feed it a hash, in which
    #   case elements not present in the hash get the values in
    #   Default.
    def initialize(*args)
      if args.length == 0
        return                  # We don't set anything in this case
      end
      
      if (h = args[0]).is_a? Hash
        for el in ELEMENTS
          if h.key?(el)
            self[el] = h[el]
          elsif Defaults.key?(el)
            self[el] = Defaults[el]
          end
        end
      else
        # If there is at least one argument, we consider them as
        # values in the order of ELEMENTS
        for el in ELEMENTS
          if args.length > 0
            self[el] =  args.shift
          else
            if Defaults.key?(el)
              self[el] = Defaults[el]
            end
          end
        end
      end

      # This is not the place where to do this. It should be implemented
      # as part of the attribute accessors.
#       # Now, if there is any entry with a value of a symbol starting with
#       # => , its value is replaced by the value for the pointed elements.
#       # That is =>color means 'replace by the color'. Be careful however
#       # with circular dependencies !!!
#       for el in ELEMENTS
#         if self[el].is_a?(Symbol) && self[el].to_s =~ /^=>(.*)/
#           self[el] = self[$1.to_sym]
#         end
#       end
    end
    
    # Overrides the style entries with the ones found in _other_.
    def override!(other)
      for iv in ELEMENTS
        if other.instance_variables.include?("@" + iv.to_s)
          # We use instance_variables.include? rather
          # than instance_variable_defined?, as the latter
          # is present only in recent versions of Ruby 1.8
          send(iv.to_s + "=", other.send(iv))
        end
      end
    end

    # Returns a hash suitable to pass on to save_legend_info, or return
    # false if there was no legend specified in the style.
    def legend_info
      if legend 
        legend_info = {
          'text' => legend,
          'marker' => marker,
          'marker_color' => marker_color,
          'marker_scale' => marker_scale, 
        }
        if color && line_style
          legend_info['line_color'] = color
          legend_info['line_type'] = line_style
        else                             # Line drawing is disabled.
          legend_info["line_width"] = -1 
        end
        return legend_info
      else
        return false
      end
    end

    # Returns true if the style element actually carries a legend
    def has_legend?
      return legend ? true : false
    end


    # Remove attributes. Can be used to 'unset' current default.
    def delete(*vars)
      for iv in vars
        begin
          remove_instance_variable('@' + iv.to_s)
        rescue NameError
        end
      end
    end

    # Hash-like accessor:
    def [](elem)
      if ELEMENTS.include?(elem)
        return self.send(elem)
      else
        raise NameError, "Unkown element of CurveStyle: #{elem}"
      end
    end

    def []=(elem, value)
      if ELEMENTS.include?(elem)
        self.send(elem.to_s + '=', value)
      else
        raise NameError, "Unkown element of CurveStyle: #{elem}"
      end
    end

    # A function to be used to output legend pictograms separately.
    # It takes all the place available in the current context of the
    # FigureMaker object _t_.
    def output_legend_pictogram(t)
      t.context do
        # output line
        if color && line_style 
          t.line_color = color
          t.line_width = linewidth if linewidth
          #           t.line_cap = dict['line_cap']
          t.line_type = line_style
          t.stroke_line(0.0, 0.5, 1.0, 0.5)
        end
        if marker
          t.line_type = Tioga::FigureConstants::Line_Type_Solid
          t.show_marker( 'x' => 0.5,
                         'y' => 0.5,
                         'marker' => marker,
                         'color' => marker_color,
                         'scale' => marker_scale,
                         'alignment' => Tioga::FigureConstants::ALIGNED_AT_MIDHEIGHT,
                         'justification' => Tioga::FigureConstants::CENTERED
                         )
        end
      end
    end

  end

end
