/*
 *  cryptmount - a utility for user-level mounting of encrypted filesystems
 *  (C)Copyright 2005-2010, RW Penney
 */

/*
    This file is part of cryptmount

    cryptmount is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    cryptmount is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <config.h>

#include <ctype.h>
#include <fcntl.h>
#include <getopt.h>
#include <inttypes.h>
#include <pwd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <sys/mman.h>
#include <sys/mount.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/wait.h>
#ifdef HAVE_SYSLOG
#  include <syslog.h>
#endif
#include <linux/fs.h>       /* Beware ordering conflict with sys/mount.h */


#include "armour.h"
#include "cryptmount.h"
#include "delegates.h"
#include "dmutils.h"
#include "fsutils.h"
#include "looputils.h"
#include "tables.h"
#include "utils.h"
#if WITH_CSWAP
#  include <sys/swap.h>
#endif
#ifdef TESTING
#  include "cmtesting.h"
#endif


typedef struct targelt          /* Element of filesystem-target list */
{
    const tgtdefn_t *tgt;

    struct targelt *nx;
} targelt_t;


typedef enum                    /* Top-level operating mode */
{
    M_DEFAULT, M_HELP,
    M_PREPARE, M_RELEASE,
    M_MOUNT, M_UNMOUNT,
    M_SWAPON, M_SWAPOFF,
    M_LIST, M_KEYMGRS,
    M_PASSWORD, M_KEYGEN, M_KEYREU,
    M_SAFETYNET, M_VERSION
} cmmode_t;


static int64_t getblkcount(const char *device);
static int execute_list(cmmode_t mode, const tgtdefn_t *tgttable,
                const km_pw_context_t *pw_ctxt,
                const char *params, const targelt_t *eltlist);
static int do_list(const tgtdefn_t *tgt);
static int do_devsetup(const km_pw_context_t *pw_ctxt,
                bound_tgtdefn_t *boundtgt, char **mntdev);
static int do_devshutdown(const bound_tgtdefn_t *boundtgt);
static int do_mount(const km_pw_context_t *pw_ctxt, bound_tgtdefn_t *boundtgt);
static int do_unmount(const bound_tgtdefn_t *boundtgt);
static int do_swapon(const km_pw_context_t *pw_ctxt, bound_tgtdefn_t *boundtgt);
static int do_swapoff(const bound_tgtdefn_t *boundtgt);
static int do_passwd(const km_pw_context_t *pw_ctxt, bound_tgtdefn_t *boundtgt);
static int do_keygen(const km_pw_context_t *pw_ctxt, bound_tgtdefn_t *boundtgt,
                const char *params, int reuse, const tgtdefn_t *tgttable);
static int do_safetynet();


static const char *USAGE_STRING = N_("\
usage: cryptmount [OPTION [target ...]]\n\
\n\
  available options are as follows:\n\
\n\
    -h | --help\n\
    -a | --all\n\
    -c | --change-password <target>\n\
    -k | --key-managers\n\
    -l | --list\n\
    -m | --mount <target>\n\
    -u | --unmount <target>\n\
    --generate-key <key-size> <target>\n\
    --reuse-key <src-target> <dst-target>\n\
    --prepare <target>\n\
    --release <target>\n\
    --config-fd <num>\n\
    --passwd-fd <num>\n\
    --swapon <target>\n\
    --swapoff <target>\n\
    --version\n\
\n\
  please report bugs to <cryptmount@rwpenney.org.uk>\n\
");


#ifdef TESTING

cm_testinfo_t test_context;
cm_testinfo_t *test_ctxtptr=&test_context;

int fs_test_blkgetsz()
    /* Check that 32bit & 64bit device size calculations agree */
{
#ifdef BLKGETSIZE64
    int fd, n_open, seclen;
    long len;
    uint64_t len64;
    const char **dev;
    const char *devices[] = {
        "/dev/hda", "/dev/hda1", "/dev/hda2", "/dev/hda3",
        "/dev/hdb", "/dev/hdb1", "/dev/hdb2", "/dev/hdb3",
        "/dev/hdc", "/dev/hdc1", "/dev/hdc2", "/dev/hdc3",
        "/dev/sda", "/dev/sda1", "/dev/sda2", "/dev/sda3",
        "/dev/scd", "/dev/scd1", "/dev/scd2", "/dev/scd3",
        NULL };
#endif

    CM_TEST_START("BLKGETSIZE ioctl calls");
#ifndef BLKGETSIZE64
    /* Assume that there is no ambiguity with BLKGETSIZE */
    CM_TEST_PASS();
#else
    dev = devices;
    n_open = 0;
    while (*dev != NULL) {
        fd = open(*dev, O_RDONLY);
        if (fd >= 0) {
            ++n_open;
            if (ioctl(fd, BLKSSZGET, &seclen) != 0
              || ioctl(fd, BLKGETSIZE, &len) != 0
              || ioctl(fd, BLKGETSIZE64, &len64) != 0) {
                CM_TEST_FAIL();
            }
            close(fd);
            if (len64 < (1<<31)) {
                CM_ASSERT_EQUAL(len64, ((int64_t)len * (int64_t)512));
            }
        }
        /* fprintf(stderr, "%s: %d  %ld  %lld\n", *dev, seclen, len * 512, len64); */
        ++dev;
    }

    if (n_open > 0) {
        CM_TEST_OK();
    } else {
        CM_TEST_ABORT();
    }
#endif  /* BLKGETSIZE64 */
}

#endif  /* TESTING */


static int64_t getblkcount(const char *device)
    /* find size of raw device in blocks */
{   int64_t count=-1;
    int fd;
#ifdef BLKGETSIZE64
    int blklen;
#else
    long len;
#endif

    fd = open(device, O_RDONLY);
    if (fd < 0) return (int64_t)-1;

#ifdef BLKGETSIZE64
    if (ioctl(fd, BLKGETSIZE64, &count) == 0
        && ioctl(fd, BLKSSZGET, &blklen) == 0) {
        count /= (int64_t)blklen;
    } else {
        count = -1;
    }
#else
    if (ioctl(fd, BLKGETSIZE, &len) == 0) {
        /*  this is the number of 512-byte blocks,
            which may be the best we can do without BLKGETSIZE64... */
        count = (int64_t)len;
    }
#endif

    (void)close(fd);
    return count;
}


static int do_list(const tgtdefn_t *tgt)
    /* Print list of available filing-system targets */
{
    /* TRANSLATORS: this string is marked as 'no-c-format' because
       some localizations may require the mount-point and filesystem type
       to be printed in a different order, but the untranslated string needs
       to remain an ordinary string that can be printed without gettext. */
    /* xgettext:no-c-format */
    printf(_("%-16s  [to mount on \"%s\" as \"%s\"]\n"),
            tgt->ident, tgt->dir, tgt->fstype);

    return ERR_NOERROR;
}


static int do_devsetup(const km_pw_context_t *pw_ctxt,
                    bound_tgtdefn_t *boundtgt, char **mntdev)
    /* Setup all devices needed to access encrypted target */
{   enum { BUFFMIN=1024 };
    unsigned char *key=NULL;
    int i, readonly, isloop=0, killloop=0, keylen=0, eflag=ERR_NOERROR;
    int64_t devlen=0, fslen=0;
    size_t dpsize;
    char *dmparams=NULL;
    const char *tgtdev=NULL;
    const tgtdefn_t *tgt=NULL;

    /* Get crypto-key for filing system: */
    eflag = cm_get_key(boundtgt, pw_ctxt, &key, &keylen);
    if (eflag != ERR_NOERROR) {
        fprintf(stderr, _("Failed to extract cipher key\n"));
        goto bail_out;
    }
    tgt = boundtgt->tgt;

    readonly = is_readonlyfs(tgt->dev);
    eflag = blockify_file(tgt->dev, (readonly ? O_RDONLY : O_RDWR),
                        tgt->loopdev, &tgtdev, &isloop);
    if (eflag != ERR_NOERROR) {
        fprintf(stderr, _("Cannot open device \"%s\" for target \"%s\"\n"),
                (tgt->dev != NULL ? tgt->dev : "(NULL)"), tgt->ident);
        goto bail_out;
    }

    /* Get size in blocks of target device: */
    devlen = getblkcount(tgtdev);
    if (devlen < 0) {
        fprintf(stderr, _("Failed to get size of \"%s\"\n"), tgtdev);
        eflag = ERR_BADIOCTL;
        goto bail_out;
    }
    if (tgt->length < 0 || (tgt->start + tgt->length) > devlen) {
        fslen = devlen - tgt->start;
    } else {
        fslen = tgt->length;
    }
    if (tgt->start < 0 || fslen <= 0) {
        fprintf(stderr,_("Bad device-mapper start/length"));
        fprintf(stderr, " (%" PRId64 ",%" PRId64 ")\n",
                tgt->start, tgt->length);
        eflag = ERR_BADDEVICE;
        goto bail_out;
    }

    /* Setup device-mapper crypt table (CIPHER KEY IV_OFFSET DEV START): */
    dpsize = 2 * keylen + BUFFMIN;
    dmparams = sec_realloc(dmparams, dpsize);
    i = snprintf(dmparams, dpsize, "%s ",
                (tgt->cipher != NULL ? tgt->cipher : DFLT_CIPHER));
    i += mk_key_string(key, (size_t)keylen, dmparams + i);
    snprintf(dmparams + i, (dpsize - i), " %" PRId64 " %s %" PRId64,
                tgt->ivoffset, tgtdev, tgt->start);

    /* Setup device-mapper target: */
    eflag = devmap_create(tgt->ident,
            (uint64_t)0, (uint64_t)fslen, "crypt", dmparams);
    if (eflag != ERR_NOERROR) {
        fprintf(stderr,
            _("Device-mapper target-creation failed for \"%s\"\n"),
            tgt->ident);
        killloop = 1;
        goto bail_out;
    }
    if (mntdev != NULL) {
        devmap_path(mntdev, tgt->ident);
    }


  bail_out:

    if (killloop) unblockify_file(&tgtdev, isloop);   /* mounting failed? */
    sec_free(dmparams);
    sec_free(key);

    return eflag;
}   /* do_devsetup() */


int do_devshutdown(const bound_tgtdefn_t *boundtgt)
    /* Remove all devices attached to encrypted target */
{   const tgtdefn_t *tgt = boundtgt->tgt;
    struct stat sbuff;
    unsigned devcnt=0;
    dev_t *devids=NULL;
    int eflag=ERR_NOERROR;

    /* Check if filing system has been configured at all: */
    if (!is_configured(tgt->ident, NULL)) {
        fprintf(stderr, _("Target \"%s\" does not appear to be configured\n"),
                        tgt->ident);
        eflag = WRN_UNCONFIG;
        goto bail_out;
    }

    /* Find any underlying (e.g. loopback) devices for device-mapper target: */
    udev_settle();
    (void)devmap_dependencies(tgt->ident, &devcnt, &devids);
#ifdef DEBUG
    fprintf(stderr, "Shutting down %s [%u dependencies]\n",
            tgt->ident, devcnt);
#endif

    if (stat(tgt->dev, &sbuff) != 0) {
        fprintf(stderr, _("Cannot stat \"%s\"\n"), tgt->dev);
        eflag = ERR_BADDEVICE;
        goto bail_out;
    }

    /* Remove demice-mapper target: */
    eflag = devmap_remove(tgt->ident);
    if (eflag != ERR_NOERROR) {
        fprintf(stderr, _("Failed to remove device-mapper target \"%s\"\n"),
            tgt->ident);
        goto bail_out;
    }
    udev_settle();

    /* Tidy-up any associated loopback devices: */
    if (S_ISREG(sbuff.st_mode) && devids != NULL) {
        (void)loop_dellist(devcnt, devids);
    }

  bail_out:

    if (devids != NULL) free((void*)devids);

    return eflag;
}


static int do_mount(const km_pw_context_t *pw_ctxt, bound_tgtdefn_t *boundtgt)
{   const tgtdefn_t *tgt=NULL;
    int freedev=0, eflag=ERR_NOERROR;
    char *mntdev=NULL;
    tgtstat_t *tstat;

    if (is_mounted(boundtgt->tgt)) {
        fprintf(stderr, _("Target \"%s\" is already mounted\n"),
                boundtgt->tgt->ident);
        eflag = ERR_BADMOUNT;
        goto bail_out;
    }

    eflag = do_devsetup(pw_ctxt, boundtgt, &mntdev);
    if (eflag != ERR_NOERROR) goto bail_out;
    tgt = boundtgt->tgt;

#if WITH_FSCK
    if ((tgt->flags & FLG_FSCK) != 0) {
        if (fs_check(mntdev, tgt) != ERR_NOERROR) {
            freedev = 1; eflag = ERR_BADMOUNT;
            goto bail_out;
        }
    }
#endif

    if (fs_mount(mntdev, tgt) != ERR_NOERROR) {
        freedev = 1; eflag = ERR_BADMOUNT;
        goto bail_out;
    }

    tstat = alloc_tgtstatus(tgt);
    tstat->uid = (unsigned long)getuid();
    put_tgtstatus(tgt, tstat);
    free_tgtstatus(tstat);

  bail_out:

    if (freedev) {
        /* Tidy-up debris if mount failed */
        udev_settle();
        do_devshutdown(boundtgt);
    }
    if (mntdev != NULL) free((void*)mntdev);

    return eflag;
}


static int do_unmount(const bound_tgtdefn_t *boundtgt)
{   const tgtdefn_t *tgt = boundtgt->tgt;
    int eflag=ERR_NOERROR;
    struct passwd *pwent;
    char *mntdev=NULL;
    tgtstat_t *tstat;

    /* Check if filing system has been configured at all: */
    if (!is_mounted(tgt) || (tstat = get_tgtstatus(tgt)) == NULL) {
        fprintf(stderr, _("Target \"%s\" does not appear to be mounted\n"),
                        tgt->ident);
        eflag = WRN_UNCONFIG;
        goto bail_out;
    }

    /* Check if filing system has been mounted & locked by another user: */
    if (getuid() != 0 && (uid_t)tstat->uid != getuid()) {
        pwent = getpwuid((uid_t)tstat->uid);
        if (pwent != NULL) {
            fprintf(stderr, _("Only \"%s\" can unmount \"%s\"\n"),
                pwent->pw_name, tgt->ident);
        } else {
            /*  TRANSLATORS: the following expands to include
                the *numerical* user-identity in place of '%lu',
                e.g. giving 'only user-16 can unmount "target"': */
            fprintf(stderr, _("Only user-%lu can unmount \"%s\"\n"),
                tstat->uid, tgt->ident);
        }
        eflag = ERR_BADPRIV;
        goto bail_out;
    }

    /* Unmount filing system: */
    if (fs_unmount(tgt) != ERR_NOERROR) {
        eflag = ERR_BADMOUNT;
        goto bail_out;
    }
    put_tgtstatus(tgt, NULL);

    /* Remove supporting device-mapper target etc */
    if (do_devshutdown(boundtgt) != ERR_NOERROR) {
        eflag = ERR_BADDEVICE;
    }


  bail_out:

    if (mntdev != NULL) free((void*)mntdev);

    return eflag;
}


static int do_swapon(const km_pw_context_t *pw_ctxt, bound_tgtdefn_t *boundtgt)
{   const tgtdefn_t *tgt=NULL;
    int freedev=0, eflag=ERR_NOERROR;
    char *mntdev=NULL;
    tgtstat_t *tstat;

#if WITH_CSWAP

    eflag = do_devsetup(pw_ctxt, boundtgt, &mntdev);
    if (eflag != ERR_NOERROR) goto bail_out;
    tgt = boundtgt->tgt;

    if (fs_swapon(mntdev, tgt) != ERR_NOERROR) {
        freedev = 1;
        eflag = ERR_BADSWAP;
        goto bail_out;
    }

    tstat = alloc_tgtstatus(tgt);
    tstat->uid = (unsigned long)getuid();
    put_tgtstatus(tgt, tstat);
    free_tgtstatus(tstat);

#else   /* !WITH_CSWAP */

    fprintf(stderr, _("Crypto-swap is not supported by this installation of cryptmount\n"));
    eflag = ERR_BADSWAP;

#endif

  bail_out:

    if (freedev) {
        /* Tidy-up debris if swapon failed */
        udev_settle();
        do_devshutdown(boundtgt);
    }

    if (mntdev != NULL) free((void*)mntdev);

    return eflag;
}


static int do_swapoff(const bound_tgtdefn_t *boundtgt)
{   const tgtdefn_t *tgt = boundtgt->tgt;
    int eflag=ERR_NOERROR;
    char *mntdev=NULL;
    tgtstat_t *tstat;

#if WITH_CSWAP

    /* Check if device has been configured at all: */
    if ((tstat = get_tgtstatus(tgt)) == NULL) {
        fprintf(stderr, _("Target \"%s\" does not appear to be configured\n"),
                        tgt->ident);
        eflag = WRN_UNCONFIG;
        goto bail_out;
    }

    /* Remove swap-partition: */
    if (fs_swapoff(tgt) != ERR_NOERROR) {
        eflag = ERR_BADSWAP;
        goto bail_out;
    }
    put_tgtstatus(tgt, NULL);

    /* Remove supporting device-mapper target etc */
    if (do_devshutdown(boundtgt) != ERR_NOERROR) {
        eflag = ERR_BADDEVICE;
    }

#else   /* !WITH_CSWAP */

    fprintf(stderr, _("Crypto-swap is not supported by this installation of cryptmount\n"));
    eflag = ERR_BADSWAP;

#endif

  bail_out:

    if (mntdev != NULL) free((void*)mntdev);

    return eflag;
}


static int do_passwd(const km_pw_context_t *pw_ctxt, bound_tgtdefn_t *boundtgt)
    /* Change password on particular target */
{   tgtdefn_t *tgt = boundtgt->tgt;
    unsigned char *key=NULL;
    unsigned keyprops;
    int keylen=0, eflag=ERR_NOERROR;
    char *newfname=NULL, *oldfname=NULL;
    struct stat sbuff;
    size_t sz;
    FILE *fp=NULL;

    keyprops = cm_get_keyproperties(boundtgt);
    if ((keyprops & KM_PROP_HASPASSWD) == 0) {
        fprintf(stderr, _("Key-file for \"%s\" isn't password-protected\n"),
                tgt->ident);
        eflag = WRN_NOPASSWD;
        goto bail_out;
    }

    /* Attempt to read current key: */
    eflag = cm_get_key(boundtgt, pw_ctxt, &key, &keylen);
    if (eflag != ERR_NOERROR) goto bail_out;

    /* Setup location to re-encrypt key: */
    if (tgt->key.filename != NULL) {
        const char *outfname=NULL;
        if ((keyprops & KM_PROP_FIXEDLOC) == 0) {
            sz = strlen(tgt->key.filename) + 16;
            oldfname = (char*)malloc(2 * sz);
            newfname = oldfname + sz;
            snprintf(oldfname, sz, "%s-old", tgt->key.filename);
            snprintf(newfname, sz, "%s-new", tgt->key.filename);
            fp = fopen(newfname, "wb");
            outfname = newfname;
        } else {
            fp = fopen(tgt->key.filename, "r+b");
            outfname = tgt->key.filename;
        }
        if (fp == NULL) {
            fprintf(stderr, _("Cannot open \"%s\" for writing\n"), outfname);
            eflag = ERR_BADFILE;
            goto bail_out;
        }
    }
    eflag = cm_put_key(boundtgt, pw_ctxt, key, keylen, fp);
    if (fclose(fp) != 0) eflag = ERR_BADFILE;
    if (eflag != ERR_NOERROR) goto bail_out;

    /* Replace old key-container with new key-container: */
    if (oldfname != NULL && newfname != NULL) {
        if (stat(tgt->key.filename, &sbuff) != 0
          || rename(tgt->key.filename, oldfname) != 0
          || chown(oldfname, 0, 0) != 0
          || chmod(oldfname, S_IRUSR | S_IWUSR) != 0) {
            fprintf(stderr, _("Retiring old key (%s -> %s) failed\n"),
                    tgt->key.filename, oldfname);
            goto bail_out;
        }

        if (rename(newfname, tgt->key.filename) != 0
          || chown(tgt->key.filename, sbuff.st_uid, sbuff.st_gid) != 0
          || chmod(tgt->key.filename, sbuff.st_mode) != 0) {
            fprintf(stderr, _("Installing new key (%s -> %s) failed\n"),
                    newfname, tgt->key.filename);
            goto bail_out;
        }

        newfname = NULL;

        fprintf(stderr, _("Backup of previous key is in \"%s\"\n"), oldfname);
    }

  bail_out:

    if (newfname != NULL) {
        unlink(newfname);
    }
    if (oldfname != NULL) free((void*)oldfname);

    return eflag;
}


static int do_keygen(const km_pw_context_t *pw_ctxt, bound_tgtdefn_t *boundtgt,
                    const char *params, int reuse, const tgtdefn_t *tgttable)
    /* Create new filesystem crypto-key */
{   unsigned char *key=NULL;
    unsigned keyprops;
    int keylen=0, fileexists=0, eflag=ERR_NOERROR;
    char *newfname=NULL;
    tgtdefn_t *tgt = boundtgt->tgt;
    const tgtdefn_t *parent=NULL;
    bound_tgtdefn_t *boundparent=NULL;
    size_t sz;
    struct stat sbuff;
    FILE *fp=NULL;
    const unsigned mask_fmtfxd = KM_PROP_FIXEDLOC | KM_PROP_FORMATTED;

    if (params != NULL) {
        if (reuse) {
            parent = get_tgtdefn(tgttable, params);
            boundparent = bind_tgtdefn(parent);
            if (parent == NULL || boundparent == NULL) {
                fprintf(stderr, _("Target name \"%s\" is not recognized\n"),
                        params);
                eflag = ERR_BADPARAM;
            }
        } else {
            if (sscanf(params, "%i", &keylen) != 1 || keylen < 1) {
                fprintf(stderr, _("Bad key-length parameter"));
                eflag = ERR_BADPARAM;
            }
        }
    }
    if (params == NULL || eflag != ERR_NOERROR) goto bail_out;

    /* Check if keyfile already exists: */
    keyprops = cm_get_keyproperties(boundtgt);
    fileexists = (tgt->key.filename != NULL
                    && stat(tgt->key.filename, &sbuff) == 0);
    if (fileexists && (keyprops & mask_fmtfxd) != KM_PROP_FIXEDLOC) {
        fprintf(stderr,_("Key-file \"%s\" already exists for target \"%s\"\n"),
            tgt->key.filename, tgt->ident);
        eflag = ERR_BADFILE;
        goto bail_out;
    }

    /* Assemble new key material: */
    if (reuse) {
        eflag = cm_get_key(boundparent, pw_ctxt, &key, &keylen);
    } else {
        fprintf(stderr, _("Generating random key; please be patient...\n"));
        key = (unsigned char*)sec_realloc(NULL, (size_t)keylen);
        eflag = get_randkey(key, (size_t)keylen);
        if (eflag != ERR_NOERROR) {
            fprintf(stderr, _("Failed to generate new key\n"));
            goto bail_out;
        }
    }

    /* Setup location for new key: */
    if (tgt->key.filename != NULL) {
        const char *outfname=NULL;
        if ((keyprops & KM_PROP_FIXEDLOC) == 0) {
            sz = strlen(tgt->key.filename) + 16;
            newfname = (char*)malloc(sz);
            snprintf(newfname, sz, "%s-new", tgt->key.filename);
            fp = fopen(newfname, "wb");
            outfname = newfname;
        } else {
            fp = fopen(tgt->key.filename, (fileexists ? "r+b" : "wb"));
            outfname = tgt->key.filename;
        }
        if (fp == NULL) {
            fprintf(stderr, _("Cannot open \"%s\" for writing\n"), outfname);
            eflag = ERR_BADFILE;
            goto bail_out;
        }
    }
    eflag = cm_put_key(boundtgt, pw_ctxt, key, keylen, fp);
    if (fp != NULL && fclose(fp) != 0) eflag = ERR_BADFILE;
    if (eflag != ERR_NOERROR) goto bail_out;

    /* Move new key file into prefered location: */
    if (newfname != NULL) {
        if (rename(newfname, tgt->key.filename) != 0
          || chown(tgt->key.filename, 0, 0) != 0
          || chmod(tgt->key.filename, S_IRUSR | S_IWUSR) != 0) {
            fprintf(stderr, _("Installation of new keyfile \"%s\" failed"),
                    tgt->key.filename);
            eflag = ERR_BADFILE;
        }
        free((void*)newfname);
        newfname = NULL;
    }

  bail_out:

    if (newfname != NULL) {
        unlink(newfname);
        free((void*)newfname);
    }
    if (key != NULL) sec_free((void*)key);
    if (boundparent != NULL) free_boundtgt(boundparent);

    return eflag;
}


static int do_safetynet()
    /* Last-ditch attempt to umount/shutdown all targets currently mounted */
{   tgtstat_t *all_tsts=NULL, *tst;
    char *devname=NULL;
    const char *old_ident=NULL;
    tgtdefn_t *tgt=NULL;
    dev_t *devids=NULL;
    unsigned devcnt=0;
    int mflag;
    struct {
        unsigned targets, unmounted, unswapped, undeviced, unlooped; }
        counts = {
            0, 0, 0, 0, 0 };

    /*  This routine is ugly, but may be the best we can do to prevent
     *  damage to filesystems that should have been properly removed
     *  by other mechanisms (e.g. --unmount, --swapoff) */

    tgt = alloc_tgtdefn(NULL);
    old_ident = tgt->ident;

    /* Get list of all targets in status-file: */
    udev_settle();
    all_tsts = get_all_tgtstatus();

    for (tst=all_tsts; tst!=NULL; tst=tst->nx) {
        ++counts.targets;
        devmap_path(&devname, tst->ident);
        tgt->ident = tst->ident;

#ifdef DLGT_UMOUNT
        /* attempt to unmount filesystem: */
        switch (fork()) {
            case -1:
                break;
            case 0:
                execl(DLGT_UMOUNT, "umount", devname, NULL);
                break;
            default:
                (void)wait(&mflag);
                break;
        }
        if (mflag == 0) ++counts.unmounted;
#endif  /* DLGT_UMOUNT */
#if WITH_CSWAP
        /* Attempt to remove swap partition: */
        if (swapoff(devname) == 0) ++counts.unswapped;
#endif  /* WITH_CSWAP */

        /* Remove device-mapper device: */
        (void)devmap_dependencies(tst->ident, &devcnt, &devids);
        if (devmap_remove(tst->ident) == ERR_NOERROR) ++counts.undeviced;
        udev_settle();

        /* Free any associated loopback devices: */
        if (devcnt > 0 && loop_dellist(devcnt, devids) == 0) ++counts.unlooped;
        if (devids != NULL) {
            free((void*)devids);
            devids = NULL;
        }

        /* Remove entry in status-file, having done our best to clean-up: */
        (void)put_tgtstatus(tgt, NULL);

        if (devname != NULL) {
            free((void*)devname);
            devname = NULL;
        }
    }

    free_tgtstatus(all_tsts);

    tgt->ident = old_ident;
    free_tgtdefn(tgt);

    if (counts.targets != 0) {
        fprintf(stderr, "Safety-net caught %u targets:\n"
                "\t%u unmounted, %u swaps removed\n"
                "\t%u devices removed, %u loopbacks freed\n",
                counts.targets,
                counts.unmounted, counts.unswapped,
                counts.undeviced, counts.unlooped);
    }

    return (counts.targets != counts.undeviced);
}


static void check_priv_opt(const char *opt)
    /* Check if ordinary user is allowed to perform privileged actions */
{

    if (getuid() != 0) {
        fprintf(stderr, _("Only root can use option \"%s\"\n"), opt);
        exit(EXIT_PRIV);
    }

    /* Remove effect of any setuid flags, reverting to real user-id: */
    seteuid(getuid());
}


static int check_priv_tgt(const tgtdefn_t *tgt)
    /* Check if ordinary user is allowed to perform privileged actions */
{
    if ((tgt->flags & FLG_USER) == 0 && getuid() != 0) {
        fprintf(stderr, _("Only root can configure \"%s\"\n"), tgt->ident);
        return ERR_BADPRIV;
    }

    return ERR_NOERROR;
}


static int check_mode(cmmode_t mode, const char *params,
                    const targelt_t *eltlist)
    /* Check that arguments supplied & privilege match selected mode */
{   enum {
        F_NORMALUSER =  0x001,      /* normal user can perform action */
        F_NEEDS_TGT =   0x010,      /* filesystem-target is required */
        F_NEEDS_PRM =   0x020       /* additional parameter is required */
    };
    struct tblent {
        cmmode_t mode;
        unsigned flags; } *mdent=NULL;
    struct tblent mdtbl[] = {
        { M_HELP,       F_NORMALUSER },
        { M_PREPARE,    F_NEEDS_TGT },
        { M_RELEASE,    F_NEEDS_TGT },
        { M_MOUNT,      F_NORMALUSER | F_NEEDS_TGT },
        { M_UNMOUNT,    F_NORMALUSER | F_NEEDS_TGT },
        { M_SWAPON,     F_NEEDS_TGT },
        { M_SWAPOFF,    F_NEEDS_TGT },
        { M_LIST,       F_NORMALUSER },
        { M_KEYMGRS,    F_NORMALUSER },
        { M_PASSWORD,   F_NORMALUSER | F_NEEDS_TGT },
        { M_KEYGEN,     F_NEEDS_TGT | F_NEEDS_PRM },
        { M_KEYREU,     F_NEEDS_TGT | F_NEEDS_PRM },
        { M_SAFETYNET,  0u },
        { M_VERSION,    F_NORMALUSER },
    };
    unsigned idx,nmds;

    nmds = sizeof(mdtbl) / sizeof(mdtbl[0]);
    for (mdent=mdtbl,idx=0; idx<nmds && mdent->mode!=mode; ++mdent,++idx);
    if (idx >= nmds) {
        fprintf(stderr, "Bad mode [%u]\n", mode);
        abort();
    }

    if ((mdent->flags & F_NORMALUSER) != F_NORMALUSER) {
        /* turn off any setuid effects for privileged modes: */
        seteuid(getuid());
    }

    if ((mdent->flags & F_NEEDS_TGT) == F_NEEDS_TGT && eltlist == NULL) {
        fprintf(stderr, _("No targets specified\n"));
        exit(EXIT_BADTGT);
    }

    if ((mdent->flags & F_NEEDS_PRM) == F_NEEDS_PRM && params == NULL) {
        fprintf(stderr, _("Missing parameter\n"));
        exit(EXIT_BADOPT);
    }

    return 0;
}


static int execute_list(cmmode_t mode, const tgtdefn_t *tgttable,
                const km_pw_context_t *pw_ctxt,
                const char *params, const targelt_t *eltlist)
    /* Apply mode-dependent operation to list of targets */
{   const targelt_t *elt=NULL;
    bound_tgtdefn_t *boundtgt=NULL;
    int i, prio=0, eflag=ERR_NOERROR;
    struct passwd *pwent=NULL;
    const char *username=NULL, **keymgrs=NULL, *syslogmsg=NULL;

    pwent = getpwuid(getuid());
    username = (pwent != NULL ? pwent->pw_name : "UNKNOWN");

#if defined(HAVE_SYSLOG) && !defined(TESTING)
    openlog(PACKAGE, LOG_PID, LOG_AUTHPRIV);
#endif

    /* Check that we have suitable privileges + parameters: */
    (void)check_mode(mode, params, eltlist);

    /* Execute special-cases of user-selected task: */
    if (mode == M_VERSION) {
        fprintf(stderr, "%s-%s\n", PACKAGE_NAME, PACKAGE_VERSION);
        eltlist = NULL;
    } else if (mode == M_KEYMGRS) {
        keymgrs = get_keymgr_list();
        for (i=0; keymgrs[i]!=NULL; ) {
            printf("%s", keymgrs[i]);
            if (keymgrs[++i] != NULL) printf(", "); else printf("\n");
        }
        free((void*)keymgrs);
        eltlist = NULL;
    } else if (mode == M_LIST) {
        for (elt=eltlist; elt!=NULL; elt=elt->nx) {
            (void)do_list(elt->tgt);
        }
        eltlist = NULL;
    } else if (mode == M_SAFETYNET) {
        do_safetynet();
        eltlist = NULL;
    }

    /* Apply user-selected operation to list of targets (if present): */
    for (elt=eltlist; elt!=NULL && eflag<ERR_threshold; elt=elt->nx) {
        boundtgt = bind_tgtdefn(elt->tgt);
        if (boundtgt == NULL) {
            fprintf(stderr, _("Cannot find key-manager to match target \"%s\"\n"), elt->tgt->ident);
            eflag = ERR_BADKEYFORMAT;
            break;
        }
        syslogmsg = NULL;
        prio = LOG_AUTHPRIV | LOG_NOTICE;

        switch (mode) {
            case M_PREPARE:
                syslogmsg = "prepare of \"%s\" by %s %s";
                eflag = do_devsetup(pw_ctxt, boundtgt, NULL);
                break;
            case M_RELEASE:
                syslogmsg = "release of \"%s\" by %s %s";
                prio = LOG_AUTHPRIV | LOG_NOTICE;
                eflag = do_devshutdown(boundtgt);
                if (eflag == WRN_UNCONFIG) syslogmsg = NULL;
                break;
            case M_MOUNT:
                if ((eflag = check_priv_tgt(boundtgt->tgt)) != ERR_NOERROR) break;
                syslogmsg = "mount of \"%s\" by %s %s";
                eflag = do_mount(pw_ctxt, boundtgt);
                break;
            case M_UNMOUNT:
                if ((eflag = check_priv_tgt(boundtgt->tgt)) != ERR_NOERROR) break;
                syslogmsg = "unmount of \"%s\" by %s %s";
                eflag = do_unmount(boundtgt);
                if (eflag == WRN_UNCONFIG) syslogmsg = NULL;
                break;
            case M_SWAPON:
                syslogmsg = "swapon \"%s\" by %s %s";
                eflag = do_swapon(pw_ctxt, boundtgt);
                break;
            case M_SWAPOFF:
                syslogmsg = "swapoff \"%s\" by %s %s";
                eflag = do_swapoff(boundtgt);
                if (eflag == WRN_UNCONFIG) syslogmsg = NULL;
                break;
            case M_PASSWORD:
                if ((eflag = check_priv_tgt(boundtgt->tgt)) != ERR_NOERROR) break;
                syslogmsg = "changing password for \"%s\" by %s %s";
                eflag = do_passwd(pw_ctxt, boundtgt);
                break;
            case M_KEYGEN:
                if ((eflag = check_priv_tgt(boundtgt->tgt)) != ERR_NOERROR) break;
                syslogmsg = "key generation for \"%s\" by %s %s";
                eflag = do_keygen(pw_ctxt, boundtgt, params, 0, NULL);
                break;
            case M_KEYREU:
                if ((eflag = check_priv_tgt(boundtgt->tgt)) != ERR_NOERROR) break;
                syslogmsg = "key generation for \"%s\" by %s %s";
                eflag = do_keygen(pw_ctxt, boundtgt, params, 1, tgttable);
                break;
            default:
                break;
        }
#if defined(HAVE_SYSLOG) && !defined(TESTING)
        if (syslogmsg != NULL) {
            syslog(prio, syslogmsg, elt->tgt->ident, username,
                (eflag == ERR_NOERROR ? "succeeded" : "failed"));
        }
#endif

        free_boundtgt(boundtgt);
    }

#ifdef HAVE_SYSLOG
    closelog();
#endif

    return eflag;
}


static cmmode_t get_defaultmode(int argc, char *argv[])
    /* Translate program-name into default action (or just assume M_MOUNT) */
{   cmmode_t mode = M_MOUNT;

#ifdef WITH_ARGV0
    struct modename {
        cmmode_t mode;
        const char *name; } modetable[] = {
            { M_MOUNT,      "cryptmount" },
            { M_UNMOUNT,    "cryptumount" },
            { M_UNMOUNT,    "cryptunmount" },
#if WITH_CSWAP
            { M_SWAPON,     "cryptswapon" },
            { M_SWAPOFF,    "cryptswapoff", },
#endif
            { M_PREPARE,    "cryptprepare" },
            { M_RELEASE,    "cryptrelease" },
            { M_DEFAULT, NULL } },
        *mp;
    const char *base;

    if (argc >= 1) {
        base = strrchr(argv[0], '/');
        if (base != NULL) ++base; else base = argv[0];

        for (mp=modetable; mp->name!=NULL; ++mp) {
            if (strcmp(base, mp->name) == 0) {
                mode = mp->mode;
                break;
            }
        }
    }

#endif

    return mode;
}


int main(int argc, char *argv[])
{   cmmode_t mode=M_DEFAULT;
    enum {
        F_ALL=0x1 }
        flags=0;
#ifdef _GNU_SOURCE
    struct option opttable[] = {
        { "all",                no_argument,        NULL, (int)'a' },
        { "change-password",    no_argument,        NULL, (int)'c' },
        { "config-fd",          required_argument,  NULL, (int)'f' },
        { "generate-key",       required_argument,  NULL, (int)'g' },
        { "help",               no_argument,        NULL, (int)'h' },
        { "key-managers",       no_argument,        NULL, (int)'k' },
        { "list",               no_argument,        NULL, (int)'l' },
        { "mount",              no_argument,        NULL, (int)'m' },
        { "passwd-fd",          required_argument,  NULL, (int)'w' },
        { "prepare",            no_argument,        NULL, (int)'p' },
        { "release",            no_argument,        NULL, (int)'r' },
        { "reuse-key",          required_argument,  NULL, (int)'e' },
        { "safetynet",          no_argument,        NULL, (int)'n' },
        { "swapon",             no_argument,        NULL, (int)'s' },
        { "swapoff",            no_argument,        NULL, (int)'x' },
        { "unmount",            no_argument,        NULL, (int)'u' },
        { "verify-password",    no_argument,        NULL, (int)'y' },
        { "version",            no_argument,        NULL, (int)'v' },
#  ifdef TESTING
        { "config-dir",         required_argument,  NULL, (int)'D' },
        { "password",           required_argument,  NULL, (int)'W' },
        { "newpassword",        required_argument,  NULL, (int)'N' },
        { "self-test",          no_argument,        NULL, (int)'T' },
#  endif
        { NULL, 0, NULL, 0 } };
#endif
    const char *params=NULL, *optlist="acde:f:g:hklmnprsuvw:xy";
    char *cmtab=NULL;
    int val, idx, config_fd=-1, passwd_fd=-1, eflag=ERR_NOERROR;
    tgtdefn_t *tgttable=NULL;
    km_pw_context_t pw_ctxt;
    const tgtdefn_t *tgt=NULL;
    targelt_t *eltlist=NULL, **eltptr=&eltlist;

#ifdef HAVE_GETTEXT
    /* setup internationalization of message-strings via gettext(): */
    setlocale(LC_ALL, "");
    bindtextdomain(PACKAGE, LOCALEDIR);
    textdomain(PACKAGE);
#endif

#ifdef TESTING
    fprintf(stderr, "WARNING!!! cryptmount has been compiled for TESTING only - DO NOT INSTALL\n");
    pw_ctxt.argpasswd[0] = pw_ctxt.argpasswd[1] = NULL;
    test_context.argconfigdir = NULL;
#endif

    if (mlockall(MCL_CURRENT | MCL_FUTURE) != 0) {
        fprintf(stderr, _("Memory-locking failed...\n"));
    }

    pw_ctxt.verify = 0;
    pw_ctxt.debug_level = 0;

    /* Parse command-line options: */
    for (;;) {
#ifdef _GNU_SOURCE
        val = getopt_long(argc, argv, optlist, opttable, &idx);
#else
        val = getopt(argc, argv, optlist);
#endif
        if (val == -1) break;
        switch (val) {
            case 'a':
                flags |= F_ALL; break;
            case 'c':
                mode = M_PASSWORD; break;
            case 'e':                       check_priv_opt("--reuse-key");
                mode = M_KEYREU; params = optarg; break;
            case 'f':                       check_priv_opt("--config-fd");
                sscanf(optarg, "%d", &config_fd); break;
            case 'g':                       check_priv_opt("--generate-key");
                mode = M_KEYGEN; params = optarg; break;
            case 'h':
                mode = M_HELP; break;
            case 'k':
                mode = M_KEYMGRS; break;
            case 'l':
                mode = M_LIST;  break;
            case 'm':
                mode = M_MOUNT; break;
            case 'n':                       check_priv_opt("--safetynet");
                mode = M_SAFETYNET; break;
            case 'p':                       check_priv_opt("--prepare");
                mode = M_PREPARE; break;
            case 'r':                       check_priv_opt("--release");
                mode = M_RELEASE; break;
            case 's':                       check_priv_opt("--swapon");
                mode = M_SWAPON; break;
            case 'x':                       check_priv_opt("--swapoff");
                mode = M_SWAPOFF; break;
            case 'u':
                mode = M_UNMOUNT; break;
            case 'v':
                mode = M_VERSION; break;
            case 'w':
                sscanf(optarg, "%d", &passwd_fd); break;
#ifdef TESTING
            case 'D':
                test_context.argconfigdir = optarg; break;
            case 'W':
                pw_ctxt.argpasswd[0] = optarg; break;
            case 'N':
                pw_ctxt.argpasswd[1] = optarg; break;
            case 'T':
                return cm_run_tests(); break;
#endif
            case '?':
                fprintf(stderr, "%s", _(USAGE_STRING));
                exit(EXIT_BADOPT);
                break;
            default:
                break;
        }
    }

    (void)cm_path(&cmtab, "cmtab");

    if (mode == M_DEFAULT) mode = get_defaultmode(argc, argv);

    if (mode == M_HELP) {
        fprintf(stderr, "%s", _(USAGE_STRING));
        exit(EXIT_OK);
    }

    /* Configure source of passwords: */
    if (passwd_fd >= 0) {
        pw_ctxt.fd_pw_source = fdopen(passwd_fd, "r");
        if (pw_ctxt.fd_pw_source == NULL) {
            fprintf(stderr, _("Bad file-descriptor (%d)\n"), passwd_fd);
            exit(EXIT_BADOPT);
        }
    } else {
        pw_ctxt.fd_pw_source = NULL;
    }

    /* Check & read-in configuration file: */
#ifndef TESTING
    if (sycheck_cmtab(cmtab) != ERR_NOERROR
      || sycheck_directory(CM_MODULE_DIR) != ERR_NOERROR) {
        fprintf(stderr, _("Security failure\n"));
        exit(EXIT_INSECURE);
    }
#endif
    if (config_fd >= 0) {
        tgttable = parse_config_fd(config_fd);
    } else {
        tgttable = parse_config(cmtab);
    }
    free((void*)cmtab);

    if (mode == M_LIST && optind >= argc) flags |= F_ALL;

    /* if '--all' given, assemble list of targets from entire config-file */
    if ((flags & F_ALL) != 0) {
        if (optind < argc) {
            fprintf(stderr, _("Trailing command-line arguments given with '--all' option\n"));
            exit(EXIT_BADOPT);
        }
        for (tgt=tgttable; tgt!=NULL; tgt=tgt->nx) {
            *eltptr = (targelt_t*)malloc(sizeof(targelt_t));
            (*eltptr)->tgt = tgt;
            (*eltptr)->nx = NULL;
            eltptr = &((*eltptr)->nx);
        }
    }

    /* Assemble list of targets from remaining command-line arguments: */
    while (optind < argc) {
        tgt = get_tgtdefn(tgttable, argv[optind]);
        if (tgt != NULL) {
            *eltptr = (targelt_t*)malloc(sizeof(targelt_t));
            (*eltptr)->tgt = tgt;
            (*eltptr)->nx = NULL;
            eltptr = &((*eltptr)->nx);
        } else {
            fprintf(stderr, _("Target name \"%s\" is not recognized\n"),
                    argv[optind]);
            exit(EXIT_BADTGT);
        }
        ++optind;
    }


    /* Check security of all targets being processed: */
    for (eltptr=&eltlist; *eltptr!=NULL; eltptr=&((*eltptr)->nx)) {
        tgt = (*eltptr)->tgt;
        if (sycheck_target(tgt) != ERR_NOERROR) {
            fprintf(stderr, _("Target security failure for \"%s\"\n"),
                    tgt->ident);
            exit(EXIT_INSECURE);
        }
    }


    /* Execute user-selected task: */
    eflag = execute_list(mode, tgttable, &pw_ctxt, params, eltlist);
    free_keymanagers();


    /* Tidy-up: */
    while (eltlist != NULL) {
        eltptr = &eltlist;
        eltlist = eltlist->nx;
        free((void*)*eltptr);
    }
    free_config(&tgttable);
    munlockall();

    return eflag;
}

/*
 *  (C)Copyright 2005-2010, RW Penney
 */
