// Crimson Fields -- a game of tactical warfare
// Copyright (C) 2000, 2001 Jens Granseuer
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

////////////////////////////////////////////////////////////////////////
// map.cpp
//
// History:
//  06-12-2000 - created
//  05-03-2001 - FindPath() now starts from source; added deviation
//             - minor code clean-up
//  08-03-2001 - preliminary fix for path blocking in FindPath()
//  12-03-2001 - path blocking fix
////////////////////////////////////////////////////////////////////////

#include "SDL_endian.h"

#include "map.h"
#include "game.h"
#include "history.h"
#include "globals.h"

////////////////////////////////////////////////////////////////////////
// NAME       : TerrainType::Load
// DESCRIPTION: Load a terrain type definition from a file.
// PARAMETERS : file - SDL_RWops file descriptor
// RETURNS    : 0 on success, -1 on error
//
// HISTORY
////////////////////////////////////////////////////////////////////////

int TerrainType::Load( SDL_RWops *file ) {
  tt_type = SDL_ReadLE16( file );
  tt_image = SDL_ReadLE16( file );
  SDL_RWread( file, &tt_att_mod, sizeof(char), 3 );
  tt_color = SDL_ReadLE32( file );
  return 0;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Map::Map
// DESCRIPTION: Create a new map instance and initilize data.
// PARAMETERS : -
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

Map::Map( void ) {
  m_data = m_path = NULL;
  m_objects = NULL;
  m_terrains = NULL;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Map::~Map
// DESCRIPTION: Free memory allocated by the map structure.
// PARAMETERS : -
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

Map::~Map( void ) {
  delete [] m_data;
  delete [] m_objects;
  delete [] m_terrains;
  delete [] m_path;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Map::Load
// DESCRIPTION: Initialize the map structure.
// PARAMETERS : file - descriptor of an opened data file from which to
//                     read the map data
// RETURNS    : 0 on success, non-zero on error
//
// HISTORY
//   21-10-2001 - fixed a bug introduced in the last release which led
//                to an arbitrary map size (and usually a crash)
////////////////////////////////////////////////////////////////////////

int Map::Load( SDL_RWops *file ) {
  m_w = SDL_ReadLE16( file );
  m_h = SDL_ReadLE16( file );

  unsigned short size = m_w * m_h;
  m_data = new short [size];
  m_objects = new MapObject * [size];
  m_path = new short [size];

  // terrain types are loaded from a
  // level set file with LoadTerrainTypes()

  for ( int i = 0; i < size; i++ ) {
    m_data[i] = SDL_ReadLE16( file );
    m_objects[i] = NULL;
  }

  return 0;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Map::LoadTerrainTypes
// DESCRIPTION: Load terrain type definitions from a level .set file.
// PARAMETERS : file - SDL_RWops descriptor of the level set file
//              n    - number of types in file
// RETURNS    : 0 on success, non-zero on error
//
// HISTORY
////////////////////////////////////////////////////////////////////////

int Map::LoadTerrainTypes( SDL_RWops *file, unsigned short n ) {
  m_terrains = new TerrainType [n];
  if ( !m_terrains ) return -1;

  for ( int i = 0; i < n; i++ )
    if ( m_terrains[i].Load( file ) ) return -1;

  return 0;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Map::Save
// DESCRIPTION: Save the current map status to a file. Only the terrain
//              type information is saved for each hex. Other data
//              (units, buildings...) must be stored separately.
// PARAMETERS : file - descriptor of the save file
// RETURNS    : 0 on success, non-zero on error
//
// HISTORY
////////////////////////////////////////////////////////////////////////

int Map::Save( SDL_RWops *file ) const {
  SDL_WriteLE16( file, m_w );
  SDL_WriteLE16( file, m_h );

  unsigned short size = m_w * m_h;
  for ( int i = 0; i < size; i++ )
    if ( SDL_WriteLE16( file, m_data[i] ) != 1 ) return -1;
  return 0;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Map::ClearPath
// DESCRIPTION: Clear the m_path array for further use.
// PARAMETERS : -
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void Map::ClearPath( void ) const {
  unsigned short size = m_w * m_h;
  for ( int i = 0; i < size; i++ ) m_path[i] = -1;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Map::FindPath
// DESCRIPTION: Try to find a way for a unit to reach a destination
//              hex.
// PARAMETERS : u       - unit to search a path for
//              dx      - destination x
//              dy      - destination y
//              qual    - path quality (1 is best, 10 worst but fastest)
//              deviate - tolerable distance to destination hex; this
//                        way you can tell a unit to stop when it comes
//                        close enough for long-range attacks
// RETURNS    : total cost to reach the destination hex, or -1 if no
//              valid path was found
//
// HISTORY
//   05-03-2001 - start from source hex, not destination
//              - added parameter 'deviate'
//   08-03-2001 - preliminary fix to take blocking into account
//   12-03-2001 - path blocking fix
//   21-08-2001 - return not the total cost of the path but the number
//                of turns required to get to the destination (or rather
//                an approximation thereof)
//   16-09-2001 - fix turn estimation
//   20-09-2001 - follow path over units
////////////////////////////////////////////////////////////////////////

int Map::FindPath( const Unit *u, short dx, short dy, short qual,
                   unsigned short deviate ) const {
  PathNode *pnode, *pnode2, *walk;
  List openlist, donelist;
  short sx = u->Position().x, sy = u->Position().y;

  ClearPath();

  // to begin, insert the starting node in openlist
  pnode = new PathNode;
  if ( !pnode ) return -1;

  // begin at the current position
  // starting from the destination would make it a bit easier to record the actual path
  // but this way we can support going towards the target until we have reached a certain
  // distance which is rather useful for AI routines...
  pnode->x = sx;
  pnode->y = sy;
  pnode->eta = Distance( sx, sy, dx, dy ) * qual;
  pnode->cost = 0;
  openlist.AddHead( pnode );

  while( !openlist.IsEmpty() ) {
    // find the shortest projected path in openlist
    pnode = static_cast<PathNode *>( openlist.Head() );
    walk = static_cast<PathNode *>( pnode->Next() );
    while ( walk ) {
      if ( (walk->cost + walk->eta) < (pnode->cost + pnode->eta) ) pnode = walk;
      walk = static_cast<PathNode *>( walk->Next() );
    }
    pnode->Remove();
    donelist.AddHead( pnode );

    // check for destination or deviation
    if ( Distance( pnode->x, pnode->y, dx, dy ) <= deviate ) break;

    // has the path been blocked on the last step? if so, the cost
    // for the next step is higher than what the unit could possibly
    // move on one turn
    bool blocked = false, narrow = false, foenear = false;
    Point adj;
    short invalid;
    Direction lastdir = (Direction)m_path[XY2Index(pnode->x, pnode->y)];
    Unit *obst;

    if ( !Dir2XY( pnode->x, pnode->y, TurnLeft( lastdir ), adj ) ) {
      if ( !(TerrainTypes(adj.x,adj.y) & u->Terrain()) ) narrow = true;
      if ( (obst = GetUnit(adj.x,adj.y)) && (obst->Owner() != u->Owner()) ) foenear = true;
    } else narrow = true;

    invalid = Dir2XY( pnode->x, pnode->y, TurnRight( lastdir ), adj );
    if ( (foenear && (invalid
               || !(TerrainTypes(adj.x, adj.y) & u->Terrain())
               || ((obst = GetUnit(adj.x,adj.y)) && (obst->Owner() != u->Owner()))) ) ||
          (narrow && !invalid && (obst = GetUnit(adj.x,adj.y)) &&
               (obst->Owner() != u->Owner())) ) blocked = true;

    // not there yet, so let's look at the hexes around it
    for ( short dir = NORTH; dir <= NORTHWEST; dir++ ) {
      Point p;

      // get the hex in that direction
      if ( !Dir2XY( pnode->x, pnode->y, (Direction)dir, p ) ) {
        short index = XY2Index( p.x, p.y );
        TerrainType *type = &m_terrains[m_data[index]];
        Unit *occ = GetUnit( p.x, p.y );
        Building *bld = GetBuilding( p.x, p.y );

        // can the unit move over that kind of terrain?
        if ( (type->tt_type & u->Terrain()) &&
           (!bld || ((p.x == dx) && (p.y == dy) && bld->Allow( u ))) ) {
          short newcost;
          if ( (bld && u->IsSheltered()) ||
               (occ && (((p.x != dx) || (p.y != dy)) || !occ->IsTransport() ||
                !static_cast<Transport *>(occ)->Allow( u ))) ) newcost = MCOST_UNIT;
          else if ( blocked ) newcost = u->Type()->ut_moves + 1;
          else if ( u->IsAircraft() ) newcost = MCOST_MIN;
          else newcost = type->tt_move;

          pnode2 = new PathNode;
          if ( !pnode2 ) return -1;

          pnode2->x = p.x;
          pnode2->y = p.y;
          pnode2->cost = pnode->cost + newcost;
          pnode2->eta = Distance( p.x, p.y, dx, dy ) * qual;

          // if the new hex is not marked in path yet, mark it and put it in openlist
          if ( m_path[ index ] == -1 ) {
            m_path[ index ] = ReverseDir( dir );
            openlist.AddTail( pnode2 );
          } else {
            // try to find the node in openlist and replace it if new cost is smaller
            walk = static_cast<PathNode *>( openlist.Head() );
            while ( walk ) {
              if ( (walk->x == pnode2->x) && (walk->y == pnode2->y) ) {
                // found the node. shall we replace it?
                if ( walk->cost > pnode2->cost ) {
                  walk->Remove();
                  delete walk;
                  m_path[ index ] = ReverseDir( dir );
                  openlist.AddTail( pnode2 );
                  pnode2 = NULL;
                }
                break;
              } else walk = static_cast<PathNode *>( walk->Next() );
            }
            delete pnode2;
          }	// endif m_path[i] == -1
        }	// endif hex free
      }		// endif valid hex
    }		// end for
  }		// end while openlist not empty

  // check if we reached our destination
  if ( Distance( pnode->x, pnode->y, dx, dy ) <= deviate ) {
    // correct the path
    // move back to beginning
    Point p = { pnode->x, pnode->y };
    Direction dir = (Direction)m_path[XY2Index(p.x,p.y)], dir2;

    while ( (p.x != sx) || (p.y != sy) ) {
      Dir2XY( p.x, p.y, dir, p );
      dir2 = (Direction)m_path[XY2Index(p.x,p.y)];
      m_path[XY2Index(p.x,p.y)] = ReverseDir( dir );
      dir = dir2;
    }
    m_path[XY2Index(pnode->x,pnode->y)] = -1;

    return (pnode->cost + u->Type()->ut_moves - 1) / u->Type()->ut_moves;
  }
  return -1;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Map::GetNeighbors
// DESCRIPTION: Get the hex coordinates of all hexes surrounding a hex.
// PARAMETERS : x      - horizontal position of hex
//              y      - vertical position of hex
//              parray - pointer to an array of at least six Points to
//                       hold the neighbors' coordinates. Each neighbor
//                       will be put in the slot corresponding to the
//                       direction the hex is in (e.g. the hex north of
//                       the source hex will reside in parray[NORTH]).
//                       If there is no valid neighbor (because the map
//                       ends here), the respective Points will contain
//                       {-1, -1}
// RETURNS    : number of adjacent hexes found
//
// HISTORY
////////////////////////////////////////////////////////////////////////

short Map::GetNeighbors( short x, short y, Point *parray ) const {
  short num = 0;
  for ( int i = NORTH; i <= NORTHWEST; i++ ) {
    if ( !Dir2XY( x, y, (Direction)i, parray[i] ) ) num++;
    else parray[i].x = parray[i].y = -1;
  }
  return num;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Map::Dir2XY
// DESCRIPTION: Get the coordinates of the hex we'll be on when we move
//              in a given direction from another hex.
// PARAMETERS : x    - hex horizontal position
//              y    - hex vertical position
//              dir  - direction to move in
//              dest - pointer to a Point to hold the coordinates of the
//                     destination hex
// RETURNS    : 0 on success, -1 on error (the contents of the Point can
//              not be relied on in this case)
//
// HISTORY
////////////////////////////////////////////////////////////////////////

int Map::Dir2XY( short x, short y, Direction dir, Point &dest ) const {
  switch ( dir ) {
    case NORTH:
      y--;
      break;
    case SOUTH:
      y++;
      break;
    case NORTHEAST:
      if ( x & 1 ) y--;    // fall through...
    case EAST:
      x++;
      break;
    case SOUTHEAST:
      if ( !(x & 1) ) y++;
      x++;
      break;
    case SOUTHWEST:
      if ( !(x & 1) ) y++; // fall through...
    case WEST:
      x--;
      break;
    case NORTHWEST:
      if ( x & 1 ) y--;
      x--;
  }

  if ( !Contains( x, y ) ) return -1;

  dest.x = x;
  dest.y = y;
  return 0;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Map::Contains
// DESCRIPTION: Check whether a hex with the given coordinates exists
//              on the map.
// PARAMETERS : x - hex horizontal position
//              y - hex vertical position
// RETURNS    : true if the coordinates denote a valid hex, false
//              otherwise
//
// HISTORY
////////////////////////////////////////////////////////////////////////

bool Map::Contains( short x, short y ) const {
  return( (x >= 0) && (x < m_w) && (y >= 0) && (y < m_h) );
}

////////////////////////////////////////////////////////////////////////
// NAME       : Map::GetUnit
// DESCRIPTION: Get a unit from the map.
// PARAMETERS : x - hex horizontal position
//              y - hex vertical position
// RETURNS    : the unit at the given coordinates, or NULL if no unit
//              was found there
//
// HISTORY
////////////////////////////////////////////////////////////////////////

Unit *Map::GetUnit( short x, short y ) const {
  MapObject *o = m_objects[XY2Index(x,y)];
  if ( o && (o->Type() == MO_UNIT) ) return static_cast<Unit *>(o);
  return NULL;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Map::SetUnit
// DESCRIPTION: Put a unit on the map.
// PARAMETERS : u - unit (may be NULL)
//              x - hex horizontal position
//              y - hex vertical position
// RETURNS    : -
//
// HISTORY
//   12-06-2001 - handle dummies specially
//              - record event for turn history if building conquered
////////////////////////////////////////////////////////////////////////

void Map::SetUnit( Unit *u, short x, short y ) {
  if ( u ) {
    u->SetPosition( x, y );

    Unit *local = GetUnit( x, y );
    if ( local && local->IsTransport() ) {	// move into transport
      if ( u->IsDummy() ) u->SetFlags( U_SHELTERED );
      else static_cast<Transport *>( local )->InsertUnit( u );
    } else {
      Building *b = GetBuilding( x, y );
      if ( b ) {					// move unit into building
        if ( b->InsertUnit( u ) == 1 )	{	// building conquered
          // exchange the building entrance
          short hextype = m_data[XY2Index(x,y)], newhex;
          if ( b->Owner() ) newhex = hextype + u->Owner()->ID() - b->Owner()->ID();
          else newhex = hextype + u->Owner()->ID() - PLAYER_NONE;
          SetHexType( x, y, newhex );
          b->SetOwner( u->Owner() );

          // record event
          History *hist = Gam->GetHistory();
          if ( hist ) hist->RecordTileEvent( newhex, hextype, x, y );
        }
      } else m_objects[XY2Index(x,y)] = u;
    }
  } else m_objects[XY2Index(x,y)] = u;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Map::GetBuilding
// DESCRIPTION: Get a building from the map.
// PARAMETERS : x - hex horizontal position
//              y - hex vertical position
// RETURNS    : the building at the given coordinates, or NULL if no
//              building was found there
//
// HISTORY
////////////////////////////////////////////////////////////////////////

Building *Map::GetBuilding( short x, short y ) const {
  MapObject *o = m_objects[XY2Index(x,y)];
  if ( o && (o->Type() == MO_BUILDING) ) return static_cast<Building *>(o);
  return NULL;
}

