// Crimson Fields -- a game of tactical warfare
// Copyright (C) 2000, 2001 Jens Granseuer
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

/////////////////////////////////////////////////////////////////////////
// unit.cpp
//
// History:
//  04-12-2000 - created
//  12-08-2001 - added sounds to unit definition
////////////////////////////////////////////////////////////////////////

#include "SDL_endian.h"

#include "game.h"
#include "globals.h"

////////////////////////////////////////////////////////////////////////
// NAME       : UnitType::Load
// DESCRIPTION: Load a unit type definition from a file.
// PARAMETERS : file   - SDL_RWops data source descriptor
//              sounds - array containing the mission set sound effects
// RETURNS    : 0 on success, -1 on error
//
// HISTORY
////////////////////////////////////////////////////////////////////////

int UnitType::Load( SDL_RWops *file, SoundEffect **sounds ) {
  ut_snd_move = ut_snd_fire = NULL;
  ut_terrain = SDL_ReadLE16( file );
  ut_image = SDL_ReadLE16( file );
  SDL_RWread( file, &ut_moves, sizeof(unsigned char), 18 );

  // set sound effects
  unsigned char sfx[2];
  SDL_RWread( file, sfx, sizeof(unsigned char), 2 );
  if ( sounds ) {
    if ( sfx[0] != UT_NO_SOUND ) ut_snd_move = sounds[sfx[0]];
    if ( sfx[1] != UT_NO_SOUND ) ut_snd_fire = sounds[sfx[1]];
  }

  SDL_RWread( file, &ut_trans_slots, sizeof(unsigned char), 3 );
  SDL_RWread( file, ut_name, sizeof(char), 20 );
  return 0;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Unit::Unit
// DESCRIPTION: Create a new unit.
// PARAMETERS : type   - pointer to a unit type definition
//              player - pointer to the player this unit belongs to
//              id     - unique unit identifier
//              x      - horizontal position on map
//              y      - vertical position on map
//              utype  - unit type identifier; required when saving
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

Unit::Unit( const UnitType *type, Player *player, unsigned short id, short x, short y ) :
      MapObject( MO_UNIT ) {
  u_type = type;
  u_player = player;
  u_id = id;
  u_pos.x = x;
  u_pos.y = y;

  u_flags = type->ut_flags;
  u_moves = 0;
  u_facing = NORTH;
  u_group = MAX_GROUP_SIZE;
  u_xp = 0;

  u_target.x = u_target.y = 0;

  if ( player ) {
    u_pid = player->ID();
    player->Units( 1 );
  } else u_pid = PLAYER_NONE;

}

////////////////////////////////////////////////////////////////////////
// NAME       : Unit::Unit
// DESCRIPTION: Load a unit from a data file.
// PARAMETERS : file    - descriptor of an open data file
//              types   - pointer to the array of unit types
//              players - pointer to the players array
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

Unit::Unit( SDL_RWops *file, const UnitType *types, Player **players ) :
      MapObject( MO_UNIT ) {
  unsigned char typeno;

  u_pos.x = SDL_ReadLE16( file );
  u_pos.y = SDL_ReadLE16( file );
  u_flags = SDL_ReadLE32( file );
  u_id = SDL_ReadLE16( file );

  SDL_RWread( file, &u_moves, sizeof(unsigned char), 5 );
  SDL_RWread( file, &typeno, sizeof(unsigned char), 1 );

  u_target.x = SDL_ReadLE16( file );
  u_target.y = SDL_ReadLE16( file );

  u_type = &types[typeno];

  if ( u_pid != PLAYER_NONE ) {
    u_player = players[u_pid];
    u_player->Units( 1 );
  } else u_player = NULL;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Unit::Save
// DESCRIPTION: Save the unit to a data file.
// PARAMETERS : file - descriptor of the save file
// RETURNS    : 0 on succes, non-zero on error
//
// HISTORY
////////////////////////////////////////////////////////////////////////

int Unit::Save( SDL_RWops *file ) const {
  SDL_WriteLE16( file, u_pos.x );
  SDL_WriteLE16( file, u_pos.y );
  SDL_WriteLE32( file, u_flags );
  SDL_WriteLE16( file, u_id );

  SDL_RWwrite( file, &u_moves, sizeof(unsigned char), 5 );
  SDL_RWwrite( file, &u_type->ut_typeid, sizeof(unsigned char), 1 );

  SDL_WriteLE16( file, u_target.x );
  SDL_WriteLE16( file, u_target.y );

  return 0;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Unit::AwardXP
// DESCRIPTION: Raise the unit's experience level. This will improve its
//              chances to survive a battle and inflict more damage to
//              the enemy.
// PARAMETERS : xp - number of experience points to give; a unit
//                   advances to a new experience level every 3 points
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void Unit::AwardXP( unsigned char xp ) {
  // XP_MAX_LEVEL is the maximum experience level a unit can reach,
  // i.e. XP_MAX_LEVEL * XP_PER_LEVEL is the upper limit in points
  u_xp = MIN( u_xp + xp, XP_MAX_LEVEL * XP_PER_LEVEL );
}

////////////////////////////////////////////////////////////////////////
// NAME       : Unit::SetOwner
// DESCRIPTION: Change the controller of the unit.
// PARAMETERS : player - pointer to new controller
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void Unit::SetOwner( Player *player ) {
  if ( u_player ) u_player->Units( -1 );

  player->Units( 1 );
  u_pid = player->ID();
  u_player = player;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Unit::SetPosition
// DESCRIPTION: Change the position of the unit on the map. This is just
//              the low level function. It does not update the display
//              or do other fancy things. It's just about data shuffling.
// PARAMETERS : x - new x position
//              y - new y position
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void Unit::SetPosition( short x, short y ) {
  u_pos.x = x;
  u_pos.y = y;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Unit::Attack
// DESCRIPTION: Store an attack target. Again, this function does not
//              handle the display update, or the game mode change.
// PARAMETERS : enemy - unit to attack
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void Unit::Attack( const Unit *enemy ) {
  u_target = enemy->Position();
  SetFlags( U_ATTACKED|U_DONE );
  u_moves = 0;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Unit::DefensiveStrength
// DESCRIPTION: Get a number measuring the defensive capabilities
//              of the unit.
// PARAMETERS : -
// RETURNS    : defensive value
//
// HISTORY
////////////////////////////////////////////////////////////////////////

unsigned char Unit::DefensiveStrength( void ) const {
  return u_type->ut_defence;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Unit::OffensiveStrength
// DESCRIPTION: Return the combat value of the unit against the given
//              target.
// PARAMETERS : target   - unit to attack
// RETURNS    : offensive strength
//
// HISTORY
////////////////////////////////////////////////////////////////////////

unsigned char Unit::OffensiveStrength( const Unit *target ) const {
  unsigned char pow = 0;

  if ( target->IsShip() && Gam->IsWater( target->Position().x,
            target->Position().y ) ) pow = u_type->ut_pow_ship;
  else if ( target->IsGround() ) pow = u_type->ut_pow_ground;
  else if ( target->IsAircraft() ) pow = u_type->ut_pow_air;
  else if ( target->IsSubmarine() ) pow = u_type->ut_pow_sub;
  return pow;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Unit::Hit
// DESCRIPTION: Inflict damage to the unit and destroy it if necessary.
// PARAMETERS : damage - how much damage is done
// RETURNS    : true if the unit was destroyed, false otherwise
//
// HISTORY
////////////////////////////////////////////////////////////////////////

bool Unit::Hit( unsigned short damage ) {
  if ( !IsAlive() ) return false;

  if ( u_group <= damage ) {     // unit destroyed
    u_group = 0;
    u_player->Units( -1 );

    SetFlags( U_DESTROYED );
    u_pos.x = u_pos.y = -1;
    return true;
  }
 
  u_group -= damage;
  return false;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Unit::CanHitType
// DESCRIPTION: Check whether the unit can shoot at an enemy unit. The
//              test is performed looking only at the enemy unit type
//              (ground, ship. aircraft, or submarine). If the function
//              returns true, this doesn't mean that the enemy is
//              actually in range for the unit's weapons systems.
// PARAMETERS : enemy - enemy unit
// RETURNS    : true if the unit type can be shot; false otherwise
//
// HISTORY
////////////////////////////////////////////////////////////////////////

bool Unit::CanHitType( const Unit *enemy ) const {
  return WeaponRange( enemy ) > 0;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Unit::CanHit
// DESCRIPTION: Check whether the unit can shoot at an enemy unit. The
//              test is performed with regard to the enemy unit type
//              (ground, ship. aircraft, or submarine) and the distance
//              of the target.
// PARAMETERS : enemy - enemy unit
// RETURNS    : true if the unit can be shot; false if it's the wrong
//              type, too far away, or not hostile
//
// HISTORY
////////////////////////////////////////////////////////////////////////

bool Unit::CanHit( const Unit *enemy ) const {
  if ( (u_player == enemy->Owner()) ||
       (u_flags & (U_DONE|U_SHELTERED)) ||
       enemy->IsSheltered() ) return false;
  return CouldHit( enemy );
}

////////////////////////////////////////////////////////////////////////
// NAME       : Unit::CouldHit
// DESCRIPTION: Check whether the unit could shoot at an enemy unit.
//              Current unit status is not taking into account, so this
//              function may return true, even though the unit already
//              attacked this turn or moved and can't attack any more.
// PARAMETERS : enemy - enemy unit
// RETURNS    : true if the unit type could be shot; false otherwise
//
// HISTORY
////////////////////////////////////////////////////////////////////////

bool Unit::CouldHit( const Unit *enemy ) const {
  unsigned short dist = Distance( u_pos, enemy->Position() );

  // units can be U_SHIP and U_GROUND at once
  if ( enemy->IsShip() &&
       Gam->IsWater( enemy->Position().x, enemy->Position().y ) )
    return ( (u_type->ut_min_range_ship <= dist)
             && (u_type->ut_max_range_ship >= dist) );

  else if ( enemy->IsGround() )
    return ( (u_type->ut_min_range_ground <= dist)
             && (u_type->ut_max_range_ground >= dist) );

  else if ( enemy->IsAircraft() )
    return ( (u_type->ut_min_range_air <= dist)
             && (u_type->ut_max_range_air >= dist) );
 
  else if ( enemy->IsSubmarine() )
    return ( (u_type->ut_min_range_sub <= dist)
             && (u_type->ut_max_range_sub >= dist) );

  return false;
}

////////////////////////////////////////////////////////////////////
// NAME       : Unit::Repair
// DESCRIPTION: Repair the unit (completely). This will get the
//              group size of the unit back to the maximum, but will
//              also lower its experience by one point per two
//              "rookies".
// PARAMETERS : resources - amount of resources available
// RETURNS    : amount of resources left after repair; if the amount
//              available was insufficient, this will be the same as
//              the original amount
//
// HISTORY
////////////////////////////////////////////////////////////////////////

unsigned short Unit::Repair( unsigned short resources ) {
  if ( u_group < MAX_GROUP_SIZE ) {
    short cost = RepairCost();
    if ( resources >= cost ) {
      u_xp = MAX( 0, u_xp - (MAX_GROUP_SIZE - u_group)/2 );
      u_group = MAX_GROUP_SIZE;
      resources -= cost;
      u_moves = 0;
      SetFlags( U_DONE );	// can't move this turn
    }
  }
  return resources;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Unit::RepairCost
// DESCRIPTION: Calculate which amount of resources would be necessary
//              to patch this unit back together. Repairing a single
//              group member requires an amount of resources equal to
//              1/10th of the build costs, but at least 1.
// PARAMETERS : -
// RETURNS    : amount of resources required for repair
//
// HISTORY
////////////////////////////////////////////////////////////////////////

unsigned short Unit::RepairCost( void ) const {
  if ( u_group == MAX_GROUP_SIZE ) return 0;
  return MAX( (MAX_GROUP_SIZE - u_group) * u_type->ut_build / 10, 1 );
}

////////////////////////////////////////////////////////////////////////
// NAME       : Unit::WeaponRange
// DESCRIPTION: Get the distance from which the unit may shoot at a
//              given target.
// PARAMETERS : u - target unit
// RETURNS    : maximum range of fire
//
// HISTORY
////////////////////////////////////////////////////////////////////////

unsigned char Unit::WeaponRange( const Unit *u ) const {

  if ( u->IsShip() &&
       Gam->IsWater( u->Position().x, u->Position().y ) )
    return u_type->ut_max_range_ship;

  else if ( u->IsGround() )
    return u_type->ut_max_range_ground;

  else if ( u->IsAircraft() )
    return u_type->ut_max_range_air;

  else if ( u->IsSubmarine() )
    return u_type->ut_max_range_sub;

  else return 0;
}

