// Crimson Fields -- a game of tactical warfare
// Copyright (C) 2000, 2001 Jens Granseuer
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

////////////////////////////////////////////////////////////////////////
// ai.cpp
//
// The model used for the computer player in Crimson Fields is a
// variation of the widely used General/Sergeant scheme. At the
// beginning of a turn, the "General" assesses the overall situation
// on the battlefield, and identifies the major objectives for the
// computer player (usually buildings). Units are assigned to those
// objectives in order of priority, i.e. more important objectives
// are served first, and less important ones may end up with less
// firepower than required.
//   Now the "Sergeants" take over. Each of them gets one objective
// and all the units assigned to it, and it's up to those division
// commanders to decide on the actual moves.
//   Right now, the AI player is recreated each turn. This, of course,
// does not allow for collecting and analyzing long-term intelligence
// data, so maybe this should be changed in the future.
//
// History:
//  12-05-2001 - created
////////////////////////////////////////////////////////////////////////

#include "ai.h"
#include "misc.h"
#include "globals.h"

////////////////////////////////////////////////////////////////////////
// NAME       : AI::AI
// DESCRIPTION: Initialize a computer controlled player.
// PARAMETERS : -
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

AI::AI( void ) {
  player = Gam->CurrentPlayer();
}

////////////////////////////////////////////////////////////////////////
// NAME       : AI::Play
// DESCRIPTION: Run the computer player.
// PARAMETERS : -
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void AI::Play( void ) {
  IdentifyObjectives();
  AssignObjectives();

  for ( AIObj *obj = static_cast<AIObj *>( objectives.Head() );
        obj; obj = static_cast<AIObj *>( obj->Next() ) )
    ProcessObjective( *obj );
}

////////////////////////////////////////////////////////////////////////
// NAME       : AI::IdentifyObjectives
// DESCRIPTION: Examine the map for targets and put them into the list
//              of objectives. Generally, all buildings are considered
//              objectives.
// PARAMETERS : -
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void AI::IdentifyObjectives( void ) {
  AIObj *obj;

  for ( Building *b = static_cast<Building *>(Gam->BuildingsList()->Head());
        b; b = static_cast<Building *>(b->Next()) ) {
    unsigned char pri = AI_PRI_LOW;
    obj = new AIObj;
    obj->pos = b->Position();
    obj->flags = 0;
    obj->needed_ground = obj->needed_ship = obj->needed_air = 0;

    if ( b->Owner() == player ) obj->type = AI_OBJ_DEFEND;
    else obj->type = AI_OBJ_CONQUER;

    // determine enemy unit presence in the vicinity
    UnitPresence( Gam->NextPlayer(), obj );

    if ( b->IsFactory() ) pri += 10;
    else if ( b->IsWorkshop() ) pri += 2;
    if ( b->IsMine() ) pri += 5;

    if ( obj->type == AI_OBJ_DEFEND ) {
      // check enemy presence; this also affects priority
      // locate closest enemy unit able to conquer buildings or transport
      Unit *u = ClosestUnit( Gam->NextPlayer(), obj->pos, U_CONQUER|U_TRANSPORT, U_DESTROYED );
      if ( u && (Distance( u->Position(), obj->pos ) <= AI_ATTENTION_RADIUS) )
        pri = AI_PRI_CRITICAL;
      else pri = MAX( pri + obj->needed_ground + obj->needed_air + obj->needed_ship, AI_PRI_MAX );
    } else if ( obj->type == AI_OBJ_CONQUER ) {
      // the farther away we are the lower the priority
      pri += AI_PRI_MEDIUM;
      Unit *u = ClosestUnit( player, obj->pos, U_CONQUER, U_DESTROYED );
      if ( u ) pri = MAX( AI_PRI_LOW, pri - Distance( obj->pos, u->Position() ) );
    }

    obj->priority = pri;

    // insert the objective into the list according to its priority
    AIObj *aio, *prev = NULL;
    for ( aio = static_cast<AIObj *>(objectives.Head());
          aio && (aio->priority > pri);
          aio = static_cast<AIObj *>(aio->Next()) ) prev = aio;
    objectives.InsertNode( obj, prev );
  }

  // create an objective which simply says: "destroy all enemy units"
  // this will be used for all units we couldn't use otherwise
  obj = new AIObj;
  obj->type = AI_OBJ_ATTACK;
  obj->pos.x = -1;
  obj->pos.y = -1;
  obj->flags = 0;
  obj->needed_ground = obj->needed_ship = obj->needed_air = 0;
  obj->priority = 0;
  objectives.AddTail( obj );
}

////////////////////////////////////////////////////////////////////////
// NAME       : AI::AssignObjectives
// DESCRIPTION: Go through the list of units available and assign them
//              to one of the objectives we identified earlier.
// PARAMETERS : -
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void AI::AssignObjectives( void ) {
  AIObj *obj = static_cast<AIObj *>(objectives.Head());
  Unit *u, *last;

  while ( obj ) {

    if ( obj->type == AI_OBJ_CONQUER ) {
      // find ourselves a unit which can take over buildings
      bool done = false, found = true;
      last = NULL;
      do {
        u = ClosestUnit( player, obj->pos, U_CONQUER, U_BUSY, last );
        if ( u ) {
          if ( !UnitCanReach( u, obj->pos, 0, obj ) ) last = u;
          else {
            obj->AssignUnit( u, UnitStrength(u) );
            done = true;
          }
        } else {
          // we can't conquer the building, so remove the objective
          AIObj *next = static_cast<AIObj *>(obj->Next());
          obj->Remove();
          delete obj;
          obj = next;
          done = true;
          found = false;
        }
      } while ( !done );

      if ( !found ) continue;      // go to next objective
    }

    // cycle through the units list
    // start with the unit closest to the target
    last = NULL;
    while ( (u = ClosestUnit( player, obj->pos, U_GROUND|U_AIR|U_SHIP, U_BUSY, last )) ) {
      if ( u->IsMine() ) u->SetFlags( U_BUSY|U_DONE );
      else if ( (!u->IsConquer() || (obj->type == AI_OBJ_CONQUER)) &&      // use infantry only
                UnitCanReach( u, obj->pos, AI_ATTENTION_RADIUS, obj ) ) {  // for taking buildings
        const UnitType *type = u->Type();
        if ( ((obj->needed_air > 0) && (type->ut_pow_air > 0)) ||
             ((obj->needed_ground > 0) && (type->ut_pow_ground > 0)) ||
             ((obj->needed_ship > 0) && (type->ut_pow_ship > 0)) ) {
          // assign unit to target
          obj->AssignUnit( u, UnitStrength(u) );
          if ( obj->needed_air + obj->needed_ground + obj->needed_ship == 0 )
            break;      // requested firepower allocated; next objective
        }
      }
      last = u;
    }

    obj = static_cast<AIObj *>(obj->Next());
  }

  // now check from the back of the list and remove any offensive objectives
  // which have not had any one unit assigned as well as those which have not
  // had the requested number of units assigned except the one with the
  // highest priority

  bool saved_hipri = false;
  obj = static_cast<AIObj *>( objectives.Head() );
  while ( obj ) {
    if ( (obj->type == AI_OBJ_CONQUER) &&
         (obj->needed_ground + obj->needed_air + obj->needed_ship > 0) ) {
      if ( obj->alloc_units.IsEmpty() || saved_hipri ) {
        AIObj *next = static_cast<AIObj *>(obj->Next());
        obj->ReleaseUnits();
        obj->Remove();
        delete obj;
        obj = next;
        continue;
      } else saved_hipri = true;
    }
    obj = static_cast<AIObj *>( obj->Next() );
  }

  // lastly, assign all unassigned units to a task
  for ( u = static_cast<Unit *>( Gam->UnitsList()->Head() );
        u; u = static_cast<Unit *>(u->Next()) ) {
    if ( (u->Owner() == player) && !(u->Flags() & U_BUSY) ) {
      AIObj *best = NULL;
      unsigned short bestval = 0;

      for ( obj = static_cast<AIObj *>(objectives.Head());
            obj; obj = static_cast<AIObj *>(obj->Next()) ) {
        const UnitType *type = u->Type();
        if ( (obj->requested_ground && type->ut_pow_ground) ||
             (obj->requested_air && type->ut_pow_air) ||
             (obj->requested_ship && type->ut_pow_ship) ) {
          unsigned short val = 1000 + obj->priority -
                               Distance( obj->pos, u->Position() );
          if ( val > bestval ) {
            bestval = val;
            best = obj;
          }
        }
      }

      // if we didn't find a suitable objective, assign the unit to
      // the general "attack-everything-that-moves" objective
      if ( best ) best->AssignUnit( u, UnitStrength(u) );
      else static_cast<AIObj *>(objectives.Tail())->AssignUnit( u, 0 );
    }
  }
}

////////////////////////////////////////////////////////////////////////
// NAME       : AI::ProcessObjective
// DESCRIPTION: This function implements the duties of the sergeant,
//              who decides on the actual moves a unit will make this
//              turn in order to accomplish the assigned objective.
// PARAMETERS : obj - objective the sergeant is responsible for
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void AI::ProcessObjective( AIObj &obj ) {

  while ( !obj.alloc_units.IsEmpty() ) {
    AIObj::AIAllocNode *n = static_cast<AIObj::AIAllocNode *>( obj.alloc_units.RemHead() );
    Unit *u = n->unit;
    delete n;

    if ( u->IsReady() ) {
      Gam->SelectUnit( u );
      CommandUnit( u, obj );
    }
  }
}

////////////////////////////////////////////////////////////////////////
// NAME       : AI::CommandUnit
// DESCRIPTION: This function does the Sergeant's dirty work. It
//              inspects the given unit and decides on what this unit
//              should do - where it should go, whom it should attack,
//              and so on.
// PARAMETERS : u   - the unit to be decided on
//              obj - objective the unit was assigned to
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void AI::CommandUnit( Unit *u, AIObj &obj ) {

  // maybe we need an overhaul?
  if ( (obj.priority < AI_PRI_CRITICAL) ||
       (u->GroupSize() >= MAX_GROUP_SIZE / 2) ||
       !CommandUnitRepair( u ) ) {

    switch ( obj.type ) {
      case AI_OBJ_DEFEND:
        CommandUnitDefend( u, obj );
        break;
      case AI_OBJ_CONQUER:
        CommandUnitConquer( u, obj );
        break;
      case AI_OBJ_ATTACK:
        CommandUnitAttack( u, obj );
        break;
    }
  }
}

////////////////////////////////////////////////////////////////////////
// NAME       : AI::CommandUnitDefend
// DESCRIPTION: Decide on how the unit can be used to defend the
//              objective.
// PARAMETERS : u   - the unit to be decided on
//              obj - objective of the unit (type AI_OBJ_DEFEND)
// RETURNS    : -
//
// HISTORY
//   15-11-2001 - if defending unit is inside the base move it out
////////////////////////////////////////////////////////////////////////

void AI::CommandUnitDefend( Unit *u, AIObj &obj ) {
  Unit *tg;

  if ( Distance( u->Position(), obj.pos ) > AI_ATTENTION_RADIUS ) {
    // find a nice target in the direction we need to go
    bool attacked = false;
    Unit *last = NULL;
    do {
      tg = ClosestUnit( Gam->NextPlayer(), u->Position(), U_GROUND|U_AIR|U_SHIP, U_DESTROYED|U_SHELTERED, last );
      if ( tg && u->CanHitType( tg ) &&
           (XY2Dir( u->Position().x, u->Position().y, tg->Position().x, tg->Position().y )
            == XY2Dir( u->Position().x, u->Position().y, obj.pos.x, obj.pos.y )) &&
           (Gam->FindPath( u, tg->Position().x, tg->Position().y, PATH_BEST, u->WeaponRange( tg )) >= 0) ) {
        attacked = true;
        Point dest = FollowPath( u );
        Gam->MoveUnit( u, dest.x, dest.y );
        if ( u->CanHit( tg ) ) Gam->RegisterCombat( u, tg );
        break;
      } else last = tg;
    } while ( tg );

    if ( !attacked && (Gam->FindPath( u, obj.pos.x, obj.pos.y, PATH_BEST, AI_ATTENTION_RADIUS ) >= 0) ) {
      Point dest = FollowPath( u );
      Gam->MoveUnit( u, dest.x, dest.y );
    }
  } else {
    tg = ClosestUnit( Gam->NextPlayer(), obj.pos, U_GROUND|U_AIR|U_SHIP, U_DESTROYED|U_SHELTERED );
    if ( tg && (Distance( u->Position(), tg->Position() ) <= AI_ATTENTION_RADIUS) ) {
      CommandUnitAttack( u, obj );
    } else if ( obj.pos == u->Position() ) CommandUnitAttack( u, obj );
  }
}

////////////////////////////////////////////////////////////////////////
// NAME       : AI::CommandUnitConquer
// DESCRIPTION: Try to conquer the objective or help another unit take
//              it.
// PARAMETERS : u   - the unit to be decided on
//              obj - objective of the unit (type AI_OBJ_CONQUER)
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void AI::CommandUnitConquer( Unit *u, AIObj &obj ) {
  Unit *enemy;
  Point dest;
  short turns;

  if ( u->IsConquer() ) {
    // this unit won't attack anyone; we need it to get the target building
    turns = Gam->FindPath( u, obj.pos.x, obj.pos.y, PATH_BEST, 0 );

    if ( turns > 1 ) {
      // if we are inside a transport that's faster than we are, stay
      Unit *t = Gam->GetUnit( u->Position().x, u->Position().y );
      if ( t && (t != u) && (t->Type()->ut_moves > u->Type()->ut_moves) && t->IsReady() &&
             (Gam->FindPath( t, obj.pos.x, obj.pos.y, PATH_BEST, 2 ) >= 0) ) {
        dest = FollowPath( t );
        Gam->SelectUnit( t );
        Gam->MoveUnit( t, dest.x, dest.y );
        Gam->SelectUnit( u );
        turns = Gam->FindPath( u, obj.pos.x, obj.pos.y, PATH_BEST, 0 );
      } else {
        dest = FollowPath( u );
        Gam->MoveUnit( u, dest.x, dest.y );
        return;
      }
    }

    if ( turns == 1 ) {
      // the building could be conquered this turn, but we only want to
      // if there are no enemies close enough to take it back on the next
      bool ok = true;

      for ( enemy = static_cast<Unit *>( Gam->UnitsList()->Head() );
            enemy; enemy = static_cast<Unit *>( enemy->Next() ) ) {
        if ( enemy->IsConquer() && (enemy->Owner() == Gam->NextPlayer())
           && (Gam->FindPath( enemy, obj.pos.x, obj.pos.y, PATH_BEST, 0 ) == 1) )
          ok = false;
      }

      if ( ok ) Gam->MoveUnit( u, obj.pos.x, obj.pos.y );
    }
  } else {          // normal units move towards the objective and attack
                    // hostile units along the way
    enemy = FindBestTarget( u );
    if ( (Gam->UnitTargets( u ) > 0) && enemy &&
         (Gam->FindPath( u, enemy->Position().x, enemy->Position().y,
                         PATH_BEST, u->WeaponRange( enemy ) ) >= 0) ) {
      dest = FollowPath( u );
      Gam->MoveUnit( u, dest.x, dest.y );
      if ( u->CanHit( enemy ) ) Gam->RegisterCombat( u, enemy );
    } else if ( !enemy ) CommandUnitReturnToBase( u );
    else {
      turns = Gam->FindPath( u, obj.pos.x, obj.pos.y, PATH_BEST, AI_ATTENTION_RADIUS );
      if ( turns > 0 ) {
        dest = FollowPath( u );
        Gam->MoveUnit( u, dest.x, dest.y );
        if ( (Gam->UnitTargets( u ) > 0) && (enemy = FindBestTarget( u )) && u->CanHit( enemy ) )
          Gam->RegisterCombat( u, enemy );
      } else if ( turns == 0 ) CommandUnitAttack( u, obj );
    }
  }
}

////////////////////////////////////////////////////////////////////////
// NAME       : AI::CommandUnitAttack
// DESCRIPTION: Attack any enemy units we can get our hands on.
// PARAMETERS : u   - the unit to be decided on
//              obj - objective of the unit (type AI_OBJ_ATTACK; unused)
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void AI::CommandUnitAttack( Unit *u, AIObj &obj ) {
  Unit *tg = FindBestTarget( u );
  if ( tg ) {
    Point dest;
    if ( Gam->FindPath( u, tg->Position().x, tg->Position().y,
                        PATH_BEST, u->WeaponRange( tg ) ) == 1 ) {
      dest = FindBestHex( u, tg );
      Gam->FindPath( u, dest.x, dest.y, PATH_BEST, 0 );
    }
    dest = FollowPath( u );
    Gam->MoveUnit( u, dest.x, dest.y );
    if ( u->CanHit( tg ) ) Gam->RegisterCombat( u, tg );
  } else CommandUnitReturnToBase( u );
}

////////////////////////////////////////////////////////////////////////
// NAME       : AI::CommandUnitRepair
// DESCRIPTION: Try to repair a unit.
// PARAMETERS : u - the unit to be repaired
// RETURNS    : TRUE if a suitable base with a sufficient amount of
//              crystals is found, FALSE otherwise
//
// HISTORY
////////////////////////////////////////////////////////////////////////

bool AI::CommandUnitRepair( Unit *u ) {
  Building *b, *last = NULL;
  unsigned short cost = u->RepairCost();

  do {
    b = ClosestBuilding( player, u->Position(), last );

    int way;
    if ( b && (b->Crystals() >= cost) &&
       ((way = Gam->FindPath( u, b->Position().x, b->Position().y, PATH_BEST, 0 )) >= 0) &&
       (way <= 5) ) {
      Point dest = FollowPath( u );
      Gam->MoveUnit( u, dest.x, dest.y );

      if ( u->Position() == b->Position() ) b->SetCrystals( u->Repair( b->Crystals() ) );
      return true;
    } else last = b;
  } while ( b );
  return false;
}

////////////////////////////////////////////////////////////////////////
// NAME       : AI::CommandUnitReturnToBase
// DESCRIPTION: Let this unit retreat to a friendly base. If we do not
//              own any buildings, the unit won't move at all...
// PARAMETERS : u - the retreating unit
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void AI::CommandUnitReturnToBase( Unit *u ) {
  Building *b, *last = NULL;

  do {
    b = ClosestBuilding( player, u->Position(), last );

    if ( b && (Gam->FindPath( u, b->Position().x, b->Position().y, PATH_BEST, 0 ) >= 0) ) {
      Point dest = FollowPath( u );
      Gam->MoveUnit( u, dest.x, dest.y );
      break;
    } else last = b;
  } while ( b );
}

////////////////////////////////////////////////////////////////////////
// NAME       : AI::UnitStrength
// DESCRIPTION: Calculate the combat strength of a unit. Putting this
//              into a single number makes this approach slightly
//              inaccurate, but easier to handle. This value is only
//              used internally by the computer player.
// PARAMETERS : u - unit
// RETURNS    : combat strength
//
// HISTORY
////////////////////////////////////////////////////////////////////////

unsigned short AI::UnitStrength( Unit *u ) const {
  const UnitType *type = u->Type();
  return (MAX( MAX( type->ut_pow_ground, type->ut_pow_ship ), type->ut_pow_air )
       + type->ut_defence + 3 * u->XPLevel()) * u->GroupSize() / MAX_GROUP_SIZE;
}

////////////////////////////////////////////////////////////////////////
// NAME       : AI::UnitPresence
// DESCRIPTION: Calculate the combined firepower of all units in a given
//              area which are controlled by the given player.
// PARAMETERS : owner  - player controlling wanted units
//              obj    - objective; the objective position defines the
//                       center of the area to be scanned, and the
//                       needed_xxx values will be filled with our
//                       findings
//              radius - area radius; if this is -1, all enemy units
//                       will be considered according to their
//                       distance to the objective (default -1)
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void AI::UnitPresence( Player *owner, AIObj *obj, short radius /* = -1 */ ) const {
  for ( Unit *u = static_cast<Unit *>(Gam->UnitsList()->Head());
        u; u = static_cast<Unit *>(u->Next()) ) {
    if ( (u->Owner() == owner) && ((radius == -1) ||
       (Distance( obj->pos, u->Position() ) <= radius)) ) {
      unsigned short str = MAX( 0, UnitStrength(u) - Distance( obj->pos, u->Position() ) * 2 );
      if ( u->IsAircraft() ) obj->needed_air += str;
      else if ( u->IsGround() ) obj->needed_ground += str;
      else obj->needed_ship += str;
    }
  }
  obj->requested_ground = obj->needed_ground > 0;
  obj->requested_air = obj->needed_air > 0;
  obj->requested_ship = obj->needed_ship > 0;
}

////////////////////////////////////////////////////////////////////////
// NAME       : AI::ClosestUnit
// DESCRIPTION: Find the unit with a given set of abilities which is
//              closest to the target location.
// PARAMETERS : owner   - player who controls the unit
//              p       - target location
//              uflags  - flags which should be looked for
//              nuflags - flags which must not be set
//              last    - last unit found. If this is given, find the
//                        next unit which is as far or further away
//                        than that one and matches the criteria. If
//                        it is NULL (default) return the closest unit.
// RETURNS    : closest unit which matches any ONE of the uflags and
//              NONE of the nuflags; or NULL if no appropriate unit found
//
// HISTORY
////////////////////////////////////////////////////////////////////////

Unit *AI::ClosestUnit( Player *owner, const Point &p, unsigned long uflags,
                       unsigned long nuflags, const Unit *last /* = NULL */ ) const {
  Unit *u, *best = NULL;
  int last_dist, best_dist = 9999;

  if ( last ) {
    last_dist = Distance( p, last->Position() );
    u = static_cast<Unit *>( last->Next() );
    if ( !u ) {
      u = static_cast<Unit *>( Gam->UnitsList()->Head() );
      last_dist++;
    }
  } else {
    u = static_cast<Unit *>( Gam->UnitsList()->Head() );
    last_dist = -1;
  }

  while ( u != last ) {
    int dist = Distance( u->Position(), p );

    if ( (u->Flags() & uflags) && !(u->Flags() & nuflags) &&
         (u->Owner() == owner) &&
         (dist >= last_dist) && (dist < best_dist) ) {
      best_dist = dist;
      best = u;
    }
    u = static_cast<Unit *>( u->Next() );
    if ( !u && last ) {
      u = static_cast<Unit *>( Gam->UnitsList()->Head() );
      last_dist++;      // increase distance for the next loop
    }
  }

  return best;
}

////////////////////////////////////////////////////////////////////////
// NAME       : AI::ClosestBuilding
// DESCRIPTION: Find the building closest to a given hex.
// PARAMETERS : owner - owning player
//              p     - target location
//              last  - last building found. If this is given, find
//                      the next building which is as far or further
//                      away than that one. If it is NULL (default)
//                      return the closest building.
// RETURNS    : closest building owned by the given player, or NULL if
//              no appropriate building found
//
// HISTORY
////////////////////////////////////////////////////////////////////////

Building *AI::ClosestBuilding( Player *owner, const Point &p,
                               const Building *last /* = NULL */ ) const {
  Building *b, *best = NULL;
  int last_dist, best_dist = 9999;

  if ( last ) {
    last_dist = Distance( p, last->Position() );
    b = static_cast<Building *>( last->Next() );
    if ( !b ) {
      b = static_cast<Building *>( Gam->BuildingsList()->Head() );
      last_dist++;
    }
  } else {
    b = static_cast<Building *>( Gam->BuildingsList()->Head() );
    last_dist = -1;
  }

  while ( b != last ) {
    int dist = Distance( b->Position(), p );

    if ( (b->Owner() == owner) && (dist >= last_dist) && (dist < best_dist) ) {
      best_dist = dist;
      best = b;
    }
    b = static_cast<Building *>( b->Next() );
    if ( !b && last ) {
      b = static_cast<Building *>( Gam->BuildingsList()->Head() );
      last_dist++;      // increase distance for the next loop
    }
  }

  return best;
}

////////////////////////////////////////////////////////////////////////
// NAME       : AI::UnitCanReach
// DESCRIPTION: Check whether a unit can get close to a given hex,
//              either by itself or by using an available transport.
// PARAMETERS : u    - unit
//              pos  - location to be checked for
//              dist - sometimes it's not necessary to get to the
//                     specified coordinates. Here you can specify the
//                     tolerable distance if you only want to get close.
//              obj  - objective which is currently being checked. This
//                     is used if a transport is required for getting
//                     the unit to the destination. If the unit does not
//                     have U_BUSY set (i.e. is not yet assigned to the
//                     objective) we'll look for an appropriate carrier
//                     among all unassigned units and, if successful,
//                     assign it to the objective. If U_BUSY is set, the
//                     transport is only looked for in the list of units
//                     assigned to the objective.
// RETURNS    : TRUE if the unit can get close enough to the destination
//              FALSE otherwise
//
// TODO       : Handling of transports is not yet implemented.
//
// HISTORY
////////////////////////////////////////////////////////////////////////

bool AI::UnitCanReach( const Unit *u, const Point &pos,
         unsigned short dist, AIObj *obj ) const {
  if ( Gam->FindPath( u, pos.x, pos.y, PATH_FAST, dist ) != -1 ) return true;
  return false;
}

////////////////////////////////////////////////////////////////////////
// NAME       : AI::FollowPath
// DESCRIPTION: If a path has been found, determine how far the unit
//              can get this turn.
// PARAMETERS : u - unit
// RETURNS    : furthest hex along the path which can be reached this
//              turn
//
// HISTORY
////////////////////////////////////////////////////////////////////////

Point AI::FollowPath( const Unit *u ) const {
  short *path = Gam->GetPath(), dir;

  Point p = u->Position();
  List hexes;
  PathNode *n;

  do {
    n = new PathNode;
    n->x = p.x;
    n->y = p.y;
    hexes.AddTail( n );

    dir = path[Gam->XY2Index(p.x,p.y)];
    if ( dir != -1 ) Gam->Dir2XY( p.x, p.y, (Direction)dir, p );
  } while ( dir != -1 );

  Gam->CreateMoveMap();

  for ( n = static_cast<PathNode *>( hexes.Head() );
        n; n = static_cast<PathNode *>( n->Next() ) ) {
    if ( path[Gam->XY2Index(n->x,n->y)] == 1 ) {
      p.x = n->x;
      p.y = n->y;
    }
  }

  return p;
}

////////////////////////////////////////////////////////////////////////
// NAME       : AI::FindBestTarget
// DESCRIPTION: Decide which enemy unit to attackk, if any.
// PARAMETERS : u - unit to check targets for
// RETURNS    : enemy unit considered the best target, or NULL if none
//
// HISTORY
////////////////////////////////////////////////////////////////////////

Unit *AI::FindBestTarget( const Unit *u ) const {
  Unit *best = NULL;
  unsigned short bestval = 0;

  for ( Unit *tg = static_cast<Unit *>( Gam->UnitsList()->Head() );
        tg; tg = static_cast<Unit *>( tg->Next() ) ) {
    if ( (tg->Owner() == Gam->NextPlayer()) && u->CanHitType( tg )
         && !tg->IsSheltered() ) {
      short cost = Gam->FindPath( u, tg->Position().x, tg->Position().y,
                                  PATH_FAST, u->WeaponRange( tg ) );
      if ( cost >= 0 ) {
        unsigned short val = MAX( 0, 1000 - UnitStrength( tg ) - cost * 10 );
        if ( tg->IsConquer() ) val += 9;
        else if ( !tg->CanHitType( u ) ) val += 8;
        else if ( tg->IsTransport() ) val += 5;

        if ( val > bestval ) {
          bestval = val;
          best = tg;
        }
      }
    }
  }
  return best;
}

////////////////////////////////////////////////////////////////////////
// NAME       : AI::FindBestHex
// DESCRIPTION: Determine the hex from which an attack can be expected
//              to bear the best result.
// PARAMETERS : u     - attacking unit
//              enemy - target unit
// RETURNS    : best attacking position or current location if target
//              cannot be reached this turn (however, the current
//              location may just as well be the best ambush point
//              indeed, so check the result)
//
// HISTORY
////////////////////////////////////////////////////////////////////////

#define _CF_BEST_HEX_INVALID	-1000
Point AI::FindBestHex( const Unit *u, const Unit *enemy ) const {
  Point hex = u->Position();

  // for long-range attacks, terrain type etc. are not important
  if ( u->WeaponRange( enemy ) > 1 ) {
    if ( Gam->FindPath( u, enemy->Position().x, enemy->Position().y,
                        PATH_BEST, u->WeaponRange( enemy ) ) >= 0 ) {
      hex = FollowPath( u );
    }
  } else {
    Unit *sup;
    Point nb[6], hlp;
    short vals[6], bestval = _CF_BEST_HEX_INVALID;
    int i;

    Gam->GetNeighbors( enemy->Position().x, enemy->Position().y, nb );
    for ( i = NORTH; i <= NORTHWEST; i++ ) {
      if ( (nb[i].x != -1) && (nb[i].y != -1) &&
           (Gam->FindPath( u, nb[i].x, nb[i].y, PATH_FAST, 0 ) == 1) ) {
        vals[i] = 0;
      } else vals[i] = _CF_BEST_HEX_INVALID;
    }

    for ( i = NORTH; i <= NORTHWEST; i++ ) {
      if ( vals[i] != _CF_BEST_HEX_INVALID ) {
        vals[i] += Gam->AttackMod( nb[i].x, nb[i].y );

        // check for support in the back of the enemy
        int j = ReverseDir( (Direction)i );
        if ( (nb[j].x != -1) && (nb[j].y != -1) ) {
          sup = Gam->GetUnit( nb[j].x, nb[j].y );
          if ( sup && (sup->Owner() == player) ) {
            if ( sup->CouldHit( enemy ) ) vals[i] += 7;
            else vals[i] += 2;
          }
        }

        // check for enemy units supporting defence
        Direction attdir = XY2Dir( nb[i].x, nb[i].y, enemy->Position().x, enemy->Position().y );
        if ( !Gam->Dir2XY( nb[i].x, nb[i].y, TurnLeft(attdir), hlp ) &&
           (sup = Gam->GetUnit( hlp.x, hlp.y )) && (sup->Owner() == player) ) vals[i] -= 8;
        if ( !Gam->Dir2XY( nb[i].x, nb[i].y, TurnRight(attdir), hlp ) &&
           (sup = Gam->GetUnit( hlp.x, hlp.y )) && (sup->Owner() == player) ) vals[i] -= 8;
      }
    }

    for ( i = NORTH; i <= NORTHWEST; i++ ) {
      if ( vals[i] > bestval ) {
        bestval = vals[i];
        hex = nb[i];
      }
    }
  }

  return hex;
}
#undef _CF_BEST_HEX_INVALID

////////////////////////////////////////////////////////////////////////
// NAME       : AI::AIObj::AssignUnit
// DESCRIPTION: Assign a unit to the objective.
// PARAMETERS : unit - unit
//              str  - unit strength value (see AI::UnitStrength())
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void AI::AIObj::AssignUnit( Unit *unit, unsigned short str ) {
  AIAllocNode *n = new AIAllocNode( unit );
  if ( unit->IsTransport() ) alloc_units.AddTail( n );
  else alloc_units.AddHead( n );
  unit->SetFlags( U_BUSY );

  const UnitType *type = unit->Type();
  unsigned char num = 0;
  if ( type->ut_pow_air > 0 ) ++num;
  if ( type->ut_pow_ground > 0 ) ++num;
  if ( type->ut_pow_ship > 0 ) ++num;

  if ( num > 0 ) {
    str /= num;
    if ( type->ut_pow_air > 0 ) needed_air = MAX( 0, needed_air - str );
    if ( type->ut_pow_ground > 0 ) needed_ground = MAX( 0, needed_ground - str );
    if ( type->ut_pow_ship > 0 ) needed_ship = MAX( 0, needed_ship - str );
  }
}

////////////////////////////////////////////////////////////////////////
// NAME       : AI::AIObj::ReleaseUnits
// DESCRIPTION: Discard the objective and release all units so they can
//              be assigned to other objectives.
// PARAMETERS : -
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void AI::AIObj::ReleaseUnits( void ) {
  while ( !alloc_units.IsEmpty() ) {
    AIAllocNode *n = static_cast<AIAllocNode *>( alloc_units.RemHead() );
    n->unit->UnsetFlags( U_BUSY );
    delete n;
  }
}

////////////////////////////////////////////////////////////////////////
// NAME       : AI::AIObj::UnitAssigned
// DESCRIPTION: Check whether a specific unit is assigned to this
//              objective.
// PARAMETERS : u - unit to check for
// RETURNS    : TRUE if unit is assigned to the objective, FALSE
//              otherwise
//
// HISTORY
////////////////////////////////////////////////////////////////////////

bool AI::AIObj::UnitAssigned( const Unit *u ) const {
  for ( AIAllocNode *n = static_cast<AIAllocNode *>( alloc_units.Head() );
        n; n = static_cast<AIAllocNode *>( n->Next() ) ) {
    if ( n->unit == u ) return true;
  }
  return false;
}

