/* cfed -- a level editor for Crimson Fields
   Copyright (C) 2000, 2001 Jens Granseuer

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* editor.c */

#include <stdlib.h>
#include <string.h>
#include <stdio.h>

#include "SDL.h"

#include "editor.h"
#include "ed_list.h"

extern struct UnitType const UModels[];
extern struct TerrainType const TModels[];

int parse( FILE *in );
unsigned short *read_map( FILE *in, unsigned short w, unsigned short h );
void crypt( char *str );
void rem_whitespace( char *str );
void load_messages( FILE *in, struct List *l, struct GameInfo *game, unsigned long *line );
int check_game( struct GameInfo *game );
int check_units( struct List *units, struct GameInfo *game );
int check_buildings( struct List *builds, struct List *units, unsigned short *map, struct GameInfo *game );
int check_events( struct List *events, struct List *units, struct List *builds, struct GameInfo *game );
int check_players( struct Player *p, struct List *units, struct List *blds, struct GameInfo *game );

/* start of program functions */
int main( int argc, char **argv ) {
  FILE *f;
  short show_help = 0;
  int rc, i;

  for ( i = argc - 1; i; i-- ) {
    if (strcmp(argv[i], "--help") == 0) show_help = 1;
    else if (strcmp(argv[i], "--version") == 0) {
      fprintf( stdout, "cfed "VERSION"\n" );
      return 0;
    }
  }
  if ( argc != 2 ) show_help = 1;
  if ( show_help ) {
    fprintf( stdout, "Usage: %s [option] file\n"                  \
                     "  --help     display this help and exit\n"  \
                     "  --version  output version information and exit\n",
             argv[0] );
    return 0;
  }

  f = fopen( argv[1], "r" );
  if ( !f ) {
    fprintf( stderr, "%s: Unable to open file %s!\n", argv[0], argv[1] );
    return 1;
  }

  rc = parse( f );
  fclose( f );
  return rc;
}

/* parse the file and create a binary mission file from it */
int parse( FILE *in ) {
  struct GameInfo game = { FID_MISSION, FILE_VERSION, -1, 0, "\0\0\0\0\0\0\0", 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, TURN_START };
  struct List units, builds, events, texts;
  struct Player players[2] = { { PLAYER_ONE, HUMAN, LEVEL_NORMAL, 0, 0, -1, "Player 1", "\0\0\0\0\0\0\0" },
                               { PLAYER_TWO, HUMAN, LEVEL_NORMAL, 0, 0, -1, "Player 2", "\0\0\0\0\0\0\0" } }, *cplayer = 0;
  struct Unit *cunit = 0;
  struct Building *cbuild = 0;
  struct Event *cevent = 0;
  struct Node *cnode;
  char buf[256], title[64], tiles[64] = "default";	/* maximum allowed line length is 256 */
  char *p;
  unsigned short mode = MODE_NONE, status = HAVE_NONE, *map = NULL;
  unsigned long line = 0;
  int err = 0, i, current_player = -1;

  NewList( &units );
  NewList( &builds );
  NewList( &events );
  NewList( &texts );

  while ( fgets( buf, 255, in ) ) {
    line++;
    rem_whitespace( buf );
    if ( (strlen(buf) > 0) && (buf[0] != '#') ) {				/* comment */
      if ( buf[0] == '[' ) {							/* starts a new entity */
        p = strchr( buf, ']' );
        if ( p == NULL ) {
          fprintf( stderr, "Syntax error in line %ld: ']' missing\n", line );
          err = 1;
          break;
        }
        *(p + 1) = '\0';							/* remove trailing chars */

        if ( strcasecmp( buf, "[map]" ) == 0 ) {				/* map definitions */
          status |= HAVE_MAP;
          map = read_map( in, game.map_width, game.map_height );
          if ( map == NULL ) {
            err = 1;
            break;
          }
          line += game.map_height;
        } else if ( strcasecmp( buf, "[mission]" ) == 0 ) {			/* global mission definitions */
          mode = MODE_GAME;
          status |= HAVE_GAME;
        } else if ( strcasecmp( buf, "[event]" ) == 0 ) {			/* events definitions */
          mode = MODE_EVENT;
          status |= HAVE_EVENT;
          cnode = malloc( sizeof(struct Event) + sizeof(struct Node) );
          if ( !cnode ) {
            fprintf( stderr, "Error in line %ld: Out of memory\n", line );
            err = 1;
            break;
          }
          cevent = (struct Event *)(((char *)cnode)+sizeof(struct Node));
          cevent->e_type = 99;
          cevent->e_player = 0;
          cevent->e_trigger = 0;
          cevent->e_tdata[0] = -9999;
          cevent->e_tdata[1] = -9999;
          cevent->e_tdata[2] = -9999;
          cevent->e_data[0] = -9999;
          cevent->e_data[1] = -9999;
          cevent->e_data[2] = -9999;
          cevent->e_title = -1;
          cevent->e_message = -1;
          cnode->ln_Data = cevent;
          AddTail( &events, (struct Node *)cnode );
        } else if ( strcasecmp( buf, "[unit]" ) == 0 ) {			/* unit definitions */
          mode = MODE_UNIT;
          status |= HAVE_UNIT;
          cnode = malloc( sizeof(struct Unit) + sizeof(struct Node) );
          if ( !cnode ) {
            fprintf( stderr, "Error in line %ld: Out of memory\n", line );
            err = 1;
            break;
          }
          cunit = (struct Unit *)(((char *)cnode)+sizeof(struct Node));
          cunit->u_pos.x = 0;
          cunit->u_pos.y = 0;
          cunit->u_id = 0;
          cunit->u_flags = 0;
          cunit->u_pid = PLAYER_NONE;
          cunit->u_utype = 0;
          cunit->u_crystals = 0;
          cnode->ln_Data = cunit;
          AddTail( &units, (struct Node *)cnode );
        } else if ( strcasecmp( buf, "[building]" ) == 0 ) {			/* building definitions */
          mode = MODE_BUILDING;
          status |= HAVE_BUILDING;
          cnode = malloc( sizeof(struct Building) + sizeof(struct Node) );
          if ( !cnode ) {
            fprintf( stderr, "Error in line %ld: Out of memory\n", line );
            err = 1;
            break;
          }
          cbuild = (struct Building *)(((char *)cnode)+sizeof(struct Node));
          cbuild->b_pos.x = 0;
          cbuild->b_pos.y = 0;
          cbuild->b_id = 0;
          cbuild->b_player = PLAYER_NONE;
          cbuild->b_crystalprod = 0;
          cbuild->b_crystals = 0;
          cbuild->b_maxcrystals = 1000;
          cbuild->b_unitprod = 0;
          cbuild->b_flags = 0;
          cbuild->b_name[0] = '\0';
          cnode->ln_Data = cbuild;
          AddTail( &builds, (struct Node *)cnode );
        } else if ( strcasecmp( buf, "[messages]" ) == 0 ) {			/* messages */
          load_messages( in, &texts, &game, &line );
        } else if ( strcasecmp( buf, "[player]" ) == 0 ) {			/* player defaults */
          mode = MODE_PLAYER;
          status |= HAVE_PLAYER;
          if ( ++current_player > 1 ) {
            fprintf( stderr, "Error: More than two [Player] sections detected\n" );
            err = 1;
            break;
          }
          cplayer = &players[current_player];
        } else {
          fprintf( stderr, "Error: Unknown entity '%s' in line %ld\n", buf, line );
          err = 1;
          break;
        }
      } else {									/* data line for current mode */
        p = strchr( buf, '=' );
        if ( p == NULL ) {
          fprintf( stderr, "Syntax error in line %ld: '=' missing\n", line );
          err = 1;
          break;
        }
        *p = '\0';								/* separate key from data */
        p++;
        rem_whitespace( p );

        switch( mode ) {
          case MODE_GAME:
            if ( strncasecmp( buf, "mapwidth", 8 ) == 0 ) game.map_width = atoi( p );		/* map width */
            else if ( strncasecmp( buf, "mapheight", 9 ) == 0 ) game.map_height = atoi( p );	/* map height */
            else if ( strncasecmp( buf, "title", 5 ) == 0 ) strcpy( title, p );			/* mission name */
            else if ( strncasecmp( buf, "info", 4 ) == 0 ) game.level_info_str = atoi( p );	/* level info */
            else if ( strncasecmp( buf, "password", 8 ) == 0 ) {				/* password */
              if ( strlen( p ) > 7 ) {
                err = 1;
                fprintf( stderr, "Error in line %ld: Password must not be longer than 7 characters\n", line );
              } else {
                strcpy( game.password, p );
                crypt( game.password );
                game.flags |= GI_PASSWORD;
              }
            } else if ( strncasecmp( buf, "set", 3 ) == 0 ) {					/* level set to use */
              strcpy( tiles, p );
            } else {
              err = 1;
              fprintf( stderr, "Error in line %ld: Illegal keyword '%s'\n", line, buf );
            }
            break;
          case MODE_UNIT:
            if ( strncasecmp( buf, "xpos", 4 ) == 0 ) cunit->u_pos.x = atoi( p );		/* horizontal position */
            else if ( strncasecmp( buf, "ypos", 4 ) == 0 ) cunit->u_pos.y = atoi( p );		/* vertical position */
            else if ( strncasecmp( buf, "id", 2 ) == 0 ) cunit->u_id = atoi( p );		/* unit id */
            else if ( strncasecmp( buf, "crystals", 8 ) == 0 ) cunit->u_crystals = atoi( p );	/* crystals */
            else if ( strncasecmp( buf, "type", 4 ) == 0 ) {					/* unit type */
              err = 1;
              for ( i = 0; i < NUM_UNIT_TYPES; i++ ) {
                if ( strcasecmp( p, UModels[i].ut_name ) == 0 ) {
                  cunit->u_utype = i;
                  err = 0;
                  break;
                }
              }
              if ( err ) fprintf( stderr, "Error: Unknown unit type '%s' in line %ld\n", p, line );
            } else if ( strncasecmp( buf, "player", 6 ) == 0 ) {				/* owning player */
              /* this is different from internal use in the game! here 0 means no owner! */
              i = atoi( p );
              if ( i == 1 ) cunit->u_pid = PLAYER_ONE;
              else if ( i == 2 ) cunit->u_pid = PLAYER_TWO;
              else if ( i == 0 ) cunit->u_pid = PLAYER_NONE;
              else {
                err = 1;
                fprintf( stderr, "Error in line %ld: Illegal owner of unit\n", line );
              }
            } else {
              err = 1;
              fprintf( stderr, "Error in line %ld: Illegal keyword '%s' in this context\n", line, buf );
            }
            break;
          case MODE_BUILDING:
            if ( strncasecmp( buf, "xpos", 4 ) == 0 ) cbuild->b_pos.x = atoi( p );	/* horizontal position */
            else if ( strncasecmp( buf, "ypos", 4 ) == 0 ) cbuild->b_pos.y = atoi( p );	/* vertical position */
            else if ( strncasecmp( buf, "id", 2 ) == 0 ) cbuild->b_id = atoi( p );	/* ID */
            else if ( strncasecmp( buf, "name", 4 ) == 0 ) {				/* name */
              if ( strlen( p ) > 19 ) {
                err = 1;
                fprintf( stderr, "Error in line %ld: Building name must not be longer than 19 characters\n", line );
              } else strcpy( cbuild->b_name, p );
            } else if ( strncasecmp( buf, "player", 6 ) == 0 ) {				/* owning player */
              /* this is different from internal use in the game! here 0 means no owner! */
              i = atoi( p );
              if ( i == 0 ) cbuild->b_player = PLAYER_NONE;
              else if ( i == 1 ) cbuild->b_player = PLAYER_ONE;
              else if ( i == 2 ) cbuild->b_player = PLAYER_TWO;
              else {
                err = 1;
                fprintf( stderr, "Error in line %ld: Illegal owner of building\n", line );
              }
            } else if ( strncasecmp( buf, "mining", 6 ) == 0 ) {					/* mining */
              cbuild->b_crystalprod = atoi( p ); 
              cbuild->b_flags |= B_MINE;
            } else if ( strncasecmp( buf, "crystals", 8 ) == 0 ) cbuild->b_crystals = atoi( p );	/* resources */
            else if ( strncasecmp( buf, "capacity", 8 ) == 0 ) cbuild->b_maxcrystals = atoi( p );	/* max. resources */
            else if ( strncasecmp( buf, "type", 4 ) == 0 ) {						/* type */
              if ( strncasecmp( p, "workshop", 8 ) == 0 ) cbuild->b_flags |= B_WORKSHOP;
              else if ( strncasecmp( p, "mine", 4 ) == 0 ) cbuild->b_flags |= B_MINE;
              else if ( strncasecmp( p, "factory", 7 ) == 0 ) cbuild->b_flags |= B_FACTORY;
              else {
                err = 1;
                fprintf( stderr, "Error in line %ld: Unknown type '%s'\n", line, p );
              }
            } else if ( strncasecmp( buf, "factory", 7 ) == 0 ) {				/* can produce which units */
              err = 1;
              cbuild->b_flags |= B_FACTORY;
              for ( i = 0; i < NUM_UNIT_TYPES; i++ ) {
                if ( strcasecmp( p, UModels[i].ut_name ) == 0 ) {
                  cbuild->b_unitprod |= (1<<i);
                  err = 0;
                  break;
                }
              }
              if ( err ) fprintf( stderr, "Error: Unknown unit type '%s' in line %ld\n", p, line );
            } else {
              err = 1;
              fprintf( stderr, "Error in line %ld: Illegal keyword '%s' in this context\n", line, buf );
            }
            break;
          case MODE_EVENT:
            if ( strncasecmp( buf, "type", 4 ) == 0 ) {						/* type */
              if ( strncasecmp( p, "message", 7 ) == 0 ) cevent->e_type = EVENT_MESSAGE;
              else if ( strncasecmp( p, "production(unit)", 16 ) == 0 ) {
                cevent->e_type = EVENT_PRODUCTION;
                cevent->e_data[2] = 1;
              } else if ( strncasecmp( p, "production(crystals)", 20 ) == 0 ) {
                cevent->e_type = EVENT_PRODUCTION;
                cevent->e_data[2] = 0;
              } else if ( strncasecmp( p, "score", 5 ) == 0 ) cevent->e_type = EVENT_SCORE;
              else if ( strncasecmp( p, "createunit", 10 ) == 0 ) cevent->e_type = EVENT_CREATE_UNIT;
              else {
                err = 1;
                fprintf( stderr, "Error in line %ld: Unknown type '%s'\n", line, p );
              }
            } else if ( strncasecmp( buf, "trigger", 7 ) == 0 )	{				/* trigger */
              if ( strncasecmp( p, "turn", 4 ) == 0 ) cevent->e_trigger = ETRIGGER_TURN;
              else if ( strncasecmp( p, "unitdestroyed", 13 ) == 0 ) cevent->e_trigger = ETRIGGER_UNIT_DESTROYED;
              else if ( strncasecmp( p, "havebuilding", 12 ) == 0 ) {
                cevent->e_trigger = ETRIGGER_HAVE_BUILDING;
                cevent->e_tdata[2] = -1;
              } else if ( strncasecmp( p, "haveunit", 8 ) == 0 ) {
                cevent->e_trigger = ETRIGGER_HAVE_UNIT;
                cevent->e_tdata[2] = -1;
              } else {
                err = 1;
                fprintf( stderr, "Error in line %ld: Unknown trigger type '%s'\n", line, p );
              }
            } else if ( strncasecmp( buf, "player", 6 ) == 0 ) {				/* player(s) */
              /* this is different from internal use in the game! here 0 means noone! */
              i = atoi( p );
              if ( i == 1 ) cevent->e_player = PLAYER_ONE;
              else if ( i == 2 ) cevent->e_player = PLAYER_TWO;
              else {
                err = 1;
                fprintf( stderr, "Error in line %ld: Illegal player\n", line );
              }
            } else if ( strncasecmp( buf, "title", 5 ) == 0 ) cevent->e_title = atoi( p );
            else if ( strncasecmp( buf, "message", 7 ) == 0 ) cevent->e_message = atoi( p );
            else {
              if ( (cevent->e_trigger == ETRIGGER_TURN) &&
                   (strncasecmp( buf, "tturn", 5 ) == 0) ) cevent->e_tdata[0] = atoi( p );	/* turn */
              else if ( ((cevent->e_trigger == ETRIGGER_UNIT_DESTROYED) ||
                         (cevent->e_trigger == ETRIGGER_HAVE_UNIT)) &&
                   (strncasecmp( buf, "tunit", 5 ) == 0) ) cevent->e_tdata[0] = atoi( p );	/* unit */
              else if ( (cevent->e_trigger == ETRIGGER_HAVE_BUILDING) &&
                   (strncasecmp( buf, "tbuilding", 9 ) == 0) ) cevent->e_tdata[0] = atoi( p );	/* building id */
              else if ( ((cevent->e_trigger == ETRIGGER_HAVE_BUILDING) ||
                         (cevent->e_trigger == ETRIGGER_HAVE_UNIT)) &&
                   (strncasecmp( buf, "towner", 6 ) == 0) ) cevent->e_tdata[1] = atoi( p )-1;	/* owner of building/unit */
              else if ( ((cevent->e_trigger == ETRIGGER_HAVE_BUILDING) ||
                         (cevent->e_trigger == ETRIGGER_HAVE_UNIT)) &&
                   (strncasecmp( buf, "tturn", 5 ) == 0) ) cevent->e_tdata[2] = atoi( p );	/* turn on which to check */
              else switch ( cevent->e_type ) {
                case EVENT_PRODUCTION:
                  if ( strncasecmp( buf, "building", 8 ) == 0 ) cevent->e_data[0] = atoi( p );	/* id of building */
                  else if ( (cevent->e_data[2] == 0) &&
                            (strncasecmp( buf, "crystals", 8 ) == 0) ) cevent->e_data[1] = atoi( p );	/* crystal production */
                  else if ( (cevent->e_data[2] == 1) &&
                            (strncasecmp( buf, "unit", 4 ) == 0) ) {				/* unit type */
                    err = 1;
                    for ( i = 0; i < NUM_UNIT_TYPES; i++ ) {
                      if ( strcasecmp( p, UModels[i].ut_name ) == 0 ) {
                        cevent->e_data[1] = i;
                        err = 0;
                        break;
                      }
                    }
                    if ( err ) fprintf( stderr, "Error: Unknown unit type '%s' in line %ld\n", p, line );
                  } else if ( (cevent->e_data[2] == 0) &&
                             (strncasecmp( buf, "crystals", 8 ) == 0) ) cevent->e_data[1] = atoi( p );	/* amount of production */
                  else {
                    err = 1;
                    fprintf( stderr, "Error in line %ld: Illegal keyword '%s'\n", line, buf );
                  }
                  break;
                case EVENT_SCORE:
                  if ( strncasecmp( buf, "score", 5 ) == 0 ) cevent->e_data[0] = atoi( p );	/* score gained */
                  else if ( strncasecmp( buf, "success", 7 ) == 0 ) cevent->e_data[1] = atoi( p );	/* success rate increase */
                  else {
                    err = 1;
                    fprintf( stderr, "Error in line %ld: Illegal keyword '%s'\n", line, buf );
                  }
                  break;
                case EVENT_CREATE_UNIT:
                  if ( strncasecmp( buf, "building", 8 ) == 0 ) cevent->e_data[1] = atoi( p );		/* building in which to create */
                  else if ( strncasecmp( buf, "unit", 4 ) == 0 ) {					/* unit type */
                    err = 1;
                    for ( i = 0; i < NUM_UNIT_TYPES; i++ ) {
                      if ( strcasecmp( p, UModels[i].ut_name ) == 0 ) {
                        cevent->e_data[0] = i;
                        err = 0;
                        break;
                      }
                    }
                    if ( err ) fprintf( stderr, "Error: Unknown unit type '%s' in line %ld\n", p, line );
                 } else {
                   err = 1;
                   fprintf( stderr, "Error in line %ld: Illegal keyword '%s'\n", line, buf );
                 }
                 break;
              }
            }
            break;
          case MODE_PLAYER:
            if ( strncasecmp( buf, "name", 4 ) == 0 ) {						/* name */
              if ( strlen( p ) > 15 ) {
                err = 1;
                fprintf( stderr, "Error in line %ld: Player name must not be longer than 15 characters\n", line );
              } else strcpy( cplayer->p_name, p );
            } else if ( strncasecmp( buf, "briefing", 8 ) == 0 ) cplayer->p_briefing = atoi( p );
            else {
              err = 1;
              fprintf( stderr, "Error in line %ld: Illegal keyword '%s' in this context\n", line, buf );
            }
            break;
          default:
            break;
        }
        if ( err ) break;
      }
    }
  }

  if ( !err ) {
    if ( (status & HAVE_MAP) == 0 ) {
      fprintf( stderr, "Error: No map\n" );
      err = 1;
    } else if ( (status & HAVE_GAME) == 0 ) {
      fprintf( stderr, "Error: Basic definitions missing\n" );
      err = 1;
    } else if ( (status & HAVE_UNIT) == 0 ) {
      fprintf( stderr, "Error: No units defined\n" );
      err = 1;
    }
  }
  if ( !err ) err = check_game( &game );
  if ( !err ) err = check_buildings( &builds, &units, map, &game );
  if ( !err ) err = check_units( &units, &game );
  if ( !err ) err = check_events( &events, &units, &builds, &game );
  if ( !err ) err = check_players( players, &units, &builds, &game );

  if ( !err ) {							/* write mission file */
    SDL_RWops *out;
    struct Node *n;
    unsigned short len, i;

    sprintf( buf, "%s.lev", title );

    out = SDL_RWFromFile( buf, "wb" );
    if ( out ) {
      /* write game info header */
      SDL_WriteLE32( out, game.file_id );
      SDL_RWwrite( out, &game.version, sizeof(unsigned char), 1 );
      SDL_RWwrite( out, &game.level_info_str, sizeof(char), 1 );
      SDL_WriteLE16( out, game.flags );
      SDL_WriteLE16( out, game.turn );
      if ( game.flags & GI_PASSWORD )
        SDL_RWwrite( out, game.password, sizeof(char), 8 );
      SDL_RWwrite( out, &game.current_player, sizeof(unsigned char), 1 );
      SDL_RWwrite( out, &game.phase, sizeof(unsigned char), 1 );

      /* write mission title */
      len = strlen( title );
      SDL_WriteLE16( out, len );
      SDL_RWwrite( out, title, sizeof(char), len );
      len = strlen( tiles );
      SDL_WriteLE16( out, len );
      SDL_RWwrite( out, tiles, sizeof(char), len );

      SDL_WriteLE16( out, game.map_width );
      SDL_WriteLE16( out, game.map_height );
      len = game.map_width * game.map_height;
      for ( i = 0; i < len; i++ )
        SDL_WriteLE16( out, map[i] );

      for ( i = 0; i < 2; i++ ) {	/* save player information */
        SDL_RWwrite( out, &players[i].p_id, sizeof(unsigned char), 5 );
        SDL_WriteLE16( out, players[i].p_score );
        SDL_RWwrite( out, &players[i].p_name, sizeof(char), 16 );
        SDL_RWwrite( out, &players[i].p_password, sizeof(char), 8 );
      }

      SDL_WriteLE16( out, game.num_buildings );
      n = builds.lh_Head;
      while ( n->ln_Next ) {
        struct Building *b = n->ln_Data;
        SDL_WriteLE16( out, b->b_id );
        SDL_WriteLE16( out, b->b_pos.x );
        SDL_WriteLE16( out, b->b_pos.y );
        SDL_WriteLE16( out, b->b_flags );
        SDL_WriteLE16( out, b->b_crystals );
        SDL_WriteLE16( out, b->b_maxcrystals );
        SDL_WriteLE32( out, b->b_unitprod );
        SDL_RWwrite( out, &b->b_crystalprod, sizeof(char), 22 );
        n = n->ln_Next;
      }

      /* save transports here; basically, transports are no different */
      /* from other units but we MUST make sure that transports having */
      /* other units on board are loaded  before those units; we need to */
      /*  make two passes through the list; we save all transports with */
      /* load in the first, and all transports without additional load in */
      /* the second */
      SDL_WriteLE16( out, game.num_transports );
      n = units.lh_Head;
      while ( n->ln_Next ) {
        struct Unit *u = n->ln_Data;
        if ( (u->u_flags & (U_TRANSPORT|U_SHELTERED)) == U_TRANSPORT ) {
          SDL_WriteLE16( out, u->u_pos.x );
          SDL_WriteLE16( out, u->u_pos.y );
          SDL_WriteLE32( out, u->u_flags );
          SDL_WriteLE16( out, u->u_id );

          SDL_RWwrite( out, &u->u_moves, sizeof(unsigned char), 6 );

          SDL_WriteLE16( out, u->u_target.x );
          SDL_WriteLE16( out, u->u_target.y );
          SDL_WriteLE16( out, u->u_crystals );
        }
        n = n->ln_Next;
      }

      n = units.lh_Head;
      while ( n->ln_Next ) {
        struct Unit *u = n->ln_Data;
        if ( (u->u_flags & (U_TRANSPORT|U_SHELTERED)) == (U_TRANSPORT|U_SHELTERED) ) {
          SDL_WriteLE16( out, u->u_pos.x );
          SDL_WriteLE16( out, u->u_pos.y );
          SDL_WriteLE32( out, u->u_flags );
          SDL_WriteLE16( out, u->u_id );

          SDL_RWwrite( out, &u->u_moves, sizeof(unsigned char), 6 );

          SDL_WriteLE16( out, u->u_target.x );
          SDL_WriteLE16( out, u->u_target.y );
          SDL_WriteLE16( out, u->u_crystals );
        }
        n = n->ln_Next;
      }

      SDL_WriteLE16( out, game.num_units );
      n = units.lh_Head;
      while ( n->ln_Next ) {
        struct Unit *u = n->ln_Data;
        if ( !(u->u_flags & U_TRANSPORT) ) {
          SDL_WriteLE16( out, u->u_pos.x );
          SDL_WriteLE16( out, u->u_pos.y );
          SDL_WriteLE32( out, u->u_flags );
          SDL_WriteLE16( out, u->u_id );

          SDL_RWwrite( out, &u->u_moves, sizeof(unsigned char), 6 );

          SDL_WriteLE16( out, u->u_target.x );
          SDL_WriteLE16( out, u->u_target.y );
        }
        n = n->ln_Next;
      }

      SDL_WriteLE16( out, game.num_combatrecords );  /* always 0 for new missions */

      SDL_WriteLE16( out, game.num_events );
      n = events.lh_Head;
      while ( n->ln_Next ) {
        struct Event *e = n->ln_Data;
        SDL_RWwrite( out, &e->e_type, sizeof(unsigned char), 2 );
        SDL_WriteLE16( out, e->e_tdata[0] );
        SDL_WriteLE16( out, e->e_tdata[1] );
        SDL_WriteLE16( out, e->e_tdata[2] );
        SDL_WriteLE16( out, e->e_data[0] );
        SDL_WriteLE16( out, e->e_data[1] );
        SDL_WriteLE16( out, e->e_data[2] );
        SDL_WriteLE16( out, e->e_title );
        SDL_WriteLE16( out, e->e_message );
        SDL_RWwrite( out, &e->e_player, sizeof(unsigned char), 1 );
        n = n->ln_Next;
      }

      SDL_WriteLE16( out, game.num_texts );
      SDL_WriteLE32( out, game.text_length );
      n = texts.lh_Head;
      while ( n->ln_Next ) {
        SDL_RWwrite( out, (char *)n->ln_Data, sizeof(char), strlen( (char *)n->ln_Data ) + 1 );
        n = n->ln_Next;
      }

      /* no turn history yet */
      SDL_WriteLE16( out, 0 );

      SDL_RWclose( out );
    } else fprintf( stderr, "Error opening file '%s' for writing\n", buf );
 }

  if ( map ) free( map );
  while ( !IsListEmpty( &units ) ) free( RemHead( &units ) );
  while ( !IsListEmpty( &builds ) ) free( RemHead( &builds ) );
  while ( !IsListEmpty( &texts ) ) free( RemHead( &texts ) );

  return err;
}

/* read the map */
unsigned short *read_map( FILE *in, unsigned short w, unsigned short h ) {
  int i, j, size, err = 0;
  unsigned short *map, terrain = 0;
  char buf[128];

  if ( (w <= 0) || (w > MAX_MAP_WIDTH) || (h <= 0) || (h > MAX_MAP_HEIGHT) ) {
    fprintf( stderr, "Error: Illegal map size\n" );
    return NULL;
  }

  size = w * h;
  map = malloc( size * sizeof( unsigned short ) );

  if ( map == NULL ) {
    fprintf( stderr, "Error: Out of memory\n" );
    return NULL;
  }

  for ( i = 0; i < h; i++ ) {
    if ( !fgets( buf, 127, in ) ) {
      fprintf( stderr, "Error reading map data\n" );
      err = 1;
      break;
    }
    buf[w] = '\0';
    if ( strlen( buf ) != w ) {
      fprintf( stderr, "Error: map row %d too short\n", i );
      err = 1;
      break;
    }

    for ( j = 0; j < w; j++ ) {
      switch ( buf[j] ) {
        case '.': terrain = 0; break;					/* plains */
        case '*': terrain = 1; break;					/* forest */
        case '%': terrain = 2; break;					/* mountains */
        case '=': terrain = 3; break;					/* shallow water */
        case '~': terrain = 4; break;					/* deep water */
        case '1': terrain = 5; break;					/* yellow hq entrance, east */
        case '2': terrain = 6; break;					/* blue hq entrance, east */
        case '0': terrain = 7; break;					/* neutral hq entrance, east */
        case '<': terrain = 8; break;					/* hq, left part */
        case '^': terrain = 9; break;					/* hq, upper part */
        case 'v': terrain = 10; break;					/* hq, lower part */
        case '#': terrain = 11; break;					/* swamp */
        case '\\': terrain = 12; break;					/* road, se-nw */
        case '|': terrain = 13; break;					/* road, s-n */
        case '/': terrain = 14; break;					/* road, sw-ne */
        case 'y': terrain = 15; break;					/* road, sw-n-ne */
        case 'Y': terrain = 16; break;					/* road, se-n-nw */
        case 'X': terrain = 17; break;					/* road, s-se-nw-n */
        case 'k': terrain = 18; break;					/* road, sw-s-ne */
        case 'K': terrain = 19; break;					/* road, s-se-nw */
        case '(': terrain = 20; break;					/* road, n-se */
        case ')': terrain = 21; break;					/* road, n-sw */
        case ']': terrain = 22; break;					/* road, nw-s */
        case '[': terrain = 23; break;					/* road, ne-s */
        case 'n': terrain = 24; break;					/* road, sw-se */
        case 'u': terrain = 25; break;					/* road, nw-ne */
        case '!': terrain = 26; break;					/* bridge, n-s */
        case '`': terrain = 27; break;					/* bridge, sw-ne */
        case '\'': terrain = 28; break;					/* bridge, se-nw */
        case '4': terrain = 29; break;					/* yellow depot entrance */
        case '5': terrain = 30; break;					/* blue depot entrance */
        case '3': terrain = 31; break;					/* neutral depot entrance */
        case '>': terrain = 32; break;					/* hq, right part */
        case 'x': terrain = 33; break;					/* road, s-sw-ne-n */
        case 'o': terrain = 34; break;					/* road, sw-nw-ne-se */
        case '7': terrain = 35; break;					/* yellow factory entrance */
        case '8': terrain = 36; break;					/* blue factory entrance */
        case '6': terrain = 37; break;					/* neutral factory entrance */
        case 'a': terrain = 38; break;					/* wire fence, se-nw end */
        case 'b': terrain = 39; break;					/* wire fence, nw-se end */
        case 'c': terrain = 40; break;					/* wire fence, ne-sw end */
        case 'd': terrain = 41; break;					/* wire fence, sw-ne end */
        case 'e': terrain = 42; break;					/* wire fence, n-s */
        case 'f': terrain = 43; break;					/* wire fence, sw-ne */
        case 'g': terrain = 44; break;					/* wire fence, nw-se */
        case 'h': terrain = 45; break;					/* wire fence, nw-s */
        case 'i': terrain = 46; break;					/* wire fence, ne-s */
        case 'j': terrain = 47; break;					/* wire fence, sw-n */
        case 'l': terrain = 48; break;					/* wire fence, se-n */
        case 'm': terrain = 49; break;					/* wire fence, nw-ne */
        case 'p': terrain = 50; break;					/* wire fence, sw-se */
        case '"': terrain = 51; break;					/* cliff */
        case 'A': terrain = 52; break;					/* yellow city */
        case 'B': terrain = 53; break;					/* blue city */
        case 'C': terrain = 54; break;					/* neutral city */
        case 'D': terrain = 55; break;					/* yellow hq entrance, west */
        case 'E': terrain = 56; break;					/* blue hq entrance, west */
        case 'F': terrain = 57; break;					/* neutral hq entrance, west */
        default:
          fprintf( stderr, "Error: Illegal character '%c' in map\n", buf[j] );
          err = 1;
      }

      if ( err ) break;
      else map[ i * w + j ] = terrain;
    }
  }

  if ( err == 0 ) return map;
  free( map );
  return NULL;
}

/* remove leading spaces and newlines from the string */
void rem_whitespace( char *str ) {
  int len = strlen( str ), i = 0, j;
  char *strbuf = malloc( len + 1 );
  if ( strbuf == NULL ) return;

  strcpy( strbuf, str );

  while ( strbuf[i] == ' ' ) i++;

  for ( j = 0; i < len; i++ ) {
    if ( strbuf[i] != '\n' ) str[j++] = strbuf[i];
  }
  str[j] = '\0';
}

/* check game data for validity */
int check_game( struct GameInfo *g ) {

  if ( (g->map_width == 0) || (g->map_height == 0) ) {
    fprintf( stderr, "Error: map width or height is 0\n" );
    return 1;
  }

  if ( g->map_width > MAX_MAP_WIDTH ) {
    fprintf( stderr, "Error: map too wide\n" );
    return 1;
  }

  if ( g->map_height > MAX_MAP_HEIGHT ) {
    fprintf( stderr, "Error: map too high\n" );
    return 1;
  }

  if ( g->level_info_str >= g->num_texts ) {
    fprintf( stderr, "Error: Illegal level info message %d\n", g->level_info_str );
    return 1;
  }

  return 0;
}

/* check players */
int check_players( struct Player *p, struct List *units, struct List *blds, struct GameInfo *g ) {
  int i;

  for ( i = 0; i < 2; i++ ) {
    if ( p[i].p_briefing >= g->num_texts ) {
      fprintf( stderr, "Error: Cannot set briefing %d for player %d; only %d messages available\n", p[i].p_briefing, i+1, g->num_texts );
      return 1;
    }
  }

  return 0;
}

/* check units */
int check_units( struct List *l, struct GameInfo *g ) {
  struct Node *n = l->lh_Head, *walk;
  struct Unit *u, *uwalk;
  unsigned short num_units = 0, num_transports = 0;

  /* if there is a transport at a unit's location put it in if possible */
  /* this only checks for transports appearing before the actual unit in the file */
  for ( n = l->lh_Head; n->ln_Next; n = n->ln_Next ) {
    u = n->ln_Data;

    u->u_moves = UModels[u->u_utype].ut_moves;
    u->u_flags |= UModels[u->u_utype].ut_flags;
    u->u_facing = 0;
    u->u_group = 6;
    u->u_xp = 0;
    u->u_target.x = u->u_target.y = -1;

    if ( (u->u_flags & (U_TRANSPORT|U_SHELTERED)) == U_TRANSPORT ) {
      short fullslots = 0;
      for ( walk = n->ln_Next; walk->ln_Next; walk = walk->ln_Next ) {
        uwalk = walk->ln_Data;
        if ( (uwalk->u_pos.x == u->u_pos.x) && (uwalk->u_pos.y == u->u_pos.y) ) {
          if ( fullslots == UModels[u->u_utype].ut_trans_slots ) {
            fprintf( stderr, "Error: Transport at %d/%d carries too many units\n",
                             u->u_pos.x, u->u_pos.y );
            return 1;
          } else if ( UModels[uwalk->u_utype].ut_weight > UModels[u->u_utype].ut_trans_weight ) {
            fprintf( stderr, "Error: Unit %s too heavy for transport at %d/%d\n",
                             UModels[uwalk->u_utype].ut_name, u->u_pos.x, u->u_pos.y );
            return 1;
          } else {
            uwalk->u_flags |= U_SHELTERED;
            fullslots++;
          }
        }
      }
    }
  }

  for ( n = l->lh_Head; n->ln_Next; n = n->ln_Next ) {
    u = n->ln_Data;

    if ( u->u_flags & U_TRANSPORT ) num_transports++;
    else num_units++;

    if ( (u->u_pos.x >= g->map_width) || (u->u_pos.y >= g->map_height) ) {
      fprintf( stderr, "Error: unit (ID: %d) out of map\n", u->u_id );
      return 1;
    }

    if ( u->u_crystals ) {
      if ( !(u->u_flags & U_TRANSPORT) )
        fprintf( stderr, "Error: non-transport unit (ID: %d) cannot carry crystals\n", u->u_id );
      else {
        unsigned short maxcrystals = UModels[u->u_utype].ut_trans_slots *
                                     UModels[u->u_utype].ut_trans_weight;
        if ( maxcrystals < u->u_crystals )
          fprintf( stderr, "Error: transport (ID: %d) can only carry a maximum of %d crystals\n",
                   u->u_id, maxcrystals );
      }
    }

    walk = n->ln_Next;
    while ( walk->ln_Next ) {
      uwalk = walk->ln_Data;
      if ( u->u_id == uwalk->u_id ) {
        fprintf( stderr, "Error: two or more units sharing one ID (%d)\n", u->u_id );
        return 1;
      }

      if ( (u->u_pos.x == uwalk->u_pos.x) && (u->u_pos.y == uwalk->u_pos.y) &&
           ( !(u->u_flags & (U_SHELTERED|U_TRANSPORT)) || !(uwalk->u_flags & U_SHELTERED) ) ) {
        fprintf( stderr, "Error: two or more units sharing one hex (%d/%d)\n", u->u_pos.x, u->u_pos.y );
        return 1;
      }
      walk = walk->ln_Next;
    }

    if ( u->u_pid == PLAYER_ONE ) u->u_facing = 0;
    else if ( u->u_pid == PLAYER_TWO ) u->u_facing = 3;
    else if ( (u->u_pid != PLAYER_NONE) || !(u->u_flags & U_SHELTERED) ) {
      fprintf( stderr, "Error: unit with ID %d has no legal owner\n", u->u_id );
      return 1;
    }
  }
  g->num_units = num_units;
  g->num_transports = num_transports;
  return 0;
}

/* check buildings */
int check_buildings( struct List *l, struct List *units, unsigned short *map, struct GameInfo *g ) {
  struct Node *n = l->lh_Head, *walk;
  struct Building *b, *bwalk;
  struct Unit *u;
  unsigned short num = 0;

  while ( n->ln_Next ) {
    num++;
    b = n->ln_Data;

    if ( (b->b_pos.x >= g->map_width) || (b->b_pos.y >= g->map_height) ) {
      fprintf( stderr, "Error: Building (%d/%d) out of map\n", b->b_pos.x, b->b_pos.y );
      return 1;
    }

    if ( !(TModels[ map[b->b_pos.y * g->map_width + b->b_pos.x] ].tt_type & TT_ENTRANCE) ) {
      fprintf( stderr, "Error: Map does not have a building at %d/%d\n", b->b_pos.x, b->b_pos.y );
      return 1;
    }

    if ( b->b_crystals > b->b_maxcrystals ) {
      fprintf( stderr, "Error: Building at %d/%d contains %d crystals, but only %d fit in\n",
                               b->b_pos.x, b->b_pos.y, b->b_crystals, b->b_maxcrystals );
      return 1;
    }

    walk = n->ln_Next;
    while ( walk->ln_Next ) {
      bwalk = walk->ln_Data;

      if ( b->b_id == bwalk->b_id ) {
        fprintf( stderr, "Error: two buildings sharing ID %d\n", b->b_id );
        return 1;
      }

      if ( (b->b_pos.x == bwalk->b_pos.x) && (b->b_pos.y == bwalk->b_pos.y) ) {
        fprintf( stderr, "Error: two buildings sharing one hex (%d/%d)\n", b->b_pos.x, b->b_pos.y );
        return 1;
      }
      walk = walk->ln_Next;
    }

    walk = units->lh_Head;
    while ( walk->ln_Next ) {
      u = walk->ln_Data;

      if ( (b->b_pos.x == u->u_pos.x) && (b->b_pos.y == u->u_pos.y) ) {
        if ( (u->u_pid != b->b_player) && (b->b_player != PLAYER_NONE) ) {
          fprintf( stderr, "Error: Hostile unit (%d) in building (%d/%d)\n", u->u_id, b->b_pos.x, b->b_pos.y );
          return 1;
        } else u->u_pid = b->b_player;

        u->u_flags |= U_SHELTERED;
      }

      walk = walk->ln_Next;
    }

    if ( b->b_name[0] == '\0' ) {
      if ( b->b_flags & B_MINE ) strcpy( b->b_name, "Mining Station" );
      else if ( b->b_flags & B_FACTORY ) strcpy( b->b_name, "Factory" );
      else strcpy( b->b_name, "Depot" );
    }

    n = n->ln_Next;
  }
  g->num_buildings = num;
  return 0;
}

/* check events for consistency */
int check_events( struct List *events, struct List *units, struct List *builds, struct GameInfo *game ) {
  struct Node *n = events->lh_Head, *walk;
  struct Event *e;
  struct Building *b;
  struct Unit *u;
  unsigned short num = 0, found = 0;

  while ( n->ln_Next ) {
    num++;
    e = n->ln_Data;

    if ( (e->e_player != PLAYER_ONE) && (e->e_player != PLAYER_TWO) ) {
      fprintf( stderr, "Error: Event has no players assigned\n" );
      return 1;
    }

    if ( e->e_message >= game->num_texts ) {
      fprintf( stderr, "Error: Event has illegal message '%d'\n", e->e_message );
      return 1;
    }
    if ( e->e_title >= game->num_texts ) {
      fprintf( stderr, "Error: Event has illegal title '%d'\n", e->e_title );
      return 1;
    }

    switch ( e->e_type ) {
      case EVENT_MESSAGE:
        if ( e->e_message < 0 ) {
          fprintf( stderr, "Error: Message event has no message\n" );
          return 1;
        }
        e->e_data[0] = 0;
        e->e_data[1] = 0;
        e->e_data[2] = 0;
        break;
      case EVENT_CREATE_UNIT:
        if ( e->e_data[0] < 0 ) {
          fprintf( stderr, "Error: No unit type specified for Create Unit event\n" );
          return 1;
        }

        walk = builds->lh_Head;
        while ( walk->ln_Next ) {
          b = walk->ln_Data;
          if ( b->b_id == e->e_data[1] ) {
            found = 1;
            break;
          }
          walk = walk->ln_Next;
        }
        if ( found ) found = 0;
        else {
          fprintf( stderr, "Error: Building with ID %d does not exist\n", e->e_data[1] );
          return 1;
        }
        e->e_data[2] = -1;
        break;
      case EVENT_PRODUCTION:
        if ( e->e_data[0] < 0 ) {
          fprintf( stderr, "Error: Unit production event with no building\n" );
          return 1;
        }
        if ( e->e_data[1] < 0 ) {
          if ( e->e_data[2] == 0 ) fprintf( stderr, "Error: Negative crystal production in event\n" );
          else fprintf( stderr, "Error: Unit production event with no unit\n" );
          return 1;
        }

        walk = builds->lh_Head;
        while ( walk->ln_Next ) {
          b = walk->ln_Data;
          if ( b->b_id == e->e_data[0] ) {
            found = 1;
            break;
          }
          walk = walk->ln_Next;
        }
        if ( found ) found = 0;
        else {
          fprintf( stderr, "Error: Building with ID %d does not exist\n", e->e_data[0] );
          return 1;
        }
        break;
      case EVENT_SCORE:
        if ( e->e_data[0] < 0 ) {
          fprintf( stderr, "Corrected score for SCORE event < 0\n" );
          e->e_data[0] = 0;
        }
        if ( e->e_data[1] < 0 ) {
          fprintf( stderr, "Corrected success rate for SCORE event < 0\n" );
          e->e_data[1] = 0;
        }
        e->e_data[2] = 0;
        break;
    }

    switch ( e->e_trigger ) {
      case ETRIGGER_TURN:
        if ( e->e_tdata[0] < 0 ) {
          fprintf( stderr, "Error: Event trigger lacks turn\n" );
          return 1;
        }
        e->e_tdata[1] = 0;
        e->e_tdata[2] = 0;
        break;
      case ETRIGGER_UNIT_DESTROYED:
        if ( e->e_tdata[0] != -1 ) {
          walk = units->lh_Head;
          while ( walk->ln_Next ) {
            u = walk->ln_Data;
            if ( u->u_id == e->e_tdata[0] ) {
              /* the event must also be triggered when the unit is not
                 destroyed but captured by the enemy. Therefore we need
                 the original owner */
              e->e_tdata[1] = u->u_pid;
              found = 1;
              break;
            }
            walk = walk->ln_Next;
          }

          if ( found ) found = 0;
          else {
            fprintf( stderr, "Error: Event trigger targets non-existing unit with ID %d\n", e->e_tdata[0] );
            return 1;
          }
        }
        e->e_tdata[2] = 0;
        break;
      case ETRIGGER_HAVE_UNIT:
        if ( (e->e_tdata[1] != PLAYER_ONE) && (e->e_tdata[1] != PLAYER_TWO) ) {
          fprintf( stderr, "Error: Event trigger wants illegal player to own a unit\n" );
          return 1;
        }
        if ( e->e_tdata[2] < 0 ) e->e_tdata[2] = -1;

        walk = units->lh_Head;
        while ( walk->ln_Next ) {
          u = walk->ln_Data;
          if ( u->u_id == e->e_tdata[0] ) {
            if ( (u->u_pid == e->e_tdata[1]) && (e->e_tdata[2] < 0)  ) {
              fprintf( stderr, "Error: Event trigger: unit %d is already owned by player %d\n", u->u_id, u->u_pid+1 );
              return 1;
            }
            found = 1;
            break;
          }
          walk = walk->ln_Next;
        }

        if ( found ) found = 0;
        else {
          fprintf( stderr, "Error: Event trigger targets non-existing unit %d\n", e->e_tdata[0] );
          return 1;
        }
        break;
      case ETRIGGER_HAVE_BUILDING:
        if ( (e->e_tdata[1] != PLAYER_ONE) && (e->e_tdata[1] != PLAYER_TWO) ) {
          fprintf( stderr, "Error: Event trigger wants illegal player to own a building\n" );
          return 1;
        }
        if ( e->e_tdata[2] < 0 ) e->e_tdata[2] = -1;

        walk = builds->lh_Head;
        while ( walk->ln_Next ) {
          b = walk->ln_Data;
          if ( b->b_id == e->e_tdata[0] ) {
            if ( (b->b_player == e->e_tdata[1]) && (e->e_tdata[2] < 0) ) {
              fprintf( stderr, "Error: Event trigger: building %d is already owned by player %d\n", b->b_id, b->b_player+1 );
              return 1;
            }
            found = 1;
            break;
          }
          walk = walk->ln_Next;
        }

        if ( found ) found = 0;
        else {
          fprintf( stderr, "Error: Event trigger targets non-existing building %d\n", e->e_tdata[0] );
          return 1;
        }
        break;
      default:
        fprintf( stderr, "Error: Illegal event type %d\n", e->e_type );
        return 1;
    }

    n = n->ln_Next;
  }
  game->num_events = num;
  return 0;
}


/* get text messages from the file */
void load_messages( FILE *in, struct List *l, struct GameInfo *game, unsigned long *line ) {
  char *buffer, *bufptr;
  struct Node *node;
  long msgleft, len;

  do {
    msgleft = 1999;
    node = malloc( 2000 + sizeof(struct Node) );	/* a single message may be up to 2000-1 characters long */
    buffer = ((char *)node) + sizeof(struct Node);
    node->ln_Data = buffer;
    AddTail( l, node );
    bufptr = buffer;
    game->num_texts++;

    while ( fgets( bufptr, msgleft, in ) ) {
      (*line)++;
      if ( bufptr[0] == '%' ) {
        *(bufptr - 1) = '\0';
        break;
      }
      len = strlen( bufptr );
      msgleft -= len;

      game->text_length += len;
      bufptr += len;
    }

  } while ( (bufptr[0] != '%') || (bufptr[1] != '%') );
}

/* crypt/decrypt ASCII text so you can't "accidently" browse through */
/* the mission files with a hex editor or something */
void crypt( char *str ) {
  while ( *str ) *str++ = ~(*str);
}

